/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.util.ArrayMap;
import android.util.SparseArray;

import androidx.annotation.IntDef;
import androidx.fragment.app.DialogFragment;

import com.foxit.pdfscan.activity.ScannerCameraActivity;
import com.foxit.pdfscan.fragment.ScannerCameraFragment;
import com.foxit.pdfscan.fragment.ScannerListFragment;
import com.foxit.pdfscan.utils.FileUtils;
import com.foxit.pdfscan.utils.IResultCallback;
import com.foxit.pdfscan.utils.SessionIntentUtils;
import com.foxitsoftware.mobile.scanning.DocumentEditingSession;
import com.foxitsoftware.mobile.scanning.DocumentSession;
import com.foxitsoftware.mobile.scanning.DocumentSessionManager;
import com.foxitsoftware.mobile.scanning.DocumentSessionManagerFactory;
import com.luratech.android.appframework.Document;
import com.luratech.android.appframework.DocumentManager;
import com.luratech.android.appframework.DocumentManagerListener;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * Foxit PDF SDK for Android Scanning Add-On
 * </p>
 *
 * 1:To use the scanning functionality, you must always declare the {@link #initializeScanner} and {@link #initializeCompression} methods
 * 2:you can determine whether the initialization is successful through {@link #isInitializeScanner} and {@link #isInitializeCompression} methods
 * 3:Once the initialization is complete, then you can use the {@link #showUI(Activity)} method to show the scan ui,
 */
public class PDFScanManager {

    /**
     * Customize the management class for the Save as actionView view
     */
    public static final int PDF_SAVE_AS_ACTION_VIEW = 1;

    /**
     * Customize the management class of save as contentView view
     */
    public static final int PDF_SAVE_AS_CONTENT_VIEW = 2;

    /**
     * The Flash of light item in the top bar of camera viewController
     */
    public static final int FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FLASH_LIGHT = 1;
    /**
     * The Single shooting button in the top bar of home view
     */
    public static final int FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_SINGLE_SHOOTING = 2;
    /**
     * The photos button in the top bar of home view
     */
    public static final int FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_PHOTOS = 3;
    /**
     * The auto dection button in the top bar of home view
     */
    public static final int FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_AUTO_DECTION = 4;
    /**
     * The format pagesize button in the top bar of home view
     */
    public static final int FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FORMAT_PAGESIZE = 5;

    private ToolBarManager mBarManager = null;
    private DocumentEditingSession mCreateDocSession;
    private static PDFScanManager mPDFScanManager = new PDFScanManager();

    private SparseArray<IContentViewPresenter> mCustomViews = null;
    private SparseArray<IActionViewPresenter> mActionViews = null;
    private Map<Integer, Boolean> mHiddenMap = null;

    public ScanPDFDoneCallBack mScanCallBackListener;
    public ObtainDocumentNameCallback mObtainDocumentNameCallBackListener;

    private PDFScanManager() {
        mPDFScanManager = this;
        mCustomViews = new SparseArray<>();
        mActionViews = new SparseArray<>();
        mHiddenMap = new ArrayMap<>();
        progressListener = null;
    }

    public static PDFScanManager instance() {
        return mPDFScanManager;
    }


    @IntDef({PDF_SAVE_AS_ACTION_VIEW,
            PDF_SAVE_AS_CONTENT_VIEW})
    @Retention(RetentionPolicy.SOURCE)
    @interface ScanManagerType {
    }


    @IntDef({FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FLASH_LIGHT
            , FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_SINGLE_SHOOTING
            , FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_PHOTOS
            , FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_AUTO_DECTION
            , FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FORMAT_PAGESIZE
    })
    @Retention(RetentionPolicy.SOURCE)
    @interface FSScanCameraControllerItemType {
    }

    public ToolBarManager getToolBarManager(Context context) {
        if (mBarManager == null) {
            mBarManager = new ToolBarManager(context);
        }
        return mBarManager;
    }


    /**
     * Set the complete custom content view management class to the save as view
     *
     *
     * @param type Distinguish the key of custom view  {@link  #PDF_SAVE_AS_CONTENT_VIEW }
     * @param view Customize the management class of contentView view
     */
    public void setContentView(@ScanManagerType int type, IContentViewPresenter view) {
        mCustomViews.put(type, view);
    }

    /**
     * Get full custom contentView
     *
     * @param type Distinguish the key of custom view {@link  #PDF_SAVE_AS_CONTENT_VIEW }
     * @return Customize the management class of contentView view
     */
    public IContentViewPresenter getContentView(@ScanManagerType int type) {
        return mCustomViews.get(type);
    }

    /**
     * Set the custom view of appending actionview
     *
     *
     * @param type Distinguish the key of custom view {@link  #PDF_SAVE_AS_ACTION_VIEW }
     * @param view Customize the management class of action view view
     */
    public void setActionView(@ScanManagerType int type, IActionViewPresenter view) {
        mActionViews.put(type, view);
    }

    /**
     * get the custom view of appending actionview
     *
     * @param type Distinguish the key of custom view {@link  #PDF_SAVE_AS_ACTION_VIEW }
     * @return Add ActionView view
     */
    public IActionViewPresenter getActionView(@ScanManagerType int type) {
        return mActionViews.get(type);
    }

    /**
     * Show the scan camera interface
     * @param context A Context of the application package implementing this class.
     */
    public void showUI(Activity context) {
        Intent intent = new Intent(context, ScannerCameraActivity.class);
        if (mCreateDocSession == null) {
            mCreateDocSession = DocumentSessionManagerFactory.getInstance().createSession();
            SessionIntentUtils.saveSessionId(intent, mCreateDocSession);
        }
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivityForResult(intent, ActRequestCode.REQ_SCAN_CAMERA);
    }
    /**
     * Set the item of photos hidden  in the camera viewController of PDFScan
     *
     * @param fsScanCameraControllerItemType Set hidden itemsPlease refer to values starting from
     *            {@link  #FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FLASH_LIGHT fs_itemTypeXXX}.
     * @param hidden                         hidden    hidden  is  YES otherwise is NO
     */
    public void setItemHiddenWithType(@FSScanCameraControllerItemType int fsScanCameraControllerItemType, boolean hidden) {
        mHiddenMap.put(fsScanCameraControllerItemType, hidden);
    }


    public void clearItemHiddenWithType() {
        mHiddenMap.clear();
    }

    public Map<Integer, Boolean> getItemHiddenWithType() {
        return mHiddenMap;
    }

    /**
     * Save as PDF with file path
     *
     * @param path The path saved by the user
     */
    public void setOnScanPDFSaveAsCallBack(String path) {
        mSaveAsPath = path;
        int startIndex = mSaveAsPath.lastIndexOf("/");
        int endIndex = mSaveAsPath.lastIndexOf(".");
        mSaveDocName = mSaveAsPath.substring(startIndex + 1, endIndex);
        mCurOperation = DOC_OPERATION.SAVE_TO_LOCAL;
        saveDocRun.run();
    }

    public void setOnObtainDocumentNameCallBack(ObtainDocumentNameCallback obtainDocumentNameCallBackListener) {
        mObtainDocumentNameCallBackListener = obtainDocumentNameCallBackListener;
    }

    public void setDoneCallBack(ScanPDFDoneCallBack scanCallBackListener) {
        mScanCallBackListener = scanCallBackListener;
    }


    public void release() {
        clearItemHiddenWithType();
        clearContentView();
        clearActionView();
        mHiddenMap = null;
        mBarManager = null;
        mCustomViews = null;
        mActionViews = null;
        mPDFScanManager = null;
    }

    public void clearContentView() {
        if (mCustomViews != null) {
            mCustomViews.clear();
        }

    }

    public void clearActionView() {

        if (mActionViews != null) {
            mActionViews.clear();
        }
    }

    /**
     * Initialize the Scanner module with additional parameters
     * This function must be called before any objects can be instantiated
     * Successful initialization of the module requires a valid serial number
     *
     * @param serial1 First part of the serial number. The trial serial can be passed to 0,
     *                if you need a formal serial , please contact us
     * @param serial2 Second part of the serial number. The trial serial can be passed to 0,
     *      *                if you need a formal serial , please contact us
     */
    public static void initializeScanner(Application application, long serial1, long serial2) {
        PDFScanSingleton.getInstance().initializeScanner(application, serial1, serial2);
    }

    /**
     * Initialize the Mobile Compression module
     * Successful initialization of the module requires a valid serial number
     *
     * @param serial1 First part of the serial number. The trial serial can be passed to 0,
     *      *                if you need a formal serial , please contact us
     * @param serial2 Second part of the serial number The trial serial can be passed to 0,
     *      *                if you need a formal serial , please contact us
     */
    public static void initializeCompression(Application application, long serial1, long serial2) {
        PDFScanSingleton.getInstance().initializeCompression(application, serial1, serial2);
    }

    /**
     * Indicates whether the Scanner module is initialized.
     *
     * @return true if the Scanner module is initialized, false otherwise
     */
    public static boolean isInitializeScanner() {
        return PDFScanSingleton.getInstance().isInitializeScanner();
    }

    /**
     * Indicates whether the Mobile Compression module is initialized.
     *
     * @return true if the Mobile Compression module is initialized, false otherwise
     */
    public static boolean isInitializeCompression() {
        return PDFScanSingleton.getInstance().isInitializeCompression();
    }

    /**
     * Return the scanning list Fragment.
     *
     * @param dismissListener Set a listener to be invoked when the scanning list Fragment is dismissed.{@link android.content.DialogInterface.OnDismissListener}
     * @return Return the scanning list Fragment.
     */
//    public static DialogFragment createScannerFragment(DialogInterface.OnDismissListener dismissListener) {
//        ScannerListFragment scannerListFragment = ScannerListFragment.newInstance();
//        scannerListFragment.setDismissListener(dismissListener);
//        return scannerListFragment;
//    }

    /**
     * Register a callback to be invoked when the scanning list is changed.
     *
     * @param listener a {@link IPDFScanManagerListener} to use
     */
    public static void registerManagerListener(IPDFScanManagerListener listener) {
        PDFScanSingleton.getInstance().registerManagerListener(listener);
    }

    /**
     * Unregister the specified {@link IPDFScanManagerListener}
     *
     * @param listener A {@link IPDFScanManagerListener} object to be unregistered.
     */
    public static void unregisterManagerListener(IPDFScanManagerListener listener) {
        PDFScanSingleton.getInstance().unregisterManagerListener(listener);
    }


    private DOC_OPERATION mCurOperation = DOC_OPERATION.NONE;
    private static final int OPER_NORMAL = 0;
    private static final int OPER_EDIT_DOC = 1;
    private static final int OPER_RENAME_DOC = 2;
    private static final int OPER_LOADING_DOC = 3;
    private static final int OPER_SAVEING_IMAGE = 4;
    private static final int OPER_SAVEING_PDF = 5;
    private static final int SAVE_AS_IMAGE = 0;
    private static final int SAVE_AS_PDF = 1;
    private int mSaveType = SAVE_AS_IMAGE;
    private int mLastOper = OPER_NORMAL;
    private String mCurOpStep = "0";
    private String mSaveAsPath;
    private String mSaveDocName;

    enum DOC_OPERATION {
        NONE,
        SAVE_TO_LOCAL,
    }

    DocumentSessionManager.ProgressListener progressListener = null;
    Runnable saveDocRun = new Runnable() {
        @Override
        public void run() {
            progressListener = new DocumentSessionManager.ProgressListener() {
                @Override
                public void onStartedAction(DocumentSession documentSession, String s) {

                }

                @Override
                public void onUpdateProgress(DocumentSession documentSession, int i, int i1) {

                }

                @Override
                public void onFinishedAction(DocumentSession documentSession, final Document document) {
                    if (mLastOper == OPER_LOADING_DOC) {
                        DocumentSessionManagerFactory.getInstance().cancelLoading(document);
                        mCreateDocSession = DocumentSessionManagerFactory.getInstance().getEditingSession(documentSession);
                        mLastOper = OPER_NORMAL;
                    }
                    if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
                        saveToPdfFlow(documentSession, document);
                    }
                }

                @Override
                public void onCanceledAction(DocumentSession documentSession) {

                }
            };
            if (mCreateDocSession.getPageCount() != 0 && mCreateDocSession.hasChanges()) {
                Document doc = DocumentManager.get().findDocument(mSaveDocName);
                mCurOpStep = "1.0";
                if (doc == null) {
                    saveToPdfFlow(null, null);
                } else {
                    DocumentManager.get().deleteDocument(doc);
                    saveToPdfFlow(null, null);
                }
            }
            DocumentManager.get().setDocumentManagerListener(new DocumentManagerListener() {
                @Override
                public void onDocumentAdded(final Document document) {
                    onDocumentMgrEvent(document);
                }

                @Override
                public void onDocumentRemoved(Document document) {
                    saveToPdfFlow(null, document);

                }

                @Override
                public void onDocumentUpdated(Document document) {

                }

                @Override
                public void onThumbnailGenerated(Document document) {

                }
            });
        }
    };

    void onDocumentMgrEvent(Document document) {
        if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
            saveToPdfFlow(null, document);
        }
    }

    void saveToImagesFlow(DocumentSession session, Document document) {

        switch (mCurOpStep) {
            case "1.0": {
                mCurOpStep = "1.1";
                mLastOper = OPER_SAVEING_IMAGE;
                DocumentSessionManagerFactory.getInstance().
                        saveSessionAsDocument(mCreateDocSession, mSaveDocName, Document.DocumentType.ImageSeries, progressListener);
                break;
            }
            case "1.1": {
                mCurOpStep = "1.2";
                DocumentManager.get().addDocument(document);
                break;
            }
            default:
                break;
        }
    }

    void saveToPdfFlow(DocumentSession session, final Document document) {
        switch (mCurOpStep) {
            case "1.0":
            case "1.1":
            case "2.0":
            case "2.1":
            case "2.2":
            case "2.3":
            case "3.0":
                saveToImagesFlow(session, document);
                break;
            case "1.2":
            case "1.3":
            case "2.4":
            case "3.1": {
                mCurOpStep = "a.0";
                Document doc = DocumentManager.get().findDocument(mSaveDocName);
                if (doc != null) {
                    try {
                        mLastOper = OPER_LOADING_DOC;
                        DocumentSessionManagerFactory.getInstance().
                                getViewingSessionForDocument(doc, progressListener);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                break;
            }
            case "a.0": {
                mCurOpStep = "a.1";
                mLastOper = OPER_SAVEING_PDF;
                DocumentSessionManagerFactory.getInstance().saveSessionAsDocument(mCreateDocSession, ScannerSession.TEMP_SAVE_PDF_NAME, Document.DocumentType.PDF, progressListener);
                break;
            }
            case "a.1": {
                if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
                    mCurOpStep = "a.2";
                    FileUtils.copyFile(document.getFilename(), mSaveAsPath, true, new IResultCallback() {
                        @Override
                        public void onResult(boolean success) {
                            mCurOpStep = "a.4";
                            String path = document.getFilename();
                            boolean ret = FileUtils.deleteFile(path);
                            onDocumentMgrEvent(document);
                        }
                    });
                }
                break;
            }
            case "a.4": {
                int errorCode = IPDFScanManagerListener.e_ErrUnknown;
                if (FileUtils.isFileExist(mSaveAsPath))
                    errorCode = IPDFScanManagerListener.e_ErrSuccess;
                List<IPDFScanManagerListener> listeners = PDFScanSingleton.getInstance().getManagerListeners();
                for (IPDFScanManagerListener listener : listeners) {
                    listener.onDocumentAdded(errorCode, mSaveAsPath);
                }
                mCreateDocSession = null;
                break;
            }
            default:
                break;
        }
    }

    public CloseEditPDFCallBack mCloseEditPDFCallBack;

    public void setOnCloseEditPDFCallBack(CloseEditPDFCallBack closeEditPDFCallBack) {
        mCloseEditPDFCallBack = closeEditPDFCallBack;
    }

    public interface CloseEditPDFCallBack {

        /**
         * Return EditPDFActivity object
         * @return EditPDFActivity
         */
        Activity getEditPDFActivity();

        /**
         * Return to close intermediate activity from EditPDFActivity
         */
        void closeEditPDFActivity();
    }
}
