/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.browser;

import com.luratech.android.appframework.Document;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

public class SectionList {
    private List<InfoListPair> mSections = new ArrayList<InfoListPair>();

    public SectionList(List<SectionInfo> sections) {
        for (SectionInfo info : sections)
            mSections.add(new InfoListPair(info, new ArrayList<Document>()));
    }

    public void add(Document document) {
        addDocumentAndSort(document, true);
    }

    public void add(List<Document> documents) {
        for (Document document : documents)
            addDocumentAndSort(document, false);

        sortAllSections();
    }

    public void remove(Document document) {
        removeDocumentAndSort(document);
    }

    public boolean isEmpty() {
        return documentCount() == 0;
    }

    public int documentCount() {
        int sum = 0;
        for (InfoListPair pair : mSections)
            sum += pair.list.size();

        return sum;
    }

    public int sectionCount() {
        return mSections.size();
    }

    public int documentCountForSection(int i) {
        return mSections.get(i).list.size();
    }

    public Document getDocument(int sectionIndex, int documentIndex) {
        return mSections.get(sectionIndex).list.get(documentIndex);
    }

    public String sectionTitle(int i) {
        InfoListPair pair = mSections.get(i);
        return pair.info.getTitle();
    }

    private void addDocumentAndSort(Document document, boolean sortList) {
        List<Document> documents = listForDate(document.getDate());
        if (find(document) == -1) {
            documents.add(document);

            if (sortList)
                sort(documents);
        }
    }

    private void removeDocumentAndSort(Document document) {
        List<Document> documents = listForDate(document.getDate());

        int docIdx = find(documents, document);

        if (docIdx != -1) {
            documents.remove(docIdx);
        }
    }

    private void sortAllSections() {
        for (InfoListPair pair : mSections)
            sort(pair.list);
    }

    private void sort(List<Document> documents) {
        Collections.sort(documents, new Comparator<Document>() {
            @Override
            public int compare(Document lhs, Document rhs) {
                return rhs.getDate().compareTo(lhs.getDate());
            }
        });
    }

    private List<Document> listForDate(Date date) {
        if (date != null) {
            for (InfoListPair pair : mSections) {
                Date sectionEndDate = pair.info.getDate();
                if (sectionEndDate == null
                        || sectionEndDate.compareTo(date) <= 0)
                    return pair.list;
            }
        }

        return mSections.get(mSections.size() - 1).list;
    }

    public static class SectionInfo {
        private String mTitle;
        private Date mDate;

        SectionInfo(String title, Date date) {
            mTitle = title;
            mDate = date;
        }

        public String getTitle() {
            return mTitle;
        }

        public Date getDate() {
            return mDate;
        }
    }

    /**
     * Find a document in a documents list.
     *
     * @param documents documents list
     * @param document  document to find
     * @return document index or -1 (not found)
     */
    private int find(List<Document> documents, Document document) {
        int docIdx = 0;
        for (Document doc : documents) {
            if (doc.getDocumentName().equals(document.getDocumentName())) {
                return docIdx;
            }
            docIdx++;
        }
        return -1;
    }

    /**
     * Find a document in a all sections.
     *
     * @param document document to find
     * @return document index or -1 (not found)
     */
    private int find(Document document) {
        for (InfoListPair pair : mSections) {
            int docIdx = find(pair.list, document);
            if (docIdx != -1) {
                return docIdx;
            }
        }
        return -1;
    }

    private class InfoListPair {
        SectionInfo info;
        List<Document> list;

        InfoListPair(SectionInfo info, List<Document> list) {
            this.info = info;
            this.list = list;
        }
    }

}
