/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.browser;

import android.content.Context;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ListView;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.views.BrowserItemView;
import com.foxit.pdfscan.views.BrowserViewPager;
import com.luratech.android.appframework.Document;
import com.luratech.android.appframework.DocumentManager;

public abstract class SectionListAdapter extends BaseAdapter {
    protected static final String TAG = SectionListAdapter.class
            .getSimpleName();

    private static final int TYPE_ITEM = 0;
    private static final int TYPE_TITLE = 1;

    private SectionList mSections;
    private Context mContext;
    private RenameDocument renameDocument;

    public SectionListAdapter(Context context, SectionList sections) {
        mContext = context;
        mSections = sections;
        renameDocument = new RenameDocument();
    }

    void setSectionList(SectionList sectionList) {
        mSections = sectionList;
    }

    @Override
    public int getCount() {
        return mSections.documentCount() + mSections.sectionCount();
    }

    @Override
    public Document getItem(int position) {
        if (getItemViewType(position) == TYPE_ITEM) {
            Document document = mSections.getDocument(sectionIndex(position),
                    documentIndex(position));
            Log.d(TAG, "document: " + document.getDocumentName());
            return document;
        } else {
            Log.d(TAG, "No document for position=" + position);
            return null;
        }
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, final ViewGroup parent) {
        if (getItemViewType(position) == TYPE_ITEM) {
            final View itemView = getItemView(position, convertView, parent);
            return itemView;
        } else {
            return getSectionTitleView(position, convertView, parent);
        }
    }

    public static class ViewHolder {
        public int row;
        public String documentName;
        public View view;
    }

    private View getItemView(int row, View convertView, ViewGroup parent) {
        renameDocument.setListView((ListView) parent);

        Document item = getItem(row);

        final ViewHolder viewHolder;
        if (convertView == null) {
            viewHolder = new ViewHolder();

            convertView = LayoutInflater.from(parent.getContext()).inflate(
                    R.layout.fx_photo2pdf_fragment_browser_viewpager, null);
            BrowserViewPager vp = (BrowserViewPager) convertView.findViewById(R.id.viewpager);
            vp.setAdapter(new ViewPagerAdapter(item) {

                @Override
                protected void delete() {
                    final BrowserViewPager vp = (BrowserViewPager) viewHolder.view
                            .findViewById(R.id.viewpager);
                    vp.setCurrentItem(0);
                    BrowserItemView browserItemView = (BrowserItemView) vp.getChildAt(0);
                    Document doc = browserItemView.getDocument();
                    DocumentManager.get().deleteDocument(doc);
                }

                @Override
                protected void rename() {
                    final BrowserViewPager vp = (BrowserViewPager) viewHolder.view
                            .findViewById(R.id.viewpager);
                    vp.setCurrentItem(0);
                    BrowserItemView browserItemView = (BrowserItemView) vp
                            .getChildAt(0);
                    Document doc = browserItemView.getDocument();

                    renameDocument.setDocument(doc);
                    renameDocument.startEditing(vp);
                }

            });
            vp.setOnClickListener(new OnClickListener() {

                @Override
                public void onClick(View v) {
                    final BrowserViewPager vp = (BrowserViewPager) viewHolder.view
                            .findViewById(R.id.viewpager);
                    vp.setCurrentItem(0);
                    BrowserItemView browserItemView = (BrowserItemView) vp
                            .getChildAt(0);
                    Document doc = browserItemView.getDocument();

                    if (renameDocument.isEditingMode()) {
                        renameDocument.stopEditing();
                    } else {
                        openDocument(doc);
                    }
                }
            });
        } else {
            viewHolder = (ViewHolder) convertView.getTag();

            BrowserViewPager vp = (BrowserViewPager) convertView
                    .findViewById(R.id.viewpager);

            BrowserItemView view = (BrowserItemView) vp.getChildAt(0);
            view.setItem(item);
        }
        viewHolder.row = row;
        viewHolder.view = convertView;
        viewHolder.documentName = item.getDocumentName();
        convertView.setTag(viewHolder);

        return convertView;
    }

    private View getSectionTitleView(int position, View convertView,
                                     ViewGroup parent) {
        int section = sectionIndex(position);
        int count = mSections.documentCountForSection(section);
        if (count == 0) {
            // make empty sections invisible
            return LayoutInflater.from(parent.getContext()).inflate(
                    R.layout.fx_photo2pdf_fragment_browser_invisible_category, null);
        }
        BrowserSectionHeaderView headerView = null;

        if (convertView instanceof BrowserSectionHeaderView)
            headerView = (BrowserSectionHeaderView) convertView;
        else
            headerView = new BrowserSectionHeaderView(mContext);

        headerView.setTitle(mSections.sectionTitle(section));

        return headerView;
    }

    @Override
    public int getItemViewType(int position) {
        int i = 0, currentIndex = 0;
        while (i < mSections.sectionCount()) {
            if (position == currentIndex)
                return TYPE_TITLE;

            currentIndex += mSections.documentCountForSection(i) + 1;
            ++i;
        }

        return TYPE_ITEM;
    }

    @Override
    public int getViewTypeCount() {
        return 2;
    }

    @Override
    public boolean areAllItemsEnabled() {
        return false;
    }

    @Override
    public boolean isEnabled(int position) {
        return getItemViewType(position) == TYPE_ITEM;
    }

    private int sectionIndex(int position) {
        int i = 0, sum = 0;
        while (i < mSections.sectionCount()) {
            sum += mSections.documentCountForSection(i) + 1;
            if (sum > position)
                return i;

            ++i;
        }

        return -1;
    }

    private int documentIndex(int position) {
        int i = 0;
        while (position >= 0) {
            int itemsInSection = mSections.documentCountForSection(i) + 1;

            if (position - itemsInSection < 0)
                return position - 1;

            position -= itemsInSection;
            ++i;
        }

        return -1;
    }

    public void add(Document document) {
        mSections.add(document);
        notifyDataSetChanged();
    }

    public void remove(Document document) {
        mSections.remove(document);
        notifyDataSetChanged();
    }

    protected abstract void openDocument(Document document);

}
