/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.editimage.cropview;

import android.app.Activity;
import android.graphics.Matrix;
import android.graphics.PointF;

import com.foxit.pdfscan.utils.PointUtils;

public class CropViewHelper {

    private final int width;
    private final int height;

    public CropViewHelper(int width, int height) {
        this.width = width;
        this.height = height;
    }

    /**
     * Translate relative coordinates [0..1] to absolute coordinates
     * [0..IMAGE_SIZE].
     */
    public PointF[] transformPointsToViewSpace(PointF[] points) {
        int i = 0;
        PointF[] result = new PointF[points.length];
        for (PointF value : points) {
            PointF p = new PointF(value.x, value.y);
            p.x *= width;
            p.y *= height;
            result[i++] = p;
        }
        return result;
    }

    /**
     * Translate absolute coordinates [0..IMAGE_SIZE] to relative coordinates
     * [0..1].
     */
    public PointF[] transformPointsToImageSpace(PointF[] points) {
        int i = 0;
        PointF[] result = new PointF[points.length];
        for (PointF value : points) {
            PointF p = new PointF(value.x, value.y);
            p.x /= width;
            p.y /= height;
            result[i++] = p;
        }
        return result;
    }

    /**
     * @return points to initially crop the whole picture.
     */
    public PointF[] createPointsAtImageCorners() {
        return new PointF[]{new PointF(0, 0), new PointF(1, 0),
                new PointF(1, 1), new PointF(0, 1)};
    }

    /**
     * @return Check, if the points in the array still form a convex polygon,
     * after replacing.
     */
    public boolean rectConvexWhenReplacing(int index, PointF point,
                                           PointF[] array) {
        int j = 0;
        PointF[] toCheck = new PointF[array.length];
        for (PointF p : array) {
            toCheck[j++] = new PointF(p.x, p.y);
        }
        toCheck[index] = point;
        return isConvex(toCheck);
    }

    /**
     * @return the points are too close to border, we possibly net margins?
     */
    public boolean pointsCloseToBorder(PointF[] points) {
        for (PointF p : points) {
            if (p.x < 0.1 || p.y < 0.05 || p.x > 0.9 || p.y < 0.1)
                return true;
        }
        return false;
    }

    /**
     * @return Check whether the points in the array form a convex polygon.
     */
    public boolean isConvex(PointF[] array) {
        int n = array.length;
        boolean sign = false;

        for (int i = 0; i < n; i++) {
            PointF p0 = array[i];
            PointF p1 = array[(i + 1) % n];
            PointF p2 = array[(i + 2) % n];

            PointF d1 = new PointF(p2.x - p1.x, p2.y - p1.y);
            PointF d2 = new PointF(p0.x - p1.x, p0.y - p1.y);

            float zcrossproduct = d1.x * d2.y - d1.y * d2.x;

            if (i == 0)
                sign = zcrossproduct > 0;
            else if (sign != (zcrossproduct > 0))
                return false;
        }
        return true;
    }

    /**
     * @param activity
     * @return the handle points are computed here. If the distance to the
     * corner point is less than the specified minimumDistance, it is
     * marked to be invisible
     */
    public PointF[] computeCenterPoints(Activity activity, PointF[] points,
                                        float squaredDistance) {
        PointF[] midPoints = new PointF[4];
        for (int side = 0; side < 4; side++) {
            PointF p1 = points[side];
            PointF p2 = points[(side + 1) % 4];
            PointF mid = PointUtils.midPoint(p1, p2);

            PointF p = (PointUtils.squaredDistance(activity, p1, mid) > squaredDistance) ? mid : new PointF(-1, -1);
            midPoints[side] = p;
        }
        return midPoints;
    }

    /**
     * @param p1 handle point after moving
     * @param p2 movedLineEnd
     * @param p3 corner point 1
     * @param p4 corner point 2
     * @return new corner point
     */
    public PointF intersectionOfLineFrom(PointF p1, PointF p2, PointF p3,
                                         PointF p4) {
        float d = (p1.x - p2.x) * (p3.y - p4.y)
                - ((p1.y - p2.y) * (p3.x - p4.x));
        if (d == 0.0)
            return null;

        float ix = ((p1.x * p2.y - p1.y * p2.x) * (p3.x - p4.x) - (p1.x - p2.x)
                * (p3.x * p4.y - p3.y * p4.x))
                / d;

        float iy = ((p1.x * p2.y - p1.y * p2.x) * (p3.y - p4.y) - (p1.y - p2.y)
                * (p3.x * p4.y - p3.y * p4.x))
                / d;

        return new PointF(ix, iy);
    }

    /**
     * Transform view point to image point (Invert matrix and map points)
     *
     * @param matrix         matrix of the layouted image inside the view
     * @param viewCoordinate coordinate of the views coordinate system [0..VIEW_SIZE]
     * @return coordinate of the image coordinate system [0..IMAGE_SIZE]
     */
    public PointF viewCoordinateToImageCoordinate(Matrix matrix,
                                                  PointF viewCoordinate) {
        Matrix inverse = new Matrix();
        matrix.invert(inverse);
        float[] pts = {viewCoordinate.x, viewCoordinate.y};
        inverse.mapPoints(pts);
        return new PointF(pts[0], pts[1]);
    }

}
