/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.fragment;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.BaseAdapter;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.ScannerSession;
import com.foxit.pdfscan.format.PageFormat;
import com.foxit.pdfscan.format.PageFormats;
import com.foxit.pdfscan.utils.AppUtils;
import com.foxit.pdfscan.utils.ResourceUtils;
import com.foxit.pdfscan.views.LastInputEditText;
import com.foxit.uiextensions.theme.BaseThemeAdapter;
import com.foxit.uiextensions.theme.IThemeChangeObserver;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;

public class FormatFragment extends Fragment implements IThemeChangeObserver {
    //	public static final int REQ_SCAN_RC_FORMAT = 3;
    public static final String PAGE_FORMAT = "page_format";
    public static final String PAGE_FORMAT_WIDTH = "page_format_width";
    public static final String PAGE_FORMAT_HEIGHT = "page_format_height";
    public static final String CUSTOM_FORMAT_WIDTH = "custom_format_width";
    public static final String CUSTOM_FORMAT_HEIGHT = "custom_format_height";

    private ListView mainListView;
    private PageSizeAdapter sizeListAdapter;

    private PageFormats pageFormats;
    private Resources resources;

    private View formatView;
    private LastInputEditText customWidthText, customHeightText;
    private TextView back, mDone;
    private CheckBox curFormat;
    private CheckBox originalCheckBox, customCheckBox;

    private int maxAspectRatio;
    private String mm;
    private int maxInput = 5587;
    private boolean done = true;
    private RelativeLayout rlTitleBar;
    private TextView mTvTitle;
    private TextView mTvLabelOriginal;
    private TextView mTvLabelCustom;
    private TextView mTvHeaderLabelWidth;
    private TextView mTvHeaderLabelHeight;
    private TextView mTvHeaderLabelWidthUnit;
    private TextView mTvHeaderLabelHeightUnit;
    private int mNightMode = -1;
    @Override
    public void onThemeChanged(String type, int color) {
        back.setTextColor(ThemeUtil.getPrimaryTextColor(getContext()));
        mDone.setTextColor(ThemeUtil.getPrimaryTextColor(getContext()));

    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        pageFormats = new PageFormats(getResources());
        if (ThemeConfig.getInstance(getContext()).getAdapter() == null) {
            ThemeConfig.getInstance(getContext()).setAdapter(new BaseThemeAdapter());
        }
        ThemeConfig.getInstance(getContext()).getAdapter().registerThemeChangeObserver(this);
        mNightMode = getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        ThemeConfig.getInstance(getContext()).getAdapter().unregisterThemeChangeObserver(this);
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
            mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
            setDarkModeColor();
        }

    }

    private void setDarkModeColor() {
        formatView.setBackgroundResource(R.color.b1);
        rlTitleBar.setBackgroundResource(R.color.b2);
        mTvTitle.setTextColor(resources.getColor(R.color.t4));
        mTvLabelOriginal.setTextColor(resources.getColor(R.color.t4));
        mTvLabelCustom.setTextColor(resources.getColor(R.color.t4));
        mTvHeaderLabelWidth.setTextColor(resources.getColor(R.color.t3));
        mTvHeaderLabelHeight.setTextColor(resources.getColor(R.color.t3));
        customWidthText.setTextColor(resources.getColor(R.color.t4));
        customWidthText.setHintTextColor(resources.getColor(R.color.t2));
        mTvHeaderLabelWidthUnit.setTextColor(resources.getColor(R.color.t3));
        customWidthText.setBackgroundResource(R.drawable.fx_photo2pdf_format_grey_edit_border);
        customHeightText.setTextColor(resources.getColor(R.color.t4));
        customHeightText.setHintTextColor(resources.getColor(R.color.t2));
        mTvHeaderLabelHeightUnit.setTextColor(resources.getColor(R.color.t3));
        customHeightText.setBackgroundResource(R.drawable.fx_photo2pdf_format_grey_edit_border);
        sizeListAdapter.notifyDataSetChanged();
    }


    @SuppressLint("ClickableViewAccessibility")
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        resources = getResources();
        formatView = inflater.inflate(R.layout.fx_photo2pdf_fragment_format, container, false);
        rlTitleBar = formatView.findViewById(R.id.photo2pdf_fragment_rl_title_bar);
        mTvTitle = formatView.findViewById(R.id.photo2pdf_fragment_format_title);
        mTvLabelOriginal= formatView.findViewById(R.id.tv_label_original);
        mTvLabelCustom= formatView.findViewById(R.id.custom);
        mTvHeaderLabelWidth= formatView.findViewById(R.id.tv_header_label_width);
        mTvHeaderLabelHeight= formatView.findViewById(R.id.tv_header_label_height);
        mTvHeaderLabelWidthUnit= formatView.findViewById(R.id.tv_header_label_width_unit);
        mTvHeaderLabelHeightUnit= formatView.findViewById(R.id.tv_header_label_height_unit);
        View originalView = formatView.findViewById(R.id.originalView);
        View customView = formatView.findViewById(R.id.customView);
//        View customTableLayout = formatView.findViewById(R.id.customTableLayout);
        mainListView = (ListView) formatView.findViewById(R.id.mainListView);
        originalCheckBox = (CheckBox) formatView.findViewById(R.id.originalFormat);
        customWidthText = (LastInputEditText) formatView.findViewById(R.id.customWidth);
        customHeightText = (LastInputEditText) formatView.findViewById(R.id.customHeight);
        if (AppDisplay.isPad()) {
            customWidthText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            customHeightText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        customCheckBox = (CheckBox) formatView.findViewById(R.id.customFormat);
        mm = resources.getString(R.string.scan_string_mm);
        maxAspectRatio = resources.getInteger(R.integer.format_custom_max_aspect_ratio);

        PageFormat pageFormat = (PageFormat) getActivity().getIntent().getSerializableExtra(ScannerSession.EXTRA_FORMAT);
        String format = null;
        int width = 1;
        int height = 1;
        if (pageFormat != null) {
            format = pageFormat.getName();
            width = pageFormat.getWidth();
            height = pageFormat.getHeight();
        }
        format = AppUtils.isEmpty(format) ? ResourceUtils.getString(getContext(), R.string.scan_string_original) : format;
        if (format.equals(ResourceUtils.getString(getContext(), R.string.scan_string_custom))) {
            customWidthText.setText(String.valueOf(width));
            customHeightText.setText(String.valueOf(height));
        } else {
            customWidthText.setText("1");
            customHeightText.setText("1");
        }

        customWidthText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                checkAndSetCustom(1, false);

            }

            @Override
            public void afterTextChanged(Editable s) {
            }
        });

        customWidthText.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                switch (event.getAction()) {
                    case MotionEvent.ACTION_DOWN:
                        checkAndSetCustom(1, false);
                        break;
                    case MotionEvent.ACTION_UP:
                        break;
                    case MotionEvent.ACTION_MOVE:
                        break;
                    default:
                        break;
                }
                return false;
            }
        });
        customWidthText.setOnFocusChangeListener(new View.OnFocusChangeListener() {
            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (!hasFocus) {
                    String widthValue = customWidthText.getText().toString();
                    if (widthValue.equals("")) {
                        customWidthText.setText("1");
                    } else {
                        if (widthValue.length() > 4 || Integer.parseInt(widthValue) < 1 || Integer.parseInt(widthValue) > maxInput) {
                            if (widthValue.length() > 4 || Integer.parseInt(widthValue) > maxInput) {
                                customWidthText.setText("" + maxInput);
                            } else {
                                customWidthText.setText("1");
                            }
                        }
                    }
                    setCustom();
                }
            }
        });

        customHeightText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                checkAndSetCustom(2, false);
            }

            @Override
            public void afterTextChanged(Editable s) {

            }
        });
        customHeightText.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                switch (event.getAction()) {
                    case MotionEvent.ACTION_DOWN:
                        checkAndSetCustom(2, false);
                        break;
                    case MotionEvent.ACTION_UP:
                        break;
                    case MotionEvent.ACTION_MOVE:
                        break;
                    default:
                        break;
                }
                return false;
            }
        });
        customHeightText.setOnFocusChangeListener(new View.OnFocusChangeListener() {
            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (!hasFocus) {
                    String heightValue = customHeightText.getText().toString();
                    if (heightValue.equals("")) {
                        customHeightText.setText("1");
                    } else {
                        if (heightValue.length() > 4 || Integer.parseInt(heightValue) < 1 || Integer.parseInt(heightValue) > maxInput) {
                            if (heightValue.length() > 4 || Integer.parseInt(heightValue) > maxInput) {
                                customHeightText.setText("" + maxInput);
                            } else {
                                customHeightText.setText("1");
                            }
                        }
                    }
                    setCustom();
                }
            }
        });

        int index = pageFormats.indexOf(format);
        if (index != -1) {
            PageFormat currentFormat = (PageFormat) pageFormats.get(index);
            pageFormats.setCurrentPageFormat(currentFormat);
        } else if (format.equals(PageFormats.ORIGINAL_PAGE_FORMAT.getName())) {
            pageFormats.setCurrentPageFormat(PageFormats.ORIGINAL_PAGE_FORMAT);
            curFormat = originalCheckBox;
            showCurrentFormat();
        } else {
            width = Integer.parseInt(customWidthText.getText().toString());
            height = Integer.parseInt(customHeightText.getText().toString());
            pageFormats.setCurrentPageFormat(new PageFormat(format, width,
                    height));
            curFormat = customCheckBox;
            showCurrentFormat();
        }

        sizeListAdapter = new PageSizeAdapter();
        mainListView.setAdapter(sizeListAdapter);

        OnClickListener originalListener = new OnClickListener() {
            @Override
            public void onClick(View view) {
                view.requestFocus();
                setOriginal(view);

            }
        };
        originalCheckBox.setOnClickListener(originalListener);
        originalView.setOnClickListener(originalListener);
        OnClickListener customListener = new OnClickListener() {
            @Override
            public void onClick(View view) {
                if (checkAndSetCustom(0, true) && validateCustomWidthHeight()) {
                    checkAndSetCustom(0, false);
                }
            }

            private boolean validateCustomWidthHeight() {
                if (!validate(customWidthText, R.string.format_custom_width_error)
                        || !validate(customHeightText, R.string.format_custom_height_error)) {
                    done = false;
                    return false;
                }
                done = true;
                int width = Integer.parseInt(customWidthText.getText().toString());
                int height = Integer.parseInt(customHeightText.getText().toString());
                float aspectRatio;
                if (width > height) {
                    aspectRatio = height / (float) width;
                } else {
                    aspectRatio = width / (float) height;
                }
                if (aspectRatio * maxAspectRatio < 1) {
                    customWidthText.setError(resources
                            .getString(R.string.format_custom_aspect_ratio,
                                    maxAspectRatio));
                    customCheckBox.setChecked(false);
                    done = false;
                    return false;
                }
                done = true;
                return true;
            }

            private boolean validate(EditText editText, int errorMsgId) {
                if (editText.length() == 0
                        || editText.getText().toString().equals("0")) {
                    editText.setError(resources.getString(errorMsgId));
                    customCheckBox.setChecked(false);
                    return false;
                }
                try {
                    Integer.parseInt(editText.getText().toString());
                } catch (NumberFormatException e) {
                    editText.setError(resources.getString(errorMsgId));
                    customCheckBox.setChecked(false);
                    return false;
                }
                return true;
            }

        };
        customCheckBox.setOnClickListener(customListener);
        customView.setOnClickListener(customListener);

        mainListView.setOnItemClickListener(new OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> parent, View view,
                                    int position, long id) {
                view.requestFocus();
                setSpecific(view, (PageFormat) sizeListAdapter.getItem(position));

            }
        });

        back = (TextView) formatView.findViewById(R.id.photo2pdf_fragment_format_back);
        mDone = (TextView) formatView.findViewById(R.id.photo2pdf_fragment_format_done);
        back.setTextColor(ThemeUtil.getPrimaryTextColor(getContext()));
        mDone.setTextColor(ThemeUtil.getPrimaryTextColor(getContext()));

        back.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                getActivity().finish();
            }
        });
        mDone.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                aspectRatioLimit();
                if (done) {
                    setResult();
                    getActivity().finish();
                }
            }
        });
        return formatView;
    }

    private void aspectRatioLimit() {
        if (!TextUtils.isEmpty(customWidthText.getText().toString()) && !TextUtils.isEmpty(customHeightText.getText().toString())) {
            int width = Integer.parseInt(customWidthText.getText().toString());
            int height = Integer.parseInt(customHeightText.getText().toString());
            float aspectRatio;
            if (width > height) {
                aspectRatio = height / (float) width;
            } else {
                aspectRatio = width / (float) height;
            }
            if (aspectRatio * maxAspectRatio < 1) {
                customWidthText.setError(resources
                        .getString(R.string.format_custom_aspect_ratio,
                                maxAspectRatio));
                customCheckBox.setChecked(false);
                done = false;
            } else {
                done = true;
            }
        }
    }

    private boolean checkAndSetCustom(int opType, boolean onlyCheck) {
        boolean canSetValue = true;

        String widthValue = customWidthText.getText().toString();
        if (widthValue.equals("")) {
            if (opType == 1) {
                done = false;
                customWidthText.setError(resources.getString(R.string.format_custom_width_error));
            } else {
                done = true;
                customWidthText.setText("1");
                customWidthText.setError(null);
            }
            canSetValue = false;
        } else {
            if (widthValue.length() > 4 || Integer.parseInt(widthValue) < 1 || Integer.parseInt(widthValue) > maxInput) {
                if (opType == 1) {
                    done = false;
                    customWidthText.setError(resources.getString(R.string.format_custom_width_error));
                } else {
                    done = true;
                    if (widthValue.length() > 4 || Integer.parseInt(widthValue) > maxInput) {
                        customWidthText.setText("" + maxInput);
                    } else {
                        customWidthText.setText("1");
                    }
                    customWidthText.setError(null);
                }
                canSetValue = false;
            } else {
                done = true;
            }
        }

        String heightValue = customHeightText.getText().toString();
        if (heightValue.equals("")) {
            if (opType == 2) {
                done = false;
                customHeightText.setError(resources.getString(R.string.format_custom_height_error));
            } else {
                done = true;
                customHeightText.setText("1");
                customHeightText.setError(null);
            }
            canSetValue = false;
        } else {
            if (heightValue.length() > 4 || Integer.parseInt(heightValue) < 1 || Integer.parseInt(heightValue) > maxInput) {
                if (opType == 2) {
                    done = false;
                    customHeightText.setError(resources.getString(R.string.format_custom_height_error));
                } else {
                    done = true;
                    if (heightValue.length() > 4 || Integer.parseInt(heightValue) > maxInput) {
                        customHeightText.setText("" + maxInput);
                    } else {
                        customHeightText.setText("1");
                    }
                    customHeightText.setError(null);
                }
                canSetValue = false;
            }
        }

        if (canSetValue && !onlyCheck)
            setCustom();
        return canSetValue;
    }

    private void setOriginal(View view) {
        hideCurrentFormat();
        curFormat = originalCheckBox;
        showCurrentFormat();
        sizeListAdapter.setOriginal();
    }

    private void setCustom() {
        PageFormat pageFormat = createCustomPageFormat();
        hideCurrentFormat();
        curFormat = customCheckBox;
        showCurrentFormat();
        sizeListAdapter.setCustom(pageFormat);
    }

    @Override
    public void onDetach() {
        super.onDetach();
    }

    protected void setSpecific(View view, PageFormat pageFormat) {
        hideCurrentFormat();
        curFormat = (CheckBox) view.findViewById(R.id.checkedFormat);
        showCurrentFormat();
        sizeListAdapter.setSpecific(pageFormat);
    }

    private void hideCurrentFormat() {
        if (curFormat != null) {
            curFormat.setChecked(false);
            curFormat.setVisibility(View.INVISIBLE);
        }
    }

    private void showCurrentFormat() {
        curFormat.setChecked(true);
        curFormat.setVisibility(View.VISIBLE);
        curFormat.setButtonDrawable(R.drawable.fx_photo2pdf_format_check);
        ThemeUtil.setTintList(curFormat, ThemeUtil.getCheckboxColor(getActivity()));
    }

    private PageFormat createCustomPageFormat() {
        int widthValue = getValueInMM(customWidthText);
        int heightValue = getValueInMM(customHeightText);
        PageFormat item = new PageFormat(getResources().getString(R.string.scan_string_custom), widthValue, heightValue);
        return item;
    }

    private int getValueInMM(TextView text) {
        String string = text.getText().toString();
        if (string.endsWith(mm)) {
            string = string.substring(0, string.length() - mm.length()).trim();
        }
        if (string.length() > 0) {
            return Integer.parseInt(string);
        }
        return 1;
    }

    protected void setResult() {
        Intent intent = new Intent();
        intent.putExtra(ScannerSession.EXTRA_FORMAT, pageFormats.getCurrentPageFormat());
        getActivity().setResult(Activity.RESULT_OK, intent);
    }

    public static class ViewHolder {
        int row;
        View view;
    }

    private final class PageSizeAdapter extends BaseAdapter {

        @Override
        public int getCount() {
            return pageFormats.getPageFormats().size();
        }

        @Override
        public Object getItem(int position) {
            return pageFormats.getPageFormats().get(position);
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        @Override
        public View getView(int position, View convertView, ViewGroup parent) {
            int row = position;
            PageFormat item = (PageFormat) getItem(row);

            final ViewHolder viewHolder;
            if (convertView == null) {
                convertView = LayoutInflater.from(parent.getContext()).inflate(R.layout.fx_photo2pdf_fragment_format_item, null);
                viewHolder = new ViewHolder();
            } else {
                viewHolder = (ViewHolder) convertView.getTag();
            }
            setItem(convertView, item);
            viewHolder.row = row;
            viewHolder.view = convertView;
            convertView.setTag(viewHolder);

            return convertView;
        }

        private void setItem(View convertView, final PageFormat item) {
            TextView formatName = (TextView) convertView.findViewById(R.id.formatName);
            formatName.setText(item.getName());
            formatName.setTextColor(getActivity().getResources().getColor(R.color.t4));
            TextView labelWidth = (TextView) convertView.findViewById(R.id.tv_label_width);
            labelWidth.setTextColor(getActivity().getResources().getColor(R.color.t3));
            TextView labelHeight = (TextView) convertView.findViewById(R.id.tv_label_height);
            labelHeight.setTextColor(getActivity().getResources().getColor(R.color.t3));
            TextView widthValue = (TextView) convertView.findViewById(R.id.widthValue);
            widthValue.setText(item.getWidth() + " " + mm);
            widthValue.setTextColor(getActivity().getResources().getColor(R.color.t3));
            TextView heightValue = (TextView) convertView.findViewById(R.id.heightValue);
            heightValue.setText(item.getHeight() + " " + mm);
            heightValue.setTextColor(getActivity().getResources().getColor(R.color.t3));
            CheckBox checkedFormat = (CheckBox) convertView.findViewById(R.id.checkedFormat);
            boolean selected = pageFormats.getCurrentPageFormat() == item;
            if (selected) {
                curFormat = checkedFormat;
                showCurrentFormat();
            }
            checkedFormat.setSelected(selected);
            checkedFormat.setChecked(selected);
            checkedFormat.setOnClickListener(new OnClickListener() {

                @Override
                public void onClick(View v) {
                    FormatFragment.this.setSpecific(v, item);
                }
            });
        }

        public void setSpecific(PageFormat item) {
            pageFormats.setCurrentPageFormat(item);
            setResult();
        }

        public void setOriginal() {
            PageFormat item = PageFormats.ORIGINAL_PAGE_FORMAT;
            pageFormats.setCurrentPageFormat(item);
            setResult();
        }

        public void setCustom(PageFormat item) {
            pageFormats.setCurrentPageFormat(item);
            setResult();
        }

    }

}