/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.utils;

import android.content.Context;
import android.net.Uri;

import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

public class FileUtils {
    private static final int BUFFER_SIZE = 1024;
    private static final String PATTERN_VALID_FILENAME = "[a-zA-Z_0-9\\.\\-\\(\\)\\%]+";
    private static final char FILENAME_REPLACEMENT_CHAR = '_';
    private static final String EXTERNAL_DIR = "ext";
    public static final String FILE_PROVIDER = "com.luratech.android.pdfscanner.provider";

    /**
     * Replace illegal filename characters by replacement character
     *
     * @param filename filename to replace
     * @return filename containing valid characters, only!
     */
    private static String fixInvalidFileCharacters(String filename) {
        Pattern pattern = Pattern.compile(PATTERN_VALID_FILENAME);
        StringBuilder result = new StringBuilder();
        for (char c : filename.toCharArray()) {
            result.append(c);
            if (!pattern.matcher(result).matches()) {
                // replace illegal filename character
                result.setCharAt(result.length() - 1, FILENAME_REPLACEMENT_CHAR);
            }
        }
        return result.toString();
    }

    /**
     * Store a temporary file in the internal storage.
     *
     * @param context current context
     * @param in      input to store
     * @param name    output file name
     * @return stored file on the external storage
     * @throws IOException I/O error
     */
    public static final File storeFileToInternalStorage(Context context,
                                                        InputStream in, String name) throws IOException {
        String filePath = context.getCacheDir() + "/" + name;
        File file = new File(filePath);
        FileOutputStream out = new FileOutputStream(file);
        save(in, out);

        return file;
    }

    private static void save(InputStream in, OutputStream out)
            throws IOException {
        int len;
        byte buf[] = new byte[BUFFER_SIZE];
        while ((len = in.read(buf)) > 0) {
            out.write(buf, 0, len);
        }
        in.close();
        out.close();
    }

    public static void copyFile(String oldPath, String newPath, IResultCallback callback) {
        try {
            int bytesum = 0;
            int byteread = 0;
            File oldFile = new File(oldPath);
            File file = new File(newPath);
//			filedir create
            if (!file.exists()) {
                InputStream inStream = new FileInputStream(oldPath);
                FileOutputStream fs = new FileOutputStream(newPath);
                byte[] buffer = new byte[1444];
                int length;
                while ((byteread = inStream.read(buffer)) != -1) {
                    bytesum += byteread;
                    fs.write(buffer, 0, byteread);
                }
                inStream.close();
            }
            if (file.length() == oldFile.length()) {
                callback.onResult(true);
            } else {
                callback.onResult(false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static void copyFile(String oldPath, String newPath, boolean deleteExist, IResultCallback callback) {
        InputStream inStream = null;
        OutputStream fs = null;
        try {
            int bytesum = 0;
            int byteread = 0;
            File oldFile = new File(oldPath);
            File file = new File(newPath);
//			filedir create
            boolean isSuccess = false;
            if (deleteExist) {
                if (file.exists()) {
                    isSuccess = file.delete();
                    if (!isSuccess && !file.canWrite()){
                        isSuccess = AppFileUtil.deleteDocumentFile(newPath);
                    }
                    file = new File(newPath);
                }
            }
            if (!file.exists()) {
                inStream = new FileInputStream(oldPath);
                File parentFile = file.getParentFile();
                if (parentFile != null && parentFile.canRead() && !parentFile.canWrite()){
                    Uri uri = AppFileUtil.toDocumentUriFromPath(newPath);
                    AppFileUtil.createNewDocument(AppFileUtil.toDocumentUriFromPath(newPath),false);
                    fs = AppUtil.getApplicationContext().getContentResolver().openOutputStream(uri);
                }else {
                   fs = new FileOutputStream(newPath);
                }
                byte[] buffer = new byte[1444];
                int length;
                while ((byteread = inStream.read(buffer)) != -1) {
                    bytesum += byteread;
                    fs.write(buffer, 0, byteread);
                }
                inStream.close();
                isSuccess = file.length() == oldFile.length();
            }
            if (callback == null){
                return;
            }
            callback.onResult(isSuccess);
        } catch (Exception e) {
            e.printStackTrace();
            if (callback != null)
                callback.onResult(false);
        }finally {
            AppFileUtil.closeQuietly(inStream);
            AppFileUtil.closeQuietly(fs);
        }
    }

    public static String generateNewDocName() {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy_MM_dd HH_mm_ss");
        Date curDate = new Date(System.currentTimeMillis());
        String str = "Scan " + formatter.format(curDate);
        return str;
    }

    public static boolean deleteFile(String path) {
        File file = new File(path);
        return file.delete();
    }

    public static boolean isFileExist(String filePath) {
        File file = new File(filePath);
        return file.exists();
    }

    public static String getAppCacheDir(Context context) {
        return context.getApplicationContext().getCacheDir().getAbsolutePath();
    }

    public static String getFileFolder(String filePath) {
        int index = filePath.lastIndexOf('/');
        if (index < 0) return "";
        return filePath.substring(0, index);
    }

}
