/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.content.Context;
import android.graphics.Rect;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.browser.SectionListAdapter;

import androidx.viewpager.widget.ViewPager;

public class BrowserViewPager extends ViewPager {
    private static final float MIN_SWIPE_DISTANCE = 10;
    private static final String TAG = BrowserViewPager.class.getSimpleName();

    private float oldX, newX;

    private OnClickListener mOnClickListener;

    public BrowserViewPager(Context context) {
        super(context);
    }

    public BrowserViewPager(Context context, AttributeSet attrs) {
        super(context, attrs);
    }


    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (event.getAction() == MotionEvent.ACTION_DOWN) // touch down
        {
            oldX = event.getX(); // get touch position

        } else if (event.getAction() == MotionEvent.ACTION_UP) // touch up
        {
            newX = event.getX(); // get last touch position
            if (Math.abs(newX - oldX) < MIN_SWIPE_DISTANCE) {
                // its a click
                handleOnClick(event);
                return true;
            }
        }
        return super.onTouchEvent(event);
    }

    public OnClickListener getOnClickListener() {
        return mOnClickListener;
    }

    /**
     * Find out, which row/view is under the mouse and trigger onClick event
     * manually.
     *
     * @param event
     */
    private void handleOnClick(MotionEvent event) {
        // search for view with specified id (a browser row)
        View convertView = findViewAtPosition(getRootView(),
                (int) event.getRawX(), (int) event.getRawY(),
                R.id.item);
        if (convertView == null) {
            return;
        }
        SectionListAdapter.ViewHolder viewHolder = (SectionListAdapter.ViewHolder) convertView.getTag();
        //Log.d(TAG, "Row clicked: " + viewHolder.row);
        BrowserViewPager vp = (BrowserViewPager) convertView.findViewById(R.id.viewpager);
        vp.getOnClickListener().onClick(convertView);
    }

    public void setOnClickListener(OnClickListener onClickListener) {
        this.mOnClickListener = onClickListener;
    }

    /**
     * Search for a view at the specified coordinates.
     *
     * @param parent   current parent view
     * @param x        x coordinate
     * @param y        y coordinate
     * @param searchId ID of the desired view
     * @return search result or null (not found)
     */
    private View findViewAtPosition(View parent, int x, int y, int searchId) {
        if (parent instanceof ViewGroup && parent.getId() != searchId) {
            ViewGroup viewGroup = (ViewGroup) parent;
            for (int i = 0; i < viewGroup.getChildCount(); i++) {
                View child = viewGroup.getChildAt(i);
                // recursive search for all children
                View viewAtPosition = findViewAtPosition(child, x, y, searchId);
                if (viewAtPosition != null) {
                    return viewAtPosition;
                }
            }
        } else {
            Rect rect = new Rect();
            parent.getGlobalVisibleRect(rect);
            if (rect.contains(x, y)) {
                // view has been identified under the mouse
                return parent;
            }
        }
        // not found
        return null;
    }

}
