/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.MotionEvent;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.widget.FrameLayout;

import com.foxit.pdfscan.R;
import com.luratech.android.appframework.CameraHandler;
import com.luratech.android.appframework.CaptureService;
import com.luratech.android.appframework.CaptureServiceResultListener;
import com.luratech.android.appframework.CaptureServiceShutterListener;
import com.luratech.android.appframework.CaptureServiceUpdateListener;

public class CameraSurfaceView extends SurfaceView implements
        SurfaceHolder.Callback, View.OnTouchListener {
    private static final String TAG = CameraSurfaceView.class.getSimpleName();

    private final CaptureService captureService;
    private CameraHandler cameraHandler;
    private SurfaceHolder holder;

    private CaptureServiceShutterListener shutterCallback;

    private CaptureServiceResultListener resultListener;

    private CaptureServiceUpdateListener updateListener;

    public CameraSurfaceView(Context context) {
        super(context);
        captureService = new CaptureService(context);
        initComponent(context);
    }

    public CameraSurfaceView(Context context, AttributeSet attrs) {
        super(context, attrs);
        captureService = new CaptureService(context);
        initComponent(context);
    }

    private void initComponent(Context context) {
        holder = getHolder();
        holder.addCallback(this);
    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        if (!(getParent() instanceof FrameLayout)) {
            throw new RuntimeException("ParentView must be FrameLayout!");
        }
        try {
            cameraHandler = new CameraHandler();
            cameraHandler.open(0, (Activity) getContext());
            cameraHandler.setPreview((FrameLayout) getParent(), holder);
            cameraHandler.setShutterCallback(shutterCallback);
            cameraHandler.setContinouousAutoFocus(true);
            cameraHandler.setAutoFocusBeforeTakingPicture(false);


            captureService.setShakeThreshold(getFloat(getResources(), R.dimen.scanner_motion_detection_acceleration_limit));
            captureService.setCamera(getContext(), cameraHandler);
            captureService.setCaptureServiceResultListener(resultListener);
            captureService.setCaptureServiceUpdateListener(updateListener);
            captureService.startPreviewProcessorThread();

            setOnTouchListener(this);
        } catch (Exception exception) {
            //Log.e(TAG, "Error setting up preview display", exception);
            cameraHandler.release();
        }
    }

    private float getFloat(Resources resources, int id) {
        TypedValue outValue = new TypedValue();
        resources.getValue(id, outValue, true);
        float float1 = outValue.getFloat();
        return float1;
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width,
                               int height) {
        cameraHandler.updatePreview(holder, format, width, height);
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
        try {
            cameraHandler.stopPreview();
        } finally {
            cameraHandler.release();
        }
        captureService.terminatePreviewProcessorThread();
    }

    public void setShutterCallback(CaptureServiceShutterListener callbackObject) {
        this.shutterCallback = callbackObject;
    }

    public void setCaptureServiceResultListener(
            CaptureServiceResultListener listener) {
        this.resultListener = listener;
    }

    public void setCaptureServiceUpdateListener(
            CaptureServiceUpdateListener listener) {
        this.updateListener = listener;
    }

    public CaptureService getCaptureService() {
        return captureService;
    }

    /**
     * Give access to the CameraHandler, in case the UI wants to
     * call some of its focus related methods.
     */
    public CameraHandler getCameraHandler() {
        return cameraHandler;
    }


    @Override
    public boolean onTouch(View v, MotionEvent event) {
        if (cameraHandler != null) {
            cameraHandler.autoFocusAtPoint(event.getX(), event.getY());
        }

        return true;
    }

    // Lint wants us to override this because we have an OnTouchListener installed.
    // However it doesn't need to do anything.
    @Override
    public boolean performClick() {
        return super.performClick();
    }
}
