/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemSelectedListener;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.SeekBar;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.Spinner;
import android.widget.TextView;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.editimage.ColorspaceAdapter;
import com.foxit.pdfscan.editimage.ColorspaceItem;
import com.foxit.pdfscan.editimage.EditImageListener;
import com.luratech.android.appframework.ImageProcessor.Colorspace;

import java.util.ArrayList;

public class EditImageActionsView extends RelativeLayout implements
        OnSeekBarChangeListener, OnItemSelectedListener {

    /**
     * Action buttons to click and start edit mode (process image changes).
     */
    private Button brightnessButton, contrastButton, rotateButton, cropButton;
    /**
     * Current slider values of the image properties.
     */
    private int brightnessValue, contrastValue;

    /**
     * The slider to change an image property.
     */
    private SeekBar seekBar;
    /**
     * The current value (brightness, contrast)
     */
    private TextView sliderText;
    /**
     * The popup menu to choose the colorspace.
     */
    private Spinner colorSpinner;

    /**
     * Are we in editing mode?
     */
    private boolean isEditingMode;
    /**
     * Listener to get notified of changes.
     */
    private EditImageListener editImageListener;

    private LinearLayout sliderWithText;

    public EditImageActionsView(Context context) {
        super(context);
        initComponent(context);
    }

    public EditImageActionsView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initComponent(context);
    }

    private void initComponent(Context context) {
        LayoutInflater inflater = LayoutInflater.from(context);
        final View v = inflater.inflate(R.layout.fx_photo2pdf_fragment_editimage_actions,
                null, false);

        seekBar = (SeekBar) v.findViewById(R.id.slider);
        sliderWithText = (LinearLayout) v.findViewById(R.id.sliderWithText);

        brightnessValue = seekBar.getProgress();
        contrastValue = seekBar.getProgress();

        seekBar.setOnSeekBarChangeListener(this);
        sliderText = (TextView) v.findViewById(R.id.sliderText);

        brightnessButton = (Button) v.findViewById(R.id.brightnessButton);
        contrastButton = (Button) v.findViewById(R.id.contrastButton);

        colorSpinner = (Spinner) v.findViewById(R.id.colorSpinner);
        ColorspaceAdapter adapter = new ColorspaceAdapter(
                (Activity) getContext(), android.R.layout.simple_spinner_item,
                populateColorspaces());
        colorSpinner.setAdapter(adapter);
        colorSpinner.setOnItemSelectedListener(this);

        rotateButton = (Button) v.findViewById(R.id.rotateButton);
        cropButton = (Button) v.findViewById(R.id.cropButton);
        brightnessButton.setOnClickListener(new OnClickListener() {

            @Override
            public void onClick(View v) {
                boolean activated = !brightnessButton.isActivated();
                brightnessButton.setActivated(activated);
                setEditingMode(activated);

                initSeekBar(sliderText, brightnessButton);
                setButtonStates(contrastButton, colorSpinner, rotateButton,
                        cropButton, !activated);
            }
        });
        contrastButton.setOnClickListener(new OnClickListener() {

            @Override
            public void onClick(View v) {
                boolean activated = !contrastButton.isActivated();
                contrastButton.setActivated(activated);
                setEditingMode(activated);

                initSeekBar(sliderText, contrastButton);
                setButtonStates(brightnessButton, colorSpinner, rotateButton,
                        cropButton, !activated);
            }
        });

        rotateButton.setOnClickListener(new OnClickListener() {

            @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
            @Override
            public void onClick(View v) {
                initSeekBar(sliderText, rotateButton);
                seekBar.setVisibility(View.INVISIBLE);
                sliderWithText.setBackground(null);
                setButtonStates(brightnessButton, contrastButton, colorSpinner,
                        cropButton, true);
                editImageListener.onButtonClicked(rotateButton);
            }
        });
        cropButton.setOnClickListener(new OnClickListener() {

            @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
            @Override
            public void onClick(View v) {
                initSeekBar(sliderText, cropButton);
                seekBar.setVisibility(View.INVISIBLE);
                sliderWithText.setBackground(null);
                setButtonStates(brightnessButton, contrastButton, colorSpinner,
                        rotateButton, true);
                editImageListener.onButtonClicked(cropButton);
            }
        });
        this.addView(v);
    }

    @Override
    public void onStopTrackingTouch(SeekBar seekBar) {
        Button button = (Button) seekBar.getTag();
        if (button == brightnessButton) {
            brightnessValue = seekBar.getProgress();
        } else if (button == contrastButton) {
            contrastValue = seekBar.getProgress();
        }
    }

    @Override
    public void onStartTrackingTouch(SeekBar seekBar) {
    }

    @Override
    public void onProgressChanged(SeekBar seekBar, int progress,
                                  boolean fromUser) {
        sliderText.setText(String.valueOf(progress - 10));
        editImageListener.seekBarChanged(seekBar, progress, fromUser, "");
    }

    private ArrayList<ColorspaceItem> populateColorspaces() {

        final ArrayList<ColorspaceItem> colorspaces = new ArrayList<ColorspaceItem>();

//		colorspaces.add(new ColorspaceItem(Colorspace.UNDEFINED, -1));
//		colorspaces.add(new ColorspaceItem(Colorspace.RGBA_WB,
//				R.drawable.editimage_colorspace_color_white_balance));
//		colorspaces.add(new ColorspaceItem(Colorspace.RGBA,
//				R.drawable.editimage_colorspace_color));
//		colorspaces.add(new ColorspaceItem(Colorspace.GRAYSCALE,
//				R.drawable.editimage_colorspace_gray));
//		colorspaces.add(new ColorspaceItem(Colorspace.BITONAL,
//				R.drawable.editimage_colorspace_black_white));

        colorspaces.add(new ColorspaceItem(Colorspace.UNDEFINED, "undefined"));
        colorspaces.add(new ColorspaceItem(Colorspace.RGBA_WB, "white_balance"));
        colorspaces.add(new ColorspaceItem(Colorspace.RGBA, "rgba"));
        colorspaces.add(new ColorspaceItem(Colorspace.GRAYSCALE, "grayscale"));
        colorspaces.add(new ColorspaceItem(Colorspace.BITONAL, "black_white"));
        return colorspaces;

    }

    public void setEditImageListener(EditImageListener listener) {
        this.editImageListener = listener;
    }

    private void setButtonStates(View btn1, View btn2, View btn3, View btn4,
                                 boolean active) {
        btn1.setEnabled(active);
        btn2.setEnabled(active);
        btn3.setEnabled(active);
        btn4.setEnabled(active);
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    private void initSeekBar(final TextView sliderText, final View button) {
        seekBar.setTag(button);
        int isVisible = button.isActivated() ? View.VISIBLE : View.INVISIBLE;
        if (button == brightnessButton) {
            seekBar.setProgress(brightnessValue);
        } else if (button == contrastButton) {
            seekBar.setProgress(contrastValue);
        }
        seekBar.setVisibility(isVisible);
        sliderWithText.setBackground(button.isActivated() ? getResources()
                .getDrawable(R.drawable.fx_photo2pdf_general_shape_grey_bg) : null);
        sliderText.setText(String.valueOf(seekBar.getProgress() - 10));
        sliderText.setVisibility(isVisible);
    }

    public boolean isEditingMode() {
        return isEditingMode;
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    public void resetEditingMode() {

        brightnessValue = 10;
        contrastValue = 10;

        brightnessButton.setEnabled(true);
        contrastButton.setEnabled(true);
        colorSpinner.setEnabled(true);
        rotateButton.setEnabled(true);
        cropButton.setEnabled(true);

        brightnessButton.setActivated(false);
        contrastButton.setActivated(false);
        colorSpinner.setActivated(false);
        rotateButton.setActivated(false);
        cropButton.setActivated(false);

        seekBar.setVisibility(View.INVISIBLE);
        sliderWithText.setBackground(null);
        sliderText.setText("");
        sliderText.setVisibility(View.INVISIBLE);
        setEditingMode(false);
    }

    private void setEditingMode(boolean activated) {
        isEditingMode = activated;
        editImageListener.editingModeChanged(isEditingMode);
    }

    @Override
    public void onNothingSelected(AdapterView<?> parent) {
    }

    @Override
    public void onItemSelected(AdapterView<?> parent, View view, int position,
                               long id) {
        if (position != 0) {
            Object item = parent.getItemAtPosition(position);
            editImageListener.onItemSelected(colorSpinner, item);
        }
    }

}
