/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.Path.Direction;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.View;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.utils.PointUtils;
import com.luratech.android.appframework.BitmapWithMetadata;

public class MagnifierView extends View {
    /**
     * Magnifier scale factor X
     */
    private static float SCALE_X;
    /**
     * Magnifier scale factor Y
     */
    private static float SCALE_Y;
    /**
     * Magnifier width
     */
    private static float WIDTH;
    /**
     * Magnifier height
     */
    private static float HEIGHT;

    private Paint crosshairPaint;
    private Path crossHairPath, clipPath;

    private Bitmap bitmapToMagnify;
    private Matrix scaleMatrix;
    private Rect src;
    private RectF dst;
    private float sx, sy;

    public MagnifierView(Context context) {
        super(context);
        initComponent();
    }

    public MagnifierView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initComponent();
    }

    protected void initComponent() {
        Activity activity = (Activity) getContext();
        Resources resources = getResources();
        SCALE_X = getFloat(resources,
                R.dimen.editimage_cropview_magnifier_scale_x);
        SCALE_Y = getFloat(resources,
                R.dimen.editimage_cropview_magnifier_scale_y);
        WIDTH = PointUtils.dpToPx(activity, resources
                .getInteger(R.integer.editimage_cropview_magnifier_width));
        HEIGHT = PointUtils.dpToPx(activity, resources
                .getInteger(R.integer.editimage_cropview_magnifier_height));
        setScale(SCALE_X, SCALE_Y);
    }

    private float getFloat(Resources resources, int id) {
        TypedValue outValue = new TypedValue();
        resources.getValue(id, outValue, true);
        float float1 = outValue.getFloat();
        return float1;
    }

    public void setBitmap(BitmapWithMetadata bitmap) {
        this.bitmapToMagnify = bitmap.getBitmap();
    }

    public void setScale(float sx, float sy) {
        this.sx = sx;
        this.sy = sy;
        scaleMatrix = new Matrix();
        scaleMatrix.postScale(sx, sy);
        createCrosshair();
    }

    public void setTouchPoint(PointF touchPoint) {
        // depending on the touch point, set magnifier location
        int middleX = bitmapToMagnify.getWidth() / 2;
        if (touchPoint.x > middleX) {
            setX(0);
            setY(0);
        } else {
            setX(getWidth() - WIDTH);
            setY(0);
        }
        // calculate bitmap region to paint and apply
        float w = WIDTH / sx;
        float h = HEIGHT / sy;

        float left = touchPoint.x - w / 2;
        float top = touchPoint.y - h / 2;
        float right = touchPoint.x + w / 2;
        float bottom = touchPoint.y + h / 2;

        src = new Rect((int) left, (int) top, (int) right, (int) bottom);

        // manually crop at the borders, otherwise android would zoom!
        float dstX = left < 0 ? Math.abs(left) : 0;
        float dstY = top < 0 ? Math.abs(top) : 0;
        if (right >= bitmapToMagnify.getWidth()) {
            w -= Math.abs(right - bitmapToMagnify.getWidth());
        }
        if (bottom >= bitmapToMagnify.getHeight()) {
            h -= Math.abs(bottom - bitmapToMagnify.getHeight());
        }
        dst = new RectF(dstX, dstY, w, h);
        invalidate();
    }

    private void createCrosshair() {
        crosshairPaint = new Paint();
        crosshairPaint.setStyle(Style.STROKE);
        crosshairPaint.setColor(getResources().getColor(
                R.color.scan_color_blue_ff007aff));

        crossHairPath = new Path();
        float w = WIDTH / sx;
        float h = HEIGHT / sy;

        crossHairPath.moveTo(w / 2, 0);
        crossHairPath.lineTo(w / 2, h);

        crossHairPath.moveTo(0, h / 2);
        crossHairPath.lineTo(w, h / 2);

        crossHairPath.addCircle(w / 2, h / 2, h / 6, Direction.CW);
        crossHairPath.addCircle(w / 2, h / 2, h / 2, Direction.CW);
        crossHairPath.close();

        clipPath = new Path();
        clipPath.addCircle(w / 2, h / 2, (h / 2) + 1, Direction.CW);
        clipPath.close();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        // apply scaling
        canvas.concat(scaleMatrix);
        // clip area around outer circle (make transparent)
        canvas.clipPath(clipPath);
        // clip bitmap (draw region of the bitmap)
        canvas.drawBitmap(bitmapToMagnify, src, dst, null);
        // draw crosshair
        canvas.drawPath(crossHairPath, crosshairPaint);
    }
}
