/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.content.Context;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.ScaleGestureDetector.OnScaleGestureListener;
import android.view.View;
import android.widget.ImageView;

import com.foxit.pdfscan.R;

import androidx.viewpager.widget.ViewPager;

public class ZoomingAndPanningViewPager extends ViewPager implements OnScaleGestureListener {
    private static final float SWIPE_THRESHOLD = 0.01f;

    /**
     * Detect zoom gestures.
     */
    private ScaleGestureDetector gestureDetector;

    /**
     * Current view to pan/zoom.
     */
    private View convertView;
    /**
     * Current image view to pan/zoom.
     */
    private ImageView imageView;

    /**
     * Minimum and maximum allowed zooming factor.
     */
    private float minZoom, maxZoom;

    private boolean swipeEnabled;

    private GestureDetector doubleClickGestureListener;

    private boolean stopPanning;

    private float swipeOffset;

    PointF mImageLeftTop = new PointF();
    float mScale = 1.0f;
    PointF mScaleCenter = new PointF();
    PointF mDownPt = new PointF();

    private class DoubleClickGestureListener extends
            GestureDetector.SimpleOnGestureListener {

        @Override
        public boolean onDown(MotionEvent e) {
            return true;
        }

        @Override
        public boolean onDoubleTap(MotionEvent e) {
            mScaleCenter.set(e.getX(), e.getY());
            if (convertView.getScaleX() == getMaxZoom()) {
                zoom(getMinZoom());
            } else {
                zoom(getMaxZoom());
            }

            return true;
        }
    }

    public ZoomingAndPanningViewPager(Context context) {
        super(context);
        init(context);
    }

    public ZoomingAndPanningViewPager(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    private void init(Context context) {
        gestureDetector = new ScaleGestureDetector(context, this);
        doubleClickGestureListener = new GestureDetector(context,
                new DoubleClickGestureListener());
        minZoom = getResources().getInteger(R.integer.photo2pdf_viewer_min_zoom);
        maxZoom = getResources().getInteger(R.integer.photo2pdf_viewer_max_zoom);
        swipeEnabled = true;
    }

    public void setSwipeEnabled(boolean swipeEnabled) {
        this.swipeEnabled = swipeEnabled;
    }

    /**
     * Handle touch events. One finger pans, and two finger pinch and zoom plus
     * panning.
     */
    @Override
    public boolean onTouchEvent(MotionEvent event) {
        // find current view to pan/zoom
        convertView = findViewWithTag(getCurrentItem());
        imageView = (ImageView) convertView.findViewById(R.id.PDFPage);

        doubleClickGestureListener.onTouchEvent(event);

        // handle zoom
        if (swipeOffset < SWIPE_THRESHOLD) {
            gestureDetector.onTouchEvent(event);
        }

        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                // remember panning start coordinate
                float x = event.getX();
                float y = event.getY();
                mDownPt.set(x, y);
                stopPanning = false;
                break;
            case MotionEvent.ACTION_UP:
                stopPanning = event.getPointerCount() == 1;
                break;
            case MotionEvent.ACTION_MOVE:
                if (event.getPointerCount() == 1) {
                    // handle panning
                    float dx = event.getX() - mDownPt.x;
                    float dy = event.getY() - mDownPt.y;
                    float scaleFactor = getScaleFactor(gestureDetector.getScaleFactor());

                    mDownPt.set(event.getX(), event.getY());

                    if (swipeOffset < SWIPE_THRESHOLD && !stopPanning) {
                        if (!panning(dx, dy, scaleFactor)) {
                            // if we pan, we don't swipe the view pager, consume
                            // event

                            // references would prevent garbage collection of
                            // underlying bitmap!
                            convertView = null;
                            imageView = null;
                            return true;
                        }
                    }
                }
                break;
            default:
                break;
        }
        // references would prevent garbage collection of underlying bitmap!
        convertView = null;
        imageView = null;

        // if panning is at the end of the visible bounds, hand over to swipe
        // event!
        if (swipeEnabled) {
            return super.onTouchEvent(event);
        }
        return true;
    }

    @Override
    public boolean onScale(ScaleGestureDetector detector) {
        float scaleFactor = getScaleFactor(detector.getScaleFactor());

        mScaleCenter.set(detector.getFocusX(), detector.getFocusY());
        zoom(scaleFactor);
        return true;
    }

    @Override
    protected void onPageScrolled(int position, float positionOffset,
                                  int positionOffsetPixels) {
        swipeOffset = positionOffset;
        super.onPageScrolled(position, positionOffset, positionOffsetPixels);
    }

    @Override
    public boolean onScaleBegin(ScaleGestureDetector detector) {
        stopPanning = true;
        return true;
    }

    @Override
    public void onScaleEnd(ScaleGestureDetector detector) {
    }

    /**
     * Get scale factor, check the minimum and maximum scale factor.
     *
     * @param scaleFactor factor to zoom
     * @return current scale factor to be used
     */
    private float getScaleFactor(float scaleFactor) {
        float targetScale = convertView.getScaleX() * scaleFactor;
        targetScale = Math.min(getMaxZoom(),
                Math.max(targetScale, getMinZoom()));
        float currentScaleFactor = Math.min(getMaxZoom(),
                Math.max(targetScale, getMinZoom()));
        return currentScaleFactor;
    }

    private float getPanningBound(int coordinate, float scaleFactor) {
        return (coordinate / scaleFactor) - coordinate;
    }

    private float getMinZoom() {
        return minZoom;
    }

    private float getMaxZoom() {
        return maxZoom;
    }

    PointF containerToImageViewPoint(PointF pt) {
        Matrix matrix = new Matrix();
        matrix.preTranslate(mImageLeftTop.x, mImageLeftTop.y);
        matrix.preScale(mScale, mScale);

        float pts[] = {pt.x, pt.y};
        //matrix.mapPoints(pts);

        PointF retPt = new PointF(pts[0], pts[1]);
        return retPt;
    }

    Point getImgDisplaySize(ImageView iv) {
        Point pt = new Point();
        Drawable imgDrawable = iv.getDrawable();
        if (imgDrawable != null) {
            int dw = imgDrawable.getBounds().width();
            int dh = imgDrawable.getBounds().height();

            Matrix m = iv.getImageMatrix();
            float[] values = new float[10];
            m.getValues(values);

            float sx = values[0];
            float sy = values[4];
            pt.set((int) (dw * sx), (int) (dh * sy));
        }
        return pt;
    }

    void adjustPosition() {
        Point ivSize = new Point(imageView.getWidth(), imageView.getHeight());
        Point bmpSize = getImgDisplaySize(imageView);

        if (mImageLeftTop.x > 0)
            mImageLeftTop.x = 0;
        if (mImageLeftTop.x < ivSize.x - (int) (ivSize.x * mScale))
            mImageLeftTop.x = ivSize.x - (int) (ivSize.x * mScale);

        if (mImageLeftTop.y > 0)
            mImageLeftTop.y = 0;
        if (mImageLeftTop.y < ivSize.y - (int) (ivSize.y * mScale))
            mImageLeftTop.y = ivSize.y - (int) (ivSize.y * mScale);
    }

    /**
     * Panning of the image. Checks, that the image stays in visible bounds
     *
     * @param dx          new X coordinate
     * @param dy          new y coordinate
     * @param scaleFactor new scale factor value
     * @return true (panning successful and within visible bounds), false
     * otherwise
     */
    private boolean panning(float dx, float dy, float scaleFactor) {
        //mScale = scaleFactor;
        mImageLeftTop.x += dx;
        mImageLeftTop.y += dy;

        adjustPosition();

        convertView.setTranslationX(mImageLeftTop.x);
        convertView.setTranslationY(mImageLeftTop.y);
        return true;
    }

    /**
     * Zoom image using scale factor
     *
     * @param scaleFactor zooming factor
     */
    private void zoom(float scaleFactor) {
        convertView.setPivotX(0);
        convertView.setPivotY(0);

        float factor = scaleFactor / mScale;
        float viewFocusX = mScaleCenter.x - mImageLeftTop.x;
        float viewFocusY = mScaleCenter.y - mImageLeftTop.y;
        float dx = viewFocusX - viewFocusX * factor;
        float dy = viewFocusY - viewFocusY * factor;

        mImageLeftTop.x += dx;
        mImageLeftTop.y += dy;

        mScale = scaleFactor;

        adjustPosition();

        convertView.setTranslationX(mImageLeftTop.x);
        convertView.setTranslationY(mImageLeftTop.y);

        convertView.setScaleX(mScale);
        convertView.setScaleY(mScale);

        invalidate();
    }
}
