/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views.util;

import android.content.Context;
import android.database.Cursor;
import android.os.Bundle;
import android.provider.MediaStore;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.utils.FileUtils;
import com.foxit.pdfscan.utils.ResourceUtils;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;
import androidx.loader.app.LoaderManager;
import androidx.loader.content.CursorLoader;
import androidx.loader.content.Loader;

import static android.provider.BaseColumns._ID;
import static android.provider.MediaStore.Images.ImageColumns.BUCKET_DISPLAY_NAME;
import static android.provider.MediaStore.Images.ImageColumns.BUCKET_ID;
import static android.provider.MediaStore.MediaColumns.DATA;
import static android.provider.MediaStore.MediaColumns.MIME_TYPE;
import static android.provider.MediaStore.MediaColumns.SIZE;

public class UilMediaStoreHelper {

    public final static String EXTRA_SHOW_GIF = "SHOW_GIF";
    public final static int INDEX_ALL_PHOTOS = 0;

    public static void getPhotoDirs(FragmentActivity activity, Bundle args, PhotosLoaderCallback resultCallback) {
        LoaderManager.getInstance(activity)
                .initLoader(0, args, new PhotoDirLoaderCallbacks(activity, resultCallback));
    }

    private static class PhotoDirLoaderCallbacks implements LoaderManager.LoaderCallbacks<Cursor> {

        private final Context context;
        private final PhotosLoaderCallback resultCallback;

        public PhotoDirLoaderCallbacks(Context context, PhotosLoaderCallback resultCallback) {
            this.context = context;
            this.resultCallback = resultCallback;
        }

        @NonNull
        @Override
        public Loader<Cursor> onCreateLoader(int id, @Nullable Bundle args) {
            return new UilPhotoDirectoryLoader(context, args != null && args.getBoolean(EXTRA_SHOW_GIF, false));
        }

        @Override
        public void onLoadFinished(@NonNull Loader<Cursor> loader, Cursor data) {
            if (data == null) return;
            List<UilPhotoDirectory> directories = new ArrayList<>();
            UilPhotoDirectory directoryRoot = new UilPhotoDirectory();
            directoryRoot.setId("ALL");
            directoryRoot.setPath("");
            directoryRoot.setName(ResourceUtils.getString(context, R.string.scanner_all_images_title));

            while (data.moveToNext()) {
                long size = data.getInt(data.getColumnIndexOrThrow(SIZE));
                if (size < 1) continue;

                int imageId = data.getInt(data.getColumnIndexOrThrow(_ID));
                String name = data.getString(data.getColumnIndexOrThrow(BUCKET_DISPLAY_NAME));
                String path = data.getString(data.getColumnIndexOrThrow(DATA));
                String bucketId = data.getString(data.getColumnIndexOrThrow(BUCKET_ID));

                UilPhotoDirectory directory = new UilPhotoDirectory();
                directory.setId(bucketId);
                directory.setName(name);

                if (!directories.contains(directory)) {
                    directory.setPath(FileUtils.getFileFolder(path));
                    directory.addPhoto(imageId, path, path, 0);
                    directories.add(directory);
                } else {
                    directories.get(directories.indexOf(directory)).addPhoto(imageId, path, path, 0);
                }

                directoryRoot.addPhoto(imageId, path, path, 0);
            }

            directories.add(INDEX_ALL_PHOTOS, directoryRoot);
            if (resultCallback != null) {
                resultCallback.onLoadFinished(directories);
            }

            LoaderManager.getInstance(((FragmentActivity) this.context)).destroyLoader(loader.getId());
            data.close();
        }

        @Override
        public void onLoaderReset(@NonNull Loader<Cursor> loader) {
        }

    }

    public interface PhotosLoaderCallback {
        void onLoadFinished(List<UilPhotoDirectory> directories);
    }
}

class UilPhotoDirectoryLoader extends CursorLoader {

    final String[] IMAGE_PROJECTION = {
            MediaStore.Images.Media._ID,
            MediaStore.Images.Media.DATA,
            MediaStore.Images.Media.BUCKET_ID,
            MediaStore.Images.Media.BUCKET_DISPLAY_NAME,
            MediaStore.Images.Media.DATE_ADDED,
            MediaStore.Images.Media.SIZE
    };

    public UilPhotoDirectoryLoader(Context context, boolean showGif) {
        super(context);

        setProjection(IMAGE_PROJECTION);
        setUri(MediaStore.Images.Media.EXTERNAL_CONTENT_URI);
        setSortOrder(MediaStore.Images.Media.DATE_ADDED + " DESC");

        setSelection(MIME_TYPE + "=? or " + MIME_TYPE + "=? or " + MIME_TYPE + "=? " + (showGif ? ("or " + MIME_TYPE + "=?") : ""));
        String[] selectionArgs;
        if (showGif) {
            selectionArgs = new String[]{"image/jpeg", "image/png", "image/jpg", "image/gif"};
        } else {
            selectionArgs = new String[]{"image/jpeg", "image/png", "image/jpg"};
        }
        setSelectionArgs(selectionArgs);
    }

}
