/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.common;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.ActionMode;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.note.NoteAnnotContent;
import com.foxit.uiextensions.annots.redaction.UIAnnotRedaction;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.SheetItemBean;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.FragmentActivity;

public class UIAnnotReply {
    public static final int TYPE_COMMENT = 0;
    public static final int TYPE_REPLY = 1;
    public static final int TYPE_DESC = 2;

    @IntDef({
            TYPE_COMMENT,
            TYPE_REPLY,
            TYPE_DESC})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ReplyType {
    }

    public interface ReplyCallback {
        void result(String content);

        String getContent();
    }

    public static void replyToAnnot(final PDFViewCtrl pdfViewCtrl, final ViewGroup parent, final Annot annot) {
        replyToAnnot(pdfViewCtrl, parent, annot, false);
    }

    public static void replyToAnnot(final PDFViewCtrl pdfViewCtrl, final ViewGroup parent, final Annot annot, final boolean isReplyGroup) {
        if (annot == null || pdfViewCtrl.getUIExtensionsManager() == null) {
            return;
        }
        Activity activity = ((UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (activity == null) {
            return;
        }
        if (!(activity instanceof FragmentActivity)) {
            Context context = activity.getApplicationContext();
            UIToast.getInstance(context).show(AppResource.getString(context, R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }
        boolean editable = !AppAnnotUtil.isReadOnly(annot);//true;

        FragmentActivity act = (FragmentActivity) activity;
        ReplyDialog fragment = (ReplyDialog) act.getSupportFragmentManager().findFragmentByTag("ReplyDialog");
        if (fragment == null) {
            fragment = new ReplyDialog();
        }

        fragment.init(pdfViewCtrl, parent, editable, UIAnnotReply.TYPE_REPLY, new ReplyCallback() {

            @Override
            public void result(String content) {
                PDFPage page;
                try {
                    page = annot.getPage();
                    addReplyAnnot(pdfViewCtrl,
                            annot,
                            page,
                            AppDmUtil.randomUUID(null),
                            content,
                            new Event.Callback() {

                                @Override
                                public void result(Event event, boolean success) {
                                    showComments(pdfViewCtrl, parent, annot, isReplyGroup);
                                }
                            });
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public String getContent() {
                return null;
            }
        });

        AppDialogManager.getInstance().showAllowManager(fragment, act.getSupportFragmentManager(), "ReplyDialog", null);
    }

    public static void replyToAnnot(final PDFViewCtrl pdfViewCtrl, final ViewGroup parent, boolean editable, @ReplyType int replyType, ReplyCallback callback) {
        if (callback == null || pdfViewCtrl.getUIExtensionsManager() == null) {
            return;
        }

        Activity activity = ((UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (activity == null) {
            return;
        }
        if (!(activity instanceof FragmentActivity)) {
            Context context = activity.getApplicationContext();
            UIToast.getInstance(context).show(AppResource.getString(context, R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }
        FragmentActivity act = (FragmentActivity) activity;
        ReplyDialog fragment = (ReplyDialog) act.getSupportFragmentManager().findFragmentByTag("ReplyDialog");
        if (fragment == null) {
            fragment = new ReplyDialog();
        }
        fragment.init(pdfViewCtrl, parent, editable, replyType, callback);
        AppDialogManager.getInstance().showAllowManager(fragment, act.getSupportFragmentManager(), "ReplyDialog", null);
    }

    public static void showComments(final PDFViewCtrl pdfViewCtrl, ViewGroup parent, Annot annot) {
        showComments(pdfViewCtrl, parent, annot, false);
    }

    public static void showComments(final PDFViewCtrl pdfViewCtrl, ViewGroup parent, Annot annot, boolean isReplyGroup) {
        if (annot == null || pdfViewCtrl.getUIExtensionsManager() == null) {
            return;
        }
        Activity activity = ((UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (activity == null) {
            return;
        }
        if (!(activity instanceof FragmentActivity)) {
            Context context = activity.getApplicationContext();
            UIToast.getInstance(context).show(AppResource.getString(context, R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }
        FragmentActivity act = (FragmentActivity) activity;
        CommentsFragment fragment = (CommentsFragment) act.getSupportFragmentManager().findFragmentByTag("CommentsFragment");
        if (fragment == null) {
            fragment = new CommentsFragment();
        }
        fragment.init(pdfViewCtrl, parent, annot, isReplyGroup);
        AppDialogManager.getInstance().showAllowManager(fragment, act.getSupportFragmentManager(), "CommentsFragment", null);
    }

    public static void addReplyAnnot(final PDFViewCtrl pdfViewCtrl,
                                     final Annot annot,
                                     final PDFPage page,
                                     final String nm,
                                     final String content,
                                     final Event.Callback callback) {
        try {
            String parentNM = AppAnnotUtil.getAnnotUniqueID(annot);
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
            uiExtensionsManager.getDocumentManager().addAnnot(page,
                    new ReplyContent(page.getIndex(), nm, content, parentNM, uiExtensionsManager.getAnnotAuthor()), true, callback);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public static class CommentContent implements AnnotContent {

        String content;
        Annot annot;

        public CommentContent(Annot annot, String content) {
            this.annot = annot;
            this.content = content;
        }

        @Override
        public int getPageIndex() {
            try {
                return annot.getPage().getIndex();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public int getType() {
            try {
                return annot.getType();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return Annot.e_UnknownType;
        }

        @Override
        public String getNM() {
            return AppAnnotUtil.getAnnotUniqueID(annot);
        }

        @Override
        public RectF getBBox() {
            try {
                return AppUtil.toRectF(annot.getRect());
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public int getColor() {
            try {
                return annot.getBorderColor();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public int getFillColor() {
            return 0;
        }

        @Override
        public int getOpacity() {
            try {
                return (int) (((Markup) annot).getOpacity() * 255f + 0.5f);
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public float getLineWidth() {
            try {
                if (annot.getBorderInfo() != null) {
                    return annot.getBorderInfo().getWidth();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public String getSubject() {
            try {
                return ((Markup) annot).getSubject();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public DateTime getModifiedDate() {
            return AppDmUtil.currentDateToDocumentDate();
        }

        @Override
        public String getContents() {
            return content;
        }

        @Override
        public String getIntent() {
            try {
                return ((Markup) annot).getIntent();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public String getAuthor() {
            return null;
        }
    }

    static class ReplyContent implements NoteAnnotContent {
        int pageIndex;
        String nm;
        String content;
        String parentNM;
        String author;

        public ReplyContent(int pageIndex, String nm, String content, String parentNM, String author) {
            this.pageIndex = pageIndex;
            this.nm = nm;
            this.content = content;
            this.parentNM = parentNM;
            this.author = author;
        }

        @Override
        public String getIcon() {
            return "";
        }

        @Override
        public String getFromType() {
            return Module.MODULE_NAME_REPLY;
        }

        @Override
        public String getParentNM() {
            return parentNM;
        }

        @Override
        public int getPageIndex() {
            return pageIndex;
        }

        @Override
        public int getType() {
            return Annot.e_Note;
        }

        @Override
        public String getNM() {
            return nm;
        }

        @Override
        public RectF getBBox() {
            return new RectF();
        }

        @Override
        public int getColor() {
            return 0;
        }

        @Override
        public int getFillColor() {
            return 0;
        }

        @Override
        public int getOpacity() {
            return 0;
        }

        @Override
        public float getLineWidth() {
            return 0;
        }

        @Override
        public String getSubject() {
            return null;
        }

        @Override
        public DateTime getModifiedDate() {
            return AppDmUtil.currentDateToDocumentDate();
        }

        @Override
        public String getContents() {
            return content;
        }

        @Override
        public String getIntent() {
            return null;
        }

        @Override
        public String getAuthor() {
            return author;
        }

    }

    public static class ReplyDialog extends DialogFragment {

        private EditText mEditText;
        private Context mContext;
        private PDFViewCtrl mPDFViewerCtrl;
        private ViewGroup mParent;
        private ReplyCallback mCallback;
        @ReplyType
        private int mReplyType;
        private boolean mDialogEditable = false;
        private int mNightMode = -1;

        public void init(PDFViewCtrl pdfViewCtrl, ViewGroup parent, boolean dialogEditable, @ReplyType int replyType, ReplyCallback callback) {
            mPDFViewerCtrl = pdfViewCtrl;
            mParent = parent;
            mDialogEditable = dialogEditable;
            mReplyType = replyType;
            mCallback = callback;
        }

        @Override
        public void onCreate(Bundle savedInstanceState) {
            mContext = this.getActivity().getApplicationContext();
            mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
            super.onCreate(savedInstanceState);
        }

        @Override
        public void onAttach(@NonNull Context context) {
            super.onAttach(context);
            if (mCallback == null) {
                ReplyDialog.this.dismiss();
                return;
            }
            setCancelable(true);
        }

        @Override
        public void onDetach() {
            super.onDetach();
            if (SystemUiHelper.getInstance().isFullScreen()) {
                if (AppDisplay.isPad())
                    SystemUiHelper.getInstance().hideSystemUI(getActivity());
                else
                    SystemUiHelper.getInstance().hideStatusBar(getActivity());
            }
            if (mEditText != null)
                AppUtil.dismissInputSoft(mEditText);
            AppDialogManager.getInstance().remove(this);
        }

        @NonNull
        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            if (mCallback == null || getActivity() == null) {
                return super.onCreateDialog(savedInstanceState);
            }
            Dialog dialog = new Dialog(getActivity(), R.style.rv_dialog_style);

            dialog.setContentView(createView());
            dialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
            dialog.getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);

            WindowManager.LayoutParams params = dialog.getWindow().getAttributes();
            params.width = AppDisplay.getUITextEditDialogWidth();
            dialog.getWindow().setAttributes(params);

            dialog.setCanceledOnTouchOutside(true);
            return dialog;
        }

        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            super.onConfigurationChanged(newConfig);

            if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
                dismiss();
                return;
            }
            if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                mEditText.setMaxLines(5);
            } else {
                mEditText.setMaxLines(10);
            }
        }

        @Override
        public void onActivityCreated(Bundle arg0) {
            super.onActivityCreated(arg0);
        }

        @TargetApi(Build.VERSION_CODES.HONEYCOMB)
        public View createView() {
            View view = View.inflate(mContext, R.layout.rd_note_dialog_edit, null);
            String title;
            if (TYPE_COMMENT == mReplyType)
                title = AppResource.getString(mContext, R.string.fx_string_note);
            else if (TYPE_DESC == mReplyType)
                title = AppResource.getString(mContext, R.string.rv_panel_edit_desc);
            else if (TYPE_REPLY == mReplyType)
                title = AppResource.getString(mContext, R.string.fx_string_reply);
            else
                title = AppResource.getString(mContext, R.string.fx_string_note);
            ((TextView) view.findViewById(R.id.rd_note_dialog_edit_title)).setText(title);
            mEditText = view.findViewById(R.id.rd_note_dialog_edit);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            Button bt_close = view.findViewById(R.id.rd_note_dialog_edit_cancel);
            final Button bt_ok = view.findViewById(R.id.rd_note_dialog_edit_ok);
            final String content = mCallback.getContent() == null ? "" : mCallback.getContent();

            mEditText.setText(content);
            if (!mDialogEditable) {
                bt_ok.setEnabled(false);

                mEditText.setFocusable(false);
                mEditText.setLongClickable(false);
                if (Build.VERSION.SDK_INT >= 11) {
                    mEditText.setCustomSelectionActionModeCallback(new ActionMode.Callback() {

                        @Override
                        public boolean onPrepareActionMode(ActionMode mode, Menu menu) {
                            return false;
                        }

                        @Override
                        public void onDestroyActionMode(ActionMode mode) {
                        }

                        @Override
                        public boolean onCreateActionMode(ActionMode mode, Menu menu) {
                            return false;
                        }

                        @Override
                        public boolean onActionItemClicked(ActionMode mode, MenuItem item) {
                            return false;
                        }
                    });
                } else {
                    mEditText.setEnabled(false);
                }
            } else {
                mEditText.setEnabled(true);
                bt_ok.setEnabled(mReplyType == TYPE_REPLY);

                mEditText.setCursorVisible(true);
                mEditText.setFocusable(true);
                mEditText.setSelection(content.length());
                AppUtil.showSoftInput(mEditText);

                mEditText.addTextChangedListener(new TextWatcher() {
                    @Override
                    public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                    }

                    @Override
                    public void onTextChanged(CharSequence s, int start, int before, int count) {
                    }

                    @Override
                    public void afterTextChanged(Editable s) {
                        if (mReplyType != TYPE_REPLY) {
                            bt_ok.setEnabled(!content.equals(s.toString()));
                        }
                    }
                });
            }
            bt_close.setOnClickListener(new OnClickListener() {

                @Override
                public void onClick(View v) {
                    if (((UIExtensionsManager) mPDFViewerCtrl.getUIExtensionsManager()).getDocumentManager().canAddAnnot()
                            && ((UIExtensionsManager) mPDFViewerCtrl.getUIExtensionsManager()).isEnableModification()
                    ) {
                        AppUtil.dismissInputSoft(mEditText);
                    }
                    AppDialogManager.getInstance().dismiss(ReplyDialog.this);
                }
            });
            bt_ok.setOnClickListener(new OnClickListener() {

                @Override
                public void onClick(View v) {
                    AppUtil.dismissInputSoft(mEditText);
                    AppDialogManager.getInstance().dismiss(ReplyDialog.this);
                    if (mCallback != null && mEditText.getText() != null) {
                        String Content = mEditText.getText().toString().trim();
                        if (!Content.equals(mCallback.getContent())) {
                            mCallback.result(Content);
                        }
                    }
                }
            });

            if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                mEditText.setMaxLines(5);
            } else {
                mEditText.setMaxLines(10);
            }

            RelativeLayout.LayoutParams editParams = (RelativeLayout.LayoutParams) mEditText.getLayoutParams();
            editParams.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
            mEditText.setLayoutParams(editParams);

            ViewGroup.LayoutParams params = new ViewGroup.LayoutParams(AppDisplay.getUITextEditDialogWidth(), ViewGroup.LayoutParams.WRAP_CONTENT);
            view.setLayoutParams(params);
            return view;
        }
    }

    public static class CommentsFragment extends DialogFragment {
        private Markup mAnnot;
        private Context mContext;
        private PDFViewCtrl mPDFViewCtrl;
        private final CommentsAdapter mAdapter;
        private final List<AnnotNode> mCheckedNodes = new ArrayList<>();

        private TextView mReplyClear;
        private ProgressDialog mDeleteDialog;
        private AnnotNode mRootNode;
        private UITextEditDialog mSRDeleteDialog;
        private View mDialogContentView;
        private UITextEditDialog mClearDialog;

        //        private final ArrayList<Boolean> mItemMoreViewShow;
        private boolean isTouchHold;
        private ViewGroup mParent;

        private boolean mIsReplyGroup;

        public CommentsFragment() {
//            mItemMoreViewShow = new ArrayList<Boolean>();
            mAdapter = new CommentsAdapter();
            mContext = null;
        }

        @Override
        public void onAttach(@NonNull Context context) {
            super.onAttach(context);

            if (mAdapter == null) {
                CommentsFragment.this.dismiss();
                return;
            }

            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    init();
                }
            });
        }

        private void init() {
            if (mAnnot == null) {
                CommentsFragment.this.dismiss();
                return;
            }

            mAdapter.clearNodes();
            mCheckedNodes.clear();

            try {
                PDFPage page = mAnnot.getPage();
                int pageIndex = page.getIndex();
                int count = page.getAnnotCount();
                DocumentManager documentManager = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getDocumentManager();
                boolean canAddAnnot = documentManager.canAddAnnot() && ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).isEnableModification();
                boolean isReadOnly = AppAnnotUtil.isReadOnly(mAnnot);
                boolean isLocked = AppAnnotUtil.isLocked(mAnnot);

                for (int i = 0; i < count; i++) {
                    Annot annot = AppAnnotUtil.createAnnot(page.getAnnot(i));

                    if (annot == null || annot.isEmpty() || AppAnnotUtil.isSupportGroupElement(annot) ||
                            !AppAnnotUtil.isSupportEditAnnot(annot))
                        continue;

                    if (annot.getType() == Annot.e_Screen) continue;

                    String name = AppAnnotUtil.getAnnotUniqueID(annot);
                    if (AppUtil.isEmpty(name)) {
                        continue;
                    }
                    String mName = AppAnnotUtil.getAnnotUniqueID(mAnnot);
                    boolean isMatch = name.equals(mName);
                    if (!AppAnnotUtil.isSupportEditAnnot(annot) || isMatch) {
                        continue;
                    }

                    Markup dmAnnot = (Markup) annot;

                    AnnotNode node = new AnnotNode(pageIndex, dmAnnot, AppAnnotUtil.getReplyToAnnot(dmAnnot));
                    node.setAuthor(dmAnnot.getTitle());
                    node.setContent(dmAnnot.getContent());
                    String date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, dmAnnot.getModifiedDateTime());

                    if (date == null || date.equals(AppDmUtil.dateOriValue)) {
                        date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, dmAnnot.getCreationDateTime());
                    }
                    node.setDate(date);
                    node.setCreationDate(AppDmUtil.getLocalDateString(dmAnnot.getCreationDateTime()));
                    node.setModifyDate(AppDmUtil.getLocalDateString(dmAnnot.getModifiedDateTime()));
                    node.setType(dmAnnot.getType());
                    boolean _isReadOnly = AppAnnotUtil.isReadOnly(dmAnnot);
                    boolean _isLocked = AppAnnotUtil.isLocked(dmAnnot);
                    boolean enable = canAddAnnot && !_isReadOnly;
                    node.setEditEnable(enable);
                    if (dmAnnot.getType() == Annot.e_Redact) {
                        boolean hasLicense = AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction);
                        node.setApplyRedaction(hasLicense && !dmAnnot.isEmpty());
                        node.setDeletable(hasLicense && enable && !_isLocked);
                    } else {
                        node.setDeletable(enable && !_isLocked);
                    }
                    node.setCanReply(AppAnnotUtil.isAnnotSupportReply(dmAnnot) && !_isReadOnly);
                    node.setCanComment(!(_isLocked || _isReadOnly));
                    node.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(documentManager, dmAnnot));
                    node.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(documentManager, dmAnnot));
                    node.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(documentManager, annot));
                    mAdapter.addNode(node);
                }

                mRootNode = new AnnotNode(pageIndex, mAnnot, AppAnnotUtil.getReplyToAnnot(mAnnot));
                mRootNode.setAuthor(mAnnot.getTitle());
                mRootNode.setContent(mAnnot.getContent());
                String date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, mAnnot.getModifiedDateTime());

                if (date == null || date.equals(AppDmUtil.dateOriValue)) {
                    date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, mAnnot.getCreationDateTime());
                }
                mRootNode.setDate(date);
                mRootNode.setCreationDate(AppDmUtil.getLocalDateString(mAnnot.getCreationDateTime()));
                mRootNode.setModifyDate(AppDmUtil.getLocalDateString(mAnnot.getModifiedDateTime()));
                mRootNode.setEditEnable(canAddAnnot && !isReadOnly);
                mRootNode.setCanComment(!(isLocked || isReadOnly));
                mRootNode.setCanReply(AppAnnotUtil.isAnnotSupportReply(mAnnot) && !isReadOnly);
                mRootNode.setType(mAnnot.getType());
                mRootNode.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(documentManager, mAnnot));
                mRootNode.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(documentManager, mAnnot));
                mRootNode.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(documentManager, mAnnot));
                if (mAnnot.getType() == Annot.e_Redact) {
                    boolean hasLicense = AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction);
                    mRootNode.setApplyRedaction(hasLicense && mAnnot != null && !mAnnot.isEmpty());
                    mRootNode.setDeletable(hasLicense && canAddAnnot && !(isReadOnly || isLocked));
                    mReplyClear.setEnabled(hasLicense && canAddAnnot && !(isReadOnly || isLocked) && mRootNode.isWithDeletePermission());
                } else {
                    mRootNode.setDeletable(canAddAnnot && !(isReadOnly || isLocked));
                    mReplyClear.setEnabled(canAddAnnot && !(isReadOnly || isLocked) && mRootNode.isWithDeletePermission());
                }

                mAdapter.addNode(mRootNode);
                mAdapter.establishReplyList(mRootNode);

                mAdapter.notifyDataSetChanged();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        public boolean canEdit(@NonNull AnnotNode node) {
            if (node.isRoot()) {
                return node.isEditEnable();
            } else {
                return node.isEditEnable() && canEdit(node.getParent());
            }
        }

        public boolean canDelete(@NonNull AnnotNode node) {
            if (canReply(node)) return node.canDelete();
            if (node.isRoot()) {
                return node.canDelete();
            } else {
                return node.canDelete() && canDelete(node.getParent());
            }
        }

        public boolean canReply(@NonNull AnnotNode node) {
            if (node.isRoot()) {
                return node.canReply();
            } else {
                return node.canReply() && canReply(node.getParent());
            }
        }

        public boolean canComment(@NonNull AnnotNode node) {
            if (node.isRoot()) {
                return node.canComment();
            } else {
                return node.canComment() && canComment(node.getParent());
            }
        }

        public void init(PDFViewCtrl pdfViewCtrl, ViewGroup parent, Annot dmAnnot, boolean isReplyGroup) {
            mPDFViewCtrl = pdfViewCtrl;
            mParent = parent;
            mAnnot = (Markup) dmAnnot;
            mAdapter.clearNodes();
            mCheckedNodes.clear();
            mIsReplyGroup = isReplyGroup;
            mAdapter.setIsReplyGroup(mIsReplyGroup);
        }

        @SuppressLint("NewApi")
        @Override
        public void onCreate(Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
            mContext = this.getActivity().getApplicationContext();

            if (!AppDisplay.isPad()) {
                int theme;
                if (SystemUiHelper.getInstance().isStatusBarShown(getActivity())) {
                    theme = AppTheme.getThemeNoTitle();
                } else {
                    theme = AppTheme.getThemeFullScreen();
                }
                setStyle(STYLE_NO_TITLE, theme);

                SystemUiHelper.getInstance().showNavigationBar(getActivity());
            }
            ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).stopHideToolbarsTimer();
        }

        @NonNull
        @SuppressLint("NewApi")
        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            if (mAdapter == null) {
                return super.onCreateDialog(savedInstanceState);
            }
            if (AppDisplay.isPad()) {
                int theme;
                if (Build.VERSION.SDK_INT >= 21) {
                    theme = android.R.style.Theme_Holo_Light_Dialog_NoActionBar;
                } else if (Build.VERSION.SDK_INT >= 14) {
                    theme = android.R.style.Theme_DeviceDefault_Light_Dialog_NoActionBar;
                } else if (Build.VERSION.SDK_INT >= 11) {
                    theme = android.R.style.Theme_Holo_Light_Dialog_NoActionBar;
                } else {
                    theme = R.style.rv_dialog_style;
                }
                Dialog dialog = new Dialog(getActivity(), theme);
                int width = AppDisplay.getDialogWidth();
                int height = AppDisplay.getDialogHeight();
                dialog.setContentView(createView());
                WindowManager.LayoutParams params = dialog.getWindow().getAttributes();
                params.width = width;
                params.height = height;
                dialog.getWindow().setAttributes(params);
                dialog.setCanceledOnTouchOutside(true);

                dialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
                dialog.getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
                return dialog;
            }
            return super.onCreateDialog(savedInstanceState);
        }

        @Override
        public void onConfigurationChanged(@NonNull Configuration newConfig) {
            super.onConfigurationChanged(newConfig);
            if (AppDisplay.isPad() && getDialog() != null) {
                if (getDialog().getWindow() != null) {
                    int width = AppDisplay.getDialogWidth();
                    int height = AppDisplay.getDialogHeight();
                    WindowManager.LayoutParams params = getDialog().getWindow().getAttributes();
                    params.width = width;
                    params.height = height;
                    getDialog().getWindow().setAttributes(params);
                }
            }
        }

        private View createView() {
            mDialogContentView = View.inflate(mContext, R.layout.annot_reply_main, null);

            TextView titleTv = mDialogContentView.findViewById(R.id.annot_reply_list_title);
            if (mAnnot != null && !mAnnot.isEmpty()) {
                try {
                    titleTv.setText(mAnnot.getTitle());
                } catch (PDFException e) {
                    titleTv.setText(AppResource.getString(mContext, R.string.rd_comment_reply_list));
                }
            } else {
                titleTv.setText(AppResource.getString(mContext, R.string.rd_comment_reply_list));
            }
            TextView replyBack = mDialogContentView.findViewById(R.id.annot_reply_back);
            replyBack.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            ListView listView = mDialogContentView.findViewById(R.id.annot_reply_list);
            mReplyClear = mDialogContentView.findViewById(R.id.annot_reply_clear);
            mReplyClear.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mReplyClear.setEnabled(false);
            mDialogContentView.setOnTouchListener(new View.OnTouchListener() {

                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    return true;
                }
            });
            mReplyClear.setOnClickListener(new OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) return;

                    if (mPDFViewCtrl.getUIExtensionsManager() == null) return;
                    Context context = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getAttachedActivity();
                    if (context == null) return;
                    mClearDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                    mClearDialog.setTitle(AppResource.getString(mContext, R.string.hm_clear));
                    mClearDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.rv_panel_annot_delete_tips));
                    mClearDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            if (AppUtil.isFastDoubleClick()) return;

                            mAdapter.resetCheckedNodes();
                            Collections.sort(mCheckedNodes);
                            mClearDialog.dismiss();

                            mDeleteDialog = new ProgressDialog(getActivity());
                            mDeleteDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
                            mDeleteDialog.setCancelable(false);
                            mDeleteDialog.setIndeterminate(false);
                            mDeleteDialog.setMessage(AppResource.getString(mContext, R.string.rv_panel_annot_deleting));
                            mDeleteDialog.show();
                            deleteItems();
                        }
                    });
                    mClearDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mClearDialog.dismiss();
                        }
                    });
                    mClearDialog.show();
                }
            });

            listView.setAdapter(mAdapter);
//            listView.setOnTouchListener(new View.OnTouchListener() {
//                @Override
//                public boolean onTouch(View v, MotionEvent event) {
//                    int action = MotionEventCompat.getActionMasked(event);
//                    switch (action) {
//                        case MotionEvent.ACTION_DOWN:
//                            boolean show = false;
//                            int position = 0;
//                            for (int i = 0; i < mAdapter.getCount(); i++) {
//                                if (mItemMoreViewShow.get(i)) {
//                                    show = true;
//                                    position = i;
//                                    break;
//                                }
//                            }
//                            if (show) {
//                                mItemMoreViewShow.set(position, false);
//                                mAdapter.notifyDataSetChanged();
//                                isTouchHold = true;
//                                return true;
//                            }
//
//                        case MotionEvent.ACTION_UP:
//                        case MotionEvent.ACTION_CANCEL:
//                            if (isTouchHold) {
//                                isTouchHold = false;
//                                return true;
//                            }
//                        default:
//                    }
//                    return false;
//                }
//            });


            mAdapter.notifyDataSetChanged();
            replyBack.setOnClickListener(new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    AppDialogManager.getInstance().dismiss(CommentsFragment.this);
                }
            });
            return mDialogContentView;
        }

        @Override
        public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
            if (AppDisplay.isPad()) {
                return super.onCreateView(inflater, container, savedInstanceState);
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                if (getDialog() != null && getActivity() != null) {
                    if (getActivity().getWindow() != null
                            && getActivity().getWindow().getStatusBarColor() != getDialog().getWindow().getStatusBarColor()) {
                        SystemUiHelper.getInstance().setStatusBarColor(getDialog().getWindow(), getActivity().getWindow().getStatusBarColor());
                    }
                }
            }
            return createView();
        }

        @Override
        public void onDetach() {
            super.onDetach();
            mCheckedNodes.clear();
            mAdapter.clearNodes();
            if (mSRDeleteDialog != null && mSRDeleteDialog.getDialog().isShowing())
                mSRDeleteDialog.dismiss();
            if (SystemUiHelper.getInstance().isFullScreen()) {
                if (AppDisplay.isPad())
                    SystemUiHelper.getInstance().hideSystemUI(getActivity());
                else
                    SystemUiHelper.getInstance().hideStatusBar(getActivity());
            }
            ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).startHideToolbarsTimer();
            mSRDeleteDialog = null;
            resetDeleteDialog();
            AppDialogManager.getInstance().remove(this);
        }

        private void beginToDelete() {
            if (checkDeleteStatus() == DELETE_SRCAN) {
                if (mSRDeleteDialog == null || mSRDeleteDialog.getDialog().getOwnerActivity() == null) {
                    mSRDeleteDialog = new UITextEditDialog(getActivity(), UIDialog.NO_INPUT);
                    mSRDeleteDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.rv_panel_annot_delete_tips));
                    mSRDeleteDialog.setTitle(AppResource.getString(mContext, R.string.cloud_delete_tv));
                }
                mSRDeleteDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

                    @Override
                    public void onClick(View v) {
                        mSRDeleteDialog.dismiss();
                        mDeleteDialog = new ProgressDialog(getActivity());
                        mDeleteDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
                        mDeleteDialog.setCancelable(false);
                        mDeleteDialog.setIndeterminate(false);
                        mDeleteDialog.setMessage(AppResource.getString(mContext, R.string.rv_panel_annot_deleting));
                        mDeleteDialog.show();
                        deleteItems();
                    }
                });
                mSRDeleteDialog.show();
            } else {
                mDeleteDialog = new ProgressDialog(getActivity());
                mDeleteDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
                mDeleteDialog.setCancelable(false);
                mDeleteDialog.setIndeterminate(false);
                mDeleteDialog.setMessage(AppResource.getString(mContext, R.string.rv_panel_annot_deleting));
                mDeleteDialog.show();
                deleteItems();
            }
        }

        private void resetDeleteDialog() {
            if (mDeleteDialog != null) {
                if (mDeleteDialog.isShowing()) {
                    AppDialogManager.getInstance().dismiss(mDeleteDialog);
                }
                mDeleteDialog = null;
            }
        }

        private void deleteItems() {
            int size = mCheckedNodes.size();
            if (size == 0) {
                resetDeleteDialog();
                notifyCounter();
                mAdapter.notifyDataSetChanged();
                return;
            }

            AnnotNode node = mCheckedNodes.get(size - 1);
            if (node == null) {
                mCheckedNodes.remove(node);
                deleteItems();
                return;
            }

            if (!canDelete(node)) {
                node.setChecked(false);
                mCheckedNodes.remove(node);
                notifyCounter();
                deleteItems();
                return;
            }
            mAdapter.removeNode(node);
        }

        private void flattenItems() {
            int size = mCheckedNodes.size();
            if (size == 0) {
                resetDeleteDialog();
                notifyCounter();
                mAdapter.notifyDataSetChanged();
                return;
            }
            AnnotNode node = mCheckedNodes.get(size - 1);
            if (node == null) {
                mCheckedNodes.remove(node);
                flattenItems();
                return;
            }
            mAdapter.flattenNode(node);
        }

        private void redactionItems() {
            int size = mCheckedNodes.size();
            if (size == 0) {
                notifyCounter();
                mAdapter.notifyDataSetChanged();
                return;
            }
            AnnotNode node = mCheckedNodes.get(size - 1);
            if (node == null) {
                mCheckedNodes.remove(node);
                redactionItems();
                return;
            }
            mAdapter.redactionNode(node);
        }

        private static final int DELETE_CAN = 0;
        private static final int DELETE_SRCAN = 1;

        private int checkDeleteStatus() {
            return DELETE_CAN;
        }

        private void notifyCounter() {
            if (mAdapter.getCount() > 0) {

                mReplyClear.setVisibility(View.VISIBLE);

            } else {

                mReplyClear.setEnabled(false);
            }
        }

        class CommentsAdapter extends BaseAdapter {
            private final List<AnnotNode> mNodes;
            private final List<AnnotNode> mNodesTmp;

            //            private final ArrayList<Boolean> mMoreViewShow;
            private boolean mIsReplyGroup;
            private ISheetMenu mSheetMenu;
            private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangeListener;
            private View mSheetMenuAnchorView;
            private AnnotNode mSheetMenuNote;

            public CommentsAdapter() {
                mNodes = new ArrayList<>();
                mNodesTmp = new ArrayList<>();
//                mMoreViewShow = moreViewShow;
            }

            public void setIsReplyGroup(boolean isReplyGroup) {
                mIsReplyGroup = isReplyGroup;
            }

            public void removeNode(final AnnotNode node) {
                if (node.getChildren() != null) {
                    node.clearChildren();
                }

                try {
                    ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getDocumentManager().removeAnnot(node.replyAnnot, mIsReplyGroup, true, null);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                mNodesTmp.remove(node);
                if (node.getParent() != null) {
                    node.getParent().removeChildNode(node);
                }
                mCheckedNodes.remove(node);
                notifyCounter();
                deleteItems();

                establishReplyList(mRootNode);
                notifyDataSetChanged();
                if (node.equals(mRootNode)) {
                    AppDialogManager.getInstance().dismiss(CommentsFragment.this);
                }
            }

            private void flattenNode(final AnnotNode node) {
                UIAnnotFlatten.flattenAnnot(mPDFViewCtrl, node.replyAnnot, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            if (node.getChildren() != null) {
                                node.clearChildren();
                            }

                            mNodesTmp.remove(node);
                            if (node.getParent() != null) {
                                node.getParent().removeChildNode(node);
                            }
                            mCheckedNodes.remove(node);
                            notifyCounter();
                            flattenItems();

                            establishReplyList(mRootNode);
                            notifyDataSetChanged();
                            if (node.equals(mRootNode)) {
                                AppDialogManager.getInstance().dismiss(CommentsFragment.this);
                            }
                        }
                    }
                });
            }

            private void redactionNode(final AnnotNode node) {
                UIAnnotRedaction.apply(mPDFViewCtrl, node.replyAnnot, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            if (node.getChildren() != null) {
                                node.clearChildren();
                            }

                            mNodesTmp.remove(node);
                            if (node.getParent() != null) {
                                node.getParent().removeChildNode(node);
                            }
                            mCheckedNodes.remove(node);
                            notifyCounter();
                            redactionItems();

                            establishReplyList(mRootNode);
                            notifyDataSetChanged();
                            if (node.equals(mRootNode)) {
                                AppDialogManager.getInstance().dismiss(CommentsFragment.this);
                            }
                        }
                    }
                });
            }

            private void resetCheckedNodes() {
                mCheckedNodes.clear();
                for (int i = 0; i < mNodes.size(); i++) {
                    AnnotNode node = mNodes.get(i);
                    if (!mCheckedNodes.contains(node) && (node.replyToAnnot == null || node.replyToAnnot.isEmpty())) {
                        mCheckedNodes.add(node);
                    }
                }
                notifyCounter();
            }

            void clearNodes() {
                mNodes.clear();
                mNodesTmp.clear();
            }

            private void establishReplyList(AnnotNode node) {
                mNodes.clear();
                int index = mNodesTmp.indexOf(node);
                if (index == -1) return;
                AnnotNode n = mNodesTmp.get(index);
                mNodes.add(n);
                establishNodeRoot(n);

//                mMoreViewShow.clear();
//                for (int i = 0; i < mNodes.size(); i++) {
//                    mMoreViewShow.add(false);
//                }
            }

            private void addNode(AnnotNode node) {
                if (mNodesTmp.contains(node)) return;
                if ((node.replyAnnot == null || node.replyAnnot.isEmpty()) && (node.replyToAnnot == null || node.replyToAnnot.isEmpty())) {
                    return;
                }
                boolean needFind = (node.replyToAnnot != null && !node.replyToAnnot.isEmpty());
                for (AnnotNode an : mNodesTmp) {
                    if (needFind) {
                        if (AppAnnotUtil.getAnnotUniqueID(an.replyAnnot).equals(AppAnnotUtil.getAnnotUniqueID(node.replyToAnnot))) {
                            node.setParent(an);
                            an.addChildNode(node);
                            needFind = false;
                            continue;
                        }
                    }
                    if (an.replyToAnnot != null && !an.replyToAnnot.isEmpty()
                            && AppAnnotUtil.getAnnotUniqueID(node.replyAnnot).equals(AppAnnotUtil.getAnnotUniqueID(an.replyToAnnot))) {
                        an.setParent(node);
                        node.addChildNode(an);
                    }
                }
                mNodesTmp.add(node);
            }

            private void establishNodeRoot(AnnotNode parent) {
                if (parent == null) return;
                if (parent.isLeafNode() || !parent.isExpanded()) return;
                if (parent.getChildren() == null) return;
                if (parent.getChildren().size() > 1) {
                    Collections.sort(parent.getChildren());
                }
                for (AnnotNode child : parent.getChildren()) {
                    mNodes.add(child);
                    establishNodeRoot(child);
                }
            }

            @Override
            public int getCount() {
                return mNodes.size();
            }

            @Override
            public Object getItem(int position) {
                return mNodes.get(position);
            }

            @Override
            public long getItemId(int position) {
                return position;
            }

            @Override
            public View getView(int position, View convertView, ViewGroup parent) {
                ViewHolder holder;
                if (convertView == null) {
                    holder = new ViewHolder();
                    convertView = View.inflate(mContext, R.layout.annot_reply_item, null);
                    holder.mReplyRoot = convertView.findViewById(R.id.annot_reply_top_layout);
                    holder.mReplyListRL = convertView.findViewById(R.id.annot_reply_list_rl);
                    holder.mAuthorTextView = convertView.findViewById(R.id.annot_reply_author_tv);
                    holder.mContentsTextView = convertView.findViewById(R.id.annot_reply_contents_tv);
                    holder.mDateTextView = convertView.findViewById(R.id.annot_reply_date_tv);
                    holder.mIconImageView = convertView.findViewById(R.id.annot_iv_reply_icon);
                    holder.mExpandImageView = convertView.findViewById(R.id.annot_reply_expand_iv);

                    holder.mIv_relist_more = convertView.findViewById(R.id.rd_annot_relist_item_more);
//                    holder.mLl_relist_moreview = convertView.findViewById(R.id.rd_annot_relist_item_moreview);

//                    holder.mLl_relist_reply = convertView.findViewById(R.id.rd_annot_relist_item_ll_reply);
//                    holder.mLl_relist_comment = convertView.findViewById(R.id.rd_annot_item_ll_comment);
//                    holder.mLl_relist_apply = convertView.findViewById(R.id.rd_annot_item_ll_redaction);
//                    holder.mLl_relist_flatten = convertView.findViewById(R.id.rd_annot_item_ll_flatten);
//                    holder.mLl_relist_delete = convertView.findViewById(R.id.rd_annot_item_ll_delete);

                    convertView.setTag(holder);
                } else {
                    holder = (ViewHolder) convertView.getTag();
                }
                ThemeUtil.setTintList(holder.mIv_relist_more, ThemeUtil.getEnableIconColor(mContext));
                ThemeUtil.setTintList(holder.mExpandImageView, ThemeUtil.getEnableIconColor(mContext));
                ThemeUtil.setTintList(holder.mIconImageView, ThemeUtil.getEnableIconColor(mContext));

                final AnnotNode node = mNodes.get(position);
//                LinearLayout.LayoutParams params = (LinearLayout.LayoutParams) holder.mContentsTextView.getLayoutParams();

                if (node.isRoot()) {
                    int drawable;
                    if (JsonConstants.TYPE_REDACTION.equals(AppAnnotUtil.getTypeString(mAnnot))) {
                        drawable = R.drawable.more_menu_redaction;
                    } else {
                        drawable = AppAnnotUtil.getIconId(AppAnnotUtil.getTypeString(mAnnot));
                    }
                    holder.mIconImageView.setImageResource(drawable);
                } else {
                    holder.mIconImageView.setImageResource(R.drawable.panel_annot_reply_arrow_small);
                }
                if (node.isRoot()) {
                    holder.mExpandImageView.setVisibility(View.GONE);

//                    params.leftMargin = AppDisplay.dp2px(0);
                } else if (node.getLevel() == 1 && !node.isLeafNode()) {
                    holder.mExpandImageView.setVisibility(View.VISIBLE);
//                    params.leftMargin = AppDisplay.dp2px(24 + 24 + 5);
                    if (node.isExpanded()) {
                        holder.mExpandImageView.setImageResource(R.drawable.ic_annot_reply_item_minus);
                    } else {
                        holder.mExpandImageView.setImageResource(R.drawable.ic_annot_reply_item_add);
                    }
                    holder.mExpandImageView.setOnClickListener(new OnClickListener() {

                        @Override
                        public void onClick(View v) {
                            node.setExpanded(!node.isExpanded());
                            establishReplyList(mRootNode);
                            notifyDataSetChanged();
                        }
                    });
                } else {
                    holder.mExpandImageView.setVisibility(View.INVISIBLE);
                    holder.mIconImageView.setVisibility(View.VISIBLE);
//                    params.leftMargin = AppDisplay.dp2px(24 + 5);
                }
//                holder.mContentsTextView.setLayoutParams(params);

                final int level = Math.min(node.getLevel(), 2);
//                int ax = AppDisplay.dp2px(32);//annotIconWidth
//                int px = AppDisplay.dp2px(5);//paddingWidth
//                int ex = AppDisplay.dp2px(21);//expandWidth
//                int dx = AppDisplay.dp2px(24);//replyIconWidth
//                if (level == 0) {
//                    convertView.setPadding(0, 0, 0, 0);
//                } else if (level == 1) {
//                    if (node.isLeafNode()) {
//                        convertView.setPadding((ax + px) + (dx + px) * (level - 1), 0, 0, 0);
//                    } else {
//                        convertView.setPadding((ax + px - ex) + (dx + px) * (level - 1), 0, 0, 0);
//                    }
//                } else {
//                    convertView.setPadding((ax + px) + (dx + px) * (level - 1), 0, 0, 0);
//                }

                holder.mDateTextView.setText(node.getDate());
                holder.mAuthorTextView.setText(node.getAuthor());
                holder.mContentsTextView.setText(node.getContent());
                final UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
                if (mSheetMenu == null) {
                    mSheetMenu = UISheetMenu.newInstance((FragmentActivity) extensionsManager.getAttachedActivity());
                    mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                        @Override
                        public void onDismiss(ISheetMenu sheetMenu) {
                            if (mConfigurationChangeListener != null) {
                                extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangeListener);
                                mConfigurationChangeListener = null;
                            }
                        }
                    });
                    if (AppDisplay.isPad())
                        mSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
                }
                holder.mIv_relist_more.setTag(position);
                holder.mIv_relist_more.setVisibility(View.GONE);
                holder.mIv_relist_more.setOnClickListener(new OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (AppUtil.isFastDoubleClick()) return;
                        showSheetMenu(v, node, level);
                    }
                });
                boolean canAddAnnot = extensionsManager.getDocumentManager().canAddAnnot();
                if (canAddAnnot && extensionsManager.isEnableModification()) {
                    int visibility = mIsReplyGroup ? View.GONE : View.VISIBLE;
                    if (!node.isWithDeletePermission() && !node.isWithModificationPermission() && !node.isWithReplyPermission())
                        visibility = View.GONE;
                    holder.mIv_relist_more.setVisibility(visibility);
                }

                holder.mContentsTextView.setTag(position);
                holder.mContentsTextView.setOnClickListener(new OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (AppUtil.isFastDoubleClick()) return;

//                        boolean hasShow = false;
//                        int index = 0;
//                        for (int i = 0; i < mMoreViewShow.size(); i++) {
//                            if (mMoreViewShow.get(i)) {
//                                hasShow = true;
//                                index = i;
//                                break;
//                            }
//                        }
//                        if (hasShow) {
//                            mMoreViewShow.set(index, false);
//                            mAdapter.notifyDataSetChanged();
//                            return;
//                        }

                        boolean editable = extensionsManager.getDocumentManager().canAddAnnot()
                                && node.canComment()
                                && node.isWithReplyPermission()
                                && extensionsManager.isEnableModification()
                                && AnnotPermissionUtil.canModifyAnnot(extensionsManager.getDocumentManager(), node.replyAnnot);

                        replyToAnnot(mPDFViewCtrl, mParent, editable, TYPE_COMMENT, new ReplyCallback() {
                            @Override
                            public void result(final String content) {
                                final Annot annot = node.replyAnnot;
                                if (annot == null) return;
                                extensionsManager.getDocumentManager().modifyAnnot(node.replyAnnot, new CommentContent(annot, content), true, new Event.Callback() {
                                    @Override
                                    public void result(Event event, boolean success) {
                                        if (success) {
                                            try {
                                                node.setAuthor(((Markup) annot).getTitle());
                                                node.setDate(AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime()));
                                                node.setModifyDate(AppDmUtil.getLocalDateString(annot.getModifiedDateTime()));
                                                node.setContent(annot.getContent());

                                                mAdapter.notifyDataSetChanged();
                                            } catch (PDFException e) {
                                                e.printStackTrace();
                                            }
                                        }
                                    }
                                });
                            }

                            @Override
                            public String getContent() {
                                return (String) node.getContent();
                            }
                        });
                    }
                });

//                LinearLayout.LayoutParams replyListRLLayoutParams = (LinearLayout.LayoutParams) holder.mReplyListRL.getLayoutParams();
//                RelativeLayout.LayoutParams replyMoreLayoutParams = (RelativeLayout.LayoutParams) holder.mIv_relist_more.getLayoutParams();
//                LinearLayout.LayoutParams contentLayoutParams = (LinearLayout.LayoutParams) holder.mContentsTextView.getLayoutParams();
//                if (AppDisplay.isPad()) {
//                    replyListRLLayoutParams.leftMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_left_margin_pad);
//                    replyMoreLayoutParams.rightMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_right_margin_pad);
//                    contentLayoutParams.rightMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_left_margin_pad);
//                } else {
//                    replyListRLLayoutParams.leftMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_left_margin_phone);
//                    replyMoreLayoutParams.rightMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_right_margin_phone);
//                    contentLayoutParams.rightMargin = (int) getActivity().getApplicationContext().getResources().getDimension(R.dimen.ux_horz_left_margin_phone);
//                }

//                holder.mReplyListRL.setLayoutParams(replyListRLLayoutParams);
//                holder.mIv_relist_more.setLayoutParams(replyMoreLayoutParams);
//                holder.mContentsTextView.setLayoutParams(contentLayoutParams);

//                RelativeLayout.LayoutParams paramsMoreView = (RelativeLayout.LayoutParams) holder.mLl_relist_moreview.getLayoutParams();
//                paramsMoreView.height = holder.mReplyRoot.getMeasuredHeight();
//                holder.mLl_relist_moreview.setLayoutParams(paramsMoreView);
//                if (mMoreViewShow.get(position)) {
//                    holder.mLl_relist_moreview.setVisibility(View.VISIBLE);
//                } else {
//                    holder.mLl_relist_moreview.setVisibility(View.GONE);
//                }

                return convertView;
            }

            public void doReply(final AnnotNode node) {
                replyToAnnot(mPDFViewCtrl, mParent, true, UIAnnotReply.TYPE_REPLY, new ReplyCallback() {
                    @Override
                    public void result(final String content) {
                        final int pageIndex = node.getPageIndex();
                        final String uid = AppDmUtil.randomUUID(null);

                        try {
                            final PDFPage page = mPDFViewCtrl.getDoc().getPage(pageIndex);

                            addReplyAnnot(mPDFViewCtrl, node.replyAnnot, page, uid, content, new Event.Callback() {

                                @Override
                                public void result(Event event, boolean success) {
                                    UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
                                    DocumentManager dm = extensionsManager.getDocumentManager();
                                    AnnotNode annotNode = new AnnotNode(pageIndex, AppAnnotUtil.getAnnot(page, uid), node.replyAnnot);
                                    annotNode.setAuthor(extensionsManager.getAnnotAuthor());
                                    String dateTemp = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, AppDmUtil.currentDateToDocumentDate());
                                    annotNode.setDate(dateTemp);
//                                                node.setCreationDate(dateTemp);
                                    node.setModifyDate(dateTemp);
                                    annotNode.setContent(content);
                                    annotNode.setType(Annot.e_Note);
                                    annotNode.setEditEnable(true);
                                    annotNode.setDeletable(true);
                                    annotNode.setCanReply(true);
                                    annotNode.setCanComment(true);
                                    annotNode.setApplyRedaction(false);
                                    annotNode.setWithReplyPermission(AnnotPermissionUtil.canReplyAnnot(dm, node.replyAnnot));
                                    annotNode.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(dm, node.replyAnnot));
                                    annotNode.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(dm, node.replyAnnot));

                                    mAdapter.addNode(annotNode);
                                    mAdapter.establishReplyList(mRootNode);

                                    node.setChecked(false);
                                    mCheckedNodes.clear();

                                    mAdapter.notifyDataSetChanged();
                                }
                            });
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }

                    @Override
                    public String getContent() {
                        return null;
                    }
                });
            }

            public void doComment(final AnnotNode node, final boolean canAddAnnot) {
                boolean editable = canEdit(node);//node.isEditEnable();
                replyToAnnot(mPDFViewCtrl, mParent, canAddAnnot && editable, TYPE_COMMENT, new ReplyCallback() {
                    @Override
                    public void result(final String content) {
                        if (!canAddAnnot) return;
                        final Annot annot = node.replyAnnot;
                        if (annot == null) return;
                        ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getDocumentManager().modifyAnnot(node.replyAnnot, new CommentContent(node.replyAnnot, content),
                                true, new Event.Callback() {
                                    @Override
                                    public void result(Event event, boolean success) {
                                        if (success) {
                                            try {
                                                node.setAuthor(((Markup) annot).getTitle());
                                                node.setDate(AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM, annot.getModifiedDateTime()));
                                                node.setModifyDate(AppDmUtil.getLocalDateString(annot.getModifiedDateTime()));
                                                node.setContent(annot.getContent());

                                                mAdapter.notifyDataSetChanged();
                                            } catch (PDFException e) {
                                                e.printStackTrace();
                                            }

                                        }
                                    }
                                });
                    }

                    @Override
                    public String getContent() {
                        return (String) node.getContent();
                    }
                });
            }

            private void showSheetMenu(View view, AnnotNode node, int level) {
                mSheetMenuAnchorView = view;
                mSheetMenuNote = node;
                final UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
                Rect rect = getViewRect(view);
                List<SheetItemBean> menuItems = new ArrayList<>();
                boolean canAddAnnot = extensionsManager.getDocumentManager().canAddAnnot() & extensionsManager.isEnableModification();
                if (node.isWithReplyPermission()) {
                    if (canAddAnnot && node.canReply())
                        menuItems.add(new SheetItemBean(ISheetMenu.REPLY_MENU, R.drawable.ic_redaction_reply, getString(R.string.fx_string_reply), true));
                }
                if (node.isWithModificationPermission()) {
                    if (!canAddAnnot || node.canComment()) {
                        menuItems.add(new SheetItemBean(ISheetMenu.COMMENT_MENU,
                                R.drawable.ic_redaction_comment, getString(R.string.fx_string_note), true));
                    }
                    if (extensionsManager.getDocumentManager().withDeletePermission()
                            && canAddAnnot
                            && level <= 0) {
                        if (!extensionsManager.getDocumentManager().isSign()
                                && extensionsManager.getDocumentManager().canModifyContents()
                                && node.canApplyRedaction()) {
                            menuItems.add(new SheetItemBean(ISheetMenu.APPLY_MENU,
                                    R.drawable.ic_redaction_apply, getString(R.string.fx_string_apply), true));
                        }
                        if (extensionsManager.canAssemble()) {
                            menuItems.add(new SheetItemBean(ISheetMenu.FLATTEN_MENU,
                                    R.drawable.ic_redaction_flatten, getString(R.string.fx_string_flatten), true));
                        }
                    }
                }
                if (node.isWithDeletePermission()) {
                    if (canAddAnnot && node.canDelete())
                        menuItems.add(new SheetItemBean(ISheetMenu.DELETE_MENU, R.drawable.ic_redaction_delete, getString(R.string.fx_string_delete), true));
                }
                mSheetMenu.setAutoResetSystemUiOnShow(false);
                mSheetMenu.setCustomSheetItem(menuItems);
                mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        switch (type) {
                            case ISheetMenu.COMMENT_MENU:
                                doComment(mSheetMenuNote, extensionsManager.getDocumentManager().canAddAnnot()
                                        && extensionsManager.isEnableModification());
                                break;
                            case ISheetMenu.REPLY_MENU:
                                doReply(mSheetMenuNote);
                                break;
                            case ISheetMenu.APPLY_MENU:
                                mCheckedNodes.clear();
                                mCheckedNodes.add(mSheetMenuNote);
                                Collections.sort(mCheckedNodes);
                                redactionItems();
                                break;
                            case ISheetMenu.FLATTEN_MENU:
                                if (extensionsManager.getPermissionProvider() != null) {
                                    extensionsManager.getPermissionProvider().checkPermission(
                                            IPermissionProvider.FUNCTION_FLATTEN, new IPermissionProvider.IPermissionState() {
                                                @Override
                                                public void onState(int state) {
                                                    if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                        mCheckedNodes.clear();
                                                        mCheckedNodes.add(mSheetMenuNote);
                                                        Collections.sort(mCheckedNodes);
                                                        flattenItems();
                                                    }
                                                }
                                            }
                                    );
                                } else {
                                    mCheckedNodes.clear();
                                    mCheckedNodes.add(mSheetMenuNote);
                                    Collections.sort(mCheckedNodes);
                                    flattenItems();
                                }
                                break;
                            case ISheetMenu.DELETE_MENU:
                                mCheckedNodes.clear();
                                mCheckedNodes.add(mSheetMenuNote);
                                Collections.sort(mCheckedNodes);
                                beginToDelete();
                                break;
                            default:
                                break;
                        }
                        mSheetMenu.dismiss();
                    }
                });
                if (mConfigurationChangeListener == null) {
                    mConfigurationChangeListener = new UIExtensionsManager.ConfigurationChangedListener() {
                        @Override
                        public void onConfigurationChanged(Configuration newConfig) {
                            mPDFViewCtrl.postDelayed(new Runnable() {
                                @Override
                                public void run() {
                                    mSheetMenu.show(extensionsManager.getRootView(), getViewRect(mSheetMenuAnchorView));
                                }
                            }, 100);
                        }
                    };
                    extensionsManager.registerConfigurationChangedListener(mConfigurationChangeListener);
                }
                mSheetMenu.show(extensionsManager.getRootView(), rect);
            }

            private Rect getViewRect(View view) {
                UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
                Rect rect = new Rect();
                view.getGlobalVisibleRect(rect);

                int[] location = new int[2];
                view.getLocationOnScreen(location);
                int screen_x = location[0];
                int screen_y = location[1];
                if (AppDevice.isChromeOs(extensionsManager.getAttachedActivity())) {
                    extensionsManager.getRootView().getLocationOnScreen(location);
                    screen_x -= location[0];
                    screen_y -= location[1];
                    rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
                } else {
                    view.getLocationInWindow(location);
                    int window_x = location[0];
                    int window_y = location[1];
                    rect.offset(screen_x - window_x, screen_y - window_y);
                }
                if (!SystemUiHelper.getInstance().isFullScreenMode(getActivity())
                        && SystemUiHelper.getInstance().isFullScreen())
                    rect.offset(0, -SystemUiHelper.getInstance().getStatusBarHeight(mContext));
                return rect;
            }

            final class ViewHolder {
                public LinearLayout mReplyRoot;
                public RelativeLayout mReplyListRL;
                public ImageView mExpandImageView;
                public ImageView mIconImageView;
                public TextView mAuthorTextView;
                public TextView mDateTextView;
                public TextView mContentsTextView;

                public ImageView mIv_relist_more;
//                public LinearLayout mLl_relist_moreview;

//                public LinearLayout mLl_relist_reply;
//                public LinearLayout mLl_relist_comment;
//                public LinearLayout mLl_relist_apply;
//                public LinearLayout mLl_relist_flatten;
//                public LinearLayout mLl_relist_delete;
            }
        }

        class AnnotNode implements Comparable<AnnotNode> {
            private final Annot replyAnnot;
            private final Annot replyToAnnot;
            private final int pageIndex;

            private boolean canReply;
            private boolean canComment;
            private boolean canDelete;
            private boolean editEnable;
            private boolean isChecked;
            private boolean isExpanded;
            private boolean canApplyRedaction;
            private boolean withModificationPermission;
            private boolean withDeletePermission;
            private boolean withReplyPermission;

            private int type;
            private String author;
            private CharSequence content;
            private String date;
            private String mModifyDate;
            private String mCreationDate;
            private List<AnnotNode> childNodes;
            private AnnotNode parent;

            AnnotNode(int pageIndex, Annot annot, Annot replyTo) {
                this.pageIndex = pageIndex;
                this.replyAnnot = annot;
                this.replyToAnnot = replyTo;
            }

            public void clearChildren() {
                if (this.childNodes != null) {
                    this.childNodes.clear();
                }
            }

            void addChildNode(AnnotNode node) {
                if (this.childNodes == null) {
                    this.childNodes = new ArrayList<>();
                }
                if (!this.childNodes.contains(node)) {
                    this.childNodes.add(node);
                }
            }

            void removeChildNode(AnnotNode node) {
                if (this.childNodes != null) {
                    this.childNodes.remove(node);
                }
            }

            public int getPageIndex() {
                return this.pageIndex;
            }

            public boolean isEditEnable() {
                return this.editEnable;
            }

            public void setEditEnable(boolean editEnable) {
                this.editEnable = editEnable;
            }

            public boolean canDelete() {
                return canDelete;
            }

            public void setDeletable(boolean canDelete) {
                this.canDelete = canDelete;
            }

            public boolean canReply() {
                return canReply;
            }

            public boolean canApplyRedaction() {
                return canApplyRedaction;
            }

            public void setApplyRedaction(boolean canApplyRedaction) {
                this.canApplyRedaction = canApplyRedaction;
            }

            public void setCanReply(boolean canReply) {
                this.canReply = canReply;
            }

            public boolean canComment() {
                return canComment;
            }

            public void setCanComment(boolean canComment) {
                this.canComment = canComment;
            }

            public List<AnnotNode> getChildren() {
                return this.childNodes;
            }

            public int getType() {
                return this.type;
            }

            public void setType(int type) {
                this.type = type;
            }

            public String getAuthor() {
                return this.author == null ? "" : this.author;
            }

            public void setAuthor(String author) {
                this.author = author;
            }

            public void setContent(CharSequence content) {
                this.content = content;
            }

            public CharSequence getContent() {
                return this.content == null ? "" : this.content;
            }

            public void setDate(String date) {
                this.date = date;
            }

            public String getDate() {
                return this.date == null ? AppDmUtil.dateOriValue : this.date;
            }

            public void setModifyDate(String modifyDate) {
                this.mModifyDate = modifyDate;
            }

            public String getModifyDate() {
                return this.mModifyDate == null ? AppDmUtil.dateOriValue : this.mModifyDate;
            }

            public void setCreationDate(String creationDate) {
                this.mCreationDate = creationDate;
            }

            public String getCreationDate() {
                return this.mCreationDate == null ? AppDmUtil.dateOriValue : this.mCreationDate;
            }

            public void setChecked(boolean isChecked) {
                this.isChecked = isChecked;
            }

            public boolean isChecked() {
                return this.isChecked;
            }

            public void setParent(AnnotNode parent) {
                this.parent = parent;
            }

            public AnnotNode getParent() {
                return this.parent;
            }

            public boolean isRoot() {
                return this.parent == null;
            }

            public boolean isLeafNode() {
                return this.childNodes == null || this.childNodes.size() == 0;
            }

            public int getLevel() {
                return this.parent == null ? 0 : parent.getLevel() + 1;
            }

            public boolean isExpanded() {
                return this.isExpanded || this.parent == null || this.getLevel() != 1;
            }

            public void setExpanded(boolean isExpanded) {
                this.isExpanded = isExpanded;
            }

            @Override
            public int compareTo(AnnotNode another) {
                if (another == null) return 0;
                if (getLevel() != another.getLevel()) {
                    return getLevel() - another.getLevel();
                }
                try {
                    String lCreationDate = getCreationDate();
                    if (lCreationDate == null || AppDmUtil.dateOriValue.equals(lCreationDate)) {
                        lCreationDate = getModifyDate();
                    }
                    String rCreationDate = another.getCreationDate();
                    if (rCreationDate == null || AppDmUtil.dateOriValue.equals(rCreationDate)) {
                        rCreationDate = another.getModifyDate();
                    }
                    Date lDate = AppDmUtil.documentDateToJavaDate(AppDmUtil.parseDocumentDate(lCreationDate));
                    Date rDate = AppDmUtil.documentDateToJavaDate(AppDmUtil.parseDocumentDate(rCreationDate));
                    if (lDate == null || rDate == null)
                        return 0;

                    return lDate.before(rDate) ? -1 : (lDate.after(rDate) ? 1 : 0);
                } catch (Exception ignored) {
                }
                return 0;
            }

            public boolean isWithModificationPermission() {
                return withModificationPermission;
            }

            public void setWithModificationPermission(boolean withModificationPermission) {
                this.withModificationPermission = withModificationPermission;
            }

            public boolean isWithDeletePermission() {
                return withDeletePermission;
            }

            public void setWithDeletePermission(boolean withDeletePermission) {
                this.withDeletePermission = withDeletePermission;
            }

            public boolean isWithReplyPermission() {
                return withReplyPermission;
            }

            public void setWithReplyPermission(boolean withReplyPermission) {
                this.withReplyPermission = withReplyPermission;
            }
        }
    }
}
