/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.common;


import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.TypedArray;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

import androidx.annotation.DrawableRes;

public class UIFillView extends FrameLayout {
    private final View mBorderView;
    private final ImageView mFillColorView;
    private final ImageView mSrcImageView;
    private final TextView mSrcTextView;

    private int mBgIcon;
    private int mShowType;
    private int mFillColor = 0xFFFFFFFF;

    private ColorStateList mBorderColorList;
    private ColorStateList mSrcColorList;

    public UIFillView(Context context) {
        this(context, null);
    }

    public UIFillView(Context context, AttributeSet attrs) {
        super(context, attrs);
        View view = View.inflate(context, R.layout.fill_view_layout, this);
        mBorderView = this;
        mFillColorView = view.findViewById(R.id.rd_fill_color_view);

        mSrcImageView = view.findViewById(R.id.rd_fill_src_iv);
        mSrcTextView = view.findViewById(R.id.rd_fill_src_tv);

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.UIFillView);
        Drawable borderDrawable = a.getDrawable(R.styleable.UIFillView_borderDrawable);
        if (borderDrawable != null)
            mBorderView.setBackground(borderDrawable);
        Drawable fillDrawable = a.getDrawable(R.styleable.UIFillView_fillDrawble);
        if (fillDrawable != null)
            mFillColorView.setImageDrawable(fillDrawable);
        float borderWidth = a.getDimension(R.styleable.UIFillView_borderWidth, 0);
        LayoutParams fillLayoutParams = (LayoutParams) mFillColorView.getLayoutParams();
        fillLayoutParams.setMargins((int) borderWidth, (int) borderWidth, (int) borderWidth, (int) borderWidth);
        mFillColorView.setLayoutParams(fillLayoutParams);

        int srcType = a.getInt(R.styleable.UIFillView_srcType, 0);
        if (srcType == 0) {//src show text
            mShowType = 0;
            mSrcTextView.setVisibility(VISIBLE);
            mSrcImageView.setVisibility(GONE);

            int defaultSize = AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_text_size_16sp);
            int textSize = a.getDimensionPixelSize(R.styleable.UIFillView_src_textSize, defaultSize);
            int textColor = a.getColor(R.styleable.UIFillView_src_textColor, AppResource.getColor(context, R.color.t4));
            String text = a.getString(R.styleable.UIFillView_src_text);
            mSrcTextView.setTextColor(textColor);
            mSrcTextView.setText(text);
            mSrcTextView.setTextSize(AppDisplay.px2sp(textSize));
        } else {//src show image
            mShowType = 1;
            mSrcTextView.setVisibility(GONE);
            mSrcImageView.setVisibility(VISIBLE);

            Drawable srcDrawable = a.getDrawable(R.styleable.UIFillView_src_image);
            mSrcImageView.setImageDrawable(srcDrawable);
        }
        a.recycle();
    }

    public void setBorderTintList(ColorStateList stateList) {
        mBorderColorList = stateList;
        ThemeUtil.setTintList(mBorderView, stateList);
    }

    public ColorStateList getBorderTintList() {
        return mBorderColorList;
    }

    public void setSrcTintList(ColorStateList stateList) {
        mSrcColorList = stateList;
        if (mShowType == 0)
            ThemeUtil.setTintList(mSrcTextView, stateList);
        else
            ThemeUtil.setTintList(mSrcImageView, stateList);
    }

    public void setTintList(ColorStateList borderStateList, ColorStateList srcStateList) {
        mBorderColorList = borderStateList;
        mSrcColorList = srcStateList;
        ThemeUtil.setTintList(mBorderView, borderStateList);
        if (mShowType == 0)
            ThemeUtil.setTintList(mSrcTextView, srcStateList);
        else
            ThemeUtil.setTintList(mSrcImageView, srcStateList);
    }

    public void setBorderWidth(int width) {
        LayoutParams fillLayoutParams = (LayoutParams) mFillColorView.getLayoutParams();
        fillLayoutParams.setMargins(width, width, width, width);
        mFillColorView.setLayoutParams(fillLayoutParams);
    }

    public void setBorderDrawale(Drawable drawale) {
        mBorderView.setBackground(drawale);
    }

    public Drawable getBorderDrawale() {
        return mBorderView.getBackground();
    }

    public void setBorderResource(@DrawableRes int resId) {
        mBgIcon = resId;
        mBorderView.setBackgroundResource(resId);
    }

    public int getBorderResource() {
        return mBgIcon;
    }

    public void setBorderColor(int color) {
        if (mBorderView.getBackground() != null)
            mBorderView.getBackground().setColorFilter(new PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN));
    }

    public void setFillDrawale(Drawable drawale) {
        mFillColorView.setImageDrawable(drawale);
    }

    public void setFillResource(@DrawableRes int resId) {
        mFillColorView.setImageResource(resId);
    }

    public Drawable getFillDrawable() {
        return mFillColorView.getDrawable();
    }

    public void setFillColorFilter(int color) {
        mFillColor = color;
        mFillColorView.setColorFilter(color);
    }

    @Override
    public void setForceDarkAllowed(boolean allow) {
        if (Build.VERSION.SDK_INT >= 29) {// 29 == Build.VERSION_CODES.Q
            super.setForceDarkAllowed(allow);
        }
    }

    public int getFillColor() {
        return mFillColor;
    }

    @Override
    public void setSelected(boolean selected) {
        super.setSelected(selected);
        if (mShowType == 0)
            mSrcTextView.setSelected(selected);
        else
            mSrcImageView.setSelected(selected);
    }

    @Override
    public void setEnabled(boolean enabled) {
        if (isEnabled() != enabled) {
            super.setEnabled(enabled);
            if (enabled)
                mBorderView.setAlpha(1.0f);
            else
                mBorderView.setAlpha(0.6f);
            mFillColorView.setEnabled(enabled);
        }
    }

}
