/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.fileattachment;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.view.KeyEvent;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.utils.AppFileUtil;

import java.io.File;


public class FileAttachmentModule implements Module, PropertyBar.PropertyChangeListener {

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private FileAttachmentAnnotHandler mAnnotHandler;
    private FileAttachmentToolHandler mToolHandler;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private PropertyBar mPropertyBar;

    @Override
    public String getName() {
        return Module.MODULE_NAME_FILEATTACHMENT;
    }

    public FileAttachmentModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new FileAttachmentToolHandler(mContext, mPdfViewCtrl, this);
        mCurrentColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        mCurrentOpacity = 100;
        mFlagType = FileAttachmentConstants.ICONTYPE_PUSHPIN;
        mPropertyBar = new PropertyBarImpl(mContext, mPdfViewCtrl);
        ((PropertyBarImpl)mPropertyBar).setNeedAutoSetCustomPropertyListener(false);
        mAnnotHandler = new FileAttachmentAnnotHandler(mContext, mPdfViewCtrl, this);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
            mCurrentColor = config.uiSettings.annotations.attachment.color;
            mCurrentOpacity = (int) (config.uiSettings.annotations.attachment.opacity * 100);
            String iconName = config.uiSettings.annotations.attachment.icon;
            mFlagType = FileAttachmentUtil.getIconType(iconName);

            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerToolHandler(mToolHandler);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.getToolsManager()
                    .addToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Attachments, mToolHandler.getToolSupply());
            uiExtensionsManager.registerConfigurationChangedListener(mConfigureChangeListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
        }
        mToolHandler.setOpacity(mCurrentOpacity);
        mToolHandler.setIconType(mFlagType);
        mToolHandler.setColor(mCurrentColor);
        mAnnotHandler.setToolHandler(mToolHandler);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterToolHandler(mToolHandler);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigureChangeListener);
            uiExtensionsManager.removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }
        // delete temp files
        String tempPath = mAnnotHandler.getTmpPath();
        File tempFile = new File(tempPath);
        if (tempFile.exists()) {
            AppFileUtil.deleteFolder(tempFile, false);
        }
        return true;
    }

    private int mCurrentColor;
    private int mCurrentOpacity;
    private int mFlagType;

    private boolean needHandleByTool(UIExtensionsManager uiExtensionsManager, AnnotHandler annotHandler) {
        return uiExtensionsManager.getCurrentToolHandler() == mToolHandler
                && (mAnnotHandler != annotHandler || mAnnotHandler.getAnnotMenu().isShowing());
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mCurrentColor = value;
                mToolHandler.setColor(mCurrentColor);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.modifyAnnotColor(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.setColor(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_OPACITY) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mCurrentOpacity = value;
                mToolHandler.setOpacity(mCurrentOpacity);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.modifyAnnotOpacity(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.setOpacity(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_FILEATTACHMENT) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mFlagType = value;
                mToolHandler.setIconType(value);
            } else if (mAnnotHandler == currentAnnotHandler) {
                mAnnotHandler.modifyIconType(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mFlagType = value;
                mToolHandler.setIconType(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(PropertyBar.PROPERTY_FILEATTACHMENT, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
    }

    @Override
    public void onValueChanged(long property, String value) {
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        return (mToolHandler.onKeyDown(keyCode, event) || mAnnotHandler.onKeyDown(keyCode, event));
    }

    public boolean onKeyBack() {
        return (mToolHandler.onKeyBack() || mAnnotHandler.onKeyBack());
    }

    private int[] mPBColors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];

    void resetPropertyBar() {
        final FileAttachmentToolHandler toolHandler = (FileAttachmentToolHandler) getToolHandler();
        long supportProperty = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_OPACITY | PropertyBar.PROPERTY_FILEATTACHMENT;
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, mPBColors, 0, mPBColors.length);

        mPropertyBar.setColors(mPBColors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, toolHandler.getColor());
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, toolHandler.getOpacity());
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FILEATTACHMENT, toolHandler.getIconType());
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(supportProperty);
        mPropertyBar.setPropertyChangeListener(this);
    }

    public PropertyBar getPropertyBar() {
        return mPropertyBar;
    }

    private UIExtensionsManager.ConfigurationChangedListener mConfigureChangeListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            mToolHandler.onConfigurationChanged(newConfig);
        }
    };


    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mPropertyBar != null)
                mPropertyBar.updateTheme();
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mCurrentColor = (int) value;
                mToolHandler.mColor = mCurrentColor;
            } else if (property == PropertyBar.PROPERTY_OPACITY) {
                mCurrentOpacity = (int) value;
                mToolHandler.mOpacity = mCurrentOpacity;
            } else if (property == PropertyBar.PROPERTY_FILEATTACHMENT) {
                mFlagType = (int) value;
                mToolHandler.mIconType = mFlagType;
            }
        }

        @Override
        public int getToolType() {
            return ToolConstants.Attachments;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_COMMENT_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
        }

    };

}
