/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.fileattachment;

import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.KeyEvent;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.FileAttachment;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import java.io.File;
import java.io.FileFilter;


public class FileAttachmentToolHandler implements ToolHandler {
    private static final int MAX_ATTACHMENT_FILE_SIZE = 1024 * 1024 * 300;

    private FileAttachmentModule mAttachmentModule;
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;

    int mColor;
    int mOpacity;
    int mIconType;
    private String mPath;
    private String attachmentName;
    private boolean mIsContinuousCreate = true;
    private UIExtensionsManager mUIExtensionsManager;

    public FileAttachmentToolHandler(Context context, PDFViewCtrl pdfViewCtrl, FileAttachmentModule module) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mAttachmentModule = module;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_FILEATTACHMENT;
    }

    @Override
    public void onActivate() {
    }

    @Override
    public void onDeactivate() {
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUIExtensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUIExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        return onFileAttachmentToolTouch(pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUIExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled){
            handled = onFileAttachmentToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        return mUIExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
    }

    private boolean onFileAttachmentToolTouch(int pageIndex, MotionEvent motionEvent) {
        int action = motionEvent.getActionMasked();
        PointF point = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);

        switch (action) {
            case MotionEvent.ACTION_DOWN:

                PDFPage page = null;
                RectF pageRect = new RectF();
                try {
                    page = mPdfViewCtrl.getDoc().getPage(pageIndex);

                    pageRect = new RectF(0, page.getHeight(), page.getWidth(), 0);

                } catch (PDFException e) {
                    e.printStackTrace();
                }
                if (point.x < pageRect.left) {
                    point.x = pageRect.left;
                }

                if (point.x > pageRect.right - 20) {
                    point.x = pageRect.right - 20;
                }

                if (point.y < 24) {
                    point.y = 24;
                }

                if (point.y > pageRect.top) {
                    point.y = pageRect.top;
                }


                showFileSelectDialog(pageIndex, point);
                break;
            default:
                break;
        }

        return true;
    }

    public void AI_addFileAttachment(int pageIndex, PointF point) {
        showFileSelectDialog(pageIndex, point);
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {

    }

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentToolHandler() == this) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                return true;
            }
        }
        return false;
    }

    public boolean onKeyBack() {
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentToolHandler() == this) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
            return true;
        }
        return false;
    }

    protected void addAnnot(final int pageIndex, AnnotContent content, final RectF rect, final Event.Callback result) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            final FileAttachment annot = (FileAttachment) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_FileAttachment, AppUtil.toFxRectF(rect)), Annot.e_FileAttachment);
            final FileAttachmentAddUndoItem undoItem = new FileAttachmentAddUndoItem(mPdfViewCtrl);

            undoItem.setCurrentValue(content);
            undoItem.mPageIndex = pageIndex;
            undoItem.mFlags = Annot.e_FlagPrint | Annot.e_FlagNoRotate | Annot.e_FlagNoZoom;
            undoItem.mIconName = ((IFileAttachmentAnnotContent) content).getIconName();
            undoItem.mPath = ((IFileAttachmentAnnotContent) content).getFilePath();
            undoItem.mPath = AppFileUtil.getAdaptedFilePath(mContext, undoItem.mPath);
            undoItem.attacheName = ((IFileAttachmentAnnotContent) content).getFileName();
            undoItem.mBBox = new RectF(rect);
            FileAttachmentEvent event = new FileAttachmentEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);

            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            try {
                                Matrix matrix = mPdfViewCtrl.getDisplayMatrix(pageIndex);
                                RectF annotRectF = AppUtil.toRectF(annot.getDeviceRect(AppUtil.toMatrix2D(matrix)));
                                Rect rect = new Rect();
                                annotRectF.roundOut(rect);
                                mPdfViewCtrl.refresh(pageIndex, rect);
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                    if (result != null) {
                        result.result(event, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            if (!mIsContinuousCreate) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected void addAnnot(final int pageIndex, final RectF rect, final Event.Callback result) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            final FileAttachment annot = (FileAttachment) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_FileAttachment, AppUtil.toFxRectF(rect)), Annot.e_FileAttachment);
            final FileAttachmentAddUndoItem undoItem = new FileAttachmentAddUndoItem(mPdfViewCtrl);

            undoItem.mPageIndex = pageIndex;
            undoItem.mNM = AppDmUtil.randomUUID(null);
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mFlags = Annot.e_FlagPrint | Annot.e_FlagNoRotate | Annot.e_FlagNoZoom;
            undoItem.mColor = mColor;
            undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mIconName = FileAttachmentUtil.getIconName(mIconType);
            undoItem.mPath = mPath;
            undoItem.mPath = AppFileUtil.getAdaptedFilePath(mContext, undoItem.mPath);
            undoItem.attacheName = attachmentName;
            undoItem.mBBox = new RectF(rect);
            FileAttachmentEvent event = new FileAttachmentEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);

            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            try {
                                Matrix matrix = mPdfViewCtrl.getDisplayMatrix(pageIndex);
                                RectF annotRectF = AppUtil.toRectF(annot.getDeviceRect(AppUtil.toMatrix2D(matrix)));
                                Rect rect = new Rect();
                                annotRectF.roundOut(rect);
                                mPdfViewCtrl.refresh(pageIndex, rect);
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                    if (result != null) {
                        result.result(event, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            if (!mIsContinuousCreate) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private UIFileSelectDialog mfileSelectDialog;

    private void showFileSelectDialog(final int pageIndex, final PointF pointf) {
        if (mfileSelectDialog != null && mfileSelectDialog.isShowing()) return;
        final PointF point = new PointF();
        if (pointf != null) {
            point.set(pointf.x, pointf.y);
        }

        Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) {
            return;
        }

        mfileSelectDialog = new UIFileSelectDialog(context);
        mfileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                if (pathname.isHidden() || !AppFileUtil.canRead(pathname)) return false;
                return true;
            }
        }, true);
        mfileSelectDialog.setListener(new UIMatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                mfileSelectDialog.dismiss();
                mPath = mfileSelectDialog.getSelectedFiles().get(0).path;
                attachmentName = mfileSelectDialog.getSelectedFiles().get(0).name;
                if (mPath == null || mPath.length() < 1) return;

                //check file size
                if (new File(mPath).length() > MAX_ATTACHMENT_FILE_SIZE) {
                    String msg = AppResource.getString(mContext, R.string.annot_fat_filesizelimit_meg,
                            MAX_ATTACHMENT_FILE_SIZE / (1024 * 1024));
                    UIToast.getInstance(mContext).showCustomToast(msg);
                    return;
                }
                //add attachment
                mfileSelectDialog.dismiss();
                RectF rectF = new RectF(point.x, point.y, point.x + 20, point.y - 20);
                addAnnot(pageIndex, rectF, null);
            }
        });
        mfileSelectDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    mfileSelectDialog.dismiss();
                }
                return true;
            }
        });

        mfileSelectDialog.showDialog(false);
    }

    public void setColor(int color) {
        mColor = color;
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    public int getColor() {
        return mColor;
    }

    public void setOpacity(int opacity) {
        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public int getIconType() {
        return mIconType;
    }

    public void setIconType(int iconType) {
        mIconType = iconType;
        if (mCurToolItem == null) return;
        mCurToolItem.property.style = iconType;
    }

    protected void onConfigurationChanged(Configuration newConfig) {
        if (mfileSelectDialog != null && mfileSelectDialog.isShowing()) {
            mfileSelectDialog.resetWH();
            mfileSelectDialog.showDialog(false);
        }
    }

    private int mLastColor;
    private int mLastOpacity;
    private int mLastStyle;
    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new AttachmentToolSupply(mContext);
        return mToolSupply;
    }

    private class AttachmentToolSupply extends ToolSupplyImpl {

        public AttachmentToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_attachment_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return R.drawable.comment_tool_attachment_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Attachments;
            property.color = mColor;
            property.opacity = mOpacity;
            property.style = mIconType;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_ATTACHMENT;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if(mUIExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Attachment);
                }


                mAttachmentModule.resetPropertyBar();

                mLastColor = mColor;
                mLastOpacity = mOpacity;
                mLastStyle = mIconType;

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mIconType = property.style;

                mAttachmentModule.resetPropertyBar();
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(FileAttachmentToolHandler.this);
            } else {
                if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentToolHandler() == FileAttachmentToolHandler.this){
                    mColor = mLastColor;
                    mOpacity = mLastOpacity;
                    mIconType = mLastStyle;
                    mCurToolItem = null;
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;
            mLastColor = mColor;
            mLastOpacity = mOpacity;
            mLastStyle = mIconType;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mIconType = property.style;

            mAttachmentModule.resetPropertyBar();
            mAttachmentModule.getPropertyBar().setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mColor = mLastColor;
                    mOpacity = mLastOpacity;
                    mIconType = mLastStyle;
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                    mAttachmentModule.getPropertyBar().setDismissListener(null);
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            mAttachmentModule.resetPropertyBar();
            return mAttachmentModule.getPropertyBar();
        }
    }

}
