/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.fillsign;


import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.view.KeyEvent;
import android.view.ScaleGestureDetector;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.AbstractUndo;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolItemsManager;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.signature.SignatureModule;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.AppBuildConfig;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.List;

public class FillSignModule implements Module {

    private FillSignToolHandler mToolHandler;
    private Context mContext;
    private PDFViewCtrl mPDFViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private SignatureModule mSignModule;
    private IPanelManager mPanelManager;
    private ToolItemBean mSigntureItem;

    private boolean mDocWillClose = false;

    public FillSignModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this.mContext = context;
        this.mPDFViewCtrl = pdfViewCtrl;
        this.mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_FIllSIGN;
    }

    @Override
    public boolean loadModule() {
        mPanelManager = mUiExtensionsManager.getPanelManager();
        mPanelManager.registerPanelEventListener(mPanelEventListener);

        mToolHandler = new FillSignToolHandler(mContext, mPDFViewCtrl);
        mSignModule = (SignatureModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_PSISIGNATURE);
        mToolHandler.setSignModule(mSignModule);

        mPDFViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPDFViewCtrl.registerDocEventListener(mDocEventListener);
        mPDFViewCtrl.registerPageEventListener(mPageEventListener);
        mPDFViewCtrl.registerScaleGestureEventListener(mScaleGestureEventListener);

        mUiExtensionsManager.registerToolHandler(mToolHandler);
        mUiExtensionsManager.registerModule(this);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.registerStateChangeListener(mStateChangeListener);
        mUiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
        mUiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().registerUndoEventListener(mUndoEventListener);
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
        initToolBar();
        return true;
    }

    @Override
    public boolean unloadModule() {
        mPDFViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPDFViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPDFViewCtrl.unregisterPageEventListener(mPageEventListener);
        mPDFViewCtrl.unregisterScaleGestureEventListener(mScaleGestureEventListener);
        mUiExtensionsManager.unregisterToolHandler(mToolHandler);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.unregisterStateChangeListener(mStateChangeListener);
        mUiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterUndoEventListener(mUndoEventListener);
        mPanelManager.unregisterPanelEventListener(mPanelEventListener);
        return true;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    private void initToolBar() {
        ToolItemsManager toolItemsManager = mUiExtensionsManager.getToolsManager();
        if (mUiExtensionsManager.getConfig().modules.isLoadFillSign) {
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeText, mToolHandler.getToolSupply());
            if (AppBuildConfig.SDK_VERSION >= 21)
                toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeComboText, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeProFile, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeCheckMark, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeCrossMark, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeDot, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeLine, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.FillSignTypeRectangle, mToolHandler.getToolSupply());
        }
        if (mSignModule != null)
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_SIGN, ToolConstants.DigitalSignature, mToolHandler.getToolSupply());
    }

    private final PDFViewCtrl.IScaleGestureEventListener mScaleGestureEventListener = new PDFViewCtrl.IScaleGestureEventListener() {
        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
            mToolHandler.onScaleEnd();
        }
    };

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            for (int i = 0; i < pageIndexes.length; i++)
                mToolHandler.onPagesRemoved(success, pageIndexes[i] - i);
        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            mToolHandler.onPageMoved(success, index, dstIndex);
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] pageRanges) {
            mToolHandler.onPagesInserted(success, dstIndex, pageRanges);
        }
    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {
        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mToolHandler.onDrawForControls(canvas);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            mDocWillClose = false;
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            mDocWillClose = true;
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            mToolHandler.mUndoItemList.clear();
            mToolHandler.release(true);
        }

        @Override
        public void onDocWillSave(PDFDoc document) {
            mToolHandler.release(true);
        }

        @Override
        public void onDocSaved(PDFDoc document, int errCode) {
        }
    };

    private final IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            resetFillSignItem();
            mToolHandler.release(true);
        }
    };

    private final IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            resetFillSignItem();
            mToolHandler.release(true);
        }

        @Override
        public void onAnnotsApplied(PDFDoc doc) {
            resetFillSignItem();
            mToolHandler.release(true);
        }
    };


    private final AbstractUndo.IUndoEventListener mUndoEventListener = new AbstractUndo.IUndoEventListener() {

        @Override
        public void itemWillAdd(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemAdded(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemWillRemoved(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemRemoved(DocumentManager dm, IUndoItem item) {
            mToolHandler.mUndoItemList.remove(item);
        }

        @Override
        public void willUndo(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void undoFinished(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willRedo(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void redoFinished(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willClearUndo(DocumentManager dm) {
            mToolHandler.mUndoItemList.clear();
        }

        @Override
        public void clearUndoFinished(DocumentManager dm) {
        }
    };

    private void resetFillSignItem() {
        UIToolBaseBar toolbar = mUiExtensionsManager.getMainFrame().getToolbar(ToolbarItemConfig.ITEM_FILLSIGN_TAB);
        if (toolbar != null) {
            toolbar.setEnabled(mUiExtensionsManager.getDocumentManager().canAddSignature()
                    && mUiExtensionsManager.isEnableModification()
                    && mUiExtensionsManager.getDocumentManager().canModifyContents());
        }
    }

    private final AnnotEventListener mAnnotEventListener = new AnnotEventListener() {

        private final List<String> mWillDeleteWidgets = new ArrayList<>();

        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            try {
                if (!annot.isEmpty() && annot.getType() == Annot.e_Widget) {
                    if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                        if (mToolHandler.getCurToolItem() != null) {
                            mToolHandler.getCurToolItem().toolItem.performClick();
                        } else {
                            mToolHandler.onExitTool(null);
                        }
                    }
                    resetFillSignItem();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {
            try {
                if (!annot.isEmpty() && annot.getType() == Annot.e_Widget) {
                    mWillDeleteWidgets.add(AppAnnotUtil.getAnnotUniqueID(annot));
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {
            if (mWillDeleteWidgets.size() == 0) return;
            mWillDeleteWidgets.remove(0);
            resetFillSignItem();
        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mToolHandler != null) {
                mToolHandler.dismissProfileDialog();
            }
        }
    };

    private final IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                    if (!AppDisplay.isPad() && mUiExtensionsManager.getPanelManager().isShowingPanel()) {
                        mUiExtensionsManager.getPanelManager().hidePanel();
                        return true;
                    }

                    if (mToolHandler.getCurToolItem() != null) {
                        mToolHandler.getCurToolItem().toolItem.performClick();
                        return true;
                    }
                    return mToolHandler.onExitTool(null);
                }
            }
            return false;
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                mToolHandler.onConfigurationChanged(newConfig);
            }
        }
    };

    private final IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            if (mDocWillClose) return;
            int curTab = mUiExtensionsManager.getMainFrame().getCurrentTab();
            if (curTab == ToolbarItemConfig.ITEM_FILLSIGN_TAB
                    && newState == ReadStateConfig.STATE_NORMAL) {
                mUiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
            } else {
                if (oldState != ReadStateConfig.STATE_FILLSIGN && newState == ReadStateConfig.STATE_FILLSIGN) {
                    mUiExtensionsManager.triggerDismissMenuEvent();
                    if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null)
                        mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);

                    if (curTab != ToolbarItemConfig.ITEM_FILLSIGN_TAB)
                        mUiExtensionsManager.getMainFrame().setCurrentTab(ToolbarItemConfig.ITEM_FILLSIGN_TAB);

                    UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
                    if (undoModule != null) {
                        undoModule.setUndoItemCallback(mUndoItemCallback);
                    }
                    return;
                }

                if (oldState == ReadStateConfig.STATE_FILLSIGN && newState != ReadStateConfig.STATE_FILLSIGN) {
                    if (mUiExtensionsManager.getCurrentToolHandler() == mToolHandler)
                        mUiExtensionsManager.setCurrentToolHandler(null);
                    else
                        mToolHandler.onExitTool(new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                mToolHandler.release(false);
                            }
                        });

                    UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
                    if (undoModule != null) {
                        undoModule.setUndoItemCallback(null);
                    }
                }
            }
        }
    };

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN)
                mToolHandler.clearFocusAnnot(null);
        }

        @Override
        public void onPanelClosed() {
        }
    };

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mToolHandler.isEditingText()) {
                mToolHandler.endAddTextBox();
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mToolHandler.isEditingText()) {
                mToolHandler.endAddTextBox();
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };

    public void activateSignature() {
        if (mPDFViewCtrl.getDoc() == null
                || mSigntureItem == null
                || !mUiExtensionsManager.isEnableModification()
                || !mUiExtensionsManager.getDocumentManager().canAddSignature())
            return;
        mToolHandler.activateSignature(mSigntureItem);
    }

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onHiddenChanged(boolean hidden) {
            if (mPDFViewCtrl.getDoc() == null) return;

            if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                if (hidden) {
                    if (mToolHandler != null) {
                        mToolHandler.onExitTool(null);
                    }
                }
            }
        }
    };

}
