/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.fillsign;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.os.Build;
import android.text.Editable;
import android.text.InputType;
import android.text.TextWatcher;
import android.util.SparseArray;
import android.view.ActionMode;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.FillSign;
import com.foxit.sdk.pdf.FillSignObject;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextFillSignObject;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.graphics.FormXObject;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.common.UIAnnotFrame;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.annots.common.UIMagnifierView;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.modules.signature.SignatureModule;
import com.foxit.uiextensions.modules.signature.SignatureToolHandler;
import com.foxit.uiextensions.pdfreader.config.AppBuildConfig;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureAnnotHandler;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;

public class FillSignToolHandler implements ToolHandler {
    static final int ADD_FORM_OBJECT = 0;

    private UIExtensionsManager mUIExtensionsManager;
    private PDFViewCtrl mPdfViewCtrl;
    private Context mContext;
    private FillSignProperty mProperty;

    private FillSignDeleteUndoItem mTempTextDeleteUndoItem;
    private SparseArray<FillSign> mFillSignArrs = new SparseArray<>();
    ArrayList<FillSignUndoItem> mUndoItemList = new ArrayList<>();

    private FormObject mOldAnnot;
    FormObject mFocusAnnot;

    private PointF mLongPressPt;
    private PointF mDownPt;
    private PointF mLastPt;

    private int mCreateType;
    private float mMoveDist;
    private int mTouchCaptured;
    private int mOp;
    private int mCtl;
    private int mPopoverAction;
    private int mLongPressPage;
    private int mLastLineCount = -1;
    private boolean mCanZoomIn = true;
    private boolean mCanZoomOut = true;
    private boolean mNeedAddDeletedUndoItem;
    private boolean mIsAddingFillSign = false;

    private ToolItemBean mCurToolItem;
    private FillSignProfileInfo mProfileInfo;
    private String mProfileStr;

    private SignatureModule mSignModule;
    private SignatureToolHandler mSignatureToolHandler;

    PointF mCreateLineSize = new PointF(45, 15);
    PointF mCreateRectSize = new PointF(45, 30);

    FillSignToolHandler(Context context, PDFViewCtrl viewCtrl) {
        mContext = context;
        mPdfViewCtrl = viewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) viewCtrl.getUIExtensionsManager();

        mDownPt = new PointF();
        mLastPt = new PointF();
        mLongPressPage = -1;

        mProfileInfo = new FillSignProfileInfo(mContext);
        mProperty = new FillSignProperty(context);
    }

    @Override
    public String getType() {
        return TH_TYPE_FILLSIGN;
    }

    @Override
    public void onActivate() {
    }

    @Override
    public void onDeactivate() {
        onExitTool(new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mUIExtensionsManager.getState() != ReadStateConfig.STATE_FILLSIGN)
                    release(false);
            }
        });
    }

    public boolean onExitTool(final Event.Callback callback) {
        if (_curToolTag() != 0) {
            if (mCurToolItem.toolItem.isChecked())
                mCurToolItem.toolItem.setChecked(false);
            if (mCreateType == ToolConstants.DigitalSignature)
                mSignModule.activeSign(mCurToolItem.toolItem, false);

            if (mCreateType != ToolConstants.FillSignTypeProFile)
                onItemClicked(mCreateType);
            mCreateType = mLastCreateType;
            mCurToolItem = null;
        }

        return clearFocusAnnot(callback);
    }

    boolean clearFocusAnnot(final Event.Callback callback) {
        mLongPressPage = -1;
        mLongPressPt = null;
        boolean ret = false;
        if (isEditingText()) {
            endAddTextBox(new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (isPopoverShowing())
                        dismissPopover();
                    if (mFocusAnnot != null)
                        focusObject(null);
                    if (callback != null)
                        callback.result(event, success);
                }
            });
            ret = true;
        } else {
            if (isPopoverShowing()) {
                dismissPopover();
                ret = true;
            }
            if (mFocusAnnot != null) {
                focusObject(null);
                ret = true;
            }
            if (callback != null)
                callback.result(null, true);
        }
        return ret;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        PointF devPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF pageViewPt = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devPoint, pageViewPt, pageIndex);
        PointF pdfPt = new PointF(pageViewPt.x, pageViewPt.y);
        mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPt, pdfPt, pageIndex);

        int itemTag = _curToolTag();
        int action = motionEvent.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (itemTag == ToolConstants.DigitalSignature && mSignatureToolHandler != null) {
                    if (mLongPressPt != null) {
                        if (isPopoverShowing())
                            dismissPopover();
                        mLongPressPt = null;
                        mLongPressPage = -1;
                    }
                    if (mFocusAnnot != null)
                        focusObject(null);
                    return mSignatureToolHandler.onTouchEvent(pageIndex, motionEvent);
                }

                if (!isEditingText() && mFocusAnnot == null) {
                    if (isPopoverShowing()) {// long press popover
                        dismissPopover();
                        mLongPressPt = null;
                        mLongPressPage = -1;
                        mTouchCaptured = 2;
                        return true;
                    }
                }
                if (mFocusAnnot != null && pageIndex == mFocusAnnot.mPageIndex) {
                    RectF pvBox = new RectF(mFocusAnnot.mBBox);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(pvBox, pvBox, pageIndex);

                    if (mFocusAnnot.mTag == ToolConstants.FillSignTypeLine) {
                        PointF p1 = new PointF(pvBox.left, pvBox.centerY());
                        PointF p2 = new PointF(pvBox.right, pvBox.centerY());
                        mCtl = UIAnnotFrame.hitLineControlTest(p1, p2, pageViewPt, AppDisplay.getFingerArea() / 5, AppDisplay.getFingerArea() / 4);
                        if (mCtl == 0) {
                            mCtl = UIAnnotFrame.CTL_LEFT_MID;
                        } else if (mCtl == 1) {
                            mCtl = UIAnnotFrame.CTL_RIGHT_MID;
                        }
                    } else if (mFocusAnnot.mTag == ToolConstants.FillSignTypeRectangle) {
                        RectF bounds = UIAnnotFrame.calculateFrameBounds(pvBox, _borderThickness(), AppDisplay.getFingerArea() / 5);
                        mCtl = UIAnnotFrame.hitControlTest(bounds, pageViewPt, AppDisplay.getFingerArea() / 4);
                    } else if (mFocusAnnot.mTag == ToolConstants.FillSignTypeCheckMark
                            || mFocusAnnot.mTag == ToolConstants.FillSignTypeCrossMark
                            || mFocusAnnot.mTag == ToolConstants.FillSignTypeDot) {
                        RectF bounds = UIAnnotFrame.calculateFrameBounds(pvBox, _borderThickness(), AppDisplay.getFingerArea() / 5);
                        mCtl = UIAnnotFrame.hitCornerControlTest(bounds, pageViewPt, AppDisplay.getFingerArea() / 4);
                    } else {
                        mCtl = UIAnnotFrame.CTL_NONE;
                    }
                    if (mCtl != UIAnnotFrame.CTL_NONE) {
                        mTouchCaptured = 1;
                        mOp = UIAnnotFrame.OP_SCALE;
                        mDownPt.set(pageViewPt);
                        mLastPt.set(pageViewPt);
                        mMoveDist = 0;
                        return true;
                    } else {
                        RectF bbox = new RectF(pvBox);
                        bbox.inset(-AppDisplay.getFingerArea() / 4, -AppDisplay.getFingerArea() / 4);
                        if (bbox.contains(pageViewPt.x, pageViewPt.y)) {
                            mTouchCaptured = 1;
                            mOp = UIAnnotFrame.OP_TRANSLATE;
                            mDownPt.set(pageViewPt);
                            mLastPt.set(pageViewPt);
                            mMoveDist = 0;
                            return true;
                        }
                    }
                }
                break;
            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (mTouchCaptured == 2) {
                    if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_CANCEL) {
                        mTouchCaptured = 0;
                    }
                    return true;
                }
                if (mTouchCaptured == 1 && mFocusAnnot != null && pageIndex == mFocusAnnot.mPageIndex) {
                    if (pageViewPt.x != mLastPt.x || pageViewPt.y != mLastPt.y) {
                        if (isPopoverShowing()) {
                            dismissPopover();
                        }

                        addMagnifierView();
                        if (mMagnifierView != null) {
                            mMagnifierView.onTouchEvent(motionEvent);
                            mMagnifierView.setVisibility(View.VISIBLE);
                        }

                        RectF pvBox = new RectF(mFocusAnnot.mBBox);
                        mPdfViewCtrl.convertPdfRectToPageViewRect(pvBox, pvBox, pageIndex);
                        RectF bounds0;
                        RectF bounds1;
                        PointF adjust = new PointF();
                        if (mFocusAnnot.mTag == ToolConstants.FillSignTypeCheckMark
                                || mFocusAnnot.mTag == ToolConstants.FillSignTypeCrossMark
                                || mFocusAnnot.mTag == ToolConstants.FillSignTypeDot) {
                            bounds0 = UIAnnotFrame.mapFrameBounds(pvBox, _borderThickness(), mOp, mCtl,
                                    mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, 5, true);
                            bounds1 = UIAnnotFrame.mapFrameBounds(pvBox, _borderThickness(), mOp, mCtl,
                                    pageViewPt.x - mDownPt.x, pageViewPt.y - mDownPt.y, 5, true);

                            if (mOp == UIAnnotFrame.OP_TRANSLATE) {
                                adjust = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).calculateCorrection(mPdfViewCtrl, pageIndex, bounds1, mOp, mCtl);
                                UIAnnotFrame.adjustBounds(bounds1, mOp, mCtl, adjust);

                                bounds1.union(bounds0);
                            } else if (mOp == UIAnnotFrame.OP_SCALE) {
                                bounds1.union(bounds0);
                                adjust = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).calculateCornerScale(mPdfViewCtrl, pageIndex, pvBox, bounds1, mCtl);
                                UIAnnotFrame.adjustBounds(bounds1, mOp, mCtl, adjust);
                            }

                            PointF lastPointF = new PointF(pageViewPt.x + adjust.x, pageViewPt.y + adjust.y);
                            if (mCtl == UIAnnotFrame.CTL_LEFT_TOP || mCtl == UIAnnotFrame.CTL_RIGHT_TOP) {
                                if (lastPointF.y <= pvBox.bottom)
                                    mLastPt.set(lastPointF.x, lastPointF.y);
                            } else if (mCtl == UIAnnotFrame.CTL_LEFT_BOTTOM || mCtl == UIAnnotFrame.CTL_RIGHT_BOTTOM) {
                                if (lastPointF.y >= pvBox.top)
                                    mLastPt.set(lastPointF.x, lastPointF.y);
                            } else {
                                mLastPt.set(lastPointF.x, lastPointF.y);
                            }
                        } else {
                            if (mFocusAnnot.mTag == ToolConstants.FillSignTypeText || mFocusAnnot.mTag == ToolConstants.FillSignTypeComboText) {
                                float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, mFocusAnnot.mFontSize);
                                float lineWidth = getLineWidth(mFocusAnnot.mTag, fontSize, mFocusAnnot.mContent);
                                pvBox.right = pvBox.left + lineWidth + _editView().getPaddingLeft() + _editView().getPaddingRight();
                            }

                            bounds1 = UIAnnotFrame.mapFrameBounds(pvBox, _borderThickness(), mOp, mCtl,
                                    pageViewPt.x - mDownPt.x, pageViewPt.y - mDownPt.y, 5);
                            adjust = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).calculateCorrection(mPdfViewCtrl, pageIndex, bounds1, mOp, mCtl);
                            UIAnnotFrame.adjustBounds(bounds1, mOp, mCtl, adjust);
                            mLastPt.set(pageViewPt.x + adjust.x, pageViewPt.y + adjust.y);
                        }

                        float dist = (float) Math.sqrt((mLastPt.x - mDownPt.x) * (mLastPt.x - mDownPt.x) + (mLastPt.y - mDownPt.y) * (mLastPt.y - mDownPt.y));
                        if (dist > mMoveDist) {
                            mMoveDist = dist;
                        }

                        UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).extentBoundsToContainControl(bounds1);
                        RectF displayRect = new RectF();
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bounds1, displayRect, pageIndex);
                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(displayRect));
                    }

                    if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_CANCEL) {
                        if (mFocusAnnot != null) {

                            if (!mLastPt.equals(mDownPt)) {
                                RectF bbox = new RectF(mFocusAnnot.mBBox);
                                mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);

                                Matrix matrix;
                                if (mFocusAnnot.mTag == ToolConstants.FillSignTypeCheckMark
                                        || mFocusAnnot.mTag == ToolConstants.FillSignTypeCrossMark
                                        || mFocusAnnot.mTag == ToolConstants.FillSignTypeDot) {
                                    matrix = UIAnnotFrame.calculateOperateMatrix(bbox, mOp, mCtl,
                                            mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, true);
                                } else {
                                    matrix = UIAnnotFrame.calculateOperateMatrix(bbox, mOp, mCtl,
                                            mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y);
                                }

                                transformAnnot(pageIndex, matrix);
                            } else {
                                showPopover(pageIndex, 0);

                                if (AppDisplay.px2dp(mMoveDist) < 2) {
                                    if (mFocusAnnot.mTag == ToolConstants.FillSignTypeText || mFocusAnnot.mTag == ToolConstants.FillSignTypeComboText) {
                                        _editFocusTextFormObj(pageIndex, mFocusAnnot.mTag);
                                    }
                                }
                            }
                        }
                        if (mMagnifierView != null) {
                            mMagnifierView.setVisibility(View.GONE);
                        }
                        mTouchCaptured = 0;
                        mDownPt.set(0, 0);
                        mLastPt.set(0, 0);
                        mOp = UIAnnotFrame.OP_DEFAULT;
                        mCtl = UIAnnotFrame.CTL_NONE;
                        mMoveDist = 0;
                    }
                    return true;
                }
                break;
            default:
                break;
        }

        if (itemTag == ToolConstants.DigitalSignature && mSignatureToolHandler != null) {
            if (mLongPressPt != null) {
                if (isPopoverShowing())
                    dismissPopover();
                mLongPressPt = null;
                mLongPressPage = -1;
            }
            if (mFocusAnnot != null)
                focusObject(null);
            return mSignatureToolHandler.onTouchEvent(pageIndex, motionEvent);
        }
        return false;
    }

    private float _borderThickness() {
        return AppDisplay.dp2px(10);
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (!mUIExtensionsManager.isEnableModification()
                || !mUIExtensionsManager.getDocumentManager().canAddSignature()) return false;
        return onPressCallback(true, pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        return onPressCallback(false, pageIndex, motionEvent);
    }

    private boolean onPressCallback(boolean isLongPress, int pageIndex, MotionEvent motionEvent) {
        if (pageIndex < 0) return false;
        int itemTag = _curToolTag();
        if (!mUIExtensionsManager.getConfig().modules.isLoadFillSign && itemTag != ToolConstants.DigitalSignature)
            return false;

        final PointF viewPointF = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
        final PointF pdfPointF = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);
        if (itemTag == ToolConstants.DigitalSignature && mSignatureToolHandler != null) {
            if (mLongPressPt != null) {
                if (isPopoverShowing())
                    dismissPopover();
                mLongPressPt = null;
                mLongPressPage = -1;
            }
            if (mFocusAnnot != null)
                focusObject(null);
            return mSignatureToolHandler.onSingleTapConfirmed(pageIndex, motionEvent);
        } else if (isEditingText()) {
            endAddTextBox();
            return true;
        } else if (mFocusAnnot != null && mFocusAnnot.mBBox.contains(pdfPointF.x, pdfPointF.y)) {
            return true;
        } else if (focusObjectAtPoint(pageIndex, pdfPointF)) {
            return true;
        } else if (focusSignatureAtPoint(pageIndex, pdfPointF, motionEvent)) {
            return true;
        } else {
            boolean handled = false;
            if (mFocusAnnot != null) {
                focusObject(null);
                handled = true;
            }

            if (mCurToolItem != null) {
                switch (itemTag) {
                    case ToolConstants.FillSignTypeText:
                    case ToolConstants.FillSignTypeComboText:
                        if (isEditingText()) {
                            endAddTextBox();
                        } else {
                            addTextBox(itemTag, pageIndex, viewPointF, true, null, null, false);
                        }
                        return true;
                    case ToolConstants.FillSignTypeCheckMark:
                    case ToolConstants.FillSignTypeCrossMark:
                    case ToolConstants.FillSignTypeDot:
                    case ToolConstants.FillSignTypeLine:
                    case ToolConstants.FillSignTypeRectangle:
                        addFormObject(itemTag, pageIndex, viewPointF, false);
                        return true;
                    case ToolConstants.FillSignTypeProFile:
                        addProfileObject(itemTag, pageIndex, viewPointF);
                        return true;
                }
            }

            if (isLongPress) {
                if (mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification()) {
                    mLongPressPage = pageIndex;
                    mLongPressPt = new PointF(pdfPointF.x, pdfPointF.y);
                    showPopover(pageIndex, 2);
                    return true;
                }
            }
            return handled;
        }
    }

    @Override
    public boolean isContinueAddAnnot() {
        return false;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
    }


    private float mLastTextSize;

    void onDrawForControls(Canvas canvas) {
        int state = mUIExtensionsManager.getState();
        if (state != ReadStateConfig.STATE_FILLSIGN) return;

        int curIndex;
        if (isEditingText()) {
            curIndex = _editView().getPageIndex();
        } else if (mFocusAnnot != null) {
            curIndex = mFocusAnnot.mPageIndex;
        } else {
            curIndex = mLongPressPage;
        }

        if (isEditingText()) {
            if (!mPdfViewCtrl.isPageVisible(curIndex)) {
                dismissPopover();
                return;
            }

            PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, curIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, curIndex);
            offset.y -= getBottomOffset();

            int rawScreenHeight = AppDisplay.getRawScreenHeight();
            int navBarHeight = AppDisplay.getRealNavBarHeight();
            int keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView());
            if ((offset.y + keyboardHeight + navBarHeight) > rawScreenHeight) {
                _editView().setVisibility(View.INVISIBLE);
                dismissPopover();
            } else {
                _editView().setVisibility(View.VISIBLE);
                setEditViewMargin((int) offset.x, (int) (offset.y));

                float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, curIndex);
                if (fontSize != mLastTextSize) {
                    mLastTextSize = fontSize;
                    _editView().setTextSize(fontSize);
                }

                RectF editBox = getEditViewBoxInDv();
                if (_editView().getTextStyle() == FillSignEditText.STYLE_COMBO_TEXT) {
                    editBox.right -= _editView().getPaddingRight() / 2f;
                }

                RectF dvBox = new RectF(0, 0,
                        mUIExtensionsManager.getRootView().getWidth(),
                        mUIExtensionsManager.getRootView().getHeight());

                if (RectF.intersects(dvBox, editBox)) {
                    updatePopover(curIndex, mPopoverAction);
                } else {
                    dismissPopover();
                }
            }
        } else if (mFocusAnnot != null) {
            if (!mPdfViewCtrl.isPageVisible(curIndex)) {
                dismissPopover();
                return;
            }

            RectF bbox = new RectF(mFocusAnnot.mBBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, curIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, curIndex);

            RectF dvBox = new RectF(0, 0,
                    mUIExtensionsManager.getRootView().getWidth(),
                    mUIExtensionsManager.getRootView().getHeight());

            if (RectF.intersects(dvBox, bbox)) {
                updatePopover(curIndex, mPopoverAction);
            } else {
                dismissPopover();
            }
        } else {
            if (mLongPressPt != null) {
                if (!mPdfViewCtrl.isPageVisible(curIndex)) {
                    dismissPopover();
                    return;
                }

                RectF bbox = new RectF(mLongPressPt.x, mLongPressPt.y, mLongPressPt.x + 2, mLongPressPt.y + 2);
                mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, curIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, curIndex);

                RectF dvBox = new RectF(0, 0,
                        mPdfViewCtrl.getDisplayViewWidth(),
                        mPdfViewCtrl.getDisplayViewHeight());

                if (RectF.intersects(dvBox, bbox)) {
                    updatePopover(curIndex, 2);
                } else {
                    dismissPopover();
                }
            }
        }
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mFocusAnnot == null || mFocusAnnot.mPageIndex != pageIndex) {
            return;
        }

        RectF bbox = new RectF(mFocusAnnot.mBBox);
        mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);

        int tag = mFocusAnnot.mTag;
        RectF mapBounds;
        RectF drawBounds;
        Matrix matrix;
        if (tag == ToolConstants.FillSignTypeCheckMark
                || tag == ToolConstants.FillSignTypeCrossMark
                || tag == ToolConstants.FillSignTypeDot) {
            matrix = UIAnnotFrame.calculateOperateMatrix(bbox, mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, true);
            mapBounds = UIAnnotFrame.mapFrameBounds(bbox, _borderThickness(), mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, 5, true);
            drawBounds = UIAnnotFrame.mapFrameBounds(bbox, 0, mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, 0, true);
        } else {
            matrix = UIAnnotFrame.calculateOperateMatrix(bbox, mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y);
            mapBounds = UIAnnotFrame.mapFrameBounds(bbox, _borderThickness(), mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, 5);
            drawBounds = UIAnnotFrame.mapFrameBounds(bbox, 0, mOp, mCtl, mLastPt.x - mDownPt.x, mLastPt.y - mDownPt.y, 0);
        }

        int color = AppResource.getColor(mContext, R.color.ux_color_blue_ff179cd8, null);
        int opacity = 255;

        if (!matrix.equals(new Matrix())) {
            switch (tag) {
                case ToolConstants.FillSignTypeCheckMark:
                    FillSignPathDraw.drawCheck(canvas, drawBounds, color);
                    break;
                case ToolConstants.FillSignTypeCrossMark:
                    FillSignPathDraw.drawX(canvas, drawBounds, color);
                    break;
                case ToolConstants.FillSignTypeDot:
                    FillSignPathDraw.drawDot(canvas, drawBounds, color);
                    break;
                case ToolConstants.FillSignTypeLine:
                    FillSignPathDraw.drawLine(mPdfViewCtrl, pageIndex, canvas, drawBounds, color);
                    break;
                case ToolConstants.FillSignTypeRectangle:
                    FillSignPathDraw.drawRect(mPdfViewCtrl, pageIndex, canvas, drawBounds, color);
                    break;
                case ToolConstants.FillSignTypeText:
                case ToolConstants.FillSignTypeComboText:
                    float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, mFocusAnnot.mFontSize);
                    float lineHeight = getLineHeight(fontSize);
                    FillSignPathDraw.drawText(mPdfViewCtrl, pageIndex, canvas, drawBounds, color, mFocusAnnot, 0, lineHeight);
                    break;
            }
        }

        switch (tag) {
            case ToolConstants.FillSignTypeCheckMark:
            case ToolConstants.FillSignTypeCrossMark:
            case ToolConstants.FillSignTypeDot:
                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).drawCorner(canvas, mapBounds, color, opacity);
                break;
            case ToolConstants.FillSignTypeLine:
                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).drawLineControls(canvas,
                        new PointF(drawBounds.left, drawBounds.centerY()), new PointF(drawBounds.right, drawBounds.centerY()),
                        color, opacity, 0);
                break;
            case ToolConstants.FillSignTypeRectangle:
                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).draw(canvas, mapBounds, color, opacity);
                break;
            case ToolConstants.FillSignTypeText:
            case ToolConstants.FillSignTypeComboText:
                float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, mFocusAnnot.mFontSize);
                float lineWidth = getLineWidth(mFocusAnnot.mTag, fontSize, mFocusAnnot.mContent) + _editView().getPaddingRight() + _editView().getPaddingLeft();
                if (lineWidth > drawBounds.width())
                    drawBounds.right = drawBounds.left + lineWidth;
                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).drawFrame(canvas, drawBounds, color, opacity, null);
                break;
        }

        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                _adjustNavBarPadding();
            }
        });
    }

    private float getLineWidth(int tag, float fontSize, String content) {
        _editView().setTextSize(fontSize);
        if (tag == ToolConstants.FillSignTypeComboText) {
            _editView().setLetterSpacing(mProperty.mFontSpacing);
            _editView().setPadding(0, 0, _editView().getStratchBmp().getWidth(), 0);
        } else {
            _editView().setTextStyle(FillSignEditText.STYLE_TEXT);
            _editView().setLetterSpacing(0);
            _editView().setPadding(0, 0, 0, 0);
        }

        Paint paint = _editView().getPaint();
        float maxWidth = 0;
        ArrayList<String> textArray = FillSignUtils.jniToJavaTextLines(content);
        for (int i = 0; i < textArray.size(); i++) {
            float width = paint.measureText(textArray.get(i));
            if (width > maxWidth) {
                maxWidth = width;
            }
        }
        return maxWidth;
    }

    private float getLineHeight(float fontSize) {
        _editView().setTextSize(fontSize);
        return (float) _editView().getLineHeight();
    }

    private int getMaxLines() {
        float lineHeight = _editView().getLineHeight();

        int pageIndex = _editView().mPageIndex;
        PointF dvPt = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
        mPdfViewCtrl.convertPdfPtToPageViewPt(dvPt, dvPt, pageIndex);
        float maxHeight = mPdfViewCtrl.getPageViewHeight(pageIndex) - dvPt.y;
        return (int) (maxHeight / lineHeight) == 0 ? 1 : (int) (maxHeight / lineHeight);
    }

    private boolean focusObjectAtPoint(final int pageIndex, PointF docPt) {
        if (!mUIExtensionsManager.getDocumentManager().canModifyContents() || !mUIExtensionsManager.isEnableModification())
            return false;

        try {
            FillSign fillSign = getFillSign(pageIndex);
            if (fillSign == null) return false;

            final FillSignObject fillSignObject = fillSign.getObjectAtPoint(AppUtil.toFxPointF(docPt));
            if (fillSignObject != null && !fillSignObject.isEmpty()) {

                final RectF objRect = AppUtil.toRectF(fillSignObject.getRect());
                final FormObject formObj = new FormObject(getFillObjTag(fillSignObject), pageIndex, objRect);
                formObj.mJniAddr = FormXObject.getCPtr(fillSignObject.getFormXObject());
                FillSignUtils.getTextFillSignInfo(fillSignObject, new IResult<String, Float, Float>() {
                    @Override
                    public void onResult(boolean success, String content, Float fontSize, Float charSpace) {
                        if (success) {
                            formObj.mContent = content;
                            formObj.mCharspace = charSpace;
                            formObj.mSpacing = charSpace / fontSize;
                            formObj.mFontSize = fontSize;
                        }
                    }
                });
                focusObject(formObj);
                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                    RectF objectRect = new RectF(objRect);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(objectRect, objectRect, pageIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(objectRect, objectRect, pageIndex);
                    mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(objectRect));
                }
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean focusSignatureAtPoint(final int pageIndex, PointF docPt, MotionEvent event) {
        try {
            PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            Annot annot = page.getAnnotAtPoint(AppUtil.toFxPointF(docPt), 1);
            if (!annot.isEmpty() && annot.getType() == Annot.e_Widget) {
                Widget widget = new Widget(annot);
                Field field = widget.getField();
                if (!field.isEmpty() && field.getType() == Field.e_TypeSignature) {
                    DigitalSignatureAnnotHandler annotHandler = (DigitalSignatureAnnotHandler)
                            mUIExtensionsManager.getAnnotHandlerByType(AnnotHandler.TYPE_FORMFIELD_SIGNATURE);
                    return annotHandler.onSingleTapConfirmed(pageIndex, event, widget);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    void onItemClicked(int type) {
        if (isEditingText()) {
            endAddTextBox();
        }

        if (type == ToolConstants.DigitalSignature || type == ToolConstants.FillSignTypeProFile) {
            if (mFocusAnnot != null) {
                focusObject(null);
            }
        }
    }

    boolean isEditingText() {
        return mEditView != null && mEditView.getParent() != null;
    }

    void setEditText(String text) {
        if (!isEditingText())
            return;

        _editView().setText(text);
    }

    private void showEditView(boolean showKeyboard) {
        _editView();

        if (mEditView.getParent() == null) {
            RelativeLayout.LayoutParams lp = new RelativeLayout.LayoutParams(
                    ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            lp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            lp.addRule(RelativeLayout.ALIGN_PARENT_TOP);

            mUIExtensionsManager.getRootView().addView(mEditView, 1, lp);
        }
        mEditView.setText("");
        mEditView.setVisibility(View.VISIBLE);
        registerKeyboardListener();
        if (showKeyboard) {
            AppUtil.showSoftInput(mEditView);
            _keyboardBar().show();
        }
    }

    private void deleteEditView() {
        mUIExtensionsManager.startHideToolbarsTimer();
        if (AppDisplay.isPad() && SystemUiHelper.getInstance().isFullScreen())
            SystemUiHelper.getInstance().hideSystemUI(mUIExtensionsManager.getAttachedActivity());

        if (mEditView != null) {
            if (mEditView.getParent() != null) {
                mLastLineCount = -1;
                AppUtil.dismissInputSoft(mEditView);
                mUIExtensionsManager.getRootView().removeView(mEditView);
            }
        }

        dismissPopover();

        AppKeyboardUtil.removeKeyboardListener(mUIExtensionsManager.getRootView());
        _keyboardBar().hide();

        jumpToEditView(true);
    }

    private void jumpToEditView(boolean editing) {
        if (getBottomOffset() != 0) {
            correctDvOffsetValue();
        }
        if (!isEditingText()) {
            setBottomOffset(0, 0);
            return;
        }

        int pageIndex = _editView().getPageIndex();
        if (!mPdfViewCtrl.isPageVisible(pageIndex)) {
            setBottomOffset(0, 0);
            return;
        }

        RectF editBox = getEditViewBoxInDv();
        mPdfViewCtrl.convertDisplayViewRectToPageViewRect(editBox, editBox, pageIndex);
        mPdfViewCtrl.convertPageViewRectToPdfRect(editBox, editBox, pageIndex);

        adjustBoxToDisplayArea(pageIndex, editBox);
        adjustCaretInDocViewer();

        if (getBottomOffset() != 0) {
            correctDvOffsetValue();
        }

        PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
        mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, pageIndex);
        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, pageIndex);
        offset.y -= getBottomOffset();
        setEditViewMargin(editing, (int) offset.x, (int) (offset.y));

        updatePopover(pageIndex, mPopoverAction);
    }

    private void updatePopover(final int pageIndex, final int action) {
        if (mMagnifierView != null && mMagnifierView.getVisibility() == View.VISIBLE) return;
        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                showPopover(pageIndex, action);
            }
        });
    }

    private void _adjustNavBarPadding() {
        if (_keyboardBar().isShow()) {
            adjustNavBarPadding();
        }
    }

    private FillSignKeyboardBar mKeyboardBar;

    private FillSignKeyboardBar _keyboardBar() {
        if (mKeyboardBar == null) {
            mKeyboardBar = new FillSignKeyboardBar(mContext, mUIExtensionsManager, this);
        }
        return mKeyboardBar;
    }

    private AppKeyboardUtil.IKeyboardListener mKeyboardListener;

    private AppKeyboardUtil.IKeyboardListener registerKeyboardListener() {
        if (mKeyboardListener == null) {
            mKeyboardListener = new AppKeyboardUtil.IKeyboardListener() {
                @Override
                public void onKeyboardOpened(int keyboardHeight) {
                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity());

                    _adjustNavBarPadding();
                    AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            _keyboardBar().show();
                            _adjustNavBarPadding();
                            jumpToEditView(true);

                            int pageIndex = _editView().getPageIndex();
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                RectF editBox = getEditViewBoxInDv();
                                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(editBox));
                            }
                        }
                    }, 100);
                }

                @Override
                public void onKeyboardClosed() {
                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity());

                    if (_editView().getVisibility() == View.INVISIBLE)
                        _editView().setVisibility(View.VISIBLE);
                    AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
//                            _adjustNavBarPadding();
                            jumpToEditView(false);
                            _keyboardBar().hide();
                        }
                    }, 100);
                }
            };
        }

        AppKeyboardUtil.setKeyboardListener(mUIExtensionsManager.getRootView(), mUIExtensionsManager.getRootView(), mKeyboardListener);
        return mKeyboardListener;
    }

    private FillSignEditText mEditView;

    private FillSignEditText _editView() {
        if (mEditView == null) {
            mEditView = new FillSignEditText(mContext.getApplicationContext()) {
                @Override
                public boolean onTouchEvent(MotionEvent event) {
                    return super.onTouchEvent(event);
                }
            };

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                mEditView.setForceDarkAllowed(false);
            }
            mEditView.setProperty(mProperty);
            mEditView.setSingleLine(false);
            mEditView.setText("");
            mEditView.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));
            mEditView.setPadding(0, 0, 0, 0);
            mEditView.setInputType(mEditView.getInputType() | InputType.TYPE_TEXT_FLAG_NO_SUGGESTIONS);
            mEditView.setTextColor(AppResource.getColor(mContext, R.color.ux_text_color_on_night));
            mEditView.setTypeface(Typeface.create(Typeface.MONOSPACE, Typeface.NORMAL));
            if (AppDisplay.isPad()) {
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            } else {
                mEditView.setImeOptions(EditorInfo.IME_ACTION_NONE);
            }
            mEditView.setOnEditorActionListener(new TextView.OnEditorActionListener() {
                @Override
                public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
                    if (actionId == EditorInfo.IME_ACTION_DONE) {
                        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
                            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    String str = mEditView.getText() + "\n";
                                    mEditView.setText(str);
                                    mEditView.setSelection(str.length());
                                }
                            });
                        } else {
                            endAddTextBox();
                        }
                        return true;
                    }
                    return false;
                }
            });

            mEditView.setOnKeyListener(new View.OnKeyListener() {
                @Override
                public boolean onKey(View v, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_ENTER && event.getAction() == KeyEvent.ACTION_DOWN) {
                        if (isEditingText()) {
                            float lineCounts = _editView().getLineCount();
                            if (lineCounts + 1 > getMaxLines())
                                return true;

                            int pageIndex = _editView().mPageIndex;
                            PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
                            mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, pageIndex);
                            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, pageIndex);
                            offset.y -= getBottomOffset();
                            setEditViewMargin((int) offset.x, (int) (offset.y));
                        }
                    }
                    return false;
                }
            });

            mEditView.addTextChangedListener(new TextWatcher() {

                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                }

                @Override
                public void afterTextChanged(Editable editable) {
                    _keyboardBar().updatePrompt(_editView().getText().toString(), mProfileInfo);
                    AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            adjustCaretInDocViewer();

                            int pageIndex = _editView().getPageIndex();
                            int lineCount = _editView().getLineCount();
                            float lineHeight = _editView().getLineHeight();
                            float needH = lineCount * lineHeight;
                            float maxHeight = _editView().getMaxHeight();
                            if (needH > maxHeight) {
                                _editView().setMaxHeight((int) needH);

                                PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
                                mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, pageIndex);
                                mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, pageIndex);
                                offset.y -= getBottomOffset();
                                offset.y -= (needH - maxHeight);
                                setEditViewMargin((int) offset.x, (int) offset.y);

                                PointF docPt = new PointF(offset.x, offset.y + getBottomOffset());
                                mPdfViewCtrl.convertDisplayViewPtToPageViewPt(docPt, docPt, pageIndex);
                                mPdfViewCtrl.convertPageViewPtToPdfPt(docPt, docPt, pageIndex);
                                _editView().setDocLtOffset(docPt);
                            } else {
                                if (mLastLineCount != -1 && mLastLineCount != lineCount) {
                                    PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
                                    mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, pageIndex);
                                    mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, pageIndex);
                                    offset.y -= getBottomOffset();
                                    setEditViewMargin((int) offset.x, (int) offset.y);
                                }
                            }
                            updatePopover(_editView().mPageIndex, 0);
                            mLastLineCount = lineCount;
                        }
                    }, 50);
                }
            });

            mEditView.setCustomSelectionActionModeCallback(new ActionMode.Callback() {

                @Override
                public boolean onCreateActionMode(ActionMode mode, Menu menu) {
                    return false;
                }

                @Override
                public boolean onPrepareActionMode(ActionMode mode, Menu menu) {
                    return false;
                }

                @Override
                public boolean onActionItemClicked(ActionMode mode, MenuItem item) {
                    return false;
                }

                @Override
                public void onDestroyActionMode(ActionMode mode) {
                }
            });
        }
        return mEditView;
    }

    private boolean mAddLTPointF;

    private void addTextBox(int tag, final int pageIndex, PointF pvPt, boolean showKeyboard, Float docFontSize, String text, boolean ltPt) {
        mUIExtensionsManager.stopHideToolbarsTimer();
        SystemUiHelper.getInstance().showNavigationBar(mUIExtensionsManager.getAttachedActivity());

        mAddLTPointF = ltPt;
        showEditView(showKeyboard);

        float textSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex);
        if (docFontSize != null) {
            textSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, docFontSize);
        }

        _editView().setTextSize(textSize);
        if (tag == ToolConstants.FillSignTypeComboText) {
            _editView().setTextStyle(FillSignEditText.STYLE_COMBO_TEXT);
            _editView().setLetterSpacing(mProperty.mFontSpacing);
            _editView().setPadding(0, 0, _editView().getStratchBmp().getWidth(), 0);
        } else {
            _editView().setTextStyle(FillSignEditText.STYLE_TEXT);
            _editView().setLetterSpacing(0);
            _editView().setPadding(0, 0, 0, 0);
        }
        if (text != null) {
            _editView().setText(text);
            _editView().setSelection(_editView().getText().length());
        }

        float textSizePx = _editView().getTextSize();
        RectF textBox = new RectF(pvPt.x, pvPt.y, (pvPt.x + textSizePx / 2), (pvPt.y + textSizePx));
        if (!ltPt) {
            textBox.offset(-textBox.width() / 2, -textBox.height() / 2);
        }

        Paint paint = _editView().getPaint();
        float wordW = paint.measureText(AppResource.getString(mContext, R.string.fx_place_holder));
        float needW = wordW + _editView().getPaddingRight() + _editView().getPaddingLeft();
        float needH = _editView().getLineHeight();

        int mw = (int) (mPdfViewCtrl.getPageViewWidth(pageIndex) - pvPt.x);
        int mh = (int) (mPdfViewCtrl.getPageViewHeight(pageIndex) - pvPt.y);
        if (mw < needW) {
            textBox.offset(mw - needW, 0);
            mw = (int) needW;
        }
        if (mh < needH) {
            textBox.offset(0, mh - needH);
            mh = (int) needH;
        }

        _editView().setMinWidth((int) textBox.width());
        _editView().setMinHeight((int) textBox.height());
        _editView().setMaxWidth(mw);
        _editView().setMaxHeight(mh);
        _editView().setMarginRight(FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, FillSignEditText.SPACING));

        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(textBox, textBox, pageIndex);
        setEditViewMargin((int) textBox.left, (int) textBox.top);

        PointF docPt = new PointF(textBox.left, textBox.top);
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(docPt, docPt, pageIndex);
        mPdfViewCtrl.convertPageViewPtToPdfPt(docPt, docPt, pageIndex);

        _editView().setPageIndex(pageIndex);
        _editView().setPvSize(new Point(mPdfViewCtrl.getPageViewWidth(pageIndex), mPdfViewCtrl.getPageViewHeight(pageIndex)));
        _editView().setDocLtOffset(docPt);

        if (showKeyboard) {
            float fontSize = mProperty.mFontSize / mProperty.mZoomScale;
            mCanZoomOut = fontSize >= FillSignProperty.MIN_FONTSIZE;

            RectF rectF = getEditViewBoxInDv();
            RectF objRect = new RectF(rectF);
            RectF viewRectF = new RectF();
            mPdfViewCtrl.convertDisplayViewRectToPageViewRect(objRect, viewRectF, pageIndex);
            float objWidth = Math.abs(viewRectF.width());
            float objHeight = Math.abs(viewRectF.height());
            float width = objWidth * mProperty.mZoomScale;
            float height = objHeight * mProperty.mZoomScale;
            float dx = (objWidth - width) / 2;
            float dy = (objHeight - height) / 2;
            viewRectF.inset(dx, dy);
            mCanZoomIn = canZoomIn(mPdfViewCtrl, pageIndex, viewRectF);

            showPopover(0, new RectF(textBox));
        }
    }

    void endAddTextBox() {
        endAddTextBox(null);
    }

    void endAddTextBox(final Event.Callback callback) {
        if (mIsAddingFillSign){
            return;
        }

        mIsAddingFillSign = true;
        final int pageIndex = _editView().getPageIndex();
        try {
            PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;

            if (AppUtil.isEmpty(_editView().getText())) {
                deleteEditView();
                dismissPopover();

                if (mOldAnnot != null) {
                    FillSignDeleteUndoItem undoItem = new FillSignDeleteUndoItem(mPdfViewCtrl, this);
                    undoItem.mPageIndex = mOldAnnot.mPageIndex;
                    undoItem.setOldValue(mOldAnnot.clone());

                    mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                    mUndoItemList.add(undoItem);

                    RectF rectF = new RectF(mOldAnnot.mBBox);
                    focusObject(null);
                    if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                        RectF viewRect = new RectF(rectF);
                        mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                        mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                    }

                    if (callback != null)
                        callback.result(null, true);
                }
                mIsAddingFillSign = false;
                return;
            }

            int tag = ToolConstants.FillSignTypeText;
            if (_editView().getTextStyle() == FillSignEditText.STYLE_COMBO_TEXT) {
                tag = ToolConstants.FillSignTypeComboText;
            }

            _editView().splitTextLines();
            ArrayList<String> texts = FillSignUtils.javaToJniTextLines(_editView().mLineTexts.mLineTexts);
            float letterSpacing = 0;
            if (AppBuildConfig.SDK_VERSION >= 21)
                letterSpacing = _editView().getLetterSpacing();

            final RectF evBox = getEditViewBoxInDv();
            final RectF docRectF = new RectF();
            mPdfViewCtrl.convertDisplayViewRectToPageViewRect(evBox, docRectF, pageIndex);
            mPdfViewCtrl.convertPageViewRectToPdfRect(docRectF, docRectF, pageIndex);

            float fontSize = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, _editView().getTextSize());

            float padding = _editView().getPaddingLeft();
            padding = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, padding);

            float lineHeight = _editView().getLineHeight();
            lineHeight = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, lineHeight);

//            deleteEditView();
            dismissPopover();

            mUIExtensionsManager.getDocumentManager().setDocModified(true);
            if (mOldAnnot != null) {
                int type = FillSign.e_FillSignObjectTypeText;
                final FillSignModifyUndoItem undoItem = new FillSignModifyUndoItem(mPdfViewCtrl, this);
                undoItem.mPageIndex = pageIndex;
                undoItem.mType = type;
                undoItem.mTexts = new ArrayList<>(texts);
                undoItem.mRectF = new RectF(docRectF);
                undoItem.mFontSize = fontSize;
                undoItem.mCharspace = letterSpacing * fontSize;
                undoItem.mLineHeight = lineHeight;
                undoItem.mIsCombText = tag == ToolConstants.FillSignTypeComboText;
                undoItem.mRotation = rotation;

                undoItem.mRedoRectF = new RectF(docRectF);
                undoItem.mRedoTexts = new ArrayList<>(texts);
                undoItem.mRedoFontSize = fontSize;
                undoItem.mRedoCharspace = letterSpacing * fontSize;

                undoItem.mUndoRectF = new RectF(mOldAnnot.mBBox);
                undoItem.mUndoTexts = FillSignUtils.jniToJavaTextLines(mOldAnnot.mContent);
                undoItem.mUndoFontSize = mOldAnnot.mFontSize;
                undoItem.mUndoCharspace = mOldAnnot.mCharspace;

                FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, mPdfViewCtrl);
                final int finalTag = tag;
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        mIsAddingFillSign = false;

                        if (success) {
                            try {
                                final FillSignObject newObj = ((FillSignEvent) event).mFillSignObj;
                                RectF newRectF = AppUtil.toRectF(newObj.getRect());
                                final FormObject formObj = new FormObject(finalTag, pageIndex, new RectF(newRectF));
                                long newAddr = FormXObject.getCPtr(newObj.getFormXObject());
                                formObj.mJniAddr = newAddr;

                                FillSignUtils.getTextFillSignInfo(newObj, new IResult<String, Float, Float>() {
                                    @Override
                                    public void onResult(boolean success, String content, Float fontsize, Float charSpace) {
                                        if (success) {
                                            formObj.mContent = content;
                                            formObj.mCharspace = charSpace;
                                            formObj.mSpacing = charSpace / fontsize;
                                            formObj.mFontSize = fontsize;
                                        }
                                    }
                                });

                                long oldAddr = mOldAnnot.mJniAddr;
                                mOldAnnot.mJniAddr = newAddr;

                                if (mOldAnnot.mTag != formObj.mTag
                                        || !mOldAnnot.mContent.equals(formObj.mContent)
                                        || mOldAnnot.mFontSize != formObj.mFontSize) {
                                    undoItem.setOldValue(mOldAnnot.clone());
                                    undoItem.setCurrentValue(formObj.clone());

                                    mUndoItemList.add(undoItem);
                                    mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                                }

                                updateUndoItemJniAddr(oldAddr, newAddr);
                                if (mUIExtensionsManager.getCurrentToolHandler() == FillSignToolHandler.this)
                                    focusObject(formObj);

                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF viewRect = new RectF(newRectF);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                                }
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }

                        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                deleteEditView();
                            }
                        }, 50);
                        if (callback != null)
                            callback.result(event, success);
                    }
                });
                mPdfViewCtrl.addTask(task);

            } else {
                final FillSignAddUndoItem undoItem = new FillSignAddUndoItem(mPdfViewCtrl, this);

                undoItem.mPageIndex = pageIndex;
                undoItem.mType = FillSign.e_FillSignObjectTypeText;
                undoItem.mTexts = new ArrayList<>(texts);
                undoItem.mRectF = new RectF(docRectF);
                undoItem.mFontSize = fontSize;
                undoItem.mCharspace = letterSpacing * fontSize;
                undoItem.mLineHeight = lineHeight;
                undoItem.mIsCombText = tag == ToolConstants.FillSignTypeComboText;
                undoItem.mRotation = rotation;

                FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, mPdfViewCtrl);
                final int finalTag = tag;
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        mIsAddingFillSign = false;

                        if (success) {
                            try {
                                FillSignObject newFillObject = ((FillSignEvent) event).mFillSignObj;
                                if (newFillObject == null || newFillObject.isEmpty()) return;
                                RectF newRectF = AppUtil.toRectF(newFillObject.getRect());

                                final FormObject formObj = new FormObject(finalTag, pageIndex, new RectF(newRectF));
                                formObj.mJniAddr = FormXObject.getCPtr(newFillObject.getFormXObject());

                                FillSignUtils.getTextFillSignInfo(newFillObject, new IResult<String, Float, Float>() {
                                    @Override
                                    public void onResult(boolean success, String content, Float fontsize, Float charSpace) {
                                        if (success) {
                                            formObj.mContent = content;
                                            formObj.mCharspace = charSpace;
                                            formObj.mSpacing = charSpace / fontsize;
                                            formObj.mFontSize = fontsize;
                                        }
                                    }
                                });

                                FormObject newObj = formObj.clone();
                                undoItem.setCurrentValue(newObj);
                                mUndoItemList.add(undoItem);
                                mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);

                                if (mUIExtensionsManager.getCurrentToolHandler() == FillSignToolHandler.this)
                                    focusObject(formObj);

                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF viewRect = new RectF(newRectF);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                                }
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }

                        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                deleteEditView();
                            }
                        }, 50);
                        if (callback != null)
                            callback.result(event, success);
                    }
                });
                mPdfViewCtrl.addTask(task);
            }
        } catch (PDFException e) {
            mIsAddingFillSign = false;
        }
    }

    private void transformAnnot(int pageIndex, Matrix matrix) {
        if (mFocusAnnot == null)
            return;

        RectF bbox = new RectF(mFocusAnnot.mBBox);
        mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
        matrix.mapRect(bbox);

        if (mFocusAnnot.mTag == ToolConstants.FillSignTypeLine) {
            float defHeight = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mProperty.mLineSize.y);
            bbox.top = bbox.centerY() - defHeight / 2;
            bbox.bottom = bbox.centerY() + defHeight / 2;
        }

        RectF displayRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, displayRect, pageIndex);
        showPopover(0, displayRect);

        RectF docRecF = new RectF();
        mPdfViewCtrl.convertPageViewRectToPdfRect(bbox, docRecF, pageIndex);
        mFocusAnnot.mBBox.set(docRecF);

        setProperty(pageIndex, docRecF);
        modifyCurFormObject(pageIndex);
    }

    void addFormObject(final int tag, final int pageIndex, PointF pvPt, boolean ltPoint) {
        PointF size = new PointF();
        switch (tag) {
            case ToolConstants.FillSignTypeCheckMark:
            case ToolConstants.FillSignTypeCrossMark:
            case ToolConstants.FillSignTypeDot:
                float checkSize = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mProperty.mCheckSize);
                size.set(checkSize, checkSize);
                break;
            case ToolConstants.FillSignTypeLine:
                //Changed to not record the last modified size   old mProperty.mLineSize
                float lineSizeX = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mCreateLineSize.x);
                float lineSizeY = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mCreateLineSize.y);
                size.set(lineSizeX, lineSizeY);
                break;

            case ToolConstants.FillSignTypeRectangle:
                //Changed to not record the last modified size   old mProperty.mRectSize
                float rectSizeX = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mCreateRectSize.x);
                float rctSizeY = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mCreateRectSize.y);
                size.set(rectSizeX, rctSizeY);
                break;
        }

        RectF viewBox = new RectF(pvPt.x, pvPt.y, pvPt.x, pvPt.y);
        if (ltPoint) {
            viewBox.right = viewBox.left + size.x;
            viewBox.bottom = viewBox.top - size.y;
        } else {
            viewBox.inset(-size.x / 2, -size.y / 2);
        }

        PointF adjust = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).calculateTranslateCorrection(mPdfViewCtrl, pageIndex, viewBox);
        viewBox.offset(adjust.x, adjust.y);

        RectF docRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToPdfRect(viewBox, docRect, pageIndex);

        addFormObject(tag, pageIndex, docRect, -1, FillSignToolHandler.ADD_FORM_OBJECT, null);
    }

    void addFormObject(final int tag, final int pageIndex, RectF docRect, int rotation, final long oldAddr, final Event.Callback callback) {
        try {
            FillSign fillSign = getFillSign(pageIndex);
            if (fillSign == null) return;

            if (rotation == -1) {
                PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            }
            int objType = getFillObjType(tag);
            final FillSignAddUndoItem undoItem = new FillSignAddUndoItem(mPdfViewCtrl, this);
            undoItem.mType = objType;
            undoItem.mRectF = new RectF(docRect);
            undoItem.mPageIndex = pageIndex;
            undoItem.mRotation = rotation;

            FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    try {
                        if (success) {
                            FillSignObject newFillObj = ((FillSignEvent) event).mFillSignObj;
                            RectF objRect = AppUtil.toRectF(newFillObj.getRect());
                            long newAddr = FormXObject.getCPtr(newFillObj.getFormXObject());

                            if (oldAddr == ADD_FORM_OBJECT) {
                                FormObject obj = new FormObject(tag, pageIndex, new RectF(objRect));
                                obj.mJniAddr = newAddr;
                                focusObject(obj);
                                FormObject newObj = obj.clone();
                                undoItem.setCurrentValue(newObj);

                                mUndoItemList.add(undoItem);
                                mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            } else {
                                updateUndoItemJniAddr(oldAddr, newAddr);
                            }

                            if (callback == null) {
                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF viewRect = new RectF();
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(objRect, viewRect, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                                }
                            }
                        }

                        if (callback != null)
                            callback.result(event, success);
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            });
            mPdfViewCtrl.addTask(task);

        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void addTextFormObj(final FormObject curVal, final Event.Callback callback) {
        if (curVal == null) return;

        final int pageIndex = curVal.mPageIndex;
        try {
            float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, curVal.mFontSize);
            float lineHeight = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, getLineHeight(fontSize));
            ArrayList<String> textLines = FillSignUtils.jniToJavaTextLines(curVal.mContent);
            PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;

            final FillSignAddUndoItem undoItem = new FillSignAddUndoItem(mPdfViewCtrl, this);
            undoItem.mPageIndex = pageIndex;
            undoItem.mType = FillSign.e_FillSignObjectTypeText;
            undoItem.mTexts = new ArrayList<>(textLines);
            undoItem.mRectF = new RectF(curVal.mBBox);
            undoItem.mFontSize = curVal.mFontSize;
            undoItem.mCharspace = curVal.mCharspace;
            undoItem.mLineHeight = lineHeight;
            undoItem.mIsCombText = curVal.mTag == ToolConstants.FillSignTypeComboText;
            undoItem.mRotation = rotation;

            FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        try {
                            if (callback == null) {
                                FillSignObject newFillObject = ((FillSignEvent) event).mFillSignObj;
                                if (newFillObject == null || newFillObject.isEmpty()) return;

                                RectF newRectF = AppUtil.toRectF(newFillObject.getRect());
                                final FormObject formObj = new FormObject(curVal.mTag, pageIndex, new RectF(newRectF));
                                long newAddr = FormXObject.getCPtr(newFillObject.getFormXObject());
                                formObj.mJniAddr = newAddr;
                                FillSignUtils.getTextFillSignInfo(newFillObject, new IResult<String, Float, Float>() {
                                    @Override
                                    public void onResult(boolean success, String content, Float fontsize, Float charSpace) {
                                        if (success) {
                                            formObj.mContent = content;
                                            formObj.mCharspace = charSpace;
                                            formObj.mSpacing = charSpace / fontsize;
                                            formObj.mFontSize = fontsize;
                                        }
                                    }
                                });

                                updateUndoItemJniAddr(curVal.mJniAddr, newAddr);
                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF viewRect = new RectF(newRectF);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                                }
                            } else {
                                callback.result(event, success);
                            }
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void addProfileObject(int tag, int pageIndex, PointF pvPt) {
        if (mCurToolItem != null)
            mCurToolItem.toolItem.performClick();

        if (AppUtil.isEmpty(mProfileStr)) return;
        addTextBox(tag, pageIndex, pvPt, false, null, mProfileStr, false);
    }

    private int getFillObjTag(FillSignObject fillSignObject) {
        try {
            int type = fillSignObject.getType();
            switch (type) {
                case FillSign.e_FillSignObjectTypeText:
                    TextFillSignObject textFillSignObject = new TextFillSignObject(fillSignObject);
                    boolean isComText = textFillSignObject.isCombFieldMode();
                    if (isComText)
                        return ToolConstants.FillSignTypeComboText;
                    else
                        return ToolConstants.FillSignTypeText;
                case FillSign.e_FillSignObjectTypeCheckMark:
                    return ToolConstants.FillSignTypeCheckMark;
                case FillSign.e_FillSignObjectTypeCrossMark:
                    return ToolConstants.FillSignTypeCrossMark;
                case FillSign.e_FillSignObjectTypeDot:
                    return ToolConstants.FillSignTypeDot;
                case FillSign.e_FillSignObjectTypeLine:
                    return ToolConstants.FillSignTypeLine;
                case FillSign.e_FillSignObjectTypeRoundRectangle:
                    return ToolConstants.FillSignTypeRectangle;
                default:
                    return -1;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return -1;
    }

    private int getFillObjType(int tag) {
        switch (tag) {
            case ToolConstants.FillSignTypeText:
            case ToolConstants.FillSignTypeComboText:
                return FillSign.e_FillSignObjectTypeText;
            case ToolConstants.FillSignTypeCheckMark:
                return FillSign.e_FillSignObjectTypeCheckMark;
            case ToolConstants.FillSignTypeCrossMark:
                return FillSign.e_FillSignObjectTypeCrossMark;
            case ToolConstants.FillSignTypeDot:
                return FillSign.e_FillSignObjectTypeDot;
            case ToolConstants.FillSignTypeLine:
                return FillSign.e_FillSignObjectTypeLine;
            case ToolConstants.FillSignTypeRectangle:
                return FillSign.e_FillSignObjectTypeRoundRectangle;
            default:
                return -1;
        }
    }

    private RectF getEditViewBoxInDv() {
        RectF bbox = new RectF();
        if (mEditView != null) {
            int pageIndex = _editView().getPageIndex();
            PointF dvPt = new PointF(mEditView.getDocLtOffset().x, mEditView.getDocLtOffset().y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(dvPt, dvPt, pageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(dvPt, dvPt, pageIndex);
            bbox.set(dvPt.x, dvPt.y, dvPt.x + mEditView.getWidth(), dvPt.y + mEditView.getHeight());
        }
        return bbox;
    }

    void focusObject(FormObject obj) {
        FormObject oldObj = mFocusAnnot;

        if (obj != mFocusAnnot) {
            if (isPopoverShowing()) {
                dismissPopover();
            }
        }

        mFocusAnnot = obj;
        if (obj != null)
            mOldAnnot = obj.clone();
        else
            mOldAnnot = null;

        if (mFocusAnnot != null) {
            int pageIndex = mFocusAnnot.mPageIndex;
            mCanZoomIn = canZoomIn(pageIndex, mFocusAnnot);
            mCanZoomOut = canZoomOut(pageIndex, mFocusAnnot);
            showPopover(pageIndex, 0);
        }

        if (oldObj != null) {
            int pageIndex = oldObj.mPageIndex;
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                RectF rect = new RectF(oldObj.mBBox);
                mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, pageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, rect, pageIndex);

                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).extentBoundsToContainControl(rect);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rect));
            }
        }

        if (mFocusAnnot != null) {
            int pageIndex = mFocusAnnot.mPageIndex;
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                RectF rect = new RectF(mFocusAnnot.mBBox);
                mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, pageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, rect, pageIndex);
                UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).extentBoundsToContainControl(rect);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rect));
            }
        }
    }

    private void adjustCaretInDocViewer() {
        if (!isEditingText())
            return;

        int pageIndex = _editView().getPageIndex();
        if (!mPdfViewCtrl.isPageVisible(pageIndex)) {
            return;
        }

        if (_editView().getText().toString().length() == 0) {
            return;
        }

        if (getBottomOffset() != 0) {
            correctDvOffsetValue();
        }

        FillSignEditText et = _editView();
        et.splitTextLines();

        ArrayList<String> lines = et.mLineTexts.mLineTexts;
        int caretIndex = et.getSelectionEnd();
        int lineCount = lines.size();
        int lineIndex = 0;
        int charCount = 0;
        for (; lineIndex < lines.size(); lineIndex++) {
            String text = lines.get(lineIndex);
            if (charCount + text.length() >= caretIndex) {
                break;
            }
            charCount += text.length();
        }

        if (lineIndex < lineCount) {
            caretIndex = caretIndex - charCount;
            String str = lines.get(lineIndex);
            str = str.substring(0, caretIndex);

            if (str.length() > 0 && str.charAt(str.length() - 1) == '\n' && lines.size() > lineIndex + 1) {
                lineIndex += 1;
                caretIndex -= str.length();
                str = lines.get(lineIndex);
                str = str.substring(0, caretIndex);
            }

            int scrollY = et.getScrollY();
            float lineHeight = et.getLineHeight();
            float caretY = lineHeight * lineIndex - scrollY;
            float caretX = et.getPaddingLeft();

            if (str.length() > 0) {
                float[] strLens = new float[str.length()];
                et.getPaint().getTextWidths(str, strLens);

                for (int j = 0; j < str.length(); j++) {
                    caretX = caretX + strLens[j];
                }
            }

            RectF caretRt = new RectF(caretX - lineHeight / 2, caretY, caretX + lineHeight / 2, caretY + lineHeight);
            PointF offset = new PointF(_editView().getDocLtOffset().x, _editView().getDocLtOffset().y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(offset, offset, pageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offset, offset, pageIndex);
            caretRt.offset(offset.x, offset.y);

            mPdfViewCtrl.convertDisplayViewRectToPageViewRect(caretRt, caretRt, pageIndex);
            mPdfViewCtrl.convertPageViewRectToPdfRect(caretRt, caretRt, pageIndex);
            adjustBoxToDisplayArea(pageIndex, caretRt);

            if (getBottomOffset() != 0) {
                correctDvOffsetValue();
            }
        }
    }

    private boolean isPopoverShowing() {
        return UIPopoverWin.isPopoverShowing();
    }

    private void dismissPopover() {
        UIPopoverWin.dismissPopover();
    }

    private void showPopover(int pageIndex, int action) {
        RectF bbox = new RectF();
        if (isEditingText()) {
            bbox.set(_editView().getLeft(), _editView().getTop(), _editView().getRight(), _editView().getBottom());
        } else if (mFocusAnnot != null) {
            bbox = new RectF(mFocusAnnot.mBBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
        } else {
            if (action != 2 || !mPdfViewCtrl.isPageVisible(mLongPressPage)) return;

            bbox.set(mLongPressPt.x, mLongPressPt.y, mLongPressPt.x + 2, mLongPressPt.y + 2);
            mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
        }

        showPopover(action, bbox);
    }

    private void showPopover(int action, RectF dvBox) {
        if (dvBox.left == 0 && dvBox.top == 0) return;

        mPopoverAction = action;
        Rect dvRect = new Rect();
        mPdfViewCtrl.getGlobalVisibleRect(dvRect);
        Rect rect = new Rect((int) dvBox.left, (int) dvBox.top, (int) dvBox.right, (int) dvBox.bottom);
        rect.offset(dvRect.left, dvRect.top);

        UIPopoverWin popoverWin = UIPopoverWin.showPopover((FragmentActivity) mUIExtensionsManager.getAttachedActivity(),
                mUIExtensionsManager.getRootView(),
                getPopoverItems(action),
                mClickListener,
                rect,
                getArrowPosition());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            popoverWin.updateTheme();
        }
    }

    private final IBaseItem.OnItemClickListener mClickListener = new IBaseItem.OnItemClickListener() {
        @Override
        public void onClick(IBaseItem item, View v) {
            int curPageIndex;
            if (isEditingText()) {
                curPageIndex = _editView().getPageIndex();
            } else if (mFocusAnnot != null) {
                curPageIndex = mFocusAnnot.mPageIndex;
            } else {
                curPageIndex = mLongPressPage;
            }

            if (!mPdfViewCtrl.isPageVisible(curPageIndex)) {
                return;
            }

            int itemTag = item.getTag();
            switch (itemTag) {
                case FillSignConstants.ZOOM_OUT_ICON: { // smaller
                    if (isEditingText()) {
                        float fontSize = mProperty.mFontSize / mProperty.mZoomScale;
                        if (fontSize < FillSignProperty.MIN_FONTSIZE) {
                            mCanZoomOut = false;
                            item.setEnable(false);
                            return;
                        }

                        mCanZoomIn = true;
                        mCanZoomOut = true;
                        mProperty.setFontSize(fontSize);
                        _editView().setTextSize(mProperty.getFontSizeDp(mPdfViewCtrl, curPageIndex));
                    } else if (mFocusAnnot != null) {
                        RectF objRect = new RectF(mFocusAnnot.mBBox);
                        RectF viewRectF = new RectF();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(objRect, viewRectF, curPageIndex);

                        float objWidth = Math.abs(viewRectF.width());
                        float objHeight = Math.abs(viewRectF.height());
                        float width = objWidth / mProperty.mZoomScale;
                        float height = objHeight / mProperty.mZoomScale;
                        float dx = (objWidth - width) / 2;
                        float dy = (objHeight - height) / 2;
                        viewRectF.inset(dx, dy);

                        RectF docRect = new RectF();
                        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, docRect, curPageIndex);
                        if (!canZoomOut(mFocusAnnot.mTag, mProperty.mZoomScale, docRect)) {
                            mCanZoomOut = false;
                            item.setEnable(false);
                            return;
                        }

                        mCanZoomIn = true;
                        mCanZoomOut = true;
                        mFocusAnnot.mBBox.set(docRect);
                        if (mFocusAnnot.mTag == ToolConstants.FillSignTypeText || mFocusAnnot.mTag == ToolConstants.FillSignTypeComboText) {
                            mProperty.setFontSize(mFocusAnnot.mFontSize / mProperty.mZoomScale);
                            mFocusAnnot.mFontSize = mProperty.mFontSize;
                            mFocusAnnot.mCharspace = mProperty.mFontSize * mFocusAnnot.mSpacing;
                        } else {
                            setProperty(curPageIndex, docRect);
                        }
                        modifyCurFormObject(curPageIndex);
                    } else {
                        break;
                    }
                    updatePopover(curPageIndex, mPopoverAction);
                    break;
                }
                case FillSignConstants.ZOOM_IN_ICON: { // larger
                    if (isEditingText()) {
                        RectF rectF = getEditViewBoxInDv();
                        RectF objRect = new RectF(rectF);
                        RectF viewRectF = new RectF();
                        mPdfViewCtrl.convertDisplayViewRectToPageViewRect(objRect, viewRectF, curPageIndex);

                        float objWidth = Math.abs(viewRectF.width());
                        float objHeight = Math.abs(viewRectF.height());

                        float width = objWidth * mProperty.mZoomScale;
                        float height = objHeight * mProperty.mZoomScale;
                        float dx = (objWidth - width) / 2;
                        float dy = (objHeight - height) / 2;
                        viewRectF.inset(dx, dy);

                        boolean canZoom = canZoomIn(mPdfViewCtrl, curPageIndex, viewRectF);
                        if (!canZoom) {
                            mCanZoomIn = false;
                            item.setEnable(false);
                            return;
                        }

                        mCanZoomIn = true;
                        mCanZoomOut = true;
                        mProperty.setFontSize(mProperty.mFontSize * mProperty.mZoomScale);
                        _editView().setTextSize(mProperty.getFontSizeDp(mPdfViewCtrl, curPageIndex));
                    } else if (mFocusAnnot != null) {
                        RectF objRect = new RectF(mFocusAnnot.mBBox);
                        RectF viewRectF = new RectF();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(objRect, viewRectF, curPageIndex);

                        float objWidth = Math.abs(viewRectF.width());
                        float objHeight = Math.abs(viewRectF.height());

                        float width = objWidth * mProperty.mZoomScale;
                        float height = objHeight * mProperty.mZoomScale;
                        float dx = (objWidth - width) / 2;
                        float dy = (objHeight - height) / 2;
                        viewRectF.inset(dx, dy);

                        boolean canZoom = canZoomIn(mPdfViewCtrl, curPageIndex, viewRectF);
                        if (!canZoom) {
                            mCanZoomIn = false;
                            item.setEnable(false);
                            return;
                        }

                        mCanZoomIn = true;
                        mCanZoomOut = true;
                        RectF docRect = new RectF();
                        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, docRect, curPageIndex);
                        mFocusAnnot.mBBox.set(docRect);

                        if (mFocusAnnot.mTag == ToolConstants.FillSignTypeText || mFocusAnnot.mTag == ToolConstants.FillSignTypeComboText) {
                            mProperty.setFontSize(mFocusAnnot.mFontSize * mProperty.mZoomScale);
                            mFocusAnnot.mFontSize = mProperty.mFontSize;
                            mFocusAnnot.mCharspace = mProperty.mFontSize * mFocusAnnot.mSpacing;
                        } else {
                            setProperty(curPageIndex, docRect);
                        }
                        modifyCurFormObject(curPageIndex);
                    } else {
                        break;
                    }
                    updatePopover(curPageIndex, mPopoverAction);
                    break;
                }
                case FillSignConstants.ONE_LEVEL_MORE: { // more
                    showPopover(curPageIndex, 1);
                    break;
                }
                case FillSignConstants.DELETE_ICON: // delete
                    if (isEditingText()) {
                        deleteEditView();
                        if (mNeedAddDeletedUndoItem) {
                            mUndoItemList.add(mTempTextDeleteUndoItem);
                            mUIExtensionsManager.getDocumentManager().addUndoItem(mTempTextDeleteUndoItem);
                            mNeedAddDeletedUndoItem = false;
                        }
                    } else if (mFocusAnnot != null) {
                        deleteCurFormObject(curPageIndex);
                    }
                    break;
                case FillSignConstants.CHECKMARK_ICON: // check
                case FillSignConstants.CROSSMARK_ICON: // x
                case FillSignConstants.DOT_ICON: // dot
                case FillSignConstants.LINE_ICON: // line
                case FillSignConstants.RECTANGLE_ICON: // rect
                {
                    int formTag = 0;
                    if (itemTag == FillSignConstants.CHECKMARK_ICON) {
                        formTag = ToolConstants.FillSignTypeCheckMark;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_V);
                    } else if (itemTag == FillSignConstants.CROSSMARK_ICON) {
                        formTag = ToolConstants.FillSignTypeCrossMark;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_X);
                    } else if (itemTag == FillSignConstants.DOT_ICON) {
                        formTag = ToolConstants.FillSignTypeDot;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Dot);
                    } else if (itemTag == FillSignConstants.LINE_ICON) {
                        formTag = ToolConstants.FillSignTypeLine;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Line);
                    } else {
                        formTag = ToolConstants.FillSignTypeRectangle;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Rectangle);
                    }

                    if (isEditingText()) {
                        deleteEditView();

                        RectF textBox = getEditViewBoxInDv();
                        PointF ltPt = new PointF(textBox.left, textBox.top);
                        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(ltPt, ltPt, curPageIndex);
                        addFormObject(formTag, curPageIndex, ltPt, true);
                    } else if (mFocusAnnot != null) {
                        mFocusAnnot.mTag = formTag;

                        RectF viewRectF = new RectF();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(mFocusAnnot.mBBox, viewRectF, curPageIndex);
                        float centerX = viewRectF.centerX();
                        float centerY = viewRectF.centerY();
                        if (itemTag == FillSignConstants.CHECKMARK_ICON
                                || itemTag == FillSignConstants.CROSSMARK_ICON
                                || itemTag == FillSignConstants.DOT_ICON) {
                            float checkSize = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, curPageIndex, mProperty.mCheckSize);
                            viewRectF.left = centerX - checkSize / 2;
                            viewRectF.right = centerX + checkSize / 2;
                            viewRectF.top = centerY - checkSize / 2;
                            viewRectF.bottom = centerY + checkSize / 2;
                        } else if (itemTag == FillSignConstants.LINE_ICON) {
                            float lineSizeX = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, curPageIndex, mProperty.mLineSize.x);
                            float lineSizeY = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, curPageIndex, mProperty.mLineSize.y);

                            viewRectF.left = centerX - lineSizeX / 2;
                            viewRectF.right = centerX + lineSizeX / 2;
                            viewRectF.top = centerY - lineSizeY / 2;
                            viewRectF.bottom = centerY + lineSizeY / 2;
                        } else {
                            float rectSizeX = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, curPageIndex, mProperty.mRectSize.x);
                            float rectSizeY = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, curPageIndex, mProperty.mRectSize.y);

                            viewRectF.left = centerX - rectSizeX / 2;
                            viewRectF.right = centerX + rectSizeX / 2;
                            viewRectF.top = centerY - rectSizeY / 2;
                            viewRectF.bottom = centerY + rectSizeY / 2;
                        }

                        PointF adjust = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).calculateTranslateCorrection(mPdfViewCtrl, curPageIndex, viewRectF);
                        viewRectF.offset(adjust.x, adjust.y);
                        RectF docRect = new RectF();
                        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, docRect, curPageIndex);
                        mFocusAnnot.mBBox.set(docRect);
                        modifyCurFormObject(curPageIndex);
                        updatePopover(curPageIndex, 0);
                    } else {
                        PointF viewPointF = new PointF();
                        mPdfViewCtrl.convertPdfPtToPageViewPt(mLongPressPt, viewPointF, curPageIndex);
                        addFormObject(formTag, curPageIndex, viewPointF, false);
                        mLongPressPage = -1;
                        mLongPressPt = null;
                    }
                    break;
                }
                case FillSignConstants.TEXT_ICON: // text
                case FillSignConstants.COMBOTEXT_ICON: { // combo text
                    int formTag;
                    if (itemTag == FillSignConstants.TEXT_ICON) {
                        formTag = ToolConstants.FillSignTypeText;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_TypeWriter);
                    } else {
                        formTag = ToolConstants.FillSignTypeComboText;
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_ComboText);
                    }

                    if (isEditingText()) {
                        if (formTag == ToolConstants.FillSignTypeComboText) {
                            _editView().setTextStyle(FillSignEditText.STYLE_COMBO_TEXT);
                            _editView().setLetterSpacing(mProperty.mFontSpacing);
                            _editView().setPadding(0, 0, _editView().getStratchBmp().getWidth(), 0);
                        } else {
                            _editView().setTextStyle(FillSignEditText.STYLE_TEXT);
                            _editView().setLetterSpacing(0);
                            _editView().setPadding(0, 0, 0, 0);
                        }

                        updatePopover(curPageIndex, 0);
                    } else if (mFocusAnnot != null) {
                        _editFocusTextFormObj(curPageIndex, formTag);
                    } else {
                        PointF viewPointF = new PointF();
                        mPdfViewCtrl.convertPdfPtToPageViewPt(mLongPressPt, viewPointF, curPageIndex);
                        addTextBox(formTag, curPageIndex, viewPointF, true, null, null, false);
                        mLongPressPage = -1;
                        mLongPressPt = null;
                    }
                    break;
                }
                case FillSignConstants.SECOND_LEVEL_MORE: { // more
                    showPopover(curPageIndex, 0);
                    break;
                }
                default:
                    break;
            }
        }
    };

    private void _editFocusTextFormObj(int pageIndex, int formTag) {
        if (mFocusAnnot == null) {
            return;
        }
        RectF pvBox = new RectF(mFocusAnnot.mBBox);
        mPdfViewCtrl.convertPdfRectToPageViewRect(pvBox, pvBox, pageIndex);

        float fontSize = 0;
        String text = null;
        int tag = mFocusAnnot.mTag;
        if (tag == ToolConstants.FillSignTypeText || tag == ToolConstants.FillSignTypeComboText) {
            fontSize = mFocusAnnot.mFontSize;
            text = mFocusAnnot.mContent;
        }

        final FormObject oldObj = mOldAnnot;
        deleteFormObject(pageIndex, mFocusAnnot, false, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    mNeedAddDeletedUndoItem = true;
                    mTempTextDeleteUndoItem = (FillSignDeleteUndoItem) ((FillSignEvent) event).mUndoItem;
                }
                mOldAnnot = oldObj;
            }
        });
        addTextBox(formTag, pageIndex, new PointF(pvBox.left, pvBox.top), true, fontSize, text, true);
    }

    private void setEditViewMargin(int l, int t) {
        setEditViewMargin(true, l, t);
    }

    private void setEditViewMargin(boolean editing, int l, int t) {
        if (!isEditingText())
            return;
        if (editing && !AppDisplay.isLandscape() && t > AppDisplay.getActivityHeight() * 2 / 3)
            return;

        RelativeLayout.LayoutParams lp = (RelativeLayout.LayoutParams) _editView().getLayoutParams();
        lp.leftMargin = l;
        lp.topMargin = t;
        _editView().setLayoutParams(lp);

        if (_editView().getParent() != null) {
            _editView().getParent().requestLayout();
        }
    }

    private void modifyCurFormObject(int pageIndex) {
        if (mFocusAnnot == null)
            return;

        modifyFormObject(pageIndex, mOldAnnot, mFocusAnnot, -1, true, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (mFocusAnnot != null)
                    mOldAnnot = mFocusAnnot.clone();
                else
                    mOldAnnot = null;
            }
        });
    }

    void modifyFormObject(final int pageIndex, final FormObject oldVal, final FormObject curVal,
                          int rotation, final boolean addUndo, final Event.Callback callback) {
        if (oldVal == null || curVal == null)
            return;

        try {
            if (addUndo) {
                final FillSignObject fillSignObject = getFillSignObject(curVal.mPageIndex, curVal.mJniAddr);
                if (fillSignObject == null) return;

                final FillSignModifyUndoItem undoItem = new FillSignModifyUndoItem(mPdfViewCtrl, this);
                if (fillSignObject.getType() == FillSign.e_FillSignObjectTypeText) {
                    float lineHeight = 0;
                    if (curVal.mFontSize > 0) {
                        float fontSize = mProperty.getFontSizeDp(mPdfViewCtrl, pageIndex, curVal.mFontSize);
                        lineHeight = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, getLineHeight(fontSize));
                    }

                    undoItem.mFontSize = curVal.mFontSize;
                    undoItem.mTexts = FillSignUtils.jniToJavaTextLines(curVal.mContent);
                    undoItem.mContent = curVal.mContent;
                    undoItem.mCharspace = curVal.mCharspace;
                    undoItem.mLineHeight = lineHeight;
                    undoItem.mIsCombText = (curVal.mTag == ToolConstants.FillSignTypeComboText);

                    undoItem.mRedoFontSize = curVal.mFontSize;
                    undoItem.mRedoContent = curVal.mContent;
                    undoItem.mRedoTexts = FillSignUtils.jniToJavaTextLines(curVal.mContent);
                    undoItem.mRedoCharspace = curVal.mCharspace;

                    undoItem.mUndoFontSize = oldVal.mFontSize;
                    undoItem.mUndoContent = oldVal.mContent;
                    undoItem.mUndoTexts = FillSignUtils.jniToJavaTextLines(oldVal.mContent);
                    undoItem.mUndoCharspace = oldVal.mCharspace;
                }

                undoItem.mPageIndex = pageIndex;
                undoItem.mRectF = new RectF(curVal.mBBox);
                undoItem.mType = getFillObjType(curVal.mTag);
                PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                undoItem.mRotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;

                undoItem.mRedoRectF = new RectF(curVal.mBBox);
                undoItem.mRedoType = getFillObjType(curVal.mTag);
                undoItem.mUndoRectF = new RectF(oldVal.mBBox);
                undoItem.mUndoType = getFillObjType(oldVal.mTag);

                final RectF lastRectF = AppUtil.toRectF(fillSignObject.getRect());
                FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_MODIFY, fillSignObject, undoItem, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            try {
                                FillSignObject newFillObj = ((FillSignEvent) event).mFillSignObj;
                                RectF newRectF = AppUtil.toRectF(newFillObj.getRect());

                                long jniAddr = FormXObject.getCPtr(newFillObj.getFormXObject());
                                updateUndoItemJniAddr(curVal.mJniAddr, jniAddr);
                                if (mFocusAnnot != null) {
                                    if (mFocusAnnot.mJniAddr == curVal.mJniAddr) {
                                        mFocusAnnot.mJniAddr = jniAddr;
                                        mFocusAnnot.mBBox = new RectF(newRectF);

                                        if (mOldAnnot != null)
                                            mOldAnnot.mJniAddr = jniAddr;
                                    }
                                    mCanZoomIn = canZoomIn(pageIndex, mFocusAnnot);
                                    mCanZoomOut = canZoomOut(pageIndex, mFocusAnnot);
                                }

                                undoItem.setOldValue(oldVal.clone());
                                undoItem.setCurrentValue(curVal.clone());
                                mUndoItemList.add(undoItem);
                                mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);

                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF objRectF = new RectF(newRectF);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(objRectF, objRectF, pageIndex);
                                    RectF oldRectF = new RectF(lastRectF);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(oldRectF, oldRectF, pageIndex);
                                    objRectF.union(oldRectF);
                                    float defHeight = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mProperty.mLineSize.y);
                                    objRectF.inset(-defHeight, -defHeight);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(objRectF));
                                }
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }

                        if (callback != null)
                            callback.result(event, success);
                    }
                });
                mPdfViewCtrl.addTask(task);
            } else {
                FillSignObject curObj = getFillSignObject(pageIndex, curVal.mJniAddr);
                if (curObj == null) return;

                final RectF lastRectF = AppUtil.toRectF(curObj.getRect());
                if (oldVal.mTag != curVal.mTag
                        || curVal.mTag == ToolConstants.FillSignTypeText
                        || curVal.mTag == ToolConstants.FillSignTypeComboText) {

                    FillSign fillSign = getFillSign(pageIndex);
                    if (fillSign == null) return;
                    fillSign.removeObject(curObj);
                    if (curVal.mTag == ToolConstants.FillSignTypeText
                            || curVal.mTag == ToolConstants.FillSignTypeComboText) {
                        addTextFormObj(curVal, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                if (success) {
                                    try {
                                        FillSignObject newFillObj = ((FillSignEvent) event).mFillSignObj;
                                        RectF newRectF = AppUtil.toRectF(newFillObj.getRect());
                                        long newAddr = FormXObject.getCPtr(newFillObj.getFormXObject());

                                        updateUndoItemJniAddr(curVal.mJniAddr, newAddr);
                                        if (mFocusAnnot != null && mFocusAnnot.mJniAddr == curVal.mJniAddr) {
                                            mFocusAnnot.mJniAddr = newAddr;
                                            if (mOldAnnot != null)
                                                mOldAnnot.mJniAddr = newAddr;
                                        }
                                        refreshPage(pageIndex, lastRectF, newRectF);
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                        });
                    } else {
                        addFormObject(curVal.mTag, curVal.mPageIndex, curVal.mBBox, rotation, curVal.mJniAddr, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                if (success) {
                                    try {
                                        FillSignObject newFillObj = ((FillSignEvent) event).mFillSignObj;
                                        RectF newRectF = AppUtil.toRectF(newFillObj.getRect());
                                        long newAddr = FormXObject.getCPtr(newFillObj.getFormXObject());

                                        updateUndoItemJniAddr(curVal.mJniAddr, newAddr);
                                        if (mFocusAnnot != null && mFocusAnnot.mJniAddr == curVal.mJniAddr) {
                                            mFocusAnnot.mJniAddr = newAddr;
                                            if (mOldAnnot != null)
                                                mOldAnnot.mJniAddr = newAddr;
                                        }
                                        refreshPage(pageIndex, lastRectF, newRectF);
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                        });
                    }
                } else {
                    RectF newRectF = new RectF(curVal.mBBox);
                    float width = Math.abs(newRectF.width());
                    float height = Math.abs(newRectF.height());
                    PointF pointF = new PointF(newRectF.centerX() - width / 2, newRectF.centerY() - height / 2);
                    if (rotation == Constants.e_Rotation90 || rotation == Constants.e_Rotation270) {
                        curObj.move(AppUtil.toFxPointF(pointF), height, width, rotation);
                    } else {
                        curObj.move(AppUtil.toFxPointF(pointF), width, height, rotation);
                    }
                    curObj.generateContent();
                    refreshPage(pageIndex, lastRectF, newRectF);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void refreshPage(int pageIndex, RectF lastRectF, RectF newRectF) {
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
            RectF objRectF = new RectF(newRectF);
            mPdfViewCtrl.convertPdfRectToPageViewRect(objRectF, objRectF, pageIndex);
            RectF oldRectF = new RectF(lastRectF);
            mPdfViewCtrl.convertPdfRectToPageViewRect(oldRectF, oldRectF, pageIndex);
            objRectF.union(oldRectF);

            float defHeight = FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, mProperty.mLineSize.y);
            objRectF.inset(-defHeight, -defHeight);
            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(objRectF));
        }
    }

    private void deleteCurFormObject(int pageIndex) {
        if (mFocusAnnot == null) return;
        deleteFormObject(pageIndex, mFocusAnnot, true, null);
    }

    void deleteFormObject(final int pageIndex, final FormObject obj, final boolean addUndo, final Event.Callback callback) {
        if (obj == null) return;

        boolean isCurObj = false;
        if (mFocusAnnot != null && mFocusAnnot.mJniAddr == obj.mJniAddr)
            isCurObj = true;

        try {
            final FillSignDeleteUndoItem undoItem = new FillSignDeleteUndoItem(mPdfViewCtrl, this);
            undoItem.mJniAddr = obj.mJniAddr;
            undoItem.mPageIndex = obj.mPageIndex;
            int type = getFillObjType(obj.mTag);
            undoItem.mType = type;
            undoItem.mRectF = new RectF(obj.mBBox);
            undoItem.mSize = new PointF(Math.abs(obj.mBBox.width()), Math.abs(obj.mBBox.height()));

            PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            undoItem.mRotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            if (obj.mTag == ToolConstants.FillSignTypeComboText || obj.mTag == ToolConstants.FillSignTypeText) {
                undoItem.mFontSize = obj.mFontSize;
                undoItem.mCharspace = obj.mCharspace;
                undoItem.mContent = obj.mContent;
                undoItem.mTexts = FillSignUtils.jniToJavaTextLines(obj.mContent);
            }

            FillSignObject fillSignObject = getFillSignObject(pageIndex, obj.mJniAddr);
            if (fillSignObject == null) return;

            FillSignEvent event = new FillSignEvent(EditAnnotEvent.EVENTTYPE_DELETE, fillSignObject, undoItem, mPdfViewCtrl);
            final boolean finalIsCurObj = isCurObj;
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        undoItem.setOldValue(obj.clone());
                        if (addUndo) {
                            mUndoItemList.add(undoItem);
                            mUIExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                        }

                        if (finalIsCurObj)
                            focusObject(null);
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF viewRect = new RectF(obj.mBBox);
                            mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                        }
                    }

                    if (callback != null)
                        callback.result(event, success);
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void deleteFormObject(final int pageIndex, final FormObject obj) {
        if (obj == null) return;

        boolean isCurObj = false;
        if (mFocusAnnot != null && mFocusAnnot.mJniAddr == obj.mJniAddr)
            isCurObj = true;

        try {
            if (obj.mJniAddr != 0) {
                FillSign fillSign = getFillSign(pageIndex);
                if (fillSign == null) return;
                FillSignObject fillSignObject = getFillSignObject(pageIndex, obj.mJniAddr);
                if (fillSignObject == null) return;
                RectF viewRectF = AppUtil.toRectF(fillSignObject.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(viewRectF, viewRectF, pageIndex);

                fillSign.removeObject(fillSignObject);
                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                    viewRectF.inset(-10, -10);
                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRectF));
                }
            }

            if (isCurObj)
                focusObject(null);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private int getArrowPosition() {
        return UIPopoverFrag.ARROW_AUTO;
    }

    private FillSignObject getFillSignObject(int pageIndex, long jniAddr) {
        try {
            FormXObject formXObject = new FormXObject(jniAddr, false);
            FillSign fillSign = getFillSign(pageIndex);
            if (fillSign == null) return null;

            FillSignObject fillSignObject = fillSign.getObjectByFormXObject(formXObject);
            return fillSignObject.isEmpty() ? null : fillSignObject;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    private ArrayList<UIPopoverWin.POPOVER_ITEM> getPopoverItems(int action) {
        ArrayList<UIPopoverWin.POPOVER_ITEM> itemList = new ArrayList<>();
        switch (action) {
            case 0:
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.ZOOM_OUT_ICON, R.drawable.fillsign_tool_smaller_white, mCanZoomOut));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.ZOOM_IN_ICON, R.drawable.fillsign_tool_larger_white, mCanZoomIn));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.ONE_LEVEL_MORE, R.drawable.fillsign_tool_more_white));

                int tag = _curItemOrToolTag();
                if (tag == ToolConstants.FillSignTypeComboText)
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.TEXT_ICON, R.drawable.fillsign_tool_text_white));
                else if (tag == ToolConstants.FillSignTypeText && AppBuildConfig.SDK_VERSION >= 21)
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.COMBOTEXT_ICON, R.drawable.fillsign_tool_combotext_white));

                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.DELETE_ICON, R.drawable.fillsign_tool_delete_white));
                break;
            case 1:
                int curTag = _curItemOrToolTag();
                if (curTag != ToolConstants.FillSignTypeCheckMark) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.CHECKMARK_ICON, R.drawable.fillsign_tool_checkmark_white));
                }
                if (curTag != ToolConstants.FillSignTypeCrossMark) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.CROSSMARK_ICON, R.drawable.fillsign_tool_crossmark_white));
                }
                if (curTag != ToolConstants.FillSignTypeDot) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.DOT_ICON, R.drawable.fillsign_tool_dot_white));
                }
                if (curTag != ToolConstants.FillSignTypeLine) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.LINE_ICON, R.drawable.fillsign_tool_line_white));
                }
                if (curTag != ToolConstants.FillSignTypeRectangle) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.RECTANGLE_ICON, R.drawable.fillsign_tool_rect_white));
                }
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.SECOND_LEVEL_MORE, R.drawable.fillsign_tool_more_white));
                break;
            case 2:
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.CHECKMARK_ICON, R.drawable.fillsign_tool_checkmark_white));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.CROSSMARK_ICON, R.drawable.fillsign_tool_crossmark_white));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.DOT_ICON, R.drawable.fillsign_tool_dot_white));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.LINE_ICON, R.drawable.fillsign_tool_line_white));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.RECTANGLE_ICON, R.drawable.fillsign_tool_rect_white));
                itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.TEXT_ICON, R.drawable.fillsign_tool_text_white));
                if (AppBuildConfig.SDK_VERSION >= 21) {
                    itemList.add(new UIPopoverWin.POPOVER_ITEM(FillSignConstants.COMBOTEXT_ICON, R.drawable.fillsign_tool_combotext_white));
                }
                break;
        }
        return itemList;
    }

    private int _curToolTag() {
        int tag = 0;
        if (mCurToolItem != null)
            tag = mCurToolItem.type;
        return tag;
    }

    private int _curItemOrToolTag() {
        if (_curItemTag() != 0)
            return _curItemTag();
        if (_curToolTag() != 0)
            return _curToolTag();
        return 0;
    }

    private int _curItemTag() {
        int tag = 0;
        if (isEditingText()) {
            if (_editView().getTextStyle() == FillSignEditText.STYLE_TEXT) {
                return ToolConstants.FillSignTypeText;
            } else {
                return ToolConstants.FillSignTypeComboText;
            }
        }

        if (mFocusAnnot != null) {
            tag = mFocusAnnot.mTag;
        }
        return tag;
    }

    private void correctDvOffsetValue() {
        int offset = getBottomOffset();
        int toolbarHeight = mKeyboardBar.getBarHeight();
        int navBarHeight = AppDisplay.getRealNavBarHeight();
        int maskHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView()) + toolbarHeight + navBarHeight;
        int offset2 = Math.max(0, Math.min(maskHeight, offset));

        if (!mPdfViewCtrl.isContinuous()) {
            int viewerHeight = mPdfViewCtrl.getDisplayViewHeight();
            int pageViewHeight = mPdfViewCtrl.getPageViewHeight(mPdfViewCtrl.getCurrentPage());
            if (pageViewHeight < viewerHeight) {
                if (offset2 + (viewerHeight - pageViewHeight) / 2 > maskHeight) {
                    offset2 = offset2 - (offset2 + (viewerHeight - pageViewHeight) / 2 - maskHeight);
                }
            }
        }

        offset2 = Math.max(0, offset2);
        if (offset2 != offset) {
            setBottomOffset(offset2);
        }
    }

    private void adjustNavBarPadding() {
        Rect rect = new Rect();
        mUIExtensionsManager.getRootView().getGlobalVisibleRect(rect);
        int padding;
        int top = rect.top;
        int bottom = rect.bottom;
        int screenHeight = AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity()) ?
                AppDisplay.getActivityHeight() : AppDisplay.getRawScreenHeight();

        int keyboardHeight;
        if (SystemUiHelper.getInstance().isFullScreenMode(mUIExtensionsManager.getAttachedActivity())) {
            keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView(), true);
        } else {
            if (SystemUiHelper.getInstance().isFullScreen())
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView(), true);
            else
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView(), false);
        }
        if ((screenHeight - bottom - AppDisplay.getNavBarHeight()) >= keyboardHeight) {
            padding = 0;
        } else if (screenHeight - top > keyboardHeight && screenHeight - bottom - AppDisplay.getNavBarHeight() < keyboardHeight) {
            if (screenHeight < (bottom + AppDisplay.getNavBarHeight())) {
                padding = keyboardHeight + AppDisplay.getNavBarHeight();
            } else {
                padding = keyboardHeight;
            }
        } else {
            padding = 0;
        }

        Rect curRect = mKeyboardBar.getPadding();
        if (padding != curRect.bottom) {
            curRect.bottom = padding;
            mKeyboardBar.setPadding(0, 0, 0, Math.max(0, curRect.bottom));
        }
    }

    private void hideStatusBar(Activity activity) {
        SystemUiHelper.getInstance().hideStatusBar(activity);
    }

    private void adjustBoxToDisplayArea(int pageIndex, RectF docBox) {
        int margin = 100;
        int toolbarHeight = mKeyboardBar.getBarHeight();
        int navBarHeight = AppDisplay.getRealNavBarHeight();
        int maskHeight = AppKeyboardUtil.getKeyboardHeight(mUIExtensionsManager.getRootView()) + navBarHeight + toolbarHeight;
        int viewerWidth = mPdfViewCtrl.getDisplayViewWidth();
        int viewerHeight = mPdfViewCtrl.getDisplayViewHeight();

        int offset = 0; //getBottomOffset();
        RectF viewArea = new RectF(0, 0, viewerWidth, viewerHeight);
        viewArea.bottom -= maskHeight;
        viewArea.offset(0, offset);

        RectF boxInPv = new RectF(docBox);
        mPdfViewCtrl.convertPdfRectToPageViewRect(boxInPv, boxInPv, pageIndex);

        RectF boxInDv = new RectF(boxInPv);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(boxInDv, boxInDv, pageIndex);

        if (viewArea.contains(boxInDv)) {
            return;
        }

        PointF jumpPt = new PointF();
        // left and right
        if (boxInDv.width() > viewArea.width() - margin) {
            if (boxInDv.left < 0) {
                jumpPt.x = boxInDv.left - margin;
            } else if (boxInDv.left > margin) {
                jumpPt.x = boxInDv.left - margin;
            }
        } else if (boxInDv.left < 0) {
            jumpPt.x = boxInDv.left - margin;
        } else if (boxInDv.right > viewArea.right) {
            jumpPt.x = boxInDv.right - viewArea.right + margin;
        }

        // top and bottom
        if (boxInDv.height() < viewArea.height() - margin) {
            if (boxInDv.bottom > viewArea.bottom) {
                jumpPt.y = boxInDv.bottom - viewArea.bottom + margin;
            } else if (boxInDv.top < viewArea.top) {
                jumpPt.y = boxInDv.top - viewArea.top - margin;
            }
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(jumpPt, jumpPt, pageIndex);
            mPdfViewCtrl.gotoPage(pageIndex, jumpPt.x, jumpPt.y);

            // recalculate box after jump to page
            boxInPv = new RectF(docBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(boxInPv, boxInPv, pageIndex);

            boxInDv = new RectF(boxInPv);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(boxInDv, boxInDv, pageIndex);

            if (boxInDv.bottom > viewArea.bottom) {
                offset += boxInDv.bottom - viewArea.bottom;
                setBottomOffset(offset, maskHeight);
            } else if (boxInDv.top < viewArea.top) {
                offset -= viewArea.top - boxInDv.top;
                setBottomOffset(offset, maskHeight);
            }
        } else {
            jumpPt.y = boxInDv.top - viewArea.top - margin;

            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(jumpPt, jumpPt, pageIndex);
            mPdfViewCtrl.gotoPage(pageIndex, jumpPt.x, jumpPt.y);

            boxInPv = new RectF(docBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(boxInPv, boxInPv, pageIndex);

            boxInDv = new RectF(boxInPv);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(boxInDv, boxInDv, pageIndex);

            if (boxInDv.top > viewArea.top + margin) {
                offset += boxInDv.top - viewArea.top - margin;
                setBottomOffset(offset, maskHeight);
            } else if (boxInDv.top < viewArea.top + margin) {
                offset -= viewArea.top + margin - boxInDv.top;
                setBottomOffset(offset, maskHeight);
            }
        }
    }

    private void setBottomOffset(int offset, int maskHeight) {
        offset = Math.max(0, Math.min(maskHeight, offset));
        setBottomOffset(offset);
    }

    private int mBottomOffset;

    private int getBottomOffset() {
        return -mBottomOffset;
    }

    private void setBottomOffset(int offset) {
        if (mBottomOffset == -offset)
            return;
        mBottomOffset = -offset;
        mPdfViewCtrl.layout(0, mBottomOffset, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight() + mBottomOffset);
    }

    private void updateUndoItemJniAddr(long oldObj, long newObj) {
        for (int i = 0; i < mUndoItemList.size(); i++) {
            mUndoItemList.get(i).updateJniAddr(oldObj, newObj);
        }
    }

    FillSign getFillSign(int pageIndex) {
        if (pageIndex < 0) return null;

        try {
            FillSign fillSign = mFillSignArrs.get(pageIndex);
            if (fillSign == null) {
                PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(pageIndex, false);
                fillSign = new FillSign(page);
                mFillSignArrs.put(pageIndex, fillSign);
            }
            return fillSign;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    void onPagesRemoved(boolean success, int index) {
        if (success) {
            for (FillSignUndoItem undoItem : mUndoItemList) {
                if (undoItem.mCurFormObj != null)
                    undoItem.mCurFormObj.mPageIndex = undoItem.mPageIndex;
                if (undoItem.mOldFormObj != null)
                    undoItem.mOldFormObj.mPageIndex = undoItem.mPageIndex;
            }

            ArrayList<Integer> invalidList = new ArrayList<>();
            SparseArray<FillSign> tempArrs = new SparseArray<>();
            int size = mFillSignArrs.size();
            for (int i = 0; i < size; i++) {
                int pageIndex = mFillSignArrs.keyAt(i);
                FillSign fillSign = mFillSignArrs.valueAt(i);
                if (pageIndex == index) {
                    invalidList.add(index);
                } else if (pageIndex > index) {
                    pageIndex -= 1;
                }
                tempArrs.put(pageIndex, fillSign);
            }

            for (Integer pageIndex : invalidList) {
                tempArrs.remove(pageIndex);
            }
            mFillSignArrs = tempArrs.clone();
            invalidList.clear();
            tempArrs.clear();
        }
    }

    void onPageMoved(boolean success, int index, int dstIndex) {
        if (success) {
            for (FillSignUndoItem undoItem : mUndoItemList) {
                if (undoItem.mCurFormObj != null)
                    undoItem.mCurFormObj.mPageIndex = undoItem.mPageIndex;
                if (undoItem.mOldFormObj != null)
                    undoItem.mOldFormObj.mPageIndex = undoItem.mPageIndex;
            }

            SparseArray<FillSign> tempArrs = new SparseArray<>();
            int size = mFillSignArrs.size();
            for (int i = 0; i < size; i++) {
                int pageIndex = mFillSignArrs.keyAt(i);
                FillSign fillSign = mFillSignArrs.valueAt(i);

                if (index < dstIndex) {
                    if (pageIndex <= dstIndex && pageIndex > index) {
                        pageIndex -= 1;
                    } else if (pageIndex == index) {
                        pageIndex = dstIndex;
                    }
                } else {
                    if (pageIndex >= dstIndex && pageIndex < index) {
                        pageIndex += 1;
                    } else if (pageIndex == index) {
                        pageIndex = dstIndex;
                    }
                }
                tempArrs.put(pageIndex, fillSign);
            }

            mFillSignArrs = tempArrs.clone();
            tempArrs.clear();
        }
    }

    void onPagesInserted(boolean success, int dstIndex, int[] pageRanges) {
        if (success) {
            int offsetIndex = 0;
            for (int i = 0; i < pageRanges.length / 2; i++) {
                offsetIndex += pageRanges[2 * i + 1];
            }

            for (FillSignUndoItem undoItem : mUndoItemList) {
                if (undoItem.mCurFormObj != null)
                    undoItem.mCurFormObj.mPageIndex = undoItem.mPageIndex;
                if (undoItem.mOldFormObj != null)
                    undoItem.mOldFormObj.mPageIndex = undoItem.mPageIndex;
            }

            SparseArray<FillSign> tempArrs = new SparseArray<>();
            int size = mFillSignArrs.size();
            for (int i = 0; i < size; i++) {
                int pageIndex = mFillSignArrs.keyAt(i);
                FillSign fillSign = mFillSignArrs.valueAt(i);

                if (pageIndex >= dstIndex) {
                    pageIndex += offsetIndex;
                }
                tempArrs.put(pageIndex, fillSign);
            }
            mFillSignArrs = tempArrs.clone();
            tempArrs.clear();
        }
    }

    void release(boolean clearFillSigns) {
        if (mFocusAnnot != null)
            focusObject(null);
        if (isPopoverShowing())
            dismissPopover();

        if (clearFillSigns) {
            int size = mFillSignArrs.size();
            for (int i = 0; i < size; i++) {
                FillSign fillSign = mFillSignArrs.valueAt(i);
                if (!fillSign.isEmpty())
                    fillSign.delete();
            }
            mFillSignArrs.clear();
        }

        mOldAnnot = null;
        mLongPressPage = -1;
        mLongPressPt = null;
        removeMagnifierView();
    }

    private void setProperty(int pageIndex, RectF docRectF) {
        RectF viewRect = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(docRectF, viewRect, pageIndex);
        float actualWidth = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, Math.abs(viewRect.width()));
        float actualHeight = FillSignUtils.pageViewToDocThickness(mPdfViewCtrl, pageIndex, Math.abs(viewRect.height()));
        if (mFocusAnnot.mTag == ToolConstants.FillSignTypeCheckMark
                || mFocusAnnot.mTag == ToolConstants.FillSignTypeCrossMark
                || mFocusAnnot.mTag == ToolConstants.FillSignTypeDot) {
            mProperty.setCheckSize(actualWidth);
        } else if (mFocusAnnot.mTag == ToolConstants.FillSignTypeLine) {
            mProperty.setLineSize(new PointF(actualWidth, mProperty.mLineSize.y));
        } else if (mFocusAnnot.mTag == ToolConstants.FillSignTypeRectangle) {
            mProperty.setRectSize(new PointF(actualWidth, actualHeight));
        }
    }

    private boolean canZoomOut(int pageIndex, FormObject formObject) {
        if (formObject == null) return false;

        RectF objRect = new RectF(formObject.mBBox);
        RectF viewRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(objRect, viewRectF, pageIndex);

        float objWidth = Math.abs(viewRectF.width());
        float objHeight = Math.abs(viewRectF.height());
        float width = objWidth / mProperty.mZoomScale;
        float height = objHeight / mProperty.mZoomScale;
        float dx = (objWidth - width) / 2;
        float dy = (objHeight - height) / 2;
        viewRectF.inset(dx, dy);

        RectF docRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, docRect, pageIndex);
        return canZoomOut(formObject.mTag, mProperty.mZoomScale, docRect);
    }

    private boolean canZoomOut(int tag, float scale, @NonNull RectF docRect) {
        float width = Math.abs(docRect.width());
        float height = Math.abs(docRect.width());

        switch (tag) {
            case ToolConstants.FillSignTypeCheckMark:
            case ToolConstants.FillSignTypeCrossMark:
            case ToolConstants.FillSignTypeDot:
                return width >= FillSignProperty.MIN_CHECKSIZE;
            case ToolConstants.FillSignTypeText:
            case ToolConstants.FillSignTypeProFile:
            case ToolConstants.FillSignTypeComboText:
                return mFocusAnnot.mFontSize / scale >= FillSignProperty.MIN_FONTSIZE;
            case ToolConstants.FillSignTypeLine:
                return width >= FillSignProperty.MIN_LINESIZE;
            case ToolConstants.FillSignTypeRectangle:
                return width >= FillSignProperty.MIN_RECT_X && height >= FillSignProperty.MIN_RECT_Y;
            default:
                break;
        }
        return true;
    }

    private boolean canZoomIn(int pageIndex, FormObject formObject) {
        if (formObject == null) return false;

        RectF objRect = new RectF(formObject.mBBox);
        RectF viewRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(objRect, viewRectF, pageIndex);

        float objWidth = Math.abs(viewRectF.width());
        float objHeight = Math.abs(viewRectF.height());

        float width = objWidth * mProperty.mZoomScale;
        float height = objHeight * mProperty.mZoomScale;
        float dx = (objWidth - width) / 2;
        float dy = (objHeight - height) / 2;
        viewRectF.inset(dx, dy);
        return canZoomIn(mPdfViewCtrl, pageIndex, viewRectF);
    }

    private boolean canZoomIn(PDFViewCtrl pdfViewCtrl, int pageIndex, RectF rectF) {
        float extent = UIAnnotFrame.getInstance(mPdfViewCtrl.getAttachedActivity()).getControlExtent();
        if (rectF.left < extent
                || rectF.right > pdfViewCtrl.getPageViewWidth(pageIndex) - extent
                || rectF.top < extent
                || rectF.bottom > pdfViewCtrl.getPageViewHeight(pageIndex) - extent) {
            return false;
        }
        return true;
    }

    void onScaleEnd() {
        int itemTag = _curItemOrToolTag();
        if (itemTag == ToolConstants.FillSignTypeText
                || itemTag == ToolConstants.FillSignTypeComboText) {
            if (isEditingText()) {
                PointF offsetPointF = _editView().getDocLtOffset();
                float textSizePx = _editView().getTextSize();
                int pageIndex = _editView().getPageIndex();

                PointF pvPt = new PointF();
                mPdfViewCtrl.convertPdfPtToPageViewPt(offsetPointF, pvPt, pageIndex);
                RectF textBox = new RectF(pvPt.x, pvPt.y, (pvPt.x + textSizePx / 2), (pvPt.y + textSizePx));
                if (!mAddLTPointF) {
                    textBox.offset(-textBox.width() / 2, -textBox.height() / 2);
                }
                int mw = (int) (mPdfViewCtrl.getPageViewWidth(pageIndex) - pvPt.x);
                int mh = (int) (mPdfViewCtrl.getPageViewHeight(pageIndex) - pvPt.y);

                _editView().setMinWidth((int) textBox.width());
                _editView().setMinHeight((int) textBox.height());
                _editView().setMaxWidth(mw);
                _editView().setMaxHeight(mh);
                _editView().setMarginRight(FillSignUtils.docToPageViewThickness(mPdfViewCtrl, pageIndex, FillSignEditText.SPACING));
            }
        }
    }

    private int mLastCreateType;
    private IToolSupply mToolSupply;

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new FillSignToolSupply(mContext);
        return mToolSupply;
    }

    ToolItemBean getCurToolItem() {
        return mCurToolItem;
    }

    private class FillSignToolSupply extends ToolSupplyImpl {

        public FillSignToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            int bgIcon = 0;
            if (ToolConstants.FillSignTypeText == toolType) {
                bgIcon = R.drawable.fillsign_tool_text;
            } else if (ToolConstants.FillSignTypeComboText == toolType) {
                bgIcon = R.drawable.fillsign_tool_combotext;
            } else if (ToolConstants.FillSignTypeProFile == toolType) {
                bgIcon = R.drawable.fillsign_tool_profile;
            } else if (ToolConstants.FillSignTypeCheckMark == toolType) {
                bgIcon = R.drawable.fillsign_tool_checkmark;
            } else if (ToolConstants.FillSignTypeCrossMark == toolType) {
                bgIcon = R.drawable.fillsign_tool_crossmark;
            } else if (ToolConstants.FillSignTypeDot == toolType) {
                bgIcon = R.drawable.fillsign_tool_dot;
            } else if (ToolConstants.FillSignTypeLine == toolType) {
                bgIcon = R.drawable.fillsign_tool_line;
            } else if (ToolConstants.FillSignTypeRectangle == toolType) {
                bgIcon = R.drawable.fillsign_tool_rect;
            } else if (ToolConstants.DigitalSignature == toolType) {
                bgIcon = R.drawable.fillsign_tool_sign;
            }
            return bgIcon;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            return null;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_FIllSIGN;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            if (itemBean.toolItem.isSelected()) {
                mCurToolItem = itemBean;
                mLastCreateType = mCreateType;
                mCreateType = itemBean.type;
                if (mUIExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_FILLSIGN_TAB) {
                    if (mCreateType == ToolConstants.FillSignTypeText) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_TypeWriter);
                    } else if (mCreateType == ToolConstants.FillSignTypeComboText) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_ComboText);
                    } else if (mCreateType == ToolConstants.FillSignTypeProFile) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Predefined);
                    } else if (mCreateType == ToolConstants.FillSignTypeCheckMark) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_V);
                    } else if (mCreateType == ToolConstants.FillSignTypeCrossMark) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_X);
                    } else if (mCreateType == ToolConstants.FillSignTypeDot) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Dot);
                    } else if (mCreateType == ToolConstants.FillSignTypeLine) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Line);
                    } else if (mCreateType == ToolConstants.FillSignTypeRectangle) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Rectangle);
                    } else if (mCreateType == ToolConstants.DigitalSignature) {
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FillSign_Sign);
                    }
                }
                if (mCreateType == ToolConstants.FillSignTypeProFile) {
                    if (isEditingText()) {
                        mPdfViewCtrl.addTask(new Task(new Task.CallBack() {
                            @Override
                            public void result(Task task) {
                                showProfileDialog();
                            }
                        }) {
                            @Override
                            protected void execute() {
                            }
                        });
                    } else {
                        showProfileDialog();
                    }
                } else {
                    if (mCreateType == ToolConstants.DigitalSignature) {
                        mSignModule.activeSign(mCurToolItem.toolItem, true);
                    }

                    onItemClicked(mCreateType);
                    mUIExtensionsManager.setCurrentToolHandler(FillSignToolHandler.this);
                }
            } else {
                int curTag = _curItemOrToolTag();
                if (curTag != 0) {
                    if (mCreateType != ToolConstants.FillSignTypeProFile) {
                        if (mCreateType == ToolConstants.DigitalSignature)
                            mSignModule.activeSign(mCurToolItem.toolItem, false);
                        onItemClicked(mCreateType);
                    }

                    mCreateType = mLastCreateType;
                    mCurToolItem = null;
                    mUIExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
        }

        @Override
        public PropertyBar getPropertyBar() {
            return null;
        }

    }

    private UIMatchDialog mProfileDialog;

    public void dismissProfileDialog() {
        if (mProfileDialog != null) {
            mProfileDialog.dismiss();
            mProfileDialog = null;
        }
    }

    private  boolean mIsEdit = false;
    private void showProfileDialog() {
        final ArrayList<ProFileItem> items = new ArrayList<>();
        final int customStartIndex = loadItemsFromSp(items);

        Activity activity = mUIExtensionsManager.getAttachedActivity();
        mProfileDialog = new UIMatchDialog(activity);
        final ViewGroup rootLayout = (ViewGroup) View.inflate(mContext, R.layout.fillsign_profile, null);

        final ScrollView scrollView = rootLayout.findViewById(R.id.fillsign_profile_scrollview);
        final ViewGroup tableView = rootLayout.findViewById(R.id.fillsign_profile_table);
        final View addCustomView = rootLayout.findViewById(R.id.profile_bottom_bar);
        UIBtnImageView addCustomViewIcon = rootLayout.findViewById(R.id.profile_add_custom_icon);
        addCustomView.setVisibility(View.VISIBLE);
        mIsEdit = false;
        ThemeUtil.setTintList(addCustomViewIcon, ThemeUtil.getPrimaryIconColor(mContext));
        final View.OnClickListener editListener = new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!mIsEdit) {
                    mIsEdit = true;
                    mProfileDialog.setRightButtonText(AppResource.getString(mContext, R.string.fx_string_done));
                    mProfileDialog.setTitle(AppResource.getString(mContext, R.string.fillsign_profile_edit_title));
                } else {
                    mIsEdit = false;
                    mProfileDialog.setRightButtonText(AppResource.getString(mContext, R.string.fx_string_edit));
                    mProfileDialog.setTitle(AppResource.getString(mContext, R.string.fillsign_profile_title));
                }

                boolean editable = mIsEdit;
                if (!editable) {
                    saveItemsToSp(tableView, items);
                    AppKeyboardUtil.hideInputMethodWindow(mContext, mProfileDialog.getWindow());
                }

                for (int i = 0; i < items.size(); i++) {
                    ProFileItem item = items.get(i);
                    if (!item.mIsGroupTitle) {
                        if (!editable) {
                            item.mSubjectEt.clearFocus();
                            item.mContentEt.clearFocus();
                        }
                        if (item.mIsCustom) {
                            item.mSubjectEt.setFocusable(editable);
                            item.mSubjectEt.setFocusableInTouchMode(editable);
                            item.mDeleteView.setVisibility(editable ? View.VISIBLE : View.GONE);
                        }
                        if (!AppUtil.isEqual(item.mSpKey, mProfileInfo.KEY_DATE)) {
                            item.mContentEt.setFocusable(editable);
                            item.mContentEt.setFocusableInTouchMode(editable);
                        }
                    }
                }
            }
        };

        for (int i = 0; i < items.size(); i++) {
            ProFileItem item = items.get(i);
            addProfileItem(mProfileDialog, tableView, addCustomView, editListener, items, item);
        }

        View.OnClickListener addCustomViewListener = new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mIsEdit = true;
                mProfileDialog.setRightButtonText(AppResource.getString(mContext, R.string.fx_string_done));
                mProfileDialog.setTitle(AppResource.getString(mContext, R.string.fillsign_profile_edit_title));

                if (items.size() == customStartIndex) {
                    ProFileItem item = new ProFileItem(true, true, AppResource.getString(mContext, R.string.fillsign_profile_custom_field).toUpperCase(), null, null, null, 0, 0);
                    items.add(item);
                    addProfileItem(mProfileDialog, tableView, addCustomView, editListener, items, item);
                }

                final ProFileItem item = new ProFileItem(false, true, "", AppResource.getString(mContext, R.string.fillsign_profile_type_value), "", null, 0, 0);
                items.add(item);
                addProfileItem(mProfileDialog, tableView, addCustomView, editListener, items, item);

                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                    @Override
                    public void run() {
                        scrollView.fullScroll(ScrollView.FOCUS_DOWN);

                        item.mSubjectEt.requestFocus();
                        AppUtil.showSoftInput(item.mSubjectEt);
                    }
                });
            }
        };
        addCustomView.setOnClickListener(addCustomViewListener);
        addCustomViewIcon.setOnClickListener(addCustomViewListener);

        final boolean[] isSelctedText = {false};
        mProfileDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
                if (btType == MatchDialog.DIALOG_OK) {
                    isSelctedText[0] = true;
                }
                mProfileDialog.dismiss();
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                editListener.onClick(mProfileDialog.getRootView());
            }
        });

        mProfileDialog.setOnDLDismissListener(new MatchDialog.DismissListener() {
            @Override
            public void onDismiss() {
                if (isSelctedText[0]) {
                    mUIExtensionsManager.setCurrentToolHandler(FillSignToolHandler.this);
                } else {
                    if (mCurToolItem != null)
                        mCurToolItem.toolItem.performClick();
                }
            }
        });

        mProfileDialog.setContentView(rootLayout);
        mProfileDialog.setTitle(AppResource.getString(mContext, R.string.fillsign_profile_title));
        mProfileDialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        mProfileDialog.setBackButtonStyle(UIMatchDialog.TEXT_BACK);
        mProfileDialog.setRightButtonVisible(View.VISIBLE);
        mProfileDialog.setRightButtonText(AppResource.getString(mContext, R.string.fx_string_edit));
        mProfileDialog.showDialog();
        if (mProfileDialog.getWindow() != null)
            mProfileDialog.getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_HIDDEN);
    }

    private void addProfileItem(final UIMatchDialog dlg, final ViewGroup tableView, final View bottomBar,
                                final View.OnClickListener editListener, final ArrayList<ProFileItem> items, final ProFileItem item) {
        if (item.mIsGroupTitle) {
            TextView tv = new TextView(mContext);
            tv.setText(item.mSubject);
            tv.setBackgroundColor(AppResource.getColor(mContext, R.color.b2, null));
            tv.setGravity(Gravity.CENTER_VERTICAL | Gravity.START);
            tv.setPadding(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp), 0, 0, 0);
            tv.setTextColor(AppResource.getColor(mContext, R.color.t3, null));
            tv.setTextSize(AppDisplay.px2dp(AppResource.getDimension(mContext, R.dimen.ux_text_size_12sp)));

            item.mItemRootView = tv;
            tableView.addView(tv, new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, AppDisplay.dp2px(24)));
        } else {
            final View itemView = View.inflate(mContext, R.layout.fillsign_profile_item, null);
            itemView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            final EditText subjectEt = itemView.findViewById(R.id.fs_item_subject);
            final EditText contentEt = itemView.findViewById(R.id.fs_item_content);
            if (item.mSubEtID > 0)
                subjectEt.setId(item.mSubEtID);
            if (item.mContentEtID > 0)
                contentEt.setId(item.mContentEtID);

            subjectEt.setOnEditorActionListener(new TextView.OnEditorActionListener() {
                @Override
                public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
                    if (actionId == EditorInfo.IME_ACTION_DONE) {
                        AppUtil.dismissInputSoft(subjectEt);
                        return true;
                    }
                    return false;
                }
            });
            contentEt.setOnEditorActionListener(new TextView.OnEditorActionListener() {
                @Override
                public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
                    if (actionId == EditorInfo.IME_ACTION_DONE) {
                        AppUtil.dismissInputSoft(contentEt);
                        return true;
                    }
                    return false;
                }
            });

            subjectEt.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {

                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    if (s.toString().contains("\n")) {
                        subjectEt.setText(s.toString().replace("\n", ""));
                    }
                }

                @Override
                public void afterTextChanged(Editable s) {

                }
            });
            contentEt.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {

                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    if (s.toString().contains("\n")) {
                        contentEt.setText(s.toString().replace("\n", ""));
                    }
                }

                @Override
                public void afterTextChanged(Editable s) {

                }
            });

            View deleteView = itemView.findViewById(R.id.fs_item_delete);
            ThemeUtil.setTintList(deleteView, ThemeUtil.getPrimaryIconColor(mContext));
            subjectEt.setHint(AppResource.getString(mContext, R.string.fillsign_profile_type_label));
            subjectEt.setText(item.mSubject);
            contentEt.setHint(item.mContentHint);
            contentEt.setText(item.mContent);

            item.mItemRootView = itemView;
            item.mSubjectEt = subjectEt;
            item.mContentEt = contentEt;
            item.mDeleteView = deleteView;

            tableView.addView(itemView, new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT));
            if (mIsEdit) {
                if (item.mIsCustom) {
                    item.mSubjectEt.setFocusable(true);
                    item.mSubjectEt.setFocusableInTouchMode(true);
                }
                item.mContentEt.setFocusable(true);
                item.mContentEt.setFocusableInTouchMode(true);

                if (item.mIsCustom) {
                    deleteView.setVisibility(View.VISIBLE);
                }
            }

            View.OnClickListener listener = new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if(mIsEdit){
                        item.mContentEt.setFocusable(true);
                        item.mContentEt.setFocusableInTouchMode(true);
                        item.mContentEt.requestFocus();
                        return;
                    } else if (AppUtil.isEmpty(item.mContentEt.getText().toString())) {
                        editListener.onClick(null);
                        item.mContentEt.requestFocus();
                        AppUtil.showSoftInput(item.mContentEt);
                        return;
                    }

                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fillsign_sign_click_prompt));
                    mProfileStr = item.mContentEt.getText().toString();
                    dlg.getDialogListerner().onResult(UIMatchDialog.DIALOG_OK);
                }
            };
           subjectEt.setOnClickListener(listener);
            contentEt.setOnClickListener(listener);

            deleteView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    _removeProfileItem(items, item, tableView);
                }
            });
        }
    }

    private int loadItemsFromSp(ArrayList<ProFileItem> items) {
        items.add(new ProFileItem(true, false, AppResource.getString(mContext, R.string.fx_string_name).toUpperCase(), null, null, null,
                0, 0));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_full_name), AppResource.getString(mContext, R.string.fillsign_profile_full_name_prompt), mProfileInfo.getFullName(), mProfileInfo.KEY_FULL_NAME,
                R.id.fillsign_pro_sub_fullname, R.id.fillsign_pro_content_fullname));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_first_name), AppResource.getString(mContext, R.string.fillsign_profile_first_name_prompt), mProfileInfo.getFirstName(), mProfileInfo.KEY_FIRST_NAME,
                R.id.fillsign_pro_sub_firstname, R.id.fillsign_pro_content_firstname));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_middle_name), AppResource.getString(mContext, R.string.fillsign_profile_middle_name_prompt), mProfileInfo.getMiddleName(), mProfileInfo.KEY_MIDDLE_NAME,
                R.id.fillsign_pro_sub_middlename, R.id.fillsign_pro_content_middlename));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_last_name), AppResource.getString(mContext, R.string.fillsign_profile_last_name_prompt), mProfileInfo.getLastName(), mProfileInfo.KEY_LAST_NAME,
                R.id.fillsign_pro_sub_lastname, R.id.fillsign_pro_content_lastname));

        items.add(new ProFileItem(true, false, AppResource.getString(mContext, R.string.fx_string_address).toUpperCase(), null, null, null, 0, 0));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_street_1), AppResource.getString(mContext, R.string.fillsign_profile_street_1_prompt), mProfileInfo.getStreet1(), mProfileInfo.KEY_STREET_1,
                R.id.fillsign_pro_sub_street1, R.id.fillsign_pro_content_street1));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_street_2), AppResource.getString(mContext, R.string.fillsign_profile_street_2_prompt), mProfileInfo.getStreet2(), mProfileInfo.KEY_STREET_2,
                R.id.fillsign_pro_sub_street2, R.id.fillsign_pro_content_street2));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_city), AppResource.getString(mContext, R.string.fillsign_profile_city_prompt), mProfileInfo.getCity(), mProfileInfo.KEY_CITY,
                R.id.fillsign_pro_sub_city, R.id.fillsign_pro_content_city));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_state), AppResource.getString(mContext, R.string.fillsign_profile_state_prompt), mProfileInfo.getState(), mProfileInfo.KEY_STATE,
                R.id.fillsign_pro_sub_state, R.id.fillsign_pro_content_state));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_zip), AppResource.getString(mContext, R.string.fillsign_profile_zip_prompt), mProfileInfo.getPostCode(), mProfileInfo.KEY_POSTCODE,
                R.id.fillsign_pro_sub_zip, R.id.fillsign_pro_content_zip));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_country), AppResource.getString(mContext, R.string.fillsign_profile_country_prompt), mProfileInfo.getCountry(), mProfileInfo.KEY_COUNTRY,
                R.id.fillsign_pro_sub_country, R.id.fillsign_pro_content_county));

        items.add(new ProFileItem(true, false, AppResource.getString(mContext, R.string.fillsign_profile_contact).toUpperCase(), null, null, null, 0, 0));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_email), AppResource.getString(mContext, R.string.fillsign_profile_email_prompt), mProfileInfo.getEmail(), mProfileInfo.KEY_EMAIL,
                R.id.fillsign_pro_sub_email, R.id.fillsign_pro_content_email));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_tel), AppResource.getString(mContext, R.string.fillsign_profile_tel_prompt), mProfileInfo.getTel(), mProfileInfo.KEY_TEL,
                R.id.fillsign_pro_sub_tel, R.id.fillsign_pro_content_tel));

        items.add(new ProFileItem(true, false, AppResource.getString(mContext, R.string.fillsign_profile_dates).toUpperCase(), null, null, null, 0, 0));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_date), AppResource.getString(mContext, R.string.fillsign_profile_date_prompt), mProfileInfo.getDate(), mProfileInfo.KEY_DATE,
                R.id.fillsign_pro_sub_date, R.id.fillsign_pro_content_date));
        items.add(new ProFileItem(false, false, AppResource.getString(mContext, R.string.fillsign_profile_birth_date), AppResource.getString(mContext, R.string.fillsign_profile_birth_date_prompt), mProfileInfo.getBirtyDate(), mProfileInfo.KEY_BIRTH_DATE,
                R.id.fillsign_pro_sub_birthdate, R.id.fillsign_pro_content_birthdate));

        final int customStartIndex = items.size();

        final ArrayList<String> customFields = mProfileInfo.getCustomFields();
        final ArrayList<String> customValues = mProfileInfo.getCustomValues();
        if (customFields.size() > 0) {
            items.add(new ProFileItem(true, true, AppResource.getString(mContext, R.string.fillsign_profile_custom_field).toUpperCase(), null, null, null, 0, 0));
            for (int i = 0; i < customFields.size(); i++) {
                String field = customFields.get(i);
                String value = customValues.get(i);
                items.add(new ProFileItem(false, true, field, AppResource.getString(mContext, R.string.fillsign_profile_type_value), value, null, 0, 0));
            }
        }

        return customStartIndex;
    }

    private void saveItemsToSp(ViewGroup tableView, ArrayList<ProFileItem> items) {
        ArrayList<ProFileItem> removeItems = new ArrayList<>();
        ArrayList<String> customFields = new ArrayList<>();
        ArrayList<String> customValues = new ArrayList<>();

        for (int i = 0; i < items.size(); i++) {
            ProFileItem item = items.get(i);
            if (item.mIsGroupTitle)
                continue;

            item.mSubject = item.mSubjectEt.getText().toString();
            item.mContent = item.mContentEt.getText().toString();

            if (!item.mIsCustom) {
                mProfileInfo.setString(item.mSpKey, item.mContent);
            } else {
                if (AppUtil.isEmpty(item.mSubject)) {
                    removeItems.add(item);
                } else {
                    customFields.add(item.mSubject);
                    customValues.add(item.mContent);
                }
            }
        }

        mProfileInfo.saveCustomFields(customFields, customValues);

        for (int i = 0; i < removeItems.size(); i++) {
            _removeProfileItem(items, removeItems.get(i), tableView);
        }

        ProFileItem lastItem = items.get(items.size() - 1);
        if (lastItem.mIsGroupTitle) {
            _removeProfileItem(items, lastItem, tableView);
        }
    }

    private void _removeProfileItem(ArrayList<ProFileItem> items, ProFileItem item, ViewGroup tableView) {
        items.remove(item);
        tableView.removeView(item.mItemRootView);
    }

    static class ProFileItem {
        private boolean mIsGroupTitle;
        private boolean mIsCustom;
        private String mSubject;
        private String mContentHint;
        private String mContent;
        private String mSpKey;

        private View mItemRootView;
        private EditText mSubjectEt;
        private EditText mContentEt;
        private View mDeleteView;

        private int mSubEtID;
        private int mContentEtID;

        ProFileItem(boolean isTitle, boolean isCustom, String subject, String hint, String content, String key, int subID, int contentID) {
            mIsGroupTitle = isTitle;
            mIsCustom = isCustom;
            mSubject = subject;
            mContentHint = hint;
            mContent = content;
            mSpKey = key;
            mSubEtID = subID;
            mContentEtID = contentID;
        }
    }

    void onConfigurationChanged(Configuration newConfig) {
        if (mProfileDialog != null && mProfileDialog.isShowing()) {
            mProfileDialog.resetWH();
            mProfileDialog.showDialog();
        }

        if (isEditingText()) {
            endAddTextBox();
        }
    }

    void setSignModule(SignatureModule signModule) {
        mSignModule = signModule;
        if (signModule != null)
            mSignatureToolHandler = (SignatureToolHandler) signModule.getToolHandler();
    }

    void activateSignature(ToolItemBean toolItemBean) {
        mLastCreateType = mCreateType;

        mCurToolItem = toolItemBean;
        toolItemBean.toolItem.setChecked(true);
        mCreateType = toolItemBean.type;
        onItemClicked(mCreateType);
        mUIExtensionsManager.setCurrentToolHandler(FillSignToolHandler.this);
    }

    private UIMagnifierView mMagnifierView;

    private void addMagnifierView() {
        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
        if (mMagnifierView == null) {
            mMagnifierView = new UIMagnifierView(mContext.getApplicationContext());
        }
        if (mainFrame.getContentView().indexOfChild(mMagnifierView) == -1) {
            mMagnifierView.setTargetView(mPdfViewCtrl);
            mMagnifierView.setVisibility(View.GONE);
            mainFrame.getContentView().addView(mMagnifierView);
        }
    }

    private void removeMagnifierView() {
        if (mMagnifierView != null) {
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
            mainFrame.getContentView().removeView(mMagnifierView);
            mMagnifierView.setTargetView(null);
        }
    }

}
