/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;


import android.content.Context;
import android.graphics.Rect;
import android.os.Handler;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.IResult;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;

public class ChoiceOptionsAdapter extends SuperAdapter<ChoiceItemInfo> {
    private static final int ADD_CHOICE_OPTION = 1;
    private static final int RENAME_CHOICE_OPTION = 2;

    private PDFViewCtrl mPDFViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private ISheetMenu mSheetMenu;
    private OnOptionItemChangedListener mOptionItemChangedListener;
    private SelectMode mSelectMode = SelectMode.SINGLE_SELECT;

    private ArrayList<ChoiceItemInfo> mChoiceInfos = new ArrayList<>();
    private ArrayList<Integer> mSelectedIndexs = new ArrayList<>();

    private boolean mIsPad;
    private int mSelectedPosition = -1;
    private int mLastSelectedIndex = -1;

    public ChoiceOptionsAdapter(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mPDFViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();

        mChoiceInfos = new ArrayList<>();
        mIsPad = AppDisplay.isPad();
    }

    void setChoiceOptions(ArrayList<ChoiceItemInfo> itemInfos) {
        mChoiceInfos = itemInfos;
    }

    public ArrayList<ChoiceItemInfo> getChoiceInfos() {
        return mChoiceInfos;
    }

    public enum SelectMode implements Serializable {
        SINGLE_SELECT, MULTI_SELECT
    }

    public void setSelectMode(SelectMode selectMode) {
        this.mSelectMode = selectMode;

        if (mSelectMode == SelectMode.SINGLE_SELECT) {
            mLastSelectedIndex = -1;
            for (int i = 0; i < mChoiceInfos.size(); i++) {
                if (mChoiceInfos.get(i).selected) {
                    mLastSelectedIndex = i;
                    break;
                }
            }
        } else if (mSelectMode == SelectMode.MULTI_SELECT) {
            mSelectedIndexs.clear();

            for (int i = 0; i < mChoiceInfos.size(); i++) {
                if (mChoiceInfos.get(i).selected) {
                    mSelectedIndexs.add(i);
                }
            }
        }
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public ChoiceItemInfo getDataItem(int position) {
        return mChoiceInfos.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        SuperViewHolder viewHolder = new ItemViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.rd_readingmark_item, parent, false));
        return viewHolder;
    }

    @Override
    public int getItemCount() {
        return mChoiceInfos.size();
    }

    private void deleteOption(int position) {
        if (SelectMode.SINGLE_SELECT == mSelectMode) {
            if (mLastSelectedIndex == position)
                mLastSelectedIndex = -1;
            else if (mLastSelectedIndex > position)
                mLastSelectedIndex -= 1;
        } else if (SelectMode.MULTI_SELECT == mSelectMode) {
            for (Integer index : mSelectedIndexs) {
                if (index == position) {
                    mSelectedIndexs.remove((Integer) position);
                    break;
                }
            }
        }
        mChoiceInfos.remove(position);
        notifyItemRemoved(position);
    }

    private void renameOption(final int position) {
        showInputDialog(RENAME_CHOICE_OPTION, mChoiceInfos.get(position).optionValue, new IResult<String, Object, Object>() {
            @Override
            public void onResult(boolean success, String value, Object p2, Object p3) {
                if (success) {
                    ChoiceItemInfo itemInfo = mChoiceInfos.get(position);
                    if (itemInfo.optionLabel.equals(itemInfo.optionValue)) {
                        itemInfo.optionLabel = value;
                    }
                    itemInfo.optionValue = value;

                    if (mOptionItemChangedListener != null)
                        mOptionItemChangedListener.onItemChanged();
                }
                notifyItemChanged(position);
            }
        });
    }

    public void addNewOption() {
        String optionName = "";
        showInputDialog(ADD_CHOICE_OPTION, optionName, new IResult<String, Object, Object>() {
            @Override
            public void onResult(boolean success, String value, Object p2, Object p3) {
                if (success) {
                    int size = mChoiceInfos.size();
                    ChoiceItemInfo itemInfo = new ChoiceItemInfo();
                    itemInfo.selected = true;
                    itemInfo.defaultSelected = false;
                    itemInfo.optionLabel = value;
                    itemInfo.optionValue = value;

                    if (SelectMode.SINGLE_SELECT == mSelectMode) {
                        if (mLastSelectedIndex != -1) {
                            ChoiceItemInfo lastItemInfo = mChoiceInfos.get(mLastSelectedIndex);
                            lastItemInfo.selected = false;
                            notifyItemChanged(mLastSelectedIndex);
                        }
                        mChoiceInfos.add(size, itemInfo);
                        notifyItemInserted(size);

                        mLastSelectedIndex = size;
                    } else if (SelectMode.MULTI_SELECT == mSelectMode) {
                        mSelectedIndexs.clear();
                        for (ChoiceItemInfo choice : mChoiceInfos) {
                            choice.selected = false;
                        }
                        mSelectedIndexs.add(size);
                        mChoiceInfos.add(size, itemInfo);
                        notifyItemInserted(size);
                        notifyUpdateData();
                    }
                }
            }
        });
    }

    private void showInputDialog(int operateType, String defaultInput, final IResult<String, Object, Object> resultCallback) {
        final UITextEditDialog textDialog = new UITextEditDialog(((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getAttachedActivity());
        String title;
        if (ADD_CHOICE_OPTION == operateType) {
            title = AppResource.getString(getContext(), R.string.fx_string_add);
            textDialog.getPromptTextView().setVisibility(View.GONE);
        } else {
            title = AppResource.getString(getContext(), R.string.fx_string_rename);
            textDialog.getPromptTextView().setText(AppResource.getString(getContext(), R.string.fx_rename_hint));
            textDialog.getOKButton().setEnabled(false);
        }
        textDialog.setTitle(title);
        textDialog.getInputEditText().setText(defaultInput);
        textDialog.getInputEditText().selectAll();
        textDialog.show();
        AppUtil.showSoftInput(textDialog.getInputEditText());

        textDialog.getInputEditText().addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
            }

            @Override
            public void afterTextChanged(Editable s) {
                boolean enabled = true;
                if (TextUtils.isEmpty(s.toString())) {
                    enabled = false;
                } else {
                    for (ChoiceItemInfo itemInfo : mChoiceInfos) {
                        if (itemInfo.optionValue.equals(s.toString())) {
                            enabled = false;
                            break;
                        }
                    }
                }
                textDialog.getOKButton().setEnabled(enabled);
            }
        });

        textDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                AppUtil.dismissInputSoft(textDialog.getInputEditText());
                textDialog.dismiss();

                if (resultCallback != null) {
                    resultCallback.onResult(true, textDialog.getInputEditText().getText().toString(), null, null);
                }
            }
        });

        textDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                textDialog.dismiss();
                AppUtil.dismissInputSoft(textDialog.getInputEditText());
                if (resultCallback != null) {
                    resultCallback.onResult(false, null, null, null);
                }
            }
        });
    }

    class ItemViewHolder extends SuperViewHolder {
        private CheckBox mCheckBox;
        private TextView mTvContent;
        private ImageView mIvMore;
        private View mItemView;

        public ItemViewHolder(View viewHolder) {
            super(viewHolder);
            mCheckBox = viewHolder.findViewById(R.id.rd_bookmark_item_checkbox);
            mTvContent = viewHolder.findViewById(R.id.rd_bookmark_item_content);
            mIvMore = viewHolder.findViewById(R.id.rd_panel_item_more);
            mItemView = viewHolder.findViewById(R.id.panel_readingmark_item_conotainer);

            mIvMore.setOnClickListener(this);
            mItemView.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            ChoiceItemInfo itemInfo = (ChoiceItemInfo) data;

            mTvContent.setText(itemInfo.optionValue);
            mCheckBox.setChecked(itemInfo.selected);
            ThemeUtil.setTintList(mCheckBox, ThemeUtil.getCheckboxColor(getContext()));
            ThemeUtil.setTintList(mIvMore, ThemeUtil.getEnableIconColor(getContext()));
            mItemView.setSelected(!mIsPad && mSelectedPosition == position);

        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            if (id == R.id.rd_panel_item_more) {
                final int position = getAdapterPosition();
                mSelectedPosition = position;
                if (mSelectedPosition >= 0)
                    notifyItemChanged(mSelectedPosition);
                mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
                if (mIsPad)
                    mSheetMenu.setWidth(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_pad_more_menu_width));
                List<Integer> items = new ArrayList<>();
                items.add(ISheetMenu.RENAME_MENU);
                items.add(ISheetMenu.DELETE_MENU);
                mSheetMenu.setSheetItems(items);
                mSheetMenu.setSheetItemClickListener(new ISheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        mSheetMenu.dismiss();
                        if (type == ISheetMenu.RENAME_MENU) {
                            renameOption(position);
                        } else if (type == ISheetMenu.DELETE_MENU) {
                            deleteOption(position);
                        }
                    }
                });
                mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                    @Override
                    public void onDismiss(ISheetMenu sheetMenu) {
                        if (mSelectedPosition != -1) {
                            int tempPosition = mSelectedPosition;
                            mSelectedPosition = -1;
                            if (tempPosition >= 0)
                                notifyItemChanged(tempPosition);
                        }
                    }
                });

                showSheetMenu(v);
            } else if (id == R.id.panel_readingmark_item_conotainer) {
                int itemPosition = getAdapterPosition();
                if (SelectMode.SINGLE_SELECT == mSelectMode) {
                    if (mLastSelectedIndex == itemPosition) {
                        ChoiceItemInfo itemInfo = mChoiceInfos.get(itemPosition);
                        itemInfo.selected = !itemInfo.selected;
                        notifyItemChanged(itemPosition);
                    } else {
                        if (mLastSelectedIndex != -1) {
                            ChoiceItemInfo lastItemInfo = mChoiceInfos.get(mLastSelectedIndex);
                            lastItemInfo.selected = false;
                            notifyItemChanged(mLastSelectedIndex);
                        }

                        ChoiceItemInfo curItemInfo = mChoiceInfos.get(itemPosition);
                        curItemInfo.selected = true;
                        notifyItemChanged(itemPosition);
                        mLastSelectedIndex = itemPosition;
                    }
                } else if (SelectMode.MULTI_SELECT == mSelectMode) {
                    ChoiceItemInfo itemInfo = mChoiceInfos.get(itemPosition);
                    if (mSelectedIndexs.contains(itemPosition)) {
                        mSelectedIndexs.remove((Integer) itemPosition);
                        itemInfo.selected = false;
                    } else {
                        mSelectedIndexs.add(itemPosition);
                        itemInfo.selected = true;
                    }
                    notifyItemChanged(itemPosition);
                }

                if (mOptionItemChangedListener != null)
                    mOptionItemChangedListener.onItemChanged();
            }
        }
    }

    private ViewGroup mRootView;

    void setRootView(ViewGroup view) {
        mRootView = view;
    }

    private void showSheetMenu(View view) {
        if (mRootView == null) return;
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);

        int[] location = new int[2];
        view.getLocationOnScreen(location);
        int screen_x = location[0];
        int screen_y = location[1];
        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())){
            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
            screen_x -= location[0];
            screen_y -= location[1];
            rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
        }else {
            view.getLocationInWindow(location);
            int window_x = location[0];
            int window_y = location[1];
            rect.offset(screen_x - window_x, screen_y - window_y);
        }
        mSheetMenu.show(mRootView, rect);
    }

    void updateSheetMenu(final LinearLayoutManager layoutManager) {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (mIsPad) {
                        View view = layoutManager.findViewByPosition(mSelectedPosition);
                        if (view != null) {
                            View moreView = view.findViewById(R.id.rd_panel_item_more);
                            showSheetMenu(moreView);
                        } else {
                            dismissSheetMenu();
                        }
                    } else {
                        showSheetMenu(mUIExtensionsManager.getRootView());
                    }
                }
            }, 200);
        }
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    void setItemChangedListener(OnOptionItemChangedListener checkedChangeListener) {
        mOptionItemChangedListener = checkedChangeListener;
    }

    interface OnOptionItemChangedListener {
        void onItemChanged();
    }
}
