/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;


import android.content.Context;
import android.content.DialogInterface;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.IResult;

import java.util.ArrayList;

import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ChoiceOptionsDialog extends UIMatchDialog {

    private IResult<ArrayList<ChoiceItemInfo>, Object, Object> mPickCallback;
    private ArrayList<ChoiceItemInfo> mItemInfos = new ArrayList<>();

    private PDFViewCtrl mPDFViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private ChoiceOptionsAdapter mOptionsAdapter;
    private RecyclerView mOptionsListView;
    private LinearLayoutManager mLinearLayoutManager;

    private LinearLayout mNoOptionsPromptLl;
    private ImageView mNoOptionsIv;

    private ChoiceOptionsAdapter.SelectMode mSelectMode = ChoiceOptionsAdapter.SelectMode.SINGLE_SELECT;

    private boolean mbDataChanged = false;

    public ChoiceOptionsDialog(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mPDFViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();

        initStyle();
        initView();
    }

    public void init(ChoiceOptionsAdapter.SelectMode selectMode,
                     ArrayList<ChoiceItemInfo> infos,
                     IResult<ArrayList<ChoiceItemInfo>, Object, Object> pickOptionsCallback) {
        mbDataChanged = false;
        mItemInfos = FormFillerUtil.cloneChoiceOptions(infos);
        mPickCallback = pickOptionsCallback;
        mSelectMode = selectMode;
        setRightButtonEnable(false);
        if (mItemInfos.size() > 0) {
            mOptionsListView.setVisibility(View.VISIBLE);
            mNoOptionsPromptLl.setVisibility(View.GONE);
        } else {
            mOptionsListView.setVisibility(View.GONE);
            mNoOptionsPromptLl.setVisibility(View.VISIBLE);
        }
        mOptionsAdapter.setChoiceOptions(mItemInfos);
        mOptionsAdapter.setSelectMode(selectMode);
        mOptionsAdapter.notifyUpdateData();
        mOptionsAdapter.registerAdapterDataObserver(mAdapterDataObserver);
        mUiExtensionsManager.registerLayoutChangeListener(mLayoutChangeListener);
    }

    private void initStyle() {
        setTitle(AppResource.getString(mContext, R.string.fx_form_item_list));
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_cancel));
        setBackButtonVisible(View.VISIBLE);
        setBackButtonStyle(TEXT_BACK);
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        setRightButtonVisible(View.VISIBLE);
        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                mUiExtensionsManager.unregisterLayoutChangeListener(mLayoutChangeListener);
                mOptionsAdapter.unregisterAdapterDataObserver(mAdapterDataObserver);
                if (mPickCallback != null)
                    mPickCallback.onResult(mbDataChanged, mItemInfos, null, null);
            }
        });
        setListener(new DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                mbDataChanged = false;
            }

            @Override
            public void onTitleRightButtonClick() {
                mbDataChanged = true;
                ChoiceOptionsDialog.this.dismiss();
            }
        });
    }

    private void initView() {
        View rootView = View.inflate(mContext, R.layout.form_choice_options_layout, null);
        mNoOptionsPromptLl = rootView.findViewById(R.id.form_no_options_prompt_ll);
        mNoOptionsIv = rootView.findViewById(R.id.form_no_options_prompt_iv);
        mNoOptionsIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mOptionsListView = rootView.findViewById(R.id.form_choice_options_list);
        mOptionsAdapter = new ChoiceOptionsAdapter(mContext, mPDFViewCtrl);
        mOptionsAdapter.setSelectMode(mSelectMode);
        mOptionsAdapter.setRootView(getRootView());
        mOptionsAdapter.setItemChangedListener(new ChoiceOptionsAdapter.OnOptionItemChangedListener() {
            @Override
            public void onItemChanged() {
                setRightButtonEnable(true);
            }
        });

        mOptionsListView.setAdapter(mOptionsAdapter);
        mLinearLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        mOptionsListView.setLayoutManager(mLinearLayoutManager);
        mOptionsListView.setItemAnimator(new DefaultItemAnimator());

        View addOptionView = rootView.findViewById(R.id.form_add_option_ll);
        ImageView ivAddOption = rootView.findViewById(R.id.form_add_option_iv);
        ThemeUtil.setTintList(ivAddOption,ThemeUtil.getPrimaryIconColor(mContext));
        addOptionView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mOptionsAdapter.addNewOption();
            }
        });
        setContentView(rootView);
    }

    private RecyclerView.AdapterDataObserver mAdapterDataObserver = new RecyclerView.AdapterDataObserver() {

        @Override
        public void onItemRangeInserted(int positionStart, int itemCount) {
            mOptionsListView.smoothScrollToPosition(positionStart + itemCount);
            setRightButtonEnable(true);
            if (mOptionsListView.getVisibility() == View.GONE) {
                mNoOptionsPromptLl.setVisibility(View.GONE);
                mOptionsListView.setVisibility(View.VISIBLE);
            }
        }

        @Override
        public void onItemRangeRemoved(int positionStart, int itemCount) {
            super.onItemRangeRemoved(positionStart, itemCount);
            setRightButtonEnable(true);
            if (mOptionsAdapter.getItemCount() == 0) {
                mNoOptionsPromptLl.setVisibility(View.VISIBLE);
                mOptionsListView.setVisibility(View.GONE);
            }
        }

    };

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (oldWidth != newWidth || oldHeight != newHeight) {
                if (isShowing()) {
                    resetWH();
                    mOptionsAdapter.updateSheetMenu(mLinearLayoutManager);
                }
            }
        }
    };

    @Override
    public void themeColorChanged() {
        super.themeColorChanged();
        mNoOptionsIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
    }

}
