/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;

import android.content.Context;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.AnimationUtils;
import android.widget.HorizontalScrollView;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.data.ISuggestionsType;
import com.foxit.uiextensions.data.SuggestionsHelper;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;

public class FormNavigationModule implements Module {
    private static final int MAX_SUGGESTIONS_COUNT = 10;
    private static final int UPDATE_SUGGESTIONS_MSG_WHAT = 100;

    private final Context mContext;
    private ViewGroup mParent;

    private RelativeLayout mFormNavigationLayout;
    private View mNavigationDividerView;
    private View mNavigationContainer;
    private ImageView mPreView;
    private ImageView mNextView;
    private TextView mClearView;
    private TextView mFinishView;

    private LinearLayout mSuggestionsNavigationLayout;
    private String mSuggestKeyWords;

    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public FormNavigationModule(Context context, ViewGroup parent, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this.mContext = context;
        this.mParent = parent;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_FORM_NAVIGATION;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
        }
        return true;
    }

    public void initView() {
        if (mFormNavigationLayout != null) return;
        if (AppDisplay.isPad()) {
            mFormNavigationLayout = (RelativeLayout) View.inflate(mContext, R.layout.rv_form_navigation_pad, null);
        } else {
            mFormNavigationLayout = (RelativeLayout) View.inflate(mContext, R.layout.rv_form_navigation_phone, null);
        }

        mSuggestionsNavigationLayout = mFormNavigationLayout.findViewById(R.id.rv_form_navigation_suggestions_bar);
        mNavigationDividerView = mFormNavigationLayout.findViewById(R.id.rv_form_navigation_divider);
        mNavigationContainer = mFormNavigationLayout.findViewById(R.id.rv_form_navigation_layout);
        mPreView = mFormNavigationLayout.findViewById(R.id.rv_form_pre);
        ThemeUtil.setTintList(mPreView, ThemeUtil.getPrimaryIconColor(mContext));
        mNextView = mFormNavigationLayout.findViewById(R.id.rv_form_next);
        ThemeUtil.setTintList(mNextView, ThemeUtil.getPrimaryIconColor(mContext));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mPreView.setForceDarkAllowed(false);
            mNextView.setForceDarkAllowed(false);
        }
        mClearView = mFormNavigationLayout.findViewById(R.id.rv_form_clear);
        mClearView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mFinishView = mFormNavigationLayout.findViewById(R.id.rv_form_finish);
        mFinishView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));

        mFormNavigationLayout.setVisibility(View.INVISIBLE);
        mFormNavigationLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
            }
        });

        RelativeLayout.LayoutParams lp = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        lp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        lp.addRule(RelativeLayout.CENTER_HORIZONTAL);
        if (AppDisplay.isPad()
                && !AppDevice.isChromeOs(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity())) {
            mNavigationDividerView.setVisibility(View.GONE);
            lp.width = AppDisplay.dp2px(400);
            lp.setMargins(0, 0, 0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_32dp));
            mFormNavigationLayout.setBackground(AppResource.getDrawable(mContext, R.drawable.dlg_bg_4circle_corner_10dp_grayf4f4f4));
        }
        mFormNavigationLayout.setPadding(0, 0, 0, 0);
        mParent.addView(mFormNavigationLayout, lp);
    }

    @Override
    public boolean unloadModule() {
        mParent.removeView(mFormNavigationLayout);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
        }
        mParent = null;
        return true;
    }

    public RelativeLayout getLayout() {
        return mFormNavigationLayout;
    }

    public void show() {
        if (getLayout().getVisibility() != View.VISIBLE) {
            Module module = ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(Module.MODULE_NAME_PAGENAV);
            if (module instanceof PageNavigationModule) {
                ((PageNavigationModule) module).setPageNavigationBarVisible(false);
            }
            getLayout().startAnimation(AnimationUtils.loadAnimation(mContext, R.anim.view_anim_btot_show));
            getLayout().setVisibility(View.VISIBLE);
            boolean containWidgets = AppAnnotUtil.containWidgets(((UIExtensionsManager) mUiExtensionsManager).getPDFViewCtrl());
            getPreView().setEnabled(containWidgets);
            getNextView().setEnabled(containWidgets);
        }
    }

    public void hide() {
        if (getLayout() == null) return;
        if (getLayout().getVisibility() != View.INVISIBLE) {
            Module module = ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(Module.MODULE_NAME_PAGENAV);
            if (module instanceof PageNavigationModule) {
                ((PageNavigationModule) module).setPageNavigationBarVisible(true);
            }
            getLayout().startAnimation(AnimationUtils.loadAnimation(mContext, R.anim.view_anim_btot_hide));
            getLayout().setVisibility(View.INVISIBLE);
        }

        mSuggestKeyWords = "";
        if (haveSuggestions()) {
            hideSuggestionsbar();
        }
    }

    public Rect getPadding() {
        Rect padding = new Rect();
        padding.left = getLayout().getPaddingLeft();
        padding.top = getLayout().getPaddingTop();
        padding.right = getLayout().getPaddingRight();
        padding.bottom = getLayout().getPaddingBottom();
        return padding;
    }

    public void setPadding(int left, int top, int right, int bottom) {
        getLayout().setPadding(left, top, right, bottom);
    }

    public void hideFormNavigationBar() {
        hide();
        setPadding(0, 0, 0, 0);
    }

    public void showFormNavigationBar() {
        if (AppDisplay.isPad()
                && !AppDevice.isChromeOs(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity())
                && mNavigationDividerView.getVisibility() == View.VISIBLE) {
            mNavigationDividerView.setVisibility(View.GONE);
            RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) getLayout().getLayoutParams();
            layoutParams.width = AppDisplay.dp2px(400);
            getLayout().setLayoutParams(layoutParams);
            layoutParams.setMargins(0, 0, 0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_32dp));
            getLayout().setBackground(AppResource.getDrawable(mContext, R.drawable.dlg_bg_4circle_corner_10dp_grayf4f4f4));
        }
        setPadding(0, 0, 0, 0);
    }

    public void showFormNavigationBarWithKeyBoard(int keyBoardHeight) {
        if (keyBoardHeight > 0 && AppDisplay.isPad()
                && !AppDevice.isChromeOs(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity())
                && mNavigationDividerView.getVisibility() == View.GONE) {
            mNavigationDividerView.setVisibility(View.VISIBLE);
            RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) getLayout().getLayoutParams();
            layoutParams.width = RelativeLayout.LayoutParams.MATCH_PARENT;
            getLayout().setLayoutParams(layoutParams);
            layoutParams.setMargins(0, 0, 0, 0);
            getLayout().setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
        }
        setPadding(0, 0, 0, keyBoardHeight);
    }

    public ImageView getPreView() {
        return mPreView;
    }

    public ImageView getNextView() {
        return mNextView;
    }

    public TextView getClearView() {
        return mClearView;
    }

    public TextView getFinishView() {
        return mFinishView;
    }

    public void setClearEnable(boolean enable) {
        mClearView.setEnabled(enable);
    }

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mFormNavigationLayout == null) return;
            mClearView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mFinishView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            ThemeUtil.setTintList(mNextView, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mPreView, ThemeUtil.getPrimaryIconColor(mContext));
            mNavigationDividerView.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mNavigationContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
        }
    };

    public void updateSuggestionsbar(String content) {
        mHandler.removeMessages(UPDATE_SUGGESTIONS_MSG_WHAT);

        Message message = Message.obtain();
        message.what = UPDATE_SUGGESTIONS_MSG_WHAT;
        message.obj = content;
        mHandler.sendMessageDelayed(message, 100);
    }

    private final Handler mHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (msg.what == UPDATE_SUGGESTIONS_MSG_WHAT) {
                updateSuggestionsbarImpl((String) msg.obj);
            }
        }
    };

    private void updateSuggestionsbarImpl(String content) {
        boolean showSuggestions = SuggestionsHelper.getInstance().isShowSuggestions(ISuggestionsType.Form_Filling);
        if (!showSuggestions) {
            hideSuggestionsbar();
            return;
        }

        ArrayList<String> suggestions = SuggestionsHelper.getInstance().getSuggestions(ISuggestionsType.Form_Filling, content);
        if (suggestions == null || suggestions.size() == 0) {
            hideSuggestionsbar();
            return;
        }
        mSuggestKeyWords = content;

        if (mSuggestionsNavigationLayout.getChildCount() > 0)
            mSuggestionsNavigationLayout.removeAllViews();
        mSuggestionsNavigationLayout.setVisibility(View.VISIBLE);

        if (!AppDisplay.isPad()) {
            View dividerView = new View(mContext);
            dividerView.setBackgroundColor(ThemeConfig.getInstance(mContext).getP1());
            LinearLayout.LayoutParams dividerParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT,
                    AppResource.getDimensionPixelSize(mContext, R.dimen.ux_list_divider_height));
            mSuggestionsNavigationLayout.addView(dividerView, dividerParams);
        }
        LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT,
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_with_text_img_height));
        View suggestionView = getSuggestionsView(suggestions);
        mSuggestionsNavigationLayout.addView(suggestionView, layoutParams);
    }

    private View getSuggestionsView(ArrayList<String> suggestions) {
        HorizontalScrollView hsView = new HorizontalScrollView(mContext);
        hsView.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
        hsView.setPadding(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp), 0,
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp), 0);
        hsView.setHorizontalScrollBarEnabled(false);

        final LinearLayout rootLayout = new LinearLayout(mContext);
        rootLayout.setGravity(Gravity.CENTER);
        rootLayout.setOrientation(LinearLayout.HORIZONTAL);
        hsView.addView(rootLayout, new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.MATCH_PARENT));

        int size = Math.min(suggestions.size(), MAX_SUGGESTIONS_COUNT);
        for (int i = 0; i < size; i++) {
            addSuggestionsChildView(suggestions, suggestions.get(i), rootLayout);
        }
        return hsView;
    }

    private void addSuggestionsChildView(final ArrayList<String> suggestions, String suggestion, final ViewGroup parentView) {
        final LinearLayout childView = new LinearLayout(mContext);
        childView.setGravity(Gravity.CENTER_VERTICAL);
        childView.setOrientation(LinearLayout.HORIZONTAL);
        childView.setBackgroundResource(R.drawable.shape_form_prompt);
        LinearLayout.LayoutParams child_lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT);
        child_lp.leftMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp);
        child_lp.rightMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp);
        parentView.addView(childView, child_lp);

        final TextView suggestTv = new TextView(mContext);
        suggestTv.setTextColor(AppResource.getColor(mContext, R.color.b1));
        suggestTv.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        suggestTv.setText(suggestion);
        suggestTv.setPadding(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp),
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp), 0,
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));
        LinearLayout.LayoutParams tv_lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT);
        childView.addView(suggestTv, tv_lp);

        ImageView clearIv = new ImageView(mContext);
        clearIv.setImageResource(R.drawable.common_white_close_small);
        clearIv.setColorFilter(ThemeConfig.getInstance(mContext).getB1());
        clearIv.setPadding(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp), 0,
                AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp), 0);
        clearIv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                final UITextEditDialog dialog = new UITextEditDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity(),
                        UIDialog.NO_INPUT);
                if (dialog.getDialog() != null) {
                    dialog.getDialog().getWindow().setFlags(WindowManager.LayoutParams.FLAG_ALT_FOCUSABLE_IM,
                            WindowManager.LayoutParams.FLAG_ALT_FOCUSABLE_IM);
                }
                dialog.setTitle(AppResource.getString(mContext, R.string.fx_form_remove_suggestion_title));
                String prompt = AppResource.getString(mContext, R.string.fx_form_remove_suggestion_prompt, suggestTv.getText());
                dialog.getPromptTextView().setText(prompt);
                dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        String suggestion = suggestTv.getText().toString();
                        SuggestionsHelper.getInstance().removeSuggestion(ISuggestionsType.Form_Filling, suggestion);
                        suggestions.remove(suggestion);

                        parentView.removeView(childView);
                        if (parentView.getChildCount() == 0) {
                            hideSuggestionsbar();
                        } else {
                            if (suggestions.size() >= 10) {
                                addSuggestionsChildView(suggestions, suggestions.get(9), parentView);
                            }
                        }
                        dialog.dismiss();
                    }
                });
                dialog.show();
            }
        });
        LinearLayout.LayoutParams iv_lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT);
        childView.addView(clearIv, iv_lp);

        childView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                String suggestion = suggestTv.getText().toString();
                SuggestionsHelper.getInstance().updateSuggestion(ISuggestionsType.Form_Filling, 0, suggestion);
                if (mSuggestionApplyCallback != null) {
                    mSuggestionApplyCallback.onSuggestionApply(suggestion);
                }
                hideSuggestionsbar();
            }
        });
    }

    void onConfigurationChanged() {
        if (haveSuggestions() && !AppUtil.isEmpty(mSuggestKeyWords)) {
            updateSuggestionsbar(mSuggestKeyWords);
        }
    }

    private boolean haveSuggestions() {
        return mSuggestionsNavigationLayout.getChildCount() > 0;
    }

    public void hideSuggestionsbar() {
        if (mSuggestionsNavigationLayout.getChildCount() > 0)
            mSuggestionsNavigationLayout.removeAllViews();

        if (AppDisplay.isPad()) {
            mSuggestionsNavigationLayout.setVisibility(View.INVISIBLE);
        } else {
            mSuggestionsNavigationLayout.setVisibility(View.GONE);
        }
    }

    private ISuggestionApplyCallback mSuggestionApplyCallback;

    public void setSuggestionApplyCallback(ISuggestionApplyCallback callback) {
        mSuggestionApplyCallback = callback;
    }

    public interface ISuggestionApplyCallback {
        void onSuggestionApply(String content);
    }

}
