/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.freetext.callout;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.MotionEvent;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.FreeText;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.freetext.FtTextUtil;
import com.foxit.uiextensions.annots.freetext.FtUtil;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

public class CalloutToolHandler implements ToolHandler {
    private Context mContext;
    private FtTextUtil mTextUtil;

    int mColor;
    int mOpacity;
    float mFontSize;
    private int mFontId;
    private int mBorderColor;
    private int mBorderType;
    private EditText mEditView;
    private boolean mCreating;
    private int mCreateIndex;
    private PointF mTextStartPt = new PointF();
    private PointF mTextStartPdfPt = new PointF();
    private PointF mDownPdfPt = new PointF();
    private PointF mEditPoint = new PointF();
    public int mLastPageIndex = -1;
    private String mAnnotText;
    private float mBBoxWidth;
    private float mBBoxHeight;
    private boolean mIsContinue = true;
    private boolean mIsSelcetEndText = false;

    private PointF mKneePoint = new PointF();
    private PointF mEndingPoint = new PointF();
    private Paint mPaintOut;
    private int mCurrentPosition;
    private boolean mCreateAlive = true;

    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;

    public interface CreateAnnotResult {
        public void callBack();
    }

    public CalloutToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
        mBorderColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];

        mPaintOut = new Paint();
        mPaintOut.setAntiAlias(true);
        mPaintOut.setStyle(Paint.Style.STROKE);
        mPaintOut.setColor(mBorderColor);

        mTextUtil = new FtTextUtil(mContext, mPdfViewCtrl);

        pdfViewCtrl.registerDocEventListener(new DocEventListener() {

            @Override
            public void onDocClosed(PDFDoc document, int errCode) {
                mAnnotText = "";
                mTextStartPt.set(0, 0);
                mEditPoint.set(0, 0);
                mLastPageIndex = -1;
                mBBoxHeight = 0;
                mBBoxWidth = 0;
                AppUtil.dismissInputSoft(mEditView);
                mUiExtensionsManager.getRootView().removeView(mEditView);
                mEditView = null;
                mBBoxHeight = 0;
                mBBoxWidth = 0;
                mCreating = false;
                mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                if (mTextUtil != null) {
                    mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
                    mTextUtil.setKeyboardOffset(0);
                }
            }
        });
    }

    @Override
    public String getType() {
        return TH_TYPE_CALLOUT;
    }

    public int getColor() {
        return mColor;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public float getFontSize() {
        return mFontSize;
    }
    public String getFontName() {
        return mTextUtil.getSupportFontName(mFontId);
    }

    public int getBorderType() {
        return mBorderType;
    }

    @Override
    public void onActivate() {
        setUndoItemCallback(mUndoItemCallback);
        mLastPageIndex = -1;
        mCreateAlive = true;
        ViewGroup parent = mUiExtensionsManager.getRootView();
        AppKeyboardUtil.setKeyboardListener(parent, parent, new AppKeyboardUtil.IKeyboardListener() {
            @Override
            public void onKeyboardOpened(int keyboardHeight) {
                if (SystemUiHelper.getInstance().isFullScreen())
                    SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
            }

            @Override
            public void onKeyboardClosed() {
                mTextUtil.setKeyboardOffset(0);
                if (SystemUiHelper.getInstance().isFullScreen())
                    SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
//                mCreateAlive = false;
                if (mContext.getResources().getConfiguration().keyboard != Configuration.KEYBOARDHIDDEN_YES) {
                    if (mEditView != null) {
                        mCreateAlive = false;
                        createCOAnnot(false);
                        mCreating = false;
                        mEditView = null;
                    }
                }
            }
        });

        resetPropertyBar();
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyBarListener() {
        mPropertyChangeListener = null;
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mFontId));
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTSIZE, mFontSize);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_FONTSIZE
                | PropertyBar.PROPERTY_FONTNAME;
    }

    @Override
    public void onDeactivate() {
        setUndoItemCallback(null);
        releaseDrawPoints();
        if (mEditView != null) {
            if (mCreateAlive) {
                mCreateAlive = false;
                createCOAnnot(false);
            }
        }
        if (SystemUiHelper.getInstance().isFullScreen())
            SystemUiHelper.getInstance().hideSystemUI(mUiExtensionsManager.getAttachedActivity());
        AppKeyboardUtil.removeKeyboardListener(mUiExtensionsManager.getRootView());
    }

    private void releaseDrawPoints() {
        mDrawStartPoint = null;
        mDrawTextStartPdfPoint = null;
        mDrawEndingPdfPoint = null;
        mDrawKneePdfPoint = null;
    }

    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent e) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
        if (!handled && mEditView == null && e.getActionMasked() != MotionEvent.ACTION_DOWN) {
            onCallOutToolTouch(pageIndex, e);
        }
        if (!handled && mEditView != null) {
            handled = onCallOutToolTouch(pageIndex, e);
        }
        return handled;
    }

    private int mCurrentTouchAction;
    private boolean mIsMoving = false;
    private PointF mDownViewPoint = null;
    private long mLastDownEventTime;

    private boolean onCallOutToolTouch(int pageIndex, MotionEvent e) {
        PointF point = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        final PointF pdfPoint = new PointF(point.x, point.y);
        mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPoint, pdfPoint, pageIndex);

        float x = point.x;
        float y = point.y;

        mCurrentTouchAction = e.getActionMasked();
        switch (mCurrentTouchAction) {
            case MotionEvent.ACTION_DOWN:
                mLastDownEventTime = System.currentTimeMillis();
                if (mDownViewPoint == null)
                    mDownViewPoint = new PointF();
                mDownViewPoint.set(point);
                mIsMoving = false;
                if (mUiExtensionsManager.getCurrentToolHandler() == this) {
                    if (mCreating) {
                        if (mFrameRectF.contains(x, y)) {
                            mPdfViewCtrl.convertPageViewPtToPdfPt(point, mEditPoint, pageIndex);
                            mTextUtil.resetEditState();
                            return true;
                        }
                        return false;
                    } else {
                        mDownPdfPt.set(x, y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mDownPdfPt, mDownPdfPt, pageIndex);
                        mBBoxWidth = 0;
                        mBBoxHeight = 0;
                        mLastPageIndex = pageIndex;
                        calculateTextPositionByDown(mPdfViewCtrl, x, y);
                        PointF textStartPoint = calculateTextStartPtByDown(mPdfViewCtrl, x, y);
                        mTextStartPt.set(textStartPoint.x,
                                textStartPoint.y);
                        mTextStartPdfPt.set(mTextStartPt.x, mTextStartPt.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mTextStartPdfPt, mTextStartPdfPt, pageIndex);

                        mCreateIndex = pageIndex;
                        return true;
                    }
                }
                return false;
            case MotionEvent.ACTION_MOVE:
                if (AppDmUtil.distanceOfTwoPoints(mDownViewPoint, point) < 8) {
                    mIsMoving = false;
                    return true;
                }
                mIsMoving = true;
                if (mDrawEndingPoint == null)
                    mDrawEndingPoint = new PointF();
                mDrawEndingPoint.set(x, y);
                mPdfViewCtrl.invalidate();
                return true;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (mIsMoving && mLastDrawPreviewEndPoint != null && AppDmUtil.distanceOfTwoPoints(mLastDrawPreviewEndPoint, new PointF(x, y)) > 10) {
                    mDrawEndingPoint.set(x, y);
                    mPdfViewCtrl.invalidate();
                }
                mIsMoving = false;
                if (mEditView == null || (mFrameRectF != null && !mFrameRectF.contains(x, y)))
                    onTouchUp(pageIndex);
                return true;
            default:
                break;
        }
        return false;
    }

    private void onTouchUp(final int pageIndex) {
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView == null) {
            mEditView = new EditText(mContext);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            mEditView.setLayoutParams(new LayoutParams(1, 1));
            mEditView.addTextChangedListener(new TextWatcher() {

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    mAnnotText = FtTextUtil.filterEmoji(String.valueOf(s));
                    if (mDrawKneePdfPoint != null)
                        mDrawKneePdfPoint = null;
                    if (mDrawEndingPdfPoint != null)
                        mDrawEndingPdfPoint = null;
                    mPdfViewCtrl.invalidate();
                }

                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {

                }

                @Override
                public void afterTextChanged(Editable s) {

                }
            });

            mTextUtil.setOnWidthChanged(new FtTextUtil.OnTextValuesChangedListener() {

                @Override
                public void onMaxWidthChanged(float maxWidth) {
                    mBBoxWidth = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, FtUtil.DEFAULTTEXTWIDTH);
                }

                @Override
                public void onMaxHeightChanged(float maxHeight) {
                    mBBoxHeight = maxHeight;
                }

                @Override
                public void onCurrentSelectIndex(int selectIndex) {
                    if (selectIndex >= mEditView.getText().length()) {
                        selectIndex = mEditView.getText().length();
                        mIsSelcetEndText = true;
                    } else {
                        mIsSelcetEndText = false;
                    }
                    mEditView.setSelection(selectIndex);
                }

                @Override
                public void onEditPointChanged(float editPointX,
                                               float editPointY) {
                    PointF point = new PointF(editPointX, editPointY);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(point, point, pageIndex);
                    mEditPoint.set(point.x, point.y);
                }

            });
            mUiExtensionsManager.getRootView().addView(mEditView);
            mPdfViewCtrl.invalidate();

            AppUtil.showSoftInput(mEditView);
            mTextUtil.getBlink().postDelayed((Runnable) mTextUtil.getBlink(), 500);
            mCreating = true;
        }
        mTextStartPt.set(0, 0);
        mEditPoint.set(0, 0);
        mCreateAlive = true;
    }


    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        if (mEditView == null) {
            mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
            onCallOutToolTouch(pageIndex, motionEvent);
        } else {
            PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
            onSingleTapOrLongPress(pageIndex, point);
        }
        return true;
    }

    private void editAnnotationOrNot(int pageIndex, MotionEvent motionEvent) {
        if (mEditView == null) {
            onCallOutToolTouch(pageIndex, motionEvent);
            onTouchUp(pageIndex);
        } else {
            PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
            onSingleTapOrLongPress(pageIndex, point);
        }
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled || mEditView != null) {
            editAnnotationOrNot(pageIndex, motionEvent);
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinue;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinue = continueAddAnnot;
    }

    private boolean onSingleTapOrLongPress(final int pageIndex, final PointF point) {
        float x = point.x;
        float y = point.y;
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView != null) {
            releaseDrawPoints();
            RectF rectF = new RectF(mTextStartPt.x, mTextStartPt.y,
                    mTextStartPt.x + getBBoxWidth(pageIndex), mTextStartPt.y + getBBoxHeight(pageIndex));
            if (rectF.contains(x, y)) {
                PointF pointF = new PointF(x, y);
                mPdfViewCtrl.convertPageViewPtToPdfPt(pointF, pointF, pageIndex);
                mEditPoint.set(pointF.x, pointF.y);
                mTextUtil.resetEditState();

                RectF _rect = new RectF(rectF);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(_rect, _rect, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(_rect));
                AppUtil.showSoftInput(mEditView);
                return true;
            } else {
                if (!mIsContinue) {
                    mUiExtensionsManager.setCurrentToolHandler(null);
                    return true;
                }

                if (mCreateAlive) {
                    mCreateAlive = false;
                    if (mUiExtensionsManager.getCurrentToolHandler() == CalloutToolHandler.this) {
                        createCOAnnot(false);
                    }
                    return true;
                }

                mCreateAlive = true;
                setCreateAnnotListener(new CreateAnnotResult() {
                    @Override
                    public void callBack() {
                        mDownPdfPt.set(point.x, point.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mDownPdfPt, mDownPdfPt, pageIndex);
                        mBBoxWidth = 0;
                        mBBoxHeight = 0;
                        if (mLastPageIndex == -1) {
                            mLastPageIndex = pageIndex;
                        }
                        calculateTextPositionByDown(mPdfViewCtrl, point.x, point.y);
                        mTextStartPt.set(calculateTextStartPtByDown(mPdfViewCtrl, point.x, point.y).x,
                                calculateTextStartPtByDown(mPdfViewCtrl, point.x, point.y).y);
                        mTextStartPdfPt.set(mTextStartPt.x, mTextStartPt.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mTextStartPdfPt, mTextStartPdfPt, pageIndex);
                        mCreateIndex = pageIndex;
                        mPdfViewCtrl.invalidate();
                        if (mEditView != null) {
                            AppUtil.showSoftInput(mEditView);
                        }
                    }
                });
                createCOAnnot(false);
                return true;
            }
        }
        return false;
    }

    private PointF mDrawStartPoint;
    private PointF mDrawKneePoint;
    private PointF mDrawKneePdfPoint;
    private PointF mDrawTextStartPoint;
    private PointF mDrawTextStartPdfPoint;
    private PointF mDrawEndingPoint;
    private PointF mDrawEndingPdfPoint;
    private PointF mLastDrawPreviewEndPoint;
    private RectF mFrameRectF;

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        canvas.save();
        if (mLastPageIndex == pageIndex && !mIsMoving) {
            if (mEditView == null) return;
            if (mDrawTextStartPdfPoint != null) {
                mTextStartPdfPt.set(mDrawTextStartPdfPoint);
            }
            if (mFrameRectF == null)
                mFrameRectF = new RectF();
            PointF textStartPoint = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(textStartPoint, textStartPoint, pageIndex);
            mFrameRectF.set(
                    textStartPoint.x,
                    textStartPoint.y,
                    textStartPoint.x + getBBoxWidth(pageIndex),
                    textStartPoint.y + getBBoxHeight(pageIndex));
            mFrameRectF.inset(-FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_BORDER_WIDTH) / 2,
                    -FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_BORDER_WIDTH) / 2);
            float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, 6);
            float adjustX = 0;
            float adjustY = 0;
            if (mFrameRectF.left < deltaXY) {
                adjustX = -mFrameRectF.left + deltaXY;
            }
            if (mFrameRectF.top < deltaXY) {
                adjustY = -mFrameRectF.top + deltaXY;
            }
            if (mFrameRectF.right > mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY) {
                adjustX = mPdfViewCtrl.getPageViewWidth(pageIndex) - mFrameRectF.right - deltaXY;
            }
            if (mFrameRectF.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY) {
                adjustY = mPdfViewCtrl.getPageViewHeight(pageIndex) - mFrameRectF.bottom - deltaXY;
            }
            mFrameRectF.offset(adjustX, adjustY);
            textStartPoint.offset(adjustX, adjustY);
            mPdfViewCtrl.convertPageViewPtToPdfPt(textStartPoint, mTextStartPdfPt, mLastPageIndex);

            PointF editPoint = new PointF(mEditPoint.x, mEditPoint.y);
            if (editPoint.x != 0 || editPoint.y != 0) {
                mPdfViewCtrl.convertPdfPtToPageViewPt(editPoint, editPoint, pageIndex);
            }

            mTextUtil.setTextString(pageIndex, mAnnotText, true);
            mTextUtil.setStartPoint(textStartPoint);
            mTextUtil.setEditPoint(editPoint);
            mTextUtil.setMaxRect(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULTTEXTWIDTH),
                    mPdfViewCtrl.getPageViewHeight(pageIndex) - textStartPoint.y);
            mTextUtil.setTextColor(mColor, AppDmUtil.opacity100To255(mOpacity));
            mTextUtil.setFont(mTextUtil.getSupportFontName(mFontId), mFontSize);

            if (mIsSelcetEndText) {
                mTextUtil.setEndSelection(mEditView.getSelectionEnd() + 1);
            } else {
                mTextUtil.setEndSelection(mEditView.getSelectionEnd());
            }

            mTextUtil.loadText(true);
            mTextUtil.drawText(canvas);

            // canvas frame board
            mPaintOut.setStrokeWidth(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH));
            mPaintOut.setPathEffect(FtUtil.getDashPathEffect(mContext, mPdfViewCtrl, pageIndex, mBorderType, true));
            PointF kneePoint = calculateKneePtByTextStartPt(mPdfViewCtrl, textStartPoint.x, textStartPoint.y);
            if (mDrawKneePdfPoint != null) {
                mPdfViewCtrl.convertPdfPtToPageViewPt(mDrawKneePdfPoint, mDrawKneePoint, mLastPageIndex);
                kneePoint.set(mDrawKneePoint);
            }
            mKneePoint.set(kneePoint);
            PointF endPoint = calculateEndingPtByTextStartPt(mPdfViewCtrl, textStartPoint.x, textStartPoint.y);
            if (mDrawEndingPdfPoint != null) {
                mPdfViewCtrl.convertPdfPtToPageViewPt(mDrawEndingPdfPoint, mDrawEndingPoint, mLastPageIndex);
                endPoint.set(mDrawEndingPoint);
            }
            mEndingPoint.set(endPoint);
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, pageIndex);
            canvas.drawLine(startPoint.x, startPoint.y, mKneePoint.x, mKneePoint.y, mPaintOut);
            canvas.drawLine(mKneePoint.x, mKneePoint.y, mEndingPoint.x, mEndingPoint.y, mPaintOut);

            if (mBorderType == FtUtil.BORDERCLOUD) {
                canvas.drawPath(FtUtil.getCloudy_Rectangle(mPdfViewCtrl, pageIndex, mFrameRectF, 0.0f), mPaintOut);
            } else {
                canvas.drawRect(mFrameRectF, mPaintOut);
            }

            Path arrowPath = FtUtil.getArrowPath(mPdfViewCtrl, mLastPageIndex, mKneePoint.x,
                    mKneePoint.y, startPoint.x, startPoint.y);
            canvas.drawPath(arrowPath, mPaintOut);
        } else if (mLastPageIndex == pageIndex) {
            onDrawPreview(pageIndex, canvas);
        }

        canvas.restore();
    }

    private void onDrawPreview(int pageIndex, Canvas canvas) {
        if (mCreating && mEditView != null && (System.currentTimeMillis() - mLastDownEventTime) < 100) {
            return;
        }
        mPaintOut.setStrokeWidth(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH));
        mPaintOut.setPathEffect(FtUtil.getDashPathEffect(mContext, mPdfViewCtrl, pageIndex, mBorderType, true));

        initDrawPointsOrNot();
        mPdfViewCtrl.convertPdfPtToPageViewPt(mDownPdfPt, mDrawStartPoint, pageIndex);
        float kneeToEnd = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_KENNTOEND_WIDTH);
        mDrawKneePoint.set(mDrawEndingPoint);
        mDrawKneePoint.offset(-kneeToEnd, 0);
        mDrawTextStartPoint.set(mDrawEndingPoint);
        mDrawTextStartPoint.offset(0, -getBBoxHeight(mLastPageIndex) / 2);
        RectF frameRectF = new RectF(
                mDrawTextStartPoint.x,
                mDrawTextStartPoint.y,
                mDrawTextStartPoint.x + getBBoxWidth(pageIndex),
                mDrawTextStartPoint.y + getBBoxHeight(pageIndex));
        frameRectF.inset(FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex,
                FtUtil.DEFAULT_BORDER_WIDTH) / 2, FtUtil.widthOnPageView(mPdfViewCtrl,
                mLastPageIndex, FtUtil.DEFAULT_BORDER_WIDTH) / 2);
        float adjustx = 0;
        float adjusty = 0;
        float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, 5);
        if (frameRectF.left < deltaXY) {
            adjustx = -frameRectF.left + deltaXY;
        }
        if (frameRectF.top < deltaXY) {
            adjusty = -frameRectF.top + deltaXY;
        }
        if (frameRectF.right > mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY) {
            adjustx = mPdfViewCtrl.getPageViewWidth(pageIndex) - frameRectF.right - deltaXY;
        }
        if (frameRectF.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY) {
            adjusty = mPdfViewCtrl.getPageViewHeight(pageIndex) - frameRectF.bottom - deltaXY;
        }
        frameRectF.offset(adjustx, adjusty);
        mDrawTextStartPoint.offset(adjustx, adjusty);
        FtUtil.resetKneeAndEndingPt(mPdfViewCtrl, pageIndex, frameRectF, mDrawStartPoint, mDrawKneePoint, mDrawEndingPoint);
        mPdfViewCtrl.convertPageViewPtToPdfPt(mDrawTextStartPoint, mDrawTextStartPdfPoint, mLastPageIndex);
        mPdfViewCtrl.convertPageViewPtToPdfPt(mDrawKneePoint, mDrawKneePdfPoint, mLastPageIndex);
        mPdfViewCtrl.convertPageViewPtToPdfPt(mDrawEndingPoint, mDrawEndingPdfPoint, mLastPageIndex);
        updateCurrentPosition();
        canvas.drawLine(mDrawStartPoint.x, mDrawStartPoint.y, mDrawKneePoint.x, mDrawKneePoint.y, mPaintOut);
        canvas.drawLine(mDrawKneePoint.x, mDrawKneePoint.y, mDrawEndingPoint.x, mDrawEndingPoint.y, mPaintOut);
        canvas.drawRect(frameRectF, mPaintOut);
        if (mLastDrawPreviewEndPoint == null) {
            mLastDrawPreviewEndPoint = new PointF();
        }
        mLastDrawPreviewEndPoint.set(mDrawEndingPoint);
        if (mCurrentTouchAction != MotionEvent.ACTION_MOVE) {
            mIsMoving = false;
            mPdfViewCtrl.invalidate();
        }
    }

    private void initDrawPointsOrNot() {
        if (mDrawStartPoint == null)
            mDrawStartPoint = new PointF();
        if (mDrawKneePoint == null)
            mDrawKneePoint = new PointF();
        if (mDrawKneePdfPoint == null)
            mDrawKneePdfPoint = new PointF();
        if (mDrawEndingPoint == null)
            mDrawEndingPoint = new PointF();
        if (mDrawEndingPdfPoint == null)
            mDrawEndingPdfPoint = new PointF();
        if (mDrawTextStartPoint == null)
            mDrawTextStartPoint = new PointF();
        if (mDrawTextStartPdfPoint == null)
            mDrawTextStartPdfPoint = new PointF();
    }

    private void createCOAnnot(final boolean disableAutoGoToPage) {
        createCOAnnot(false, disableAutoGoToPage);
    }

    private void createCOAnnot(boolean fromUndoClick, final boolean disableAutoGoToPage) {
        releaseDrawPoints();
        PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
        mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mCreateIndex);
        float boxHeight = getBBoxHeight(mCreateIndex);
        final RectF pdfRectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mCreateIndex), pdfPointF.y
                + boxHeight);
        final RectF textRectF = new RectF(pdfPointF.x, pdfPointF.y - 6, pdfPointF.x + getBBoxWidth(mCreateIndex), pdfPointF.y
                + boxHeight + 3);

        PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
        mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mCreateIndex);
        RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
        pdfRectF.union(borderRect);
        mPdfViewCtrl.convertPageViewRectToPdfRect(pdfRectF, pdfRectF, mCreateIndex);
        mPdfViewCtrl.convertPageViewRectToPdfRect(textRectF, textRectF, mCreateIndex);

        String content = "";
        try {
            if (!TextUtils.isEmpty(mAnnotText))
                content = new String(mAnnotText.getBytes(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mCreateIndex);
            final CalloutAddUndoItem undoItem = createUndoItem(page, content, startPoint, pdfRectF, textRectF);
            if (undoItem == null) return;

            if (fromUndoClick) {
                ArrayList<IUndoItem> redoItems = new ArrayList<>();
                redoItems.add(undoItem);
                mUiExtensionsManager.getDocumentManager().addRedoItems(redoItems);
                UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
                if (undoModule != null)
                    undoModule.changeButtonStatus();

                refreshPage(true, disableAutoGoToPage, pdfRectF);
                return;
            }

            final Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_FreeText, AppUtil.toFxRectF(undoItem.mBBox)), Annot.e_FreeText);
            CalloutEvent event = new CalloutEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (FreeText) annot, mPdfViewCtrl);
            event.mDisallowTextOverflow = true;
            mUiExtensionsManager.getDocumentManager().setHasModifyTask(true);
            mUiExtensionsManager.getDocumentManager().setDocModified(true);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                        mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                        mUiExtensionsManager.getDocumentManager().setHasModifyTask(false);

                        try {
                            refreshPage(false, disableAutoGoToPage, AppUtil.toRectF(annot.getRect()));
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    } else {
                        dismissEditView();
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void refreshPage(boolean fromUndoClick, boolean disableAutoGoToPage, RectF annotRect) {
        if (mPdfViewCtrl.isPageVisible(mCreateIndex)) {
            RectF viewRect = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, viewRect, mCreateIndex);
            if (fromUndoClick) {
                RectF displayRect = new RectF();
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRect, displayRect, mCreateIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(displayRect));
            } else {
                mPdfViewCtrl.refresh(mCreateIndex, AppDmUtil.rectFToRect(viewRect));
            }

            if (mIsContinue && mCreateAlive) {
                if (mEditView != null)
                    mEditView.setText("");
            } else {
                AppUtil.dismissInputSoft(mEditView);
                mUiExtensionsManager.getRootView().removeView(mEditView);
                mEditView = null;
                mCreating = false;
                mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
                mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                if (mPdfViewCtrl.isPageVisible(mCreateIndex) &&
                        (mCreateIndex == mPdfViewCtrl.getPageCount() - 1 ||
                                (!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE)) &&
                        mCreateIndex == mPdfViewCtrl.getCurrentPage()) {
                    int pageWidth = mPdfViewCtrl.getPageViewWidth(mCreateIndex);
                    int pageHeight = mPdfViewCtrl.getPageViewHeight(mCreateIndex);
                    PointF endPoint = new PointF(pageWidth, pageHeight);
                    mPdfViewCtrl.convertPageViewPtToDisplayViewPt(endPoint, endPoint, mCreateIndex);
                    if (AppDisplay.getRawScreenHeight() - (endPoint.y - mTextUtil.getKeyboardOffset()) > 0) {
                        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                        mTextUtil.setKeyboardOffset(0);
                        PointF startPoint = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
                        mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mCreateIndex);
                        PointF _point = mTextUtil.getPageViewOrigin(mPdfViewCtrl, mCreateIndex, startPoint.x, startPoint.y);
                        if (!disableAutoGoToPage)
                            mPdfViewCtrl.gotoPage(mCreateIndex, _point.x, _point.y);
                    }
                }
            }

            mAnnotText = "";
            mTextStartPt.set(0, 0);
            mEditPoint.set(0, 0);
            mLastPageIndex = -1;
            mBBoxHeight = 0;
            mBBoxWidth = 0;
            mTextUtil.resetEditState();
            if (mIsContinue && mListener != null) {
                mListener.callBack();
            }
        } else {
            dismissEditView();
        }
    }

    private CalloutAddUndoItem createUndoItem(PDFPage page, String content, PointF startPoint, RectF pdfRectF, RectF textRectF) {
        try {
            CalloutAddUndoItem undoItem = new CalloutAddUndoItem(mPdfViewCtrl);
            undoItem.mNM = AppDmUtil.randomUUID(null);
            undoItem.mPageIndex = mCreateIndex;
            undoItem.mColor = mBorderColor;
            undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mContents = content;
            undoItem.mFontId = mFontId;
            undoItem.mFontSize = mFontSize;
            undoItem.mFlags = Annot.e_FlagPrint;
            undoItem.mTextColor = mColor;
            undoItem.mDaFlags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagTextColor | DefaultAppearance.e_FlagFontSize;
            undoItem.mSubject = "Callout";
            undoItem.mIntent = "FreeTextCallout";
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mBBox = new RectF(pdfRectF);
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            PointF startingPt = new PointF(startPoint.x, startPoint.y);
            PointF kneePt = new PointF(mKneePoint.x, mKneePoint.y);
            PointF endingPt = new PointF(mEndingPoint.x, mEndingPoint.y);
            mPdfViewCtrl.convertPageViewPtToPdfPt(startingPt, startingPt, mCreateIndex);
            mPdfViewCtrl.convertPageViewPtToPdfPt(kneePt, kneePt, mCreateIndex);
            mPdfViewCtrl.convertPageViewPtToPdfPt(endingPt, endingPt, mCreateIndex);
            undoItem.mStartingPt = new PointF(startingPt.x, startingPt.y);
            undoItem.mKneePt = new PointF(kneePt.x, kneePt.y);
            undoItem.mEndingPt = new PointF(endingPt.x, endingPt.y);
            undoItem.mTextBBox = new RectF(textRectF);
            undoItem.mBorderType = mBorderType;

            RectF annotRect = new RectF(undoItem.mTextBBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, mCreateIndex);
            undoItem.mComposedText = mTextUtil.getComposedText(mPdfViewCtrl, mCreateIndex, annotRect, content, mTextUtil.getSupportFontName(mFontId), mFontSize);
            undoItem.mTextLineCount = undoItem.mComposedText.size();

            if (!undoItem.mContents.equals(" ")) {
                RectF adjustBBox = new RectF(annotRect);
                mTextUtil.adjustTextRect(mPdfViewCtrl, mCreateIndex, mTextUtil.getSupportFontName(undoItem.mFontId), undoItem.mFontSize, adjustBBox, undoItem.mComposedText);
                mPdfViewCtrl.convertPageViewRectToPdfRect(adjustBBox, adjustBBox, mCreateIndex);
                RectF bbox = new RectF(undoItem.mBBox);
                FtUtil.adjustKneeAndEndingPt(bbox, adjustBBox, undoItem.mStartingPt, undoItem.mKneePt, undoItem.mEndingPt);
                undoItem.mBBox = new RectF(bbox);
                undoItem.mTextBBox = new RectF(adjustBBox);
                undoItem.mFillColor = Color.WHITE;
            }

            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            undoItem.mRotation = rotation == 0 ? rotation : 4 - rotation;
            return undoItem;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    private void dismissEditView() {
        mAnnotText = "";
        mTextStartPt.set(0, 0);
        mEditPoint.set(0, 0);
        mLastPageIndex = -1;
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        AppUtil.dismissInputSoft(mEditView);
        mUiExtensionsManager.getRootView().removeView(mEditView);
        mEditView = null;
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        mCreating = false;
        mTextUtil.resetEditState();
        mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
        mTextUtil.setKeyboardOffset(0);
    }

    protected void setBorderColor(int color, int opacity) {
        mBorderColor = color;
        mPaintOut.setColor(AppDmUtil.calColorByMultiply(mBorderColor, AppDmUtil.opacity100To255(opacity)));
    }

    protected void onColorValueChanged(int color) {
        mColor = color;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mLastPageIndex), pdfPointF.y + getBBoxHeight(mLastPageIndex));
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mLastPageIndex);
            RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                    mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
            borderRect.union(rectF);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(borderRect));
        }

        setProItemColor(color);
    }

    private void setProItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    protected void onOpacityValueChanged(int opacity) {
        mOpacity = opacity;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mLastPageIndex), pdfPointF.y + getBBoxHeight(mLastPageIndex));
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mLastPageIndex);
            RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                    mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
            borderRect.union(rectF);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(borderRect));
        }
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void changeFontDefaultValue(String font) {
        mFontId = mTextUtil.getSupportFontID(font);
    }

    protected void onFontValueChanged(String font) {
        mFontId = mTextUtil.getSupportFontID(font);
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mLastPageIndex), pdfPointF.y + getBBoxHeight(mLastPageIndex));
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mLastPageIndex);
            RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                    mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
            borderRect.union(rectF);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(borderRect));
        }

        if (mCurToolItem == null) return;
        mCurToolItem.property.fontName = font;
    }

    protected void onFontSizeValueChanged(float fontSize) {
        mFontSize = fontSize;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mLastPageIndex), pdfPointF.y + getBBoxHeight(mLastPageIndex));
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mLastPageIndex);
            RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                    mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
            borderRect.union(rectF);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(borderRect));
        }

        if (mCurToolItem == null) return;
        mCurToolItem.property.fontSize = fontSize;
    }

    protected void onBorderTypeValueChanged(int borderType) {
        mBorderType = 1;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + getBBoxWidth(mLastPageIndex), pdfPointF.y + getBBoxHeight(mLastPageIndex));
            PointF startPoint = new PointF(mDownPdfPt.x, mDownPdfPt.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mLastPageIndex);
            RectF borderRect = FtUtil.getBorderRectByStartKneeAndEnding(startPoint.x, startPoint.y, mKneePoint.x,
                    mKneePoint.y, mEndingPoint.x, mEndingPoint.y);
            borderRect.union(rectF);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(borderRect));
        }
    }

    private PointF calculateKneePtByTextStartPt(PDFViewCtrl pdfViewCtrl, float stX, float stY) {
        float width = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_KENNTOEND_WIDTH);
        PointF pointKnee = new PointF();
        switch (mCurrentPosition) {
            case FtUtil.LEFTTOTOP:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x + width,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y);
                break;
            case FtUtil.LEFTTOBOTTOM:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x + width,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y);
                break;
            case FtUtil.RIGHTTOTOP:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x - width,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y);
                break;
            case FtUtil.RIGHTTOBOTTOM:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x - width,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y);
                break;
            case FtUtil.MIDBOTTOM:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y - width);
                break;
            case FtUtil.MIDTOP:
                pointKnee.set(calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).x,
                        calculateEndingPtByTextStartPt(pdfViewCtrl, stX, stY).y + width);
                break;
            default:
                break;
        }
        return pointKnee;
    }

    private PointF calculateEndingPtByTextStartPt(PDFViewCtrl pdfViewCtrl, float stX, float stY) {
        PointF pointEnd = new PointF();
        float width = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULTTEXTWIDTH);
        switch (mCurrentPosition) {
            case FtUtil.LEFTTOTOP:
            case FtUtil.LEFTTOBOTTOM:
                pointEnd.set(stX + width, stY + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.RIGHTTOBOTTOM:
            case FtUtil.RIGHTTOTOP:
                pointEnd.set(stX, stY + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.MIDBOTTOM:
                pointEnd.set(stX + width / 2, stY);
                break;
            case FtUtil.MIDTOP:
                pointEnd.set(stX + width / 2, stY + getBBoxHeight(mLastPageIndex));
                break;
            default:
                break;
        }
        return pointEnd;
    }

    private PointF calculateTextStartPtByDown(PDFViewCtrl pdfViewCtrl, float stX, float stY) {

        PointF pointknee = new PointF();
        PointF pointend = new PointF();
        PointF pointTextStart = new PointF();

        float kneeToend = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_KENNTOEND_WIDTH);
        float startToKnee = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_STARTTOKNEE_WIDTH);
        float rectText = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULTTEXTWIDTH);
        switch (mCurrentPosition) {
            case FtUtil.LEFTTOTOP:
                pointknee.set(stX - startToKnee, stY - startToKnee);
                pointend.set(pointknee.x - kneeToend, pointknee.y);
                pointTextStart.set(pointend.x - rectText, pointend.y + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.LEFTTOBOTTOM:
                pointknee.set(stX - startToKnee, stY + startToKnee);
                pointend.set(pointknee.x - kneeToend, pointknee.y);
                pointTextStart.set(pointend.x - rectText, pointend.y + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.RIGHTTOBOTTOM:
                pointknee.set(stX + startToKnee, stY + startToKnee);
                pointend.set(pointknee.x + kneeToend, pointknee.y);
                pointTextStart.set(pointend.x, pointend.y + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.RIGHTTOTOP:
                pointknee.set(stX + startToKnee, stY - startToKnee);
                pointend.set(pointknee.x + kneeToend, pointknee.y);
                pointTextStart.set(pointend.x, pointend.y + getBBoxHeight(mLastPageIndex) / 2);
                break;
            case FtUtil.MIDBOTTOM:
                pointknee.set(stX, stY + startToKnee);
                pointend.set(pointknee.x, pointknee.y + kneeToend);
                pointTextStart.set(pointend.x - rectText / 2, pointend.y);
                break;
            case FtUtil.MIDTOP:
                pointknee.set(stX, stY - startToKnee);
                pointend.set(pointknee.x, pointknee.y - kneeToend);
                pointTextStart.set(pointend.x - rectText / 2, pointend.y - getBBoxHeight(mLastPageIndex));
                break;
            default:
                break;
        }
        return pointTextStart;
    }

    private void calculateTextPositionByDown(PDFViewCtrl pdfViewCtrl, float stX, float stY) {
        float kneeToend = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_KENNTOEND_WIDTH);
        float startToKnee = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULT_STARTTOKNEE_WIDTH);
        float rectText = FtUtil.widthOnPageView(pdfViewCtrl, mLastPageIndex, FtUtil.DEFAULTTEXTWIDTH);

        // left top

        if (stX > kneeToend + startToKnee + rectText
                && stY > startToKnee + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.LEFTTOTOP;
        }
        // left bottom
        else if (stX > kneeToend + startToKnee + rectText
                && stY < startToKnee + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.LEFTTOBOTTOM;
        }
        // right bottom
        else if (mPdfViewCtrl.getPageViewWidth(mLastPageIndex) - stX > kneeToend + startToKnee + rectText
                && stY < startToKnee + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.RIGHTTOBOTTOM;
        }
        // right top
        else if (mPdfViewCtrl.getPageViewWidth(mLastPageIndex) - stX > kneeToend + startToKnee + rectText
                && stY > startToKnee + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.RIGHTTOTOP;
        }
        // bottom
        else if (mPdfViewCtrl.getPageViewWidth(mLastPageIndex) - stX < kneeToend + startToKnee + rectText
                && stX < kneeToend + startToKnee + rectText
                && stY < startToKnee + kneeToend + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.MIDBOTTOM;
        }
        // top
        else if (mPdfViewCtrl.getPageViewWidth(mLastPageIndex) - stX < kneeToend + startToKnee + rectText
                && stX < kneeToend + startToKnee + rectText
                && stY > startToKnee + kneeToend + mTextUtil.getFontHeight(pdfViewCtrl, mLastPageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) / 2) {
            mCurrentPosition = FtUtil.MIDTOP;
        } else {
            mCurrentPosition = FtUtil.MIDTOP;
        }
    }

    private CreateAnnotResult mListener;

    private void setCreateAnnotListener(CreateAnnotResult listener) {
        mListener = listener;
    }

    private float getBBoxHeight(int pageIndex) {
        return mBBoxHeight == 0 ? mTextUtil.getFontHeight(mPdfViewCtrl, pageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) : mBBoxHeight;
    }

    private float getBBoxWidth(int pageIndex) {
        return mBBoxWidth == 0 ? FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULTTEXTWIDTH) : mBBoxWidth;
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new CalloutToolSupply(mContext);
        return mToolSupply;
    }

    private class CalloutToolSupply extends ToolSupplyImpl {

        public CalloutToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_callout_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return R.drawable.comment_tool_callout_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Callout;
            property.color = mColor;
            property.opacity = mOpacity;
            property.fontName = mTextUtil.getSupportFontName(mFontId);
            property.fontSize = mFontSize;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_CALLOUT;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Callout);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mFontId = mTextUtil.getSupportFontID(property.fontName);
                mFontSize = property.fontSize;
                mUiExtensionsManager.setCurrentToolHandler(CalloutToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == CalloutToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mFontId = mTextUtil.getSupportFontID(property.fontName);
            mFontSize = property.fontSize;
            CalloutToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void handlePageChanged() {
        if (mEditView == null) {
            return;
        }

        // MOBRD-7200
        if (mLastPageIndex == mPdfViewCtrl.getPageCount() - 1) {
            Rect rect = mPdfViewCtrl.getPageViewRect(mLastPageIndex);
            int h = mPdfViewCtrl.getHeight();
            if (rect.bottom <= mPdfViewCtrl.getHeight())
                return;
        }

        mCreateAlive = false;
        createCOAnnot(true);
    }

    private void updateCurrentPosition() {
        mCurrentPosition = FtUtil.RIGHTTOTOP;
        if (mDrawKneePoint.x == mDrawEndingPoint.x) {
            mCurrentPosition = mDrawKneePoint.y > mDrawEndingPoint.y ? FtUtil.MIDTOP : FtUtil.MIDBOTTOM;
        } else if (mDrawKneePoint.x < mDrawEndingPoint.x) {
            mCurrentPosition = mDrawStartPoint.y > mDrawEndingPoint.y ? FtUtil.RIGHTTOTOP : FtUtil.RIGHTTOBOTTOM;
        } else {
            mCurrentPosition = mDrawStartPoint.y > mDrawEndingPoint.y ? FtUtil.LEFTTOTOP : FtUtil.LEFTTOBOTTOM;
        }
    }

    private void setUndoItemCallback(UndoModule.IUndoItemCallback callback) {
        UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
        if (undoModule != null) {
            undoModule.setUndoItemCallback(callback);
        }
    }

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mCurToolItem != null && mUiExtensionsManager.getCurrentAnnotHandler() instanceof CalloutAnnotHandler) {
                ((CalloutAnnotHandler) mUiExtensionsManager.getCurrentAnnotHandler()).handleKeyboardClosed();
                return true;
            }
            if (mEditView != null && mCurToolItem != null) {
//                mCurToolItem.toolItem.performClick();
                mCreateAlive = false;
                createCOAnnot(true, false);
                mCreating = false;
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mEditView != null && mCurToolItem != null) {
                mCurToolItem.toolItem.performClick();
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };

}
