/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.freetext.textbox;


import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.MotionEvent;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.FreeText;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.freetext.FtTextUtil;
import com.foxit.uiextensions.annots.freetext.FtUtil;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

public class TextBoxToolHandler implements ToolHandler {
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private FtTextUtil mTextUtil;

    int mColor;
    int mOpacity;
    float mFontSize;
    private int mFontId;
    private int mBorderColor;
    private EditText mEditView;
    private boolean mCreating;
    private int mCreateIndex = -1;
    private PointF mTextStartPt = new PointF();
    private PointF mTextStartPdfPt = new PointF();
    private PointF mDownPoint = new PointF();
    private PointF mMovePoint = new PointF();

    private PointF mEditPoint = new PointF();
    private String mAnnotText;
    private float mBBoxWidth;
    private float mBBoxHeight;
    private boolean mIsContinue = true;
    private boolean mIsSelcetEndText = false;

    private Paint mPaintOut;
    private Paint mPaintFill;
    private boolean mCreateAlive = true;

    private CreateAnnotResult mListener;

    public int mLastPageIndex = -1;
    private int mPageViewWidth;
    private int mPageViewHeigh;
    private int mTextBoxWidth;

    public interface CreateAnnotResult {
        public void callBack();
    }

    public TextBoxToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        this.mContext = context;
        this.mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
        mBorderColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];

        mPaintOut = new Paint();
        mPaintOut.setAntiAlias(true);
        mPaintOut.setStyle(Paint.Style.STROKE);
        mPaintOut.setColor(mBorderColor);

        mPaintFill = new Paint();
        mPaintFill.setStyle(Paint.Style.FILL);
        mPaintFill.setAntiAlias(true);
        mPaintFill.setDither(true);
        mPaintFill.setColor(AppResource.getColor(context, R.color.ux_color_white));

        mTextUtil = new FtTextUtil(mContext, mPdfViewCtrl);

        if (AppDisplay.isPad()) {
            mTextBoxWidth = (int) mContext.getResources().getDimension(R.dimen.annot_textbox_width_pad);
        } else {
            mTextBoxWidth = (int) mContext.getResources().getDimension(R.dimen.annot_textbox_width_phone);
        }

        pdfViewCtrl.registerDocEventListener(new DocEventListener() {

            @Override
            public void onDocClosed(PDFDoc pdfDoc, int i) {
                mAnnotText = "";
                mTextStartPt.set(0, 0);
                mEditPoint.set(0, 0);
                mLastPageIndex = -1;
                mBBoxHeight = 0;
                mBBoxWidth = 0;
                AppUtil.dismissInputSoft(mEditView);
                mUiExtensionsManager.getRootView().removeView(mEditView);
                mEditView = null;
                mBBoxHeight = 0;
                mBBoxWidth = 0;
                mCreating = false;
                mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                if (mTextUtil != null) {
                    mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
                    mTextUtil.setKeyboardOffset(0);
                }
            }

        });
    }

    public int getColor() {
        return mColor;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public float getFontSize() {
        return mFontSize;
    }

    public String getFontName() {
        return mTextUtil.getSupportFontName(mFontId);
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_TEXTBOX;
    }

    @Override
    public void onActivate() {
        mLastPageIndex = -1;
        mCreateAlive = true;
        setUndoItemCallback(mUndoItemCallback);

        AppKeyboardUtil.setKeyboardListener(mUiExtensionsManager.getRootView(),
                mUiExtensionsManager.getRootView(), new AppKeyboardUtil.IKeyboardListener() {
                    @Override
                    public void onKeyboardOpened(int keyboardHeight) {
                        if (SystemUiHelper.getInstance().isFullScreen())
                            SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
                    }

                    @Override
                    public void onKeyboardClosed() {
                        mTextUtil.setKeyboardOffset(0);
                        if (SystemUiHelper.getInstance().isFullScreen())
                            SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
                        if (mContext.getResources().getConfiguration().keyboard != Configuration.KEYBOARDHIDDEN_YES) {
                            mCreateAlive = false;
                            if (mEditView != null) {
                                createTBAnnot(false);
                                mCreating = false;
                                mEditView = null;
                            }
                        }
                    }
                });

        resetPropertyBar();
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyBarListener() {
        mPropertyChangeListener = null;
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);

        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mFontId));
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTSIZE, mFontSize);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_FONTSIZE
                | PropertyBar.PROPERTY_FONTNAME;
    }

    @Override
    public void onDeactivate() {
        mCreateAlive = false;
        mIsInterceptTouchEvent = false;
        setUndoItemCallback(null);

        if (mEditView != null) {
            createTBAnnot(false);
        }

        if (SystemUiHelper.getInstance().isFullScreen())
            SystemUiHelper.getInstance().hideSystemUI(mUiExtensionsManager.getAttachedActivity());
        AppKeyboardUtil.removeKeyboardListener(mUiExtensionsManager.getRootView());
    }

    private void createTBAnnot(final boolean disableAutoGoToPage) {
        createTBAnnot(false, disableAutoGoToPage);
    }

    private void createTBAnnot(boolean fromUndoClick, final boolean disableAutoGoToPage) {
        final RectF rect = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rect, mCreateIndex);
        PointF adjustF = FtUtil.adjustPointF(mPdfViewCtrl, mCreateIndex, rect);
        rect.offset(adjustF.x, adjustF.y);

        RectF pdfRectF = new RectF(rect);
        RectF textRectF = new RectF(rect);
        mPdfViewCtrl.convertPageViewRectToPdfRect(pdfRectF, pdfRectF, mCreateIndex);
        mPdfViewCtrl.convertPageViewRectToPdfRect(textRectF, textRectF, mCreateIndex);
        String content = "";
        try {
            if (!TextUtils.isEmpty(mAnnotText))
                content = new String(mAnnotText.getBytes(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mCreateIndex);
            final TextBoxAddUndoItem undoItem = createUndoItem(page, content, pdfRectF, textRectF);
            if (undoItem == null) return;

            if (fromUndoClick) {
                ArrayList<IUndoItem> redoItems = new ArrayList<>();
                redoItems.add(undoItem);
                mUiExtensionsManager.getDocumentManager().addRedoItems(redoItems);
                UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
                if (undoModule != null)
                    undoModule.changeButtonStatus();

                refreshPage(true, disableAutoGoToPage, rect);
                return;
            }

            final Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_FreeText, AppUtil.toFxRectF(undoItem.mBBox)), Annot.e_FreeText);
            TextBoxEvent addEvent = new TextBoxEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (FreeText) annot, mPdfViewCtrl);
            mUiExtensionsManager.getDocumentManager().setHasModifyTask(true);
            mUiExtensionsManager.getDocumentManager().setDocModified(true);

            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                        mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                        mUiExtensionsManager.getDocumentManager().setHasModifyTask(false);

                        try {
                            RectF annotRectF = AppUtil.toRectF(annot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, mCreateIndex);
                            refreshPage(false, disableAutoGoToPage, rect);
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    } else {
                        dismissEditView();
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void refreshPage(boolean fromUndoClick, boolean disableAutoGoToPage, RectF viewRect) {
        if (mPdfViewCtrl.isPageVisible(mCreateIndex)) {
            if (fromUndoClick) {
                RectF displayRect = new RectF();
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRect, displayRect, mCreateIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(displayRect));
            } else {
                mPdfViewCtrl.refresh(mCreateIndex, AppDmUtil.rectFToRect(viewRect));
            }

            if (mIsContinue && mCreateAlive) {
                if (mEditView != null)
                    mEditView.setText("");
            } else {
                AppUtil.dismissInputSoft(mEditView);
                mUiExtensionsManager.getRootView().removeView(mEditView);
                mEditView = null;
                mCreating = false;
                mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
                mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                if ((mCreateIndex == mPdfViewCtrl.getPageCount() - 1
                        || (!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE))
                        && mCreateIndex == mPdfViewCtrl.getCurrentPage()) {
                    int pageWidth = mPdfViewCtrl.getPageViewWidth(mCreateIndex);
                    int pageHeight = mPdfViewCtrl.getPageViewHeight(mCreateIndex);
                    PointF endPoint = new PointF(pageWidth, pageHeight);
                    mPdfViewCtrl.convertPageViewPtToDisplayViewPt(endPoint, endPoint, mCreateIndex);

                    if (AppDisplay.getRawScreenHeight() - (endPoint.y - mTextUtil.getKeyboardOffset()) > 0) {
                        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                        mTextUtil.setKeyboardOffset(0);
                        PointF startPoint = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
                        mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mCreateIndex);
                        if (!disableAutoGoToPage) {
                            PointF pointF = mTextUtil.getPageViewOrigin(mPdfViewCtrl, mCreateIndex, startPoint.x, startPoint.y);
                            mPdfViewCtrl.gotoPage(mCreateIndex, pointF.x, pointF.y);
                        }
                    }
                }
            }

            mDownPoint.set(0, 0);
            mMovePoint.set(0, 0);
            mEditRect.setEmpty();
            mAnnotText = "";
            mTextStartPt.set(0, 0);
            mEditPoint.set(0, 0);
            mLastPageIndex = -1;
            mBBoxHeight = 0;
            mBBoxWidth = 0;
            mCreateIndex = -1;
            mTextUtil.resetEditState();
            if (mIsContinue && mListener != null) {
                mListener.callBack();
            }
        } else {
            dismissEditView();
        }
    }

    private TextBoxAddUndoItem createUndoItem(PDFPage page, String content, RectF pdfRectF, RectF textRectF) {
        try {
            TextBoxAddUndoItem undoItem = new TextBoxAddUndoItem(mPdfViewCtrl);
            undoItem.mNM = AppDmUtil.randomUUID(null);
            undoItem.mPageIndex = mCreateIndex;
            undoItem.mColor = mBorderColor;
            undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mContents = content;
            undoItem.mFont = mTextUtil.getSupportFont(mFontId);
            undoItem.mFontSize = mFontSize;
            undoItem.mTextColor = mColor;
            undoItem.mFillColor = Color.WHITE;
            undoItem.mDaFlags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagTextColor | DefaultAppearance.e_FlagFontSize;
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mBBox = new RectF(pdfRectF);
            undoItem.mTextRectF = new RectF(textRectF);
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mFlags = Annot.e_FlagPrint;
            undoItem.mIntent = null;
            undoItem.mSubject = "Textbox";

            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            undoItem.mRotation = rotation == 0 ? rotation : 4 - rotation;
            return undoItem;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    private void dismissEditView() {
        mDownPoint.set(0, 0);
        mMovePoint.set(0, 0);
        mEditRect.setEmpty();
        mAnnotText = "";
        mTextStartPt.set(0, 0);
        mEditPoint.set(0, 0);
        mLastPageIndex = -1;
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        AppUtil.dismissInputSoft(mEditView);
        mUiExtensionsManager.getRootView().removeView(mEditView);
        mEditView = null;
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        mCreating = false;
        mTextUtil.resetEditState();
        mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
        mTextUtil.setKeyboardOffset(0);
        mCreateIndex = -1;
    }

    protected void setBorderColor(int color, int opacity) {
        mBorderColor = color;
        mPaintOut.setColor(AppDmUtil.calColorByMultiply(mBorderColor, AppDmUtil.opacity100To255(opacity)));
    }

    protected void onColorValueChanged(int color) {
        mColor = color;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF rectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rectF, mLastPageIndex);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(rectF));
        }

        setProItemColor(color);
    }

    private void setProItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    protected void onOpacityValueChanged(int opacity) {
        mOpacity = opacity;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF rectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rectF, mLastPageIndex);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(rectF));
        }
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void changeFontDefaultValue(String font) {
        mFontId = mTextUtil.getSupportFontID(font);
    }

    protected void onFontValueChanged(String fontName) {
        mFontId = mTextUtil.getSupportFontID(fontName);
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF rectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rectF, mLastPageIndex);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(rectF));
        }
        if (mCurToolItem == null) return;
        mCurToolItem.property.fontName = fontName;
    }

    protected void onFontSizeValueChanged(float fontSize) {
        mFontSize = fontSize;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF rectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rectF, mLastPageIndex);
            mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(rectF));
        }
        if (mCurToolItem == null) return;
        mCurToolItem.property.fontSize = fontSize;
    }

    private boolean mIsInterceptTouchEvent;

    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled) {
            if (motionEvent.getAction() == MotionEvent.ACTION_DOWN) {
                if (mEditView != null) {
                    mIsInterceptTouchEvent = true;
                    PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
                    mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
                    onSingleTapOrLongPress(pageIndex, point);
                    return true;
                }
            } else {
                if (!mIsInterceptTouchEvent) {
                    handled = onTextBoxToolTouch(pageIndex, motionEvent);
                }

                if (motionEvent.getAction() == MotionEvent.ACTION_UP) {
                    mIsInterceptTouchEvent = false;
                }
            }
        }
        return handled;
    }

    private boolean onTextBoxToolTouch(final int pageIndex, MotionEvent motionEvent) {
        PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        PointF pdfPoint = new PointF(point.x, point.y);
        mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPoint, pdfPoint, pageIndex);

        float x = point.x;
        float y = point.y;

        int action = motionEvent.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (mUiExtensionsManager.getCurrentToolHandler() == this) {
                    if (mCreating) {
                        return false;
                    } else {
                        mDownPoint.set(x, y);
                        mMovePoint.set(x, y);

                        mBBoxWidth = 0;
                        mBBoxHeight = 0;

                        if (mLastPageIndex == -1) {
                            mLastPageIndex = pageIndex;
                        }

                        mPageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                        mPageViewHeigh = mPdfViewCtrl.getPageViewHeight(pageIndex);
                        mTextStartPt.set(x, y);
                        adjustStartPt(mPdfViewCtrl, pageIndex, mTextStartPt);
                        mTextStartPdfPt.set(mTextStartPt.x, mTextStartPt.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mTextStartPdfPt, mTextStartPdfPt, pageIndex);

                        mCreateIndex = pageIndex;
                        return true;
                    }
                }
                return false;
            case MotionEvent.ACTION_MOVE:
                if (!mMovePoint.equals(x, y)) {
                    RectF rectF = onTouchMove((int) x, (int) y);
                    mPdfViewCtrl.convertPageViewRectToPdfRect(rectF, mEditRect, pageIndex);

                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, pageIndex);
                    mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectF));
                    mMovePoint.set(x, y);
                }
                return true;
            case MotionEvent.ACTION_UP:
                if (Math.abs(mMovePoint.length() - mDownPoint.length()) < 5) {
                    PointF startPoint = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
                    mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, pageIndex);

                    RectF rectF = new RectF();
                    rectF.set(startPoint.x, startPoint.y,
                            startPoint.x + getBBoxWidth(pageIndex), startPoint.y + getBBoxHeight(pageIndex));
                    rectF.inset(-FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH) / 2,
                            -FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH) / 2);
                    PointF adjustPointF = FtUtil.adjustPointF(mPdfViewCtrl, pageIndex, rectF);
                    rectF.offset(adjustPointF.x, adjustPointF.y);
                    mPdfViewCtrl.convertPageViewRectToPdfRect(rectF, mEditRect, pageIndex);
                }
                onTouchUp(pageIndex);
                return false;
            case MotionEvent.ACTION_CANCEL:
                mTextStartPt.set(0, 0);
                mEditPoint.set(0, 0);
                mDownPoint.set(0, 0);
                mMovePoint.set(0, 0);
                mCreateAlive = true;
                return true;
            default:
                break;
        }
        return true;
    }

    private final RectF mEditRect = new RectF();

    private RectF onTouchMove(int x, int y) {
        if (x < 0) x = 0;
        if (y < 0) y = 0;
        if (x > mPageViewWidth) x = mPageViewWidth;
        if (y > mPageViewHeigh) y = mPageViewHeigh;

        float minx = Math.min(mDownPoint.x, x);
        float miny = Math.min(mDownPoint.y, y);
        float maxx = Math.max(mDownPoint.x, x);
        float maxy = Math.max(mDownPoint.y, y);
        return new RectF(minx, miny, maxx, maxy);
    }

    private void onTouchUp(final int pageIndex) {
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView == null) {
            mEditView = new EditText(mContext);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            mEditView.setLayoutParams(new ViewGroup.LayoutParams(1, 1));
            mEditView.addTextChangedListener(new TextWatcher() {

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    mAnnotText = FtTextUtil.filterEmoji(String.valueOf(s));
                    mTextUtil.setTextString(pageIndex, mAnnotText, true);
                    mTextUtil.loadText();
                    mPdfViewCtrl.invalidate();
                }

                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                }

                @Override
                public void afterTextChanged(Editable s) {
                }
            });

            mTextUtil.setOnWidthChanged(new FtTextUtil.OnTextValuesChangedListener() {

                @Override
                public void onMaxWidthChanged(float maxWidth) {
                    mBBoxWidth = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, mTextBoxWidth);
                }

                @Override
                public void onMaxHeightChanged(float maxHeight) {
                    mBBoxHeight = maxHeight;
                }

                @Override
                public void onCurrentSelectIndex(int selectIndex) {
                    if (selectIndex >= mEditView.getText().length()) {
                        selectIndex = mEditView.getText().length();
                        mIsSelcetEndText = true;
                    } else {
                        mIsSelcetEndText = false;
                    }
                    mEditView.setSelection(selectIndex);
                }

                @Override
                public void onEditPointChanged(float editPointX,
                                               float editPointY) {
                    PointF point = new PointF(editPointX, editPointY);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(point, point, pageIndex);
                    mEditPoint.set(point.x, point.y);
                }

            });
            mUiExtensionsManager.getRootView().addView(mEditView);
            mPdfViewCtrl.invalidate();
            AppUtil.showSoftInput(mEditView);
            mTextUtil.getBlink().postDelayed((Runnable) mTextUtil.getBlink(), 500);
            mCreating = true;
        }
        mCreateAlive = true;
    }

    private void adjustStartPt(PDFViewCtrl pdfViewCtrl, int pageIndex, PointF point) {
        float fontWidth = FtUtil.widthOnPageView(mPdfViewCtrl, mLastPageIndex, mTextBoxWidth);
        if (pdfViewCtrl.getPageViewWidth(pageIndex) - point.x < fontWidth) {
            point.x = mPageViewWidth - fontWidth;
        }

        float fontHeight = mTextUtil.getFontHeight(pdfViewCtrl, pageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize);
        if (pdfViewCtrl.getPageViewHeight(pageIndex) - point.y < fontHeight) {
            point.y = mPageViewHeigh - fontHeight;
        }
    }


    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        editAnnotationOrNot(false, pageIndex, motionEvent);
        return true;
    }

    private void editAnnotationOrNot(boolean isSingleTap, int pageIndex, MotionEvent motionEvent) {
        if (mEditView == null) {
            onTextBoxToolTouch(pageIndex, motionEvent);
            if (isSingleTap) {
                MotionEvent upEvent = MotionEvent.obtain(motionEvent);
                upEvent.setAction(MotionEvent.ACTION_UP);
                upEvent.setLocation(mMovePoint.x, mMovePoint.y);
                onTextBoxToolTouch(pageIndex, upEvent);
                upEvent.recycle();
            } else {
                mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
            }
        } else {
            PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
            onSingleTapOrLongPress(pageIndex, point);
        }
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled || mEditView != null) {
            editAnnotationOrNot(true, pageIndex, motionEvent);
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinue;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinue = continueAddAnnot;
    }

    private boolean onSingleTapOrLongPress(final int pageIndex, final PointF point) {
        float x = point.x;
        float y = point.y;
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView != null) {
            RectF rectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, rectF, pageIndex);
            if (rectF.contains(x, y)) {
                PointF pointF = new PointF(x, y);
                mPdfViewCtrl.convertPageViewPtToPdfPt(pointF, pointF, pageIndex);
                mEditPoint.set(pointF.x, pointF.y);
                mTextUtil.resetEditState();

                RectF _rect = new RectF(rectF);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(_rect, _rect, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(_rect));
                AppUtil.showSoftInput(mEditView);
                return true;
            } else {
                if (!mIsContinue) {
                    mUiExtensionsManager.setCurrentToolHandler(null);
                    return true;
                }

                if (mCreateAlive) {
                    mCreateAlive = false;
                    if (mUiExtensionsManager.getCurrentToolHandler() == this) {
                        createTBAnnot(false);
                    }
                    return true;
                }

                mCreateAlive = true;
                setCreateAnnotListener(new CreateAnnotResult() {

                    @Override
                    public void callBack() {
                        mBBoxWidth = 0;
                        mBBoxHeight = 0;
                        if (mLastPageIndex == -1) {
                            mLastPageIndex = pageIndex;
                        }

                        mTextStartPt.set(point.x, point.y);
                        adjustStartPt(mPdfViewCtrl, pageIndex, mTextStartPt);
                        mTextStartPdfPt.set(mTextStartPt.x, mTextStartPt.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mTextStartPdfPt, mTextStartPdfPt, pageIndex);
                        mCreateIndex = pageIndex;
                        mPdfViewCtrl.invalidate();
                        if (mEditView != null) {
                            AppUtil.showSoftInput(mEditView);
                        }
                    }
                });
                createTBAnnot(false);
                return true;
            }
        }
        return false;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCreateIndex == pageIndex) {
            canvas.save();

            RectF frameRectF = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mEditRect, frameRectF, pageIndex);
            PointF adjustPointF = FtUtil.adjustPointF(mPdfViewCtrl, pageIndex, frameRectF);
            frameRectF.offset(adjustPointF.x, adjustPointF.y);
            if (mEditView != null) {
                if (getBBoxHeight(pageIndex) > frameRectF.height()) {
                    frameRectF.bottom = frameRectF.top + getBBoxHeight(pageIndex);
                    mPdfViewCtrl.convertPageViewRectToPdfRect(frameRectF, mEditRect, pageIndex);
                }
                canvas.drawRect(frameRectF, mPaintFill);

                PointF textStartPoint = new PointF(mTextStartPdfPt.x, mTextStartPdfPt.y);
                mPdfViewCtrl.convertPdfPtToPageViewPt(textStartPoint, textStartPoint, pageIndex);
                textStartPoint.offset(adjustPointF.x, adjustPointF.y);
                PointF editPoint = new PointF(mEditPoint.x, mEditPoint.y);
                if (editPoint.x != 0 || editPoint.y != 0) {
                    mPdfViewCtrl.convertPdfPtToPageViewPt(editPoint, editPoint, pageIndex);
                }
                mTextUtil.setTextString(pageIndex, mAnnotText, true);
                mTextUtil.setStartPoint(textStartPoint);
                mTextUtil.setEditPoint(editPoint);
                mTextUtil.setMaxRect(Math.max(mFontSize / 5, frameRectF.width() - mFontSize / 5),
                        mPdfViewCtrl.getPageViewHeight(pageIndex) - textStartPoint.y);

                mTextUtil.setTextColor(mColor, AppDmUtil.opacity100To255(mOpacity));
                mTextUtil.setFont(mTextUtil.getSupportFontName(mFontId), mFontSize);

                if (mIsSelcetEndText) {
                    mTextUtil.setEndSelection(mEditView.getSelectionEnd() + 1);
                } else {
                    mTextUtil.setEndSelection(mEditView.getSelectionEnd());
                }

                mTextUtil.loadText();
                mTextUtil.drawText(canvas);
            } else {
                canvas.drawRect(frameRectF, mPaintFill);
            }
            // canvas frame board
            mPaintOut.setStrokeWidth(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH));
            canvas.drawRect(frameRectF, mPaintOut);
            canvas.restore();
        }
    }

    private void setCreateAnnotListener(CreateAnnotResult listener) {
        mListener = listener;
    }

    private float getBBoxHeight(int pageIndex) {
        return mBBoxHeight == 0 ? mTextUtil.getFontHeight(mPdfViewCtrl, pageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize) : mBBoxHeight;
    }

    private float getBBoxWidth(int pageIndex) {
        return mBBoxWidth == 0 ? FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, mTextBoxWidth) : mBBoxWidth;
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new TextBoxToolSupply(mContext);
        return mToolSupply;
    }

    private class TextBoxToolSupply extends ToolSupplyImpl {

        public TextBoxToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_textbox_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return R.drawable.comment_tool_textbox_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Textbox;
            property.color = mColor;
            property.opacity = mOpacity;
            property.fontName = mTextUtil.getSupportFontName(mFontId);
            property.fontSize = mFontSize;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_TEXTBOX;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Textbox);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mFontId = mTextUtil.getSupportFontID(property.fontName);
                mFontSize = property.fontSize;
                mUiExtensionsManager.setCurrentToolHandler(TextBoxToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == TextBoxToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mFontId = mTextUtil.getSupportFontID(property.fontName);
            mFontSize = property.fontSize;
            TextBoxToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void handlePageChanged() {
        if (mEditView == null) {
            return;
        }

        // MOBRD-7200
        if (mLastPageIndex == mPdfViewCtrl.getPageCount() - 1) {
            Rect rect = mPdfViewCtrl.getPageViewRect(mLastPageIndex);
            int h = mPdfViewCtrl.getHeight();
            if (rect.bottom <= mPdfViewCtrl.getHeight())
                return;
        }

        mCreateAlive = false;
        createTBAnnot(true);
    }

    private void setUndoItemCallback(UndoModule.IUndoItemCallback callback) {
        UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
        if (undoModule != null) {
            undoModule.setUndoItemCallback(callback);
        }
    }

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mCurToolItem != null && mUiExtensionsManager.getCurrentAnnotHandler() instanceof TextBoxAnnotHandler) {
                ((TextBoxAnnotHandler) mUiExtensionsManager.getCurrentAnnotHandler()).handleKeyboardClosed();
                return true;
            }
            if (mEditView != null && mCurToolItem != null) {
//                mCurToolItem.toolItem.performClick();
                mCreateAlive = false;
                createTBAnnot(true, false);
                mCreating = false;
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mEditView != null && mCurToolItem != null) {
                mCurToolItem.toolItem.performClick();
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };

}
