/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import static com.foxit.uiextensions.utils.AppDmUtil.rectFToRect;

import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.util.SparseArray;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.sdk.pdf.objects.PDFArray;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.annots.AbstractToolHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;

import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;


public class EraserToolHandler extends AbstractToolHandler {
    private static final int MIN_ERASER_LINE_WIDTH = 1;
    private static final int MAX_ERASER_LINE_WIDTH = 36;

    private int mShapeMode;

    private final Paint mEraserPaint;
    private final Paint mPaint;
    protected float mRadius = 15.0f;
    private int mCtlPtRadius = 5;

    private final ArrayList<AnnotInfo> mRootList;
    private ArrayList<Path> mPaths;

    private final ArrayList<Annot> mTempRemovingAnnots;
    private final ArrayList<AnnotInfo> mTempRootList;
    private ArrayList<Path> mTempPaths;
    private final SparseArray<FSPageDisplayMatrix> mPageMatrixArray;
    private int mTempCapturedPage = -1;
    private boolean mIsScaling = false;

    private boolean mTouchCaptured = false;
    private int mCapturedPage = -1;
    private final PointF mLastPt = new PointF();

    private ToolItemBean mCurToolItem;

    private boolean mIsPen = false;
    private boolean mShowPenOnlySwitch = true;

    public EraserToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context, pdfViewCtrl, ToolHandler.TH_TYPE_ERASER, "ERASER");

        mCtlPtRadius = AppDisplay.dp2px((float) mCtlPtRadius);
        mRootList = new ArrayList<>();
        mPaths = new ArrayList<>();

        mTempRemovingAnnots = new ArrayList<>();
        mTempRootList = new ArrayList<>();
        mTempPaths = new ArrayList<>();
        mPageMatrixArray = new SparseArray<>();

        mEraserPaint = new Paint();
        mEraserPaint.setStyle(Style.STROKE);
        mEraserPaint.setAntiAlias(true);
        mEraserPaint.setDither(true);
        mEraserPaint.setColor(Color.RED);

        mPaint = new Paint();
        mPaint.setStyle(Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);
        mPaint.setStrokeCap(Paint.Cap.ROUND);
        mColor = Color.LTGRAY;

        mThickness = 18;
        mShapeMode = InkConstants.ERASER_SHAPE_PARTIAL;
        mShowPenOnlySwitch = mUiExtensionsManager.getConfig().uiSettings.showPenOnlySwitch;
    }

    void initUiElements() {
    }

    @Override
    public void onActivate() {
        mIsPen = mUiExtensionsManager.getInkDrawToolType() == InkDrawToolType.STYLUS;
        mCapturedPage = -1;
        mRootList.clear();
        mPaths.clear();
        mTempRemovingAnnots.clear();
        mInvalidateRect.setEmpty();
        resetPropertyBar();
    }

    @Override
    protected void resetPropertyBar() {
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH_2, mThickness);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_ERASERSHAPE, mShapeMode);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_MAX_LINEWIDTH, MAX_ERASER_LINE_WIDTH);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_MIN_LINEWIDTH, MIN_ERASER_LINE_WIDTH);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(getSupportedProperties());
        if (mShowPenOnlySwitch)
            mPropertyBar.addCustomItem(PropertyBar.PROPERTY_PEN_ONLY, getPenOnlyView(), 0, -1);
        mPropertyBar.setPropertyChangeListener(this);
    }

    private View mPenOnlyView;
    private RelativeLayout mPenOnlySwitchRela;
    private ImageView mPenOnlySwitchIv;
    private TextView mPenOnlyTitleTv;

    private View getPenOnlyView() {
        mPenOnlyView = View.inflate(mContext, R.layout.pb_ink_draw_type, null);
        mPenOnlyTitleTv = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_tv);
        mPenOnlySwitchRela = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_rl);
        mPenOnlySwitchIv = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_iv);
        ThemeUtil.setBackgroundTintList(mPenOnlySwitchRela, getSelectedButtonColorStateList());
        setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, mUiExtensionsManager.getInkDrawToolType() == InkDrawToolType.STYLUS);

        mPenOnlySwitchRela.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mPenOnlySwitchRela.isSelected()) {
                    mIsPen = false;
                    setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, false);
                    if (mCurToolItem != null) {
                        mUiExtensionsManager.setInkDrawToolType(InkDrawToolType.STYLUS_OR_FINGER);
                    }
                } else {
                    mIsPen = true;
                    setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, true);
                    if (mCurToolItem != null) {
                        mUiExtensionsManager.setInkDrawToolType(InkDrawToolType.STYLUS);
                    }
                }
            }
        });
        return mPenOnlyView;
    }

    private void setSelectedButtonState(RelativeLayout checkLayout, ImageView checkView, boolean isSelected) {
        checkLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) checkView.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        checkView.setLayoutParams(params);
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    void updateTheme() {
        if (mPenOnlyView != null) {
            mPenOnlyTitleTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            ThemeUtil.setBackgroundTintList(mPenOnlySwitchRela, getSelectedButtonColorStateList());
            setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, mUiExtensionsManager.getInkDrawToolType() == InkDrawToolType.STYLUS);
        }
    }

    @Override
    protected void onCreateValueChanged(long property, Object value) {
    }

    private final RectF mInvalidateRect = new RectF();
    private CompositeDisposable mDisposable;

    @Override
    public void onDeactivate() {
        if (mDisposable != null) {
            mDisposable.dispose();
            mDisposable = null;
        }
        mIsPen = false;
        mTempCapturedPage = -1;
        mTempRootList.clear();
        mTempPaths.clear();
        mIsScaling = false;
        mPageMatrixArray.clear();
        resetEraser(mCapturedPage, mInvalidateRect);
    }

    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent e) {
        if (!mUiExtensionsManager.getDocumentManager().canAddAnnot() || !mUiExtensionsManager.isEnableModification()) {
            return false;
        }
        if (e.getToolType(0) == MotionEvent.TOOL_TYPE_STYLUS) {
            mIsPen = true;
        }

        final PointF point = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);

        float x = point.x;
        float y = point.y;
        switch (e.getActionMasked()) {
            case MotionEvent.ACTION_DOWN:
                if (e.getToolType(0) != MotionEvent.TOOL_TYPE_STYLUS && mIsPen) {
                    return mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
                }
                if (!mTouchCaptured) {
                    if (mCapturedPage == -1) {
                        mTouchCaptured = true;
                        mCapturedPage = pageIndex;
                    } else if (pageIndex == mCapturedPage) {
                        mTouchCaptured = true;
                    } else {
                        resetEraser(mCapturedPage, pageIndex, mInvalidateRect);
                        mTouchCaptured = true;
                        mCapturedPage = pageIndex;
                    }

                    mLastPt.set(x, y);
                }
                return true;
            case MotionEvent.ACTION_POINTER_DOWN:
                invalidateEraserZone(pageIndex, point);
                MotionEvent downEvent = MotionEvent.obtain(e);
                downEvent.setAction(MotionEvent.ACTION_DOWN);
                boolean handled = mPdfViewCtrl.defaultTouchEvent(downEvent);
                downEvent.recycle();
                return handled;
            case MotionEvent.ACTION_MOVE:
                if (e.getPointerCount() > 1) {
                    return mPdfViewCtrl.defaultTouchEvent(e);
                }
                if (e.getToolType(0) != MotionEvent.TOOL_TYPE_STYLUS && mIsPen) {
                    return mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
                }
                if (mTouchCaptured && mCapturedPage == pageIndex) {
                    if (!mLastPt.equals(x, y)) {
                        if (mShapeMode == InkConstants.ERASER_SHAPE_WHOLE)
                            cleanWholeShapeLines(mPdfViewCtrl, pageIndex, point);
                        else
                            calculateNewLines(mPdfViewCtrl, pageIndex, point);
                        RectF invaRect = getEraserBBox(mLastPt, point);
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invaRect, invaRect, pageIndex);
                        mPdfViewCtrl.invalidate(rectFToRect(invaRect));
                        mLastPt.set(x, y);
                    }
                }
                return true;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mTouchCaptured) {
                    return mPdfViewCtrl.defaultTouchEvent(e);
                }
                if (e.getToolType(0) != MotionEvent.TOOL_TYPE_STYLUS && mIsPen) {
                    return mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
                }
                onTouchUp(new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (mTouchCaptured) {
                            invalidateEraserZone(pageIndex, point);
                        }
                    }
                });
                return true;
            default:
        }
        return true;
    }

    void invalidateEraserZone(int pageIndex, PointF point) {
        mTouchCaptured = false;
        RectF invaRect2 = getEraserBBox(mLastPt, point);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invaRect2, invaRect2, pageIndex);
        mPdfViewCtrl.invalidate(rectFToRect(invaRect2));
        mLastPt.set(-mRadius, -mRadius);
    }

    private void onTouchUp(final Event.Callback callback) {
        if (mDisposable == null)
            mDisposable = new CompositeDisposable();

        mDisposable.add(onTouchUpImpl().subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<TouchInfo>() {
                    @Override
                    public void accept(final TouchInfo touchInfo) throws Exception {
                        if (touchInfo.success) {
                            int pageIndex = touchInfo.pageIndex;

                            if (touchInfo.noModifyRectF != null) {
                                if (touchInfo.noModifyInfos != null) {
                                    mRootList.removeAll(touchInfo.noModifyInfos);
                                    touchInfo.noModifyInfos.clear();
                                }

                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    RectF viewRectF = new RectF();
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(touchInfo.noModifyRectF, viewRectF, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRectF));
                                }
                            }

                            if (touchInfo.undoItem != null) {
                                ArrayList<EditAnnotEvent> onRemovingEvents = new ArrayList<>();
                                for (EditAnnotEvent event : touchInfo.events) {
                                    if (event.mType == EditAnnotEvent.EVENTTYPE_DELETE) {
                                        if (mTempRemovingAnnots.contains(event.mAnnot)) {
                                            onRemovingEvents.add(event);
                                        } else {
                                            mTempRemovingAnnots.add(event.mAnnot);
                                        }
                                    }
                                }
                                touchInfo.events.removeAll(onRemovingEvents);

                                RectF rectF = new RectF(touchInfo.refreshRectF);
                                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                                if (mInvalidateRect.isEmpty()) {
                                    mInvalidateRect.set(rectF);
                                } else {
                                    mInvalidateRect.union(rectF);
                                }

                                if (touchInfo.events.size() == 0) return;
                                touchInfo.undoItem.doTask(pageIndex, rectF, touchInfo.events, new Event.Callback() {
                                    @Override
                                    public void result(Event event, boolean success) {
                                        if (touchInfo.removeInfos != null) {
                                            for (AnnotInfo annotInfo : touchInfo.removeInfos) {
                                                mRootList.remove(annotInfo);
                                                mTempRemovingAnnots.remove(annotInfo.mAnnot);
                                                invalidateJniAnnots(annotInfo, 1, null);
                                            }
                                        }

                                        mUiExtensionsManager.getDocumentManager().addUndoItem(touchInfo.undoItem);
                                        mUiExtensionsManager.getDocumentManager().setHasModifyTask(false);

                                        touchInfo.release();
                                        if (callback != null)
                                            callback.result(event, success);
                                    }
                                });
                            } else {
                                touchInfo.release();
                                if (callback != null)
                                    callback.result(null, touchInfo.success);
                            }
                        } else {
                            touchInfo.release();
                            if (callback != null)
                                callback.result(null, false);
                        }
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) {
                        if (callback != null)
                            callback.result(null, false);
                    }
                }));
    }

    private Single<TouchInfo> onTouchUpImpl() {
        return Single.fromCallable(new Callable<TouchInfo>() {
            @Override
            public TouchInfo call() throws Exception {
                TouchInfo touchInfo = new TouchInfo();

                // if the ink annotation is not really modified,
                // remove it from the list, and update data in JNI.
                RectF noModifyRect = new RectF();
                boolean hasNoModify = false;
                ArrayList<AnnotInfo> noModifyInfos = new ArrayList<>();
                if (mRootList.size() > 1) {
                    try {
                        for (int i = 0; i < mRootList.size(); i++) {
                            if (mDisposable == null || mDisposable.isDisposed()) break;

                            AnnotInfo annotInfo = mRootList.get(i);
                            if (!annotInfo.mModifyFlag) {
                                noModifyInfos.add(annotInfo);
                                invalidateJniAnnots(annotInfo, 1, null);

                                if (!hasNoModify) {
                                    noModifyRect.set(AppUtil.toRectF(annotInfo.mAnnot.getRect()));
                                } else {
                                    AppUtil.unionFxRectF(noModifyRect, AppUtil.toRectF(annotInfo.mAnnot.getRect()));
                                }
                                hasNoModify = true;
                            }
                        }
//                        mRootList.removeAll(noModifyInfos);

                        touchInfo.pageIndex = mRootList.get(0).mAnnot.getPage().getIndex();
                        touchInfo.noModifyRectF = noModifyRect;
                        touchInfo.noModifyInfos = noModifyInfos;
                    } catch (Exception ignored) {
                        touchInfo.success = false;
                    }
                }

                if (noModifyInfos.size() != mRootList.size() && !mRootList.isEmpty()) {
                    Collections.sort(mRootList);
                    try {
                        if (mDisposable == null || mDisposable.isDisposed()) {
                            touchInfo.success = false;
                            return touchInfo;
                        }
                        int pageIndex = mRootList.get(0).mAnnot.getPage().getIndex();
                        RectF rect = AppUtil.toRectF(mRootList.get(0).mAnnot.getRect());
                        RectF rmRect = new RectF(rect);
                        EraserUndoItem undoItems = new EraserUndoItem(mPdfViewCtrl, EraserToolHandler.this);
                        undoItems.mPageIndex = pageIndex;
                        boolean isLast = false;

                        ArrayList<AnnotInfo> removeAnnots = new ArrayList<>();
                        ArrayList<EditAnnotEvent> events = new ArrayList<>();

                        for (int i = mRootList.size() - 1; i >= 0; i--) {
                            if (mDisposable == null || mDisposable.isDisposed()) break;

                            if (i == 0) {
                                isLast = true;
                            }
                            AnnotInfo annotInfo = mRootList.get(i);
                            if (annotInfo.mModifyFlag) {
                                if (!annotInfo.mNewLines.isEmpty()) {
                                    addModifyAnnotEvent(annotInfo, undoItems, events);
                                } else {
                                    addDeleteAnnotEvent(annotInfo.mAnnot, undoItems, events);
                                    removeAnnots.add(annotInfo);
                                }
                            }
                            AppUtil.unionFxRectF(rmRect, AppUtil.toRectF(annotInfo.mAnnot.getRect()));
                        }
                        mUiExtensionsManager.getDocumentManager().setHasModifyTask(isLast);

                        if (hasNoModify)
                            AppUtil.unionFxRectF(rmRect, noModifyRect);

                        touchInfo.success = true;
                        touchInfo.refreshRectF = rmRect;
                        touchInfo.removeInfos = removeAnnots;
                        touchInfo.undoItem = undoItems;
                        touchInfo.events = events;
                        touchInfo.pageIndex = pageIndex;
                    } catch (Exception ignored) {
                        touchInfo.success = false;
                    }
                } else {
                    touchInfo.success = true;
                }
                return touchInfo;
            }
        });
    }

    static class TouchInfo {
        boolean success;
        int pageIndex;
        RectF noModifyRectF;
        RectF refreshRectF;
        EraserUndoItem undoItem;
        ArrayList<AnnotInfo> noModifyInfos;
        ArrayList<AnnotInfo> removeInfos;
        ArrayList<EditAnnotEvent> events;

        void release() {
            success = false;
            pageIndex = -1;
            noModifyRectF = null;
            refreshRectF = null;
            undoItem = null;
            if (removeInfos != null) {
                removeInfos.clear();
                removeInfos = null;
            }
            if (noModifyInfos != null) {
                noModifyInfos.clear();
                noModifyInfos = null;
            }
            if (events != null) {
                events.clear();
                events = null;
            }
        }
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        return false;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        return false;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCapturedPage == pageIndex) {
            canvas.drawCircle(mLastPt.x, mLastPt.y, mRadius, mEraserPaint);
            if (mRootList.isEmpty()) return;

            FSPageDisplayMatrix matrix = this.getPageMatrix(pageIndex, mPdfViewCtrl);
            for (int i = 0; i < mRootList.size(); i++) {
                AnnotInfo annotInfo = mRootList.get(i);
                if (!annotInfo.mDrawAtJava) continue;
                if (!annotInfo.mIsPSIMode) {
                    setPaint(matrix, pageIndex, annotInfo);
                    mPaths = getNewPaths(matrix, mPdfViewCtrl, pageIndex, annotInfo);
                    int count = mPaths.size();
                    for (int j = 0; j < count; j++) {
                        canvas.drawPath(mPaths.get(j), mPaint);
                    }
                }
            }
        }

        if (mTempCapturedPage == pageIndex) {
            if (mTempRootList.isEmpty()) return;

            FSPageDisplayMatrix matrix = this.getPageMatrix(pageIndex, mPdfViewCtrl);
            for (int i = 0; i < mTempRootList.size(); i++) {
                AnnotInfo annotInfo = mTempRootList.get(i);
                if (!annotInfo.mDrawAtJava) continue;
                if (!annotInfo.mIsPSIMode) {
                    setPaint(matrix, pageIndex, annotInfo);
                    mTempPaths = getNewPaths(matrix, mPdfViewCtrl, pageIndex, annotInfo);
                    int count = mTempPaths.size();
                    for (int j = 0; j < count; j++) {
                        canvas.drawPath(mTempPaths.get(j), mPaint);
                    }
                }
            }
        }
    }


    static class AnnotInfo implements Comparable<AnnotInfo> {
        Ink mAnnot;
        float mBorderWidth;
        int mBorderColor;
        float mOpacity;

        boolean mModifyFlag;
        boolean mDrawAtJava;
        ArrayList<LineInfo> mNewLines;
        boolean mIsPSIMode = false; //true : psi, false: ink

        public AnnotInfo() {
            mModifyFlag = false;
            mDrawAtJava = false;
            mNewLines = new ArrayList<LineInfo>();
            mIsPSIMode = false;
            mBorderWidth = -1;
            mBorderColor = Color.RED;
            mOpacity = 1.0f;
        }

        //sort mRootlist :delete  and  modify  from small to big by annotation index
        @Override
        public int compareTo(AnnotInfo another) {
            try {
                if (another.mNewLines.isEmpty()) {
                    if (mNewLines.isEmpty()) {
                        return Integer.compare(mAnnot.getIndex(), another.mAnnot.getIndex());
                    } else {
                        return 1;
                    }
                } else {
                    if (mNewLines.isEmpty()) {
                        return -1;
                    } else {
                        return Integer.compare(mAnnot.getIndex(), another.mAnnot.getIndex());
                    }
                }
            } catch (PDFException ignored) {
            }
            return 1;
        }

    }

    static class LineInfo {
        RectF mLineBBox;
        ArrayList<PointF> mLine;
        ArrayList<Float> mPresses;

        public LineInfo() {
            mLineBBox = new RectF();
            mLine = new ArrayList<PointF>();
            mPresses = new ArrayList<Float>();
        }
    }

    private List<Integer> mDeletingAnnots = new ArrayList<>();

    private void cleanWholeShapeLines(final PDFViewCtrl pdfViewCtrl, final int pageIndex, PointF point) {
        if (!mUiExtensionsManager.getDocumentManager().withModifyPermission()
                || !mUiExtensionsManager.getDocumentManager().withDeletePermission()) {
            return;
        }

        RectF rect = new RectF(point.x, point.y, point.x, point.y);
        rect.union(mLastPt.x, mLastPt.y);
        rect.inset(-mRadius, -mRadius);
        mPdfViewCtrl.convertPageViewRectToPdfRect(rect, rect, pageIndex);

        try {
            PDFPage page = pdfViewCtrl.getDoc().getPage(pageIndex);
            ArrayList<Annot> annotList = mUiExtensionsManager.getDocumentManager().getAnnotsInteractRect(page, new RectF(rect), Annot.e_Ink);
            RectF eraseBBox = getEraserBBox(mLastPt, point);
            mPdfViewCtrl.convertPageViewRectToPdfRect(eraseBBox, eraseBBox, pageIndex);

            int count = 0;
            for (final Annot annot : annotList) {
                if (AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)
                        || !mUiExtensionsManager.getDocumentManager().withDeletePermission(annot)
                        || !mUiExtensionsManager.getDocumentManager().withModifyPermission(annot))
                    continue;
                RectF annotBBox = AppUtil.toRectF(annot.getRect());
                if (DocumentManager.intersects(annotBBox, eraseBBox)) {
                    final int objNum = AppAnnotUtil.getAnnotObjNum(annot);
                    if (mDeletingAnnots.contains(objNum)) {
                        continue;
                    }
                    mDeletingAnnots.add(objNum);

                    final RectF rmRect = new RectF(rect);
                    //annot Info
                    final AnnotInfo annotInfo = new AnnotInfo();
                    annotInfo.mAnnot = (Ink) annot;
                    com.foxit.sdk.common.Path path = ((Ink) annot).getInkList();

                    LineInfo lineInfo = null;
                    int ptCount = path.getPointCount();
                    for (int j = 0; j < ptCount; j++) {
                        if (path.getPointType(j) == com.foxit.sdk.common.Path.e_TypeMoveTo) {
                            lineInfo = new LineInfo();
                        }
                        lineInfo.mLine.add(AppUtil.toPointF(path.getPoint(j)));
                        if (j == ptCount - 1 || ((j + 1) < ptCount && path.getPointType(j + 1) == com.foxit.sdk.common.Path.e_TypeMoveTo)) {
                            lineInfo.mLineBBox = getLineBBox(lineInfo.mLine, annot.getBorderInfo().getWidth());
                            annotInfo.mNewLines.add(lineInfo);
                        }
                    }

                    ArrayList<EditAnnotEvent> events = new ArrayList<>();
                    final EraserUndoItem undoItems = new EraserUndoItem(pdfViewCtrl, EraserToolHandler.this);
                    undoItems.mPageIndex = pageIndex;
                    addDeleteAnnotEvent(annot, undoItems, events);

                    AppUtil.unionFxRectF(rmRect, AppUtil.toRectF(annotInfo.mAnnot.getRect()));
                    mInvalidateRect.set(rmRect);

                    mUiExtensionsManager.getDocumentManager().setHasModifyTask(count == annotList.size());
                    RectF rectF = new RectF(mInvalidateRect);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                    undoItems.doTask(pageIndex, rectF, events, new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            invalidateJniAnnots(annotInfo, 1, null);
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                RectF viewRectF = new RectF();
                                mPdfViewCtrl.convertPdfRectToPageViewRect(rmRect, viewRectF, pageIndex);
                                mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRectF));
                            }

                            mDeletingAnnots.remove((Integer) objNum);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItems);
                            mUiExtensionsManager.getDocumentManager().setHasModifyTask(false);
                        }
                    });
                }
                count++;
            }

        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void calculateNewLines(final PDFViewCtrl pdfViewCtrl, final int pageIndex, PointF point) {
        if (!mUiExtensionsManager.getDocumentManager().withModifyPermission()
                || !mUiExtensionsManager.getDocumentManager().withDeletePermission()) {
            return;
        }

        RectF rect = new RectF(point.x, point.y, point.x, point.y);
        rect.union(mLastPt.x, mLastPt.y);
        rect.inset(-mRadius, -mRadius);
        mPdfViewCtrl.convertPageViewRectToPdfRect(rect, rect, pageIndex);
        try {
            PDFPage page = pdfViewCtrl.getDoc().getPage(pageIndex);
            ArrayList<Annot> annotList = mUiExtensionsManager.getDocumentManager().getAnnotsInteractRect(page, new RectF(rect), Annot.e_Ink);
            PointF tv_pt = new PointF();
            RectF tv_rectF = new RectF();
            RectF eraseBBox = getEraserBBox(mLastPt, point);
            mPdfViewCtrl.convertPageViewRectToPdfRect(eraseBBox, eraseBBox, pageIndex);
            for (final Annot annot : annotList) {
                if (AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)
                        || !mUiExtensionsManager.getDocumentManager().withDeletePermission(annot)
                        || !mUiExtensionsManager.getDocumentManager().withModifyPermission(annot))
                    continue;
                RectF annotBBox = AppUtil.toRectF(annot.getRect());
                if (DocumentManager.intersects(annotBBox, eraseBBox)) {
                    boolean isExist = false;
                    for (int i = 0; i < mRootList.size(); i++) {
                        if (AppAnnotUtil.getAnnotUniqueID(mRootList.get(i).mAnnot).equalsIgnoreCase(AppAnnotUtil.getAnnotUniqueID(annot))) {
                            isExist = true;
                            break;
                        }
                    }
                    if (!isExist) {
                        final AnnotInfo annotInfo = new AnnotInfo();
                        annotInfo.mAnnot = (Ink) annot;
                        annotInfo.mBorderWidth = annot.getBorderInfo().getWidth();
                        annotInfo.mBorderColor = annot.getBorderColor();
                        annotInfo.mOpacity = ((Ink) annot).getOpacity();

                        com.foxit.sdk.common.Path path = ((Ink) annot).getInkList();
                        LineInfo lineInfo = null;
                        int ptCount = path.getPointCount();
                        for (int j = 0; j < ptCount; j++) {
                            if (path.getPointType(j) == com.foxit.sdk.common.Path.e_TypeMoveTo) {
                                lineInfo = new LineInfo();
                            }
                            lineInfo.mLine.add(AppUtil.toPointF(path.getPoint(j)));
                            if (j == ptCount - 1 || ((j + 1) < ptCount && path.getPointType(j + 1) == com.foxit.sdk.common.Path.e_TypeMoveTo)) {
                                lineInfo.mLineBBox = getLineBBox(lineInfo.mLine, annot.getBorderInfo().getWidth());
                                annotInfo.mNewLines.add(lineInfo);
                            }
                        }
                        mRootList.add(annotInfo);

                        invalidateJniAnnots(annotInfo, 0, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                try {
                                    RectF viewRect = AppUtil.toRectF(annot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    mPdfViewCtrl.refresh(pageIndex, rectFToRect(viewRect));
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                                annotInfo.mDrawAtJava = true;
                            }
                        });
                    }
                }
            }

            if (mRootList.size() == 0) return;
            PointF pdfDP = new PointF(mLastPt.x, mLastPt.y);
            PointF pdfCP = new PointF(point.x, point.y);
            RectF eBBox = getEraserBBox(mLastPt, point);
            RectF radiusRect = new RectF(0, 0, mRadius, mRadius);
            mPdfViewCtrl.convertPageViewPtToPdfPt(pdfDP, pdfDP, pageIndex);
            mPdfViewCtrl.convertPageViewPtToPdfPt(pdfCP, pdfCP, pageIndex);
            mPdfViewCtrl.convertPageViewRectToPdfRect(eBBox, eBBox, pageIndex);
            mPdfViewCtrl.convertPageViewRectToPdfRect(radiusRect, radiusRect, pageIndex);

            float pdfR = radiusRect.width();
            PointF intersectPoint = new PointF();
            PointF pdfPoint1 = new PointF();
            PointF pdfPoint2 = new PointF();

            for (int i = 0; i < mRootList.size(); i++) {
                AnnotInfo annotNode = mRootList.get(i);
                if (!DocumentManager.intersects(AppUtil.toRectF(annotNode.mAnnot.getRect()), eBBox))
                    continue;
                for (int lineIndex = 0; lineIndex < annotNode.mNewLines.size(); lineIndex++) {
                    LineInfo lineNode = annotNode.mNewLines.get(lineIndex);
                    ArrayList<PointF> pdfLine = lineNode.mLine;
                    ArrayList<Float> presses = lineNode.mPresses;
                    int end1_PointIndex = -1, begin2_PointIndex = -1;

                    //if lineRect intersect eraseRect
                    RectF lineRect = lineNode.mLineBBox;
                    if (!DocumentManager.intersects(lineRect, eBBox))
                        continue;

                    for (int j = 0; j < pdfLine.size(); j++) {
                        // out of circle point  or  first point
                        pdfPoint1.set(pdfLine.get(j).x, pdfLine.get(j).y);
                        boolean createNewLine = false;
                        boolean reachEnd = false;
                        if (j == pdfLine.size() - 1) {
                            reachEnd = true;
                        } else {
                            // in circle point  or  second point
                            pdfPoint2.set(pdfLine.get(j + 1).x, pdfLine.get(j + 1).y);
                        }

                        int type = getIntersection(pdfPoint1, pdfPoint2, pdfDP, pdfCP, intersectPoint);
                        if (!reachEnd && type == 1) {
                            createNewLine = true;
                            tv_rectF.set(intersectPoint.x, intersectPoint.y, intersectPoint.x, intersectPoint.y);
                            for (int p = j; p >= 0; p--) {
                                tv_pt.set(pdfLine.get(p).x, pdfLine.get(p).y);
                                tv_rectF.union(tv_pt.x, tv_pt.y);
                                if (getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, intersectPoint.x, intersectPoint.y) > pdfR) {
                                    end1_PointIndex = p;
                                    if (p > 0) {
                                        tv_rectF.union(pdfLine.get(p - 1).x, pdfLine.get(p - 1).y);
                                    }
                                    break;
                                }
                            }
                            for (int q = j + 1; q < pdfLine.size(); q++) {
                                tv_pt.set(pdfLine.get(q).x, pdfLine.get(q).y);
                                tv_rectF.union(tv_pt.x, tv_pt.y);
                                if (getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, intersectPoint.x, intersectPoint.y) > pdfR) {
                                    begin2_PointIndex = q;
                                    if (q < pdfLine.size() - 1) {
                                        tv_rectF.union(pdfLine.get(q + 1).x, pdfLine.get(q + 1).y);
                                    }
                                    break;
                                }
                            }
                        } else if (getDistanceOfPointToLine(pdfPoint1.x, pdfPoint1.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y) < pdfR) {
                            if (isIntersectPointInLine(pdfPoint1.x, pdfPoint1.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y)
                                    || getDistanceOfTwoPoints(pdfPoint1.x, pdfPoint1.y, pdfDP.x, pdfDP.y) < pdfR
                                    || getDistanceOfTwoPoints(pdfPoint1.x, pdfPoint1.y, pdfCP.x, pdfCP.y) < pdfR) {
                                createNewLine = true;
                                for (int p = j; p >= 0; p--) {
                                    tv_pt.set(pdfLine.get(p).x, pdfLine.get(p).y);
                                    tv_rectF.union(tv_pt.x, tv_pt.y);
                                    if (getDistanceOfPointToLine(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y) < pdfR &&
                                            (isIntersectPointInLine(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y) ||
                                                    getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y) < pdfR ||
                                                    getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, pdfCP.x, pdfCP.y) < pdfR)) {
                                        continue;
                                    }
                                    end1_PointIndex = p;
                                    if (p > 0) {
                                        tv_rectF.union(pdfLine.get(p - 1).x, pdfLine.get(p - 1).y);
                                    }
                                    break;
                                }
                                for (int q = j + 1; q < pdfLine.size(); q++) {
                                    tv_pt.set(pdfLine.get(q).x, pdfLine.get(q).y);
                                    tv_rectF.union(tv_pt.x, tv_pt.y);
                                    if (getDistanceOfPointToLine(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y) < pdfR &&
                                            (isIntersectPointInLine(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y, pdfCP.x, pdfCP.y) ||
                                                    getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, pdfDP.x, pdfDP.y) < pdfR ||
                                                    getDistanceOfTwoPoints(tv_pt.x, tv_pt.y, pdfCP.x, pdfCP.y) < pdfR)) {
                                        continue;
                                    }
                                    begin2_PointIndex = q;
                                    if (q < pdfLine.size() - 1) {
                                        tv_rectF.union(pdfLine.get(q + 1).x, pdfLine.get(q + 1).y);
                                    }
                                    break;
                                }
                            }
                        }

                        if (createNewLine) {
                            createNewLine = false;

                            ArrayList<PointF> newLine1 = new ArrayList<PointF>();
                            ArrayList<Float> newPresses1 = new ArrayList<Float>();
                            if (0 <= end1_PointIndex && end1_PointIndex < pdfLine.size()) {
                                for (int k = 0; k <= end1_PointIndex; k++) {
                                    newLine1.add(pdfLine.get(k));
                                }
                            }

                            ArrayList<PointF> newLine2 = new ArrayList<PointF>();
                            ArrayList<Float> newPresses2 = new ArrayList<Float>();
                            if (0 <= begin2_PointIndex && begin2_PointIndex < pdfLine.size()) {
                                for (int k = pdfLine.size() - 1; k >= begin2_PointIndex; k--) {
                                    newLine2.add(pdfLine.get(k));
                                }
                            }

                            annotNode.mNewLines.remove(lineIndex);
                            if (newLine1.size() == 0 && newLine2.size() == 0) {
                                // current line is removed, and no new line is added
                                // lineIndex -- adjust index continue to erase next line
                                lineIndex--;
                            } else {
                                // insert line2 first, then line1
                                // make sure the line1 is before line2
                                if (newLine2.size() > 1) {
                                    LineInfo info = new LineInfo();
                                    info.mLine = newLine2;
                                    info.mPresses = newPresses2;
                                    info.mLineBBox = getLineBBox(newLine2, annotNode.mAnnot.getBorderInfo().getWidth());
                                    annotNode.mNewLines.add(lineIndex, info);
                                }
                                if (newLine1.size() > 1) {
                                    LineInfo info = new LineInfo();
                                    info.mLine = newLine1;
                                    info.mPresses = newPresses1;
                                    info.mLineBBox = getLineBBox(newLine1, annotNode.mAnnot.getBorderInfo().getWidth());
                                    annotNode.mNewLines.add(lineIndex, info);
                                } else {
                                    // if line1 have no point, add index -- for continue erase line2
                                    lineIndex--;
                                }
                            }
                            annotNode.mModifyFlag = true;
                            invalidateNewLine(pdfViewCtrl, pageIndex, annotNode.mAnnot, tv_rectF);
                            break;
                        }
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void invalidateNewLine(PDFViewCtrl pdfViewCtrl, int pageIndex, Annot annot, RectF rect) {
        try {
            rect.inset(annot.getBorderInfo().getWidth(), annot.getBorderInfo().getWidth());
            float tmp = rect.top;
            rect.top = rect.bottom;
            rect.bottom = tmp;

            pdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, pageIndex);
            pdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, rect, pageIndex);
            pdfViewCtrl.invalidate(rectFToRect(rect));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private double getDistanceOfPointToLine(float x, float y, float x1, float y1, float x2, float y2) {
        float k = 0;
        float b = 0;

        if (x1 == x2) {
            return Math.abs(x - x1);
        } else if (y1 == y2) {
            return Math.abs(y - y1);
        } else {
            k = (y2 - y1) / (x2 - x1);
            b = y2 - k * x2;
            return Math.abs(k * x - y + b) / Math.sqrt(k * k + 1);
        }
    }

    private boolean isIntersectPointInLine(float x, float y, float x1, float y1, float x2, float y2) {
        boolean result = false;
        double cross = (x2 - x1) * (x - x1) + (y2 - y1) * (y - y1);
        double d = (x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1);
        double r = cross / d;
        if (r > 0 && r < 1) {
            result = true;
        }
        return result;
    }


    private int getIntersection(PointF a, PointF b, PointF c, PointF d, PointF intersection) {
        if (Math.abs(b.y - a.y) + Math.abs(b.x - a.x) + Math.abs(d.y - c.y) + Math.abs(d.x - c.x) == 0) {
            if ((c.x - a.x) + (c.y - a.y) == 0) {
                //System.out.println("A B C D are the same point");
            } else {
                //System.out.println("A B are the same point.C D are the same point, and different from AC.");
            }
            return 0;
        }

        if (Math.abs(b.y - a.y) + Math.abs(b.x - a.x) == 0) {
            if ((a.x - d.x) * (c.y - d.y) - (a.y - d.y) * (c.x - d.x) == 0) {
                //System.out.println("A、B are the same point,and in line CD.");
            } else {
                //System.out.println("A、B are the same point,and not in line CD.");
            }
            return 0;
        }

        if (Math.abs(d.y - c.y) + Math.abs(d.x - c.x) == 0) {
            if ((d.x - b.x) * (a.y - b.y) - (d.y - b.y) * (a.x - b.x) == 0) {
                //System.out.println("C、D are the same point,and in line AB.");
            } else {
                //System.out.println("C、D are the same point,and not in line AB.");
            }
            return 0;
        }

        if ((b.y - a.y) * (c.x - d.x) - (b.x - a.x) * (c.y - d.y) == 0) {
            //System.out.println("Parallel lines, no intersection!");
            return 0;
        }

        intersection.x = ((b.x - a.x) * (c.x - d.x) * (c.y - a.y) - c.x
                * (b.x - a.x) * (c.y - d.y) + a.x * (b.y - a.y) * (c.x - d.x))
                / ((b.y - a.y) * (c.x - d.x) - (b.x - a.x) * (c.y - d.y));
        intersection.y = ((b.y - a.y) * (c.y - d.y) * (c.x - a.x) - c.y
                * (b.y - a.y) * (c.x - d.x) + a.y * (b.x - a.x) * (c.y - d.y))
                / ((b.x - a.x) * (c.y - d.y) - (b.y - a.y) * (c.x - d.x));

        if ((intersection.x - a.x) * (intersection.x - b.x) <= 0
                && (intersection.x - c.x) * (intersection.x - d.x) <= 0
                && (intersection.y - a.y) * (intersection.y - b.y) <= 0
                && (intersection.y - c.y) * (intersection.y - d.y) <= 0) {
            //System.out.println("Lines intersect at the intersection point(" + intersection.x + "," + intersection.y + ")!");
            return 1;
        } else {
            //System.out.println("Lines intersect at the virtual intersection point(" + intersection.x + "," + intersection.y + ")!");
            return -1;
        }
    }

    private RectF getEraserBBox(PointF downPoint, PointF point) {
        RectF eraserBBox = new RectF();
        eraserBBox.left = Math.min(downPoint.x, point.x);
        eraserBBox.top = Math.min(downPoint.y, point.y);
        eraserBBox.right = Math.max(downPoint.x, point.x);
        eraserBBox.bottom = Math.max(downPoint.y, point.y);
        eraserBBox.inset(-mRadius - 2, -mRadius - 2);
        return eraserBBox;
    }

    private RectF getLineBBox(ArrayList<PointF> line, float thickness) {
        if (line.size() == 0) {
            return new RectF(0, 0, 0, 0);
        }

        RectF lineBBox = new RectF(line.get(0).x, line.get(0).y, line.get(0).x, line.get(0).y);
        for (int i = 0; i < line.size(); i++) {
            lineBBox.left = Math.min(lineBBox.left, line.get(i).x);
            lineBBox.top = Math.max(lineBBox.top, line.get(i).y);
            lineBBox.right = Math.max(lineBBox.right, line.get(i).x);
            lineBBox.bottom = Math.min(lineBBox.bottom, line.get(i).y);
        }
        lineBBox.inset(-thickness / 2, -thickness / 2);
        return lineBBox;
    }


    private float getDistanceOfTwoPoints(float x1, float y1, float x2, float y2) {
        return (float) (Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)));
    }

    private ArrayList<Path> getNewPaths(FSPageDisplayMatrix matrix, PDFViewCtrl pdfViewCtrl, int pageIndex, AnnotInfo info) {
        ArrayList<LineInfo> pdfLines = info.mNewLines;
        ArrayList<Path> paths = new ArrayList<Path>();
        PointF pointF = new PointF();

        float cx = 0, cy = 0, ex, ey;
        for (int i = 0; i < pdfLines.size(); i++) {
            ArrayList<PointF> pdfLine = pdfLines.get(i).mLine;
            int ptCount = pdfLine.size();
            if (ptCount == 0) {
                continue;
            } else if (ptCount == 1) {
                Path path = new Path();
                pointF.set(pdfLine.get(0).x, pdfLine.get(0).y);
                if (matrix == null) {
                    pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                } else {
                    pointF = matrix.convertPdfPtToPageViewPt(pointF);
                }
                path.moveTo(pointF.x, pointF.y);
                path.lineTo(pointF.x + 0.1f, pointF.y + 0.1f);
                paths.add(path);
            } else {
                Path path = new Path();
                for (int j = 0; j < ptCount; j++) {
                    pointF.set(pdfLine.get(j).x, pdfLine.get(j).y);
                    if (matrix == null) {
                        pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                    } else {
                        pointF = matrix.convertPdfPtToPageViewPt(pointF);
                    }
                    if (j == 0) {
                        path.moveTo(pointF.x, pointF.y);
                        cx = pointF.x;
                        cy = pointF.y;
                    } else {
                        ex = (cx + pointF.x) / 2;
                        ey = (cy + pointF.y) / 2;
                        path.quadTo(cx, cy, ex, ey);
                        cx = pointF.x;
                        cy = pointF.y;
                        if (j == pdfLine.size() - 1) {
                            ex = pointF.x;
                            ey = pointF.y;
                            path.lineTo(ex, ey);
                        }
                    }
                }
                paths.add(path);
            }
        }
        return paths;
    }

    private void invalidateJniAnnots(AnnotInfo annotInfo, int flag, final Event.Callback result) {
        if (flag == 0) {
            mUiExtensionsManager.getDocumentManager().onAnnotStartEraser(annotInfo.mAnnot);
        } else if (flag == 1) {
            mUiExtensionsManager.getDocumentManager().onAnnotEndEraser(annotInfo.mAnnot);
        }

        if (result != null) {
            result.result(null, true);
        }
    }

    private RectF getNewBBox(AnnotInfo annotInfo) {
        Ink annot = annotInfo.mAnnot;
        ArrayList<ArrayList<PointF>> pdfLines = getNewPdfLines(annotInfo);
        RectF newBBox = null;
        for (int i = 0; i < pdfLines.size(); i++) {
            for (int j = 0; j < pdfLines.get(i).size(); j++) {
                PointF pdfPt = pdfLines.get(i).get(j);
                if (newBBox == null) {
                    newBBox = new RectF(pdfPt.x, pdfPt.y, pdfPt.x, pdfPt.y);
                } else {
                    newBBox.left = Math.min(newBBox.left, pdfPt.x);
                    newBBox.bottom = Math.min(newBBox.bottom, pdfPt.y);
                    newBBox.right = Math.max(newBBox.right, pdfPt.x);
                    newBBox.top = Math.max(newBBox.top, pdfPt.y);
                }
            }
        }
        try {
            newBBox.inset(-annot.getBorderInfo().getWidth() * 0.5f - mCtlPtRadius, -annot.getBorderInfo().getWidth() * 0.5f - mCtlPtRadius);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return newBBox;
    }

    private ArrayList<ArrayList<PointF>> getNewPdfLines(AnnotInfo annotInfo) {
        ArrayList<ArrayList<PointF>> pdfLines = new ArrayList<ArrayList<PointF>>();
        for (int i = 0; i < annotInfo.mNewLines.size(); i++) {
            ArrayList<PointF> oldLine = annotInfo.mNewLines.get(i).mLine;
            ArrayList<PointF> newLine = new ArrayList<PointF>();
            for (int j = 0; j < oldLine.size(); j++) {
                newLine.add(oldLine.get(j));
            }
            pdfLines.add(newLine);
        }
        return pdfLines;
    }


    private void addModifyAnnotEvent(final AnnotInfo annotInfo, final EraserUndoItem undoItems, final ArrayList<EditAnnotEvent> events) {
        try {
            final Ink annot = annotInfo.mAnnot;
            InkAnnotHandler annotHandler = (InkAnnotHandler) mUiExtensionsManager.getAnnotHandlerByType(Annot.e_Ink);
            final InkModifyUndoItem undoItem = new InkModifyUndoItem(annotHandler, mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.setOldValue(annot);
            RectF newBBox = getNewBBox(annotInfo);
            undoItem.mBBox = new RectF(newBBox);
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            ArrayList<ArrayList<PointF>> pdfLines = getNewPdfLines(annotInfo);
            undoItem.mPath = new com.foxit.sdk.common.Path();
            for (int i = 0; i < pdfLines.size(); i++) {
                ArrayList<PointF> line =pdfLines.get(i);
                int size = line.size();
                if (size == 1) {
                    undoItem.mPath.moveTo(AppUtil.toFxPointF(line.get(0)));
                    undoItem.mPath.lineTo(new com.foxit.sdk.common.fxcrt.PointF(line.get(0).x + 0.1f, line.get(0).y + 0.1f));
                } else {
                    for (int j = 0; j < size; j++) {
                        if (j == 0) {
                            undoItem.mPath.moveTo(AppUtil.toFxPointF(line.get(j)));
                        } else {
                            undoItem.mPath.lineTo(AppUtil.toFxPointF(line.get(j)));
                        }
                    }
                }
            }

            undoItem.setOldValue(annot);
            undoItem.mOldPath = annot.getInkList();
            undoItem.isFromEraser = true;
            annotHandler.modifyAnnot(annot, undoItem, false, false, false, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    undoItems.addUndoItem(undoItem);
                    events.add((EditAnnotEvent) event);
                }
            });
        } catch (PDFException e) {
            e.printStackTrace();
        }

    }

    private void addDeleteAnnotEvent(final Annot annot, final EraserUndoItem undoItems, final ArrayList<EditAnnotEvent> events) {
        // step 1: set current annot to null
        if (annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
            mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null, false);
        }

        try {
            final PDFPage page = annot.getPage();
            mUiExtensionsManager.getDocumentManager().onAnnotWillDelete(page, annot);
            InkAnnotHandler annotHandler = (InkAnnotHandler) mUiExtensionsManager.getAnnotHandlerByType(Annot.e_Ink);
            final InkDeleteUndoItem undoItem = new InkDeleteUndoItem(annotHandler, mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.isFromEraser = true;
            try {
                undoItem.mPath = ((Ink) annot).getInkList();
                undoItem.mGroupNMList = GroupManager.getInstance().getGroupUniqueIDs(mPdfViewCtrl, annot);
                PDFObject pdfObject = annot.getDict().getElement("FxPList");
                if (pdfObject != null) {
                    PDFArray array = pdfObject.getArray();
                    if (array != null) {
                        ArrayList<Float> pressures = new ArrayList<>();
                        int size = array.getElementCount();
                        for (int i = 0; i < size; i++) {
                            pressures.add(array.getElement(i).getFloat());
                        }
                        if (pressures.size() > 0) {
                            undoItem.mPencilType = InkConstants.PENCIL_WITH_BRUSH;
                            undoItem.mPSIPressures = pressures;
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }

            annotHandler.removeAnnot(annot, undoItem, false, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    undoItems.addUndoItem(undoItem);
                    events.add((EditAnnotEvent) event);
                }
            });
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void setPaint(FSPageDisplayMatrix matrix, int pageIndex, AnnotInfo annotInfo) {
        try {
            if (annotInfo.mAnnot == null) return;

            if (annotInfo.mBorderWidth == -1)
                annotInfo.mBorderWidth = annotInfo.mAnnot.getBorderInfo().getWidth();
            float thickness;
            if (matrix == null) {
                thickness = thicknessOnPageView(pageIndex, annotInfo.mBorderWidth);
            } else {
                thickness = matrix.convertPdfToViewLength(annotInfo.mBorderWidth);
            }
            mPaint.setColor(annotInfo.mBorderColor);
            mPaint.setStrokeWidth(thickness);
            int opacity = (int) (annotInfo.mOpacity * 255f + 0.5f);
            mPaint.setAlpha(opacity);
            mPaint.setStyle(Style.STROKE);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private float thicknessOnPageView(int pageIndex, float thickness) {
        RectF rectF = new RectF(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        return Math.abs(rectF.width());
    }

    public void setRadius(float radius) {
        mRadius = AppDisplay.dp2px(radius);
    }

    @Override
    public void setThickness(float thickness) {
        super.setThickness(thickness);
        setRadius(thickness);
        if (mCurToolItem == null) return;
        mCurToolItem.property.lineWidth = thickness;
    }

    @Override
    public void onValueChanged(long property, int value) {
        if (property == PropertyBar.PROPERTY_ERASERSHAPE) {
            resetEraser(mCapturedPage, mInvalidateRect);
            mShapeMode = value;
            if (mCurToolItem != null)
                mCurToolItem.property.eraserShape = mShapeMode;
        }
    }

    @Override
    protected void setPaintProperty(PDFViewCtrl pdfViewCtrl, int pageIndex, Paint paint) {

    }

    @Override
    public long getSupportedProperties() {
        return PropertyBar.PROPERTY_ERASERSHAPE;
    }

    @Override
    protected void setPropertyBarProperties(PropertyBar propertyBar) {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        propertyBar.setColors(colors);
        propertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        propertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH_2, getThickness());
        if (AppDisplay.isPad()) {
            propertyBar.setArrowVisible(true);
        } else {
            propertyBar.setArrowVisible(false);
        }
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_ERASER;
    }

    private float mLastLineWidth;
    private int mLastShape;
    private IToolSupply mToolSupply;

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new EraserToolSupply(mContext);
        return mToolSupply;
    }

    private class EraserToolSupply extends ToolSupplyImpl {

        public EraserToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.drawing_tool_eraser;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Eraser;
            property.lineWidth = mThickness;
            property.color = AppResource.getColor(mContext, R.color.p3);
            property.eraserShape = mShapeMode;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_ERASER;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_DRAWING_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Eraser);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mLastLineWidth = mThickness;
                mThickness = property.lineWidth;

                mLastShape = mShapeMode;
                mShapeMode = property.eraserShape;

                setRadius(mThickness);
                mUiExtensionsManager.setCurrentToolHandler(EraserToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == EraserToolHandler.this) {
                    mThickness = mLastLineWidth;
                    mShapeMode = mLastShape;
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCurToolItem = itemBean;
            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mLastLineWidth = mThickness;
            mThickness = property.lineWidth;
            mLastShape = mShapeMode;
            mShapeMode = property.eraserShape;
            EraserToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mThickness = mLastLineWidth;
                    mShapeMode = mLastShape;
                    mCurToolItem = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void resetEraser(int pageIndex, RectF refreshRectF) {
        for (int i = mRootList.size() - 1; i >= 0; i--) {
            AnnotInfo annotInfo = mRootList.get(i);
            invalidateJniAnnots(annotInfo, 1, null);
        }

        mCapturedPage = -1;
        mRootList.clear();
        mPaths.clear();
        mTempRemovingAnnots.clear();
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(refreshRectF));
        }
    }

    void resetEraser(final int oldPageIndex, int curPageIndex, RectF refreshRectF) {
        for (int i = mRootList.size() - 1; i >= 0; i--) {
            AnnotInfo annotInfo = mRootList.get(i);
            invalidateJniAnnots(annotInfo, 1, null);
        }

        Task refreshFinishTask = null;
        if (oldPageIndex != curPageIndex) {
            mTempCapturedPage = oldPageIndex;
            mTempRootList.clear();
            mTempPaths.clear();
            mTempRootList.addAll(mRootList);
            mTempPaths.addAll(mPaths);
            final RectF invalideRect = new RectF(refreshRectF);
            refreshFinishTask = new Task(null) {
                @Override
                protected void execute() {
                    AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                        @Override
                        public void run() {
                            mTempRootList.clear();
                            mTempPaths.clear();
                            mTempCapturedPage = -1;
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalideRect, invalideRect, oldPageIndex);
                            mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalideRect));
                        }
                    });
                }
            };
        }

        mCapturedPage = -1;
        mRootList.clear();
        mPaths.clear();
        mTempRemovingAnnots.clear();
        if (mPdfViewCtrl.isPageVisible(oldPageIndex)) {
            mPdfViewCtrl.refresh(oldPageIndex, AppDmUtil.rectFToRect(refreshRectF));
        }
        if (refreshFinishTask != null)
            mPdfViewCtrl.addTask(refreshFinishTask);
    }


    private FSPageDisplayMatrix getPageMatrix(int pageIndex, PDFViewCtrl pdfViewCtrl) {
        if (mIsScaling) {
            return null;
        }
        FSPageDisplayMatrix disMatrix = this.mPageMatrixArray.get(pageIndex);
        if (disMatrix == null) {
            Matrix matrix = pdfViewCtrl.getDisplayMatrix(pageIndex);
            if (matrix != null) {
                disMatrix = new FSPageDisplayMatrix(matrix);
                this.mPageMatrixArray.put(pageIndex, disMatrix);
            }
        }
        return disMatrix;
    }

    protected void onScaleBegin(ScaleGestureDetector detector) {
        this.mIsScaling = true;
        this.mPageMatrixArray.clear();
    }

    protected void onScaleEnd(ScaleGestureDetector detector) {
        this.mIsScaling = false;
    }
}
