/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;

public class EraserUndoItem extends AnnotUndoItem {
    private final ArrayList<InkUndoItem> undoItems;
    private final ArrayList<EditAnnotEvent> mEvents = new ArrayList<>();
    private final EraserToolHandler mToolHandler;

    public EraserUndoItem(PDFViewCtrl pdfViewCtrl, EraserToolHandler toolHandler) {
        mPdfViewCtrl = pdfViewCtrl;
        mToolHandler = toolHandler;
        undoItems = new ArrayList<>();
    }

    public void addUndoItem(InkUndoItem undoItem) {
        undoItems.add(undoItem);
    }

    public ArrayList<InkUndoItem> getUndoItems() {
        return undoItems;
    }

    @Override
    public boolean undo() {
        mEvents.clear();
        int size = undoItems.size();
        for (int i = size - 1; i >= 0; i--) {
            InkUndoItem undoItem = undoItems.get(i);
            if (undoItem instanceof InkModifyUndoItem) {
                InkModifyUndoItem item = (InkModifyUndoItem) undoItem;
                item.undo(callback);
            } else if (undoItem instanceof InkDeleteUndoItem) {
                InkDeleteUndoItem item = (InkDeleteUndoItem) undoItem;
                item.undo(callback);
            }
        }
        final RectF rectF = calculateInvalidateRect(mEvents);
        doTask(true, mPageIndex, rectF, mEvents, null);
        return true;
    }

    @Override
    public boolean redo() {
        mEvents.clear();
        for (InkUndoItem undoItem : undoItems) {
            if (undoItem instanceof InkModifyUndoItem) {
                InkModifyUndoItem item = (InkModifyUndoItem) undoItem;
                item.redo(callback);
            } else if (undoItem instanceof InkDeleteUndoItem) {
                InkDeleteUndoItem item = (InkDeleteUndoItem) undoItem;
                item.redo(callback);
            }
        }

        final RectF rectF = calculateInvalidateRect(mEvents);
        doTask(true, mPageIndex, rectF, mEvents, null);
        return true;
    }

    Event.Callback callback = new Event.Callback() {
        @Override
        public void result(Event event, boolean success) {
            if (success && event instanceof EditAnnotEvent) {
                mEvents.add((EditAnnotEvent) event);
            }
        }
    };

    private RectF calculateInvalidateRect(ArrayList<EditAnnotEvent> eventList) {
        int count = 0;
        RectF rect = new RectF();
        try {
            for (EditAnnotEvent event : eventList) {
                RectF invalidateRect;
                if (event.mUndoItem instanceof InkModifyUndoItem) {
                    RectF oldRectF = event.mUndoItem.mOldBBox;
                    RectF nowRectF = event.mUndoItem.mBBox;
                    if (oldRectF != null && nowRectF != null) {
                        AppUtil.unionFxRectF(nowRectF, oldRectF);
                        invalidateRect = new RectF(nowRectF);
                    } else {
                        invalidateRect = AppUtil.toRectF(event.mAnnot.getRect());
                    }
                } else {
                    invalidateRect = AppUtil.toRectF(event.mAnnot.getRect());
                }
                mPdfViewCtrl.convertPdfRectToPageViewRect(invalidateRect, invalidateRect, event.mUndoItem.mPageIndex);

                if (count == 0) {
                    rect.set(invalidateRect);
                } else {
                    rect.union(invalidateRect);
                }
                count++;
            }
            return rect;
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return null;
    }

    void doTask(final int pageIndex, final RectF refreshRectF,
                final ArrayList<EditAnnotEvent> events, final Event.Callback callback) {
        doTask(false, pageIndex, refreshRectF, events, callback);
    }

    private void doTask(final boolean fromUndo,
                        final int pageIndex,
                        final RectF refreshRectF,
                        final ArrayList<EditAnnotEvent> events,
                        final Event.Callback callback) {
        EraserEvent eraserEvent = new EraserEvent(events);
        EditAnnotTask task = new EditAnnotTask(eraserEvent, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success)
                    doEvents(events);

                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                if (fromUndo && uiExtensionsManager.getCurrentToolHandler() == mToolHandler) {
                    mToolHandler.resetEraser(pageIndex, refreshRectF);
                } else {
                    if (callback != null) {
                        callback.result(event, success);
                    } else {
                        if (mPdfViewCtrl.isPageVisible(pageIndex))
                            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(refreshRectF));
                    }
                }

            }
        });
        mPdfViewCtrl.addTask(task);
    }

    private void doEvents(ArrayList<EditAnnotEvent> events) {
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        for (int i = 0; i < events.size(); i++) {
            EditAnnotEvent event = events.get(i);
            try {
                PDFPage pdfPage = mPdfViewCtrl.getDoc().getPage(event.mUndoItem.mPageIndex);
                switch (event.mType) {
                    case EditAnnotEvent.EVENTTYPE_ADD:
                        documentManager.onAnnotAdded(pdfPage, event.mAnnot);
                        break;
                    case EditAnnotEvent.EVENTTYPE_MODIFY:
                        documentManager.onAnnotModified(pdfPage, event.mAnnot);
                        break;
                    case EditAnnotEvent.EVENTTYPE_DELETE:
                        documentManager.onAnnotDeleted(pdfPage, event.mAnnot);

                        if (event.mUndoItem instanceof InkUndoItem) {
                            InkUndoItem undoItem = (InkUndoItem) event.mUndoItem;
                            if (undoItem.mGroupNMList.size() >= 2) {
                                ArrayList<String> newGroupList = new ArrayList<>(undoItem.mGroupNMList);
                                newGroupList.remove(undoItem.mNM);
                                if (newGroupList.size() >= 2)
                                    GroupManager.getInstance().setAnnotGroup(mPdfViewCtrl, pdfPage, newGroupList);
                                else
                                    GroupManager.getInstance().unGroup(pdfPage, newGroupList.get(0));
                            }
                        }
                        break;
                    default:
                        break;
                }
            } catch (PDFException ignored) {
            }
        }
    }

}
