/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.PathArray;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.ResultInfo;

import java.util.ArrayList;


public class InkAnnotUtil {

    public static long getSupportedProperties(boolean createAnnot, boolean highlighter) {
        if (createAnnot) {
            if (highlighter)
                return PropertyBar.PROPERTY_COLOR
                        | PropertyBar.PROPERTY_OPACITY
                        | PropertyBar.PROPERTY_LINEWIDTH;
            else
                return PropertyBar.PROPERTY_COLOR
                        | PropertyBar.PROPERTY_OPACITY
                        | PropertyBar.PROPERTY_LINEWIDTH
                        | PropertyBar.PROPERTY_PENCIL_TYPE;
        } else
            return PropertyBar.PROPERTY_COLOR
                    | PropertyBar.PROPERTY_OPACITY
                    | PropertyBar.PROPERTY_LINEWIDTH;
    }

    protected static float widthOnPageView(PDFViewCtrl pdfViewCtrl, int pageIndex, float width) {
        RectF rectF = new RectF(0, 0, width, width);
        pdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        return Math.abs(rectF.width());
    }

    protected ArrayList<ArrayList<PointF>> docLinesFromPageView(PDFViewCtrl pdfViewCtrl, int pageIndex,
                                                                ArrayList<ArrayList<PointF>> lines, RectF bbox) {
        RectF bboxF = null;
        ArrayList<ArrayList<PointF>> docLines = new ArrayList<ArrayList<PointF>>();
        try {
            for (int i = 0; i < lines.size(); i++) {
                ArrayList<PointF> newLine = new ArrayList<PointF>();
                for (int j = 0; j < lines.get(i).size(); j++) {
                    PointF curPoint = new PointF();
                    curPoint.set(lines.get(i).get(j));
                    if (bboxF == null) {
                        bboxF = new RectF(curPoint.x, curPoint.y, curPoint.x, curPoint.y);
                    } else {
                        bboxF.union(curPoint.x, curPoint.y);
                    }
                    pdfViewCtrl.convertPageViewPtToPdfPt(curPoint, curPoint, pageIndex);
                    newLine.add(curPoint);
                }
                docLines.add(newLine);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (bboxF != null) {
            pdfViewCtrl.convertPageViewRectToPdfRect(bboxF, bboxF, pageIndex);
            bbox.set(bboxF.left, bboxF.top, bboxF.right, bboxF.bottom);
        }
        return docLines;
    }

    protected ArrayList<ArrayList<PSIData>> clonePSIDataLists(ArrayList<ArrayList<PSIData>> srcDatas) {
        if (srcDatas == null || srcDatas.size() == 0) return null;

        ArrayList<ArrayList<PSIData>> dstDatas = new ArrayList<>();
        for (ArrayList<PSIData> psiDataArrayList : srcDatas) {
            ArrayList<PSIData> dstDataArrayList = new ArrayList<>();
            for (PSIData srcData : psiDataArrayList) {
                dstDataArrayList.add(new PSIData(srcData));
            }
            dstDatas.add(dstDataArrayList);
        }
        return dstDatas;
    }

    protected static ArrayList<Path> generatePathData(PDFViewCtrl pdfViewCtrl, int pageIndex, Ink annot) {
        int type = getInkType(annot);
        if (type == InkConstants.PENCIL_WITH_BRUSH)
            return generatePsiPaths(pdfViewCtrl, pageIndex, annot);
        else
            return generateInkPaths(pdfViewCtrl, pageIndex, annot);
    }

    protected static ArrayList<Path> generateInkPaths(PDFViewCtrl pdfViewCtrl, int pageIndex, Ink annot) {
        try {
            com.foxit.sdk.common.Path pdfPath = annot.getInkList();
            if (pdfPath == null) return null;
            ArrayList<Path> paths = new ArrayList<Path>();
            PointF pointF = new PointF();

            int ptCount = pdfPath.getPointCount();
            if (ptCount == 1) {
                Path path = new Path();
                pointF.set(pdfPath.getPoint(0).getX(), pdfPath.getPoint(0).getY());
                pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                path.moveTo(pointF.x, pointF.y);
                path.lineTo(pointF.x + 0.1f, pointF.y + 0.1f);
                paths.add(path);
                return paths;
            }
            float cx = 0, cy = 0, ex, ey;
            Path path = null;
            for (int i = 0; i < ptCount; i++) {
                pointF.set(pdfPath.getPoint(i).getX(), pdfPath.getPoint(i).getY());
                pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                if (pdfPath.getPointType(i) == com.foxit.sdk.common.Path.e_TypeMoveTo) {
                    path = new Path();
                    path.moveTo(pointF.x, pointF.y);
                    cx = pointF.x;
                    cy = pointF.y;
                } else {
                    ex = (cx + pointF.x) / 2;
                    ey = (cy + pointF.y) / 2;
                    path.quadTo(cx, cy, ex, ey);
                    cx = pointF.x;
                    cy = pointF.y;
                }

                if (i == ptCount - 1 || ((i + 1) < ptCount && pdfPath.getPointType(i + 1) == com.foxit.sdk.common.Path.e_TypeMoveTo)) {
                    ex = pointF.x;
                    ey = pointF.y;
                    path.lineTo(ex, ey);

                    paths.add(path);
                }
            }
            return paths;
        } catch (PDFException e) {
            return null;
        }
    }

    protected static ArrayList<Path> generatePsiPaths(PDFViewCtrl pdfViewCtrl, int pageIndex, Ink annot) {
        ArrayList<Path> paths = new ArrayList<Path>();
        try {
            PathArray pathPts = annot.getEIAInkList();
            long size = pathPts.getSize();
            if (size == 0) return paths;

            for (int i = 0; i < size; i++) {
                com.foxit.sdk.common.Path psiPath = pathPts.getAt(i);
                Path path = new Path();
                PointF pointF = new PointF();
                PointF point0 = new PointF();
                PointF point1 = new PointF();
                int flag = 0;
                int bezier = 0;
                int count = psiPath.getPointCount();
                for (int j = 0; j < count; j++) {
                    pointF.set(AppUtil.toPointF(psiPath.getPoint(j)));
                    pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                    flag = psiPath.getPointType(j);
                    if (flag == com.foxit.sdk.common.Path.e_TypeMoveTo) {
                        path.moveTo(pointF.x, pointF.y);
                    } else if ((flag & com.foxit.sdk.common.Path.e_TypeLineTo) != 0) {
                        path.lineTo(pointF.x, pointF.y);
                    } else if ((flag & com.foxit.sdk.common.Path.e_TypeBezierTo) != 0) {
                        if (bezier == 0) {
                            point0.set(pointF);
                        } else if (bezier == 1) {
                            point1.set(pointF);
                        }
                        bezier++;
                        if (bezier == 3) {
                            path.cubicTo(point0.x, point0.y, point1.x, point1.y, pointF.x, pointF.y);
                            bezier = 0;
                        }
                    }
                    if ((flag & com.foxit.sdk.common.Path.e_TypeLineToCloseFigure) != 0) {
                        path.close();
                    }
                }
                paths.add(path);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return paths;
    }

    public static void correctPvPoint(PDFViewCtrl pdfViewCtrl, int pageIndex, PointF pt) {
        pt.x = Math.max(0, pt.x);
        pt.y = Math.max(0, pt.y);
        pt.x = Math.min(pdfViewCtrl.getPageViewWidth(pageIndex), pt.x);
        pt.y = Math.min(pdfViewCtrl.getPageViewHeight(pageIndex), pt.y);
    }

    public static void correctPvPoint(PointF pt, int pvWidth, int pvHeight) {
        pt.x = Math.max(0, pt.x);
        pt.y = Math.max(0, pt.y);
        pt.x = Math.min(pvWidth, pt.x);
        pt.y = Math.min(pvHeight, pt.y);
    }

    public static ArrayList<ArrayList<PointF>> cloneInkList(ArrayList<ArrayList<PointF>> lines) {
        if (lines == null) return null;
        ArrayList<ArrayList<PointF>> newLines = new ArrayList<ArrayList<PointF>>();
        for (int i = 0; i < lines.size(); i++) {
            ArrayList<PointF> line = lines.get(i);
            ArrayList<PointF> newLine = new ArrayList<PointF>();
            for (int j = 0; j < line.size(); j++) {
                newLine.add(new PointF(line.get(j).x, line.get(j).y));
            }
            newLines.add(newLine);
        }
        return newLines;
    }

    public static ArrayList<ArrayList<PointF>> generateInkList(com.foxit.sdk.common.Path pdfPath) {
        if (pdfPath == null) return null;
        ArrayList<ArrayList<PointF>> newLines = new ArrayList<ArrayList<PointF>>();
        ArrayList<PointF> newLine = null;

        try {
            int ptCount = pdfPath.getPointCount();
            for (int i = 0; i < ptCount; i++) {
                if (pdfPath.getPointType(i) == com.foxit.sdk.common.Path.e_TypeMoveTo) {
                    newLine = new ArrayList<PointF>();
                }
                if (newLine != null)
                    newLine.add(AppUtil.toPointF(pdfPath.getPoint(i)));

                if (i == ptCount - 1 || ((i + 1) < ptCount && pdfPath.getPointType(i + 1) == com.foxit.sdk.common.Path.e_TypeMoveTo)) {
                    newLines.add(newLine);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return newLines;
    }

    public static com.foxit.sdk.common.Path linesToPath(ArrayList<ArrayList<PointF>> lines) {
        if (lines == null || lines.size() == 0) return null;

        try {
            com.foxit.sdk.common.Path path = new com.foxit.sdk.common.Path();
            for (int li = 0; li < lines.size(); li++) { //li: line index
                ArrayList<PointF> line = lines.get(li);
                int size = line.size();
                if (size == 1) {
                    path.moveTo(AppUtil.toFxPointF(line.get(0)));
                    path.lineTo(new com.foxit.sdk.common.fxcrt.PointF(line.get(0).x + 0.1f, line.get(0).y + 0.1f));
                } else {
                    for (int pi = 0; pi < size; pi++) {//pi: point index
                        if (pi == 0) {
                            path.moveTo(AppUtil.toFxPointF(line.get(pi)));
                        } else {
                            path.lineTo(AppUtil.toFxPointF(line.get(pi)));
                        }
                    }
                }
            }
            return path;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static com.foxit.sdk.common.Path pointsToPath(ArrayList<PointF> points) {
        if (points == null || points.isEmpty()) return null;

        try {
            com.foxit.sdk.common.Path path = new com.foxit.sdk.common.Path();
            int size = points.size();
            if (size == 1) {
                path.moveTo(AppUtil.toFxPointF(points.get(0)));
                path.lineTo(new com.foxit.sdk.common.fxcrt.PointF(points.get(0).x + 0.1f, points.get(0).y + 0.1f));
            } else {
                for (int i = 0; i < size; i++) {//pi: point index
                    if (i == 0) {
                        path.moveTo(AppUtil.toFxPointF(points.get(i)));
                    } else {
                        path.lineTo(AppUtil.toFxPointF(points.get(i)));
                    }
                }
            }
            return path;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static int getInkType(Ink annot) {
        if (annot == null || annot.isEmpty()) return -1;
        try {
            PDFDictionary dict = annot.getDict();
            if (dict != null) {
                PDFObject object = dict.getElement(InkConstants.PSINK_DICT_KEY);
                if (object != null && object.getWideString().equals(InkConstants.PSINK_DICT_VALUE))
                    return InkConstants.PENCIL_WITH_BRUSH;
                else
                    return InkConstants.PENCIL_WITH_PEN;
            } else {
                return InkConstants.PENCIL_WITH_PEN;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return -1;
    }

    public static PointF[] calcBesselControlPoints(PointF[] pLinePoints, float k) {
        if (pLinePoints[0].equals(pLinePoints[1]) && pLinePoints[1].equals(pLinePoints[2])) {
            return null;
        }

        float xc1 = (pLinePoints[0].x + pLinePoints[1].x) / 2.0f;
        float yc1 = (pLinePoints[0].y + pLinePoints[1].y) / 2.0f;
        float xc2 = (pLinePoints[1].x + pLinePoints[2].x) / 2.0f;
        float yc2 = (pLinePoints[1].y + pLinePoints[2].y) / 2.0f;
        float xc3 = (pLinePoints[2].x + pLinePoints[3].x) / 2.0f;
        float yc3 = (pLinePoints[2].y + pLinePoints[3].y) / 2.0f;

        float xDiff, yDiff;
        xDiff = pLinePoints[1].x - pLinePoints[0].x;
        yDiff = pLinePoints[1].y - pLinePoints[0].y;
        double len1 = Math.sqrt(xDiff * xDiff + yDiff * yDiff);
        xDiff = pLinePoints[2].x - pLinePoints[1].x;
        yDiff = pLinePoints[2].y - pLinePoints[1].y;
        double len2 = Math.sqrt(xDiff * xDiff + yDiff * yDiff);
        xDiff = pLinePoints[3].x - pLinePoints[2].x;
        yDiff = pLinePoints[3].y - pLinePoints[2].y;
        double len3 = Math.sqrt(xDiff * xDiff + yDiff * yDiff);

        float k1 = 0.5f;
        float k2 = 0.5f;
        if (len1 + len2 > 0.0001f && len2 + len3 > 0.0001f) {
            k1 = (float) (len1 / (len1 + len2));
            k2 = (float) (len2 / (len2 + len3));
        }

        float xm1 = xc1 + (xc2 - xc1) * k1;
        float ym1 = yc1 + (yc2 - yc1) * k1;
        float xm2 = xc2 + (xc3 - xc2) * k2;
        float ym2 = yc2 + (yc3 - yc2) * k2;

        PointF point1 = new PointF();
        PointF point2 = new PointF();

        point1.x = xm1 + (xc2 - xm1) * k + pLinePoints[1].x - xm1;
        point1.y = ym1 + (yc2 - ym1) * k + pLinePoints[1].y - ym1;
        point2.x = xm2 + (xc2 - xm2) * k + pLinePoints[2].x - xm2;
        point2.y = ym2 + (yc2 - ym2) * k + pLinePoints[2].y - ym2;
        return new PointF[]{point1, point2};
    }

    public static ResultInfo<ArrayList<PointF>, RectF, Void> convertPageLineToPdfLine(PDFViewCtrl pdfViewCtrl, int pageIndex, ArrayList<PointF> line) {
        RectF bboxF = null;
        ArrayList<PointF> newLine = new ArrayList<>();
        for (int i = 0; i < line.size(); i++) {
            PointF linePoint = line.get(i);

            if (bboxF == null) {
                bboxF = new RectF(linePoint.x, linePoint.y, linePoint.x, linePoint.y);
            } else {
                bboxF.union(linePoint.x, linePoint.y);
            }
            PointF pdfPt = new PointF();
            pdfViewCtrl.convertPageViewPtToPdfPt(linePoint, pdfPt, pageIndex);
            newLine.add(pdfPt);
        }

        RectF pdfRect = new RectF();
        if (bboxF != null) {
            pdfViewCtrl.convertPageViewRectToPdfRect(bboxF, pdfRect, pageIndex);
        }
        return new ResultInfo<>(true, newLine, pdfRect);
    }

    public static com.foxit.sdk.common.Path appendPointsToPath(com.foxit.sdk.common.Path path, ArrayList<PointF> points) {
        if (points == null || points.isEmpty()) {
            return null;
        }

        try {
            int size = points.size();
            if (size == 1) {
                path.moveTo(AppUtil.toFxPointF(points.get(0)));
                path.lineTo(AppUtil.toFxPointF(points.get(0).x + 0.1f, points.get(0).y + 0.1f));
            } else {
                for (int j = 0; j < size; j++) { //pi: point index
                    if (j == 0) {
                        path.moveTo(AppUtil.toFxPointF(points.get(j)));
                    } else {
                        path.lineTo(AppUtil.toFxPointF(points.get(j)));
                    }
                }
            }
            return path;
        } catch (PDFException ignored) {
        }
        return null;
    }
}
