/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.view.ScaleGestureDetector;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.AbstractUndo;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.utils.Event;


public class InkModule implements Module {
    protected InkAnnotHandler mAnnotHandler;
    protected InkToolHandler mToolHandler;
    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public InkModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_INK;
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new InkToolHandler(mContext, mPdfViewCtrl);
        mAnnotHandler = new InkAnnotHandler(mContext, mPdfViewCtrl, mUiExtensionsManager, mToolHandler);
        mToolHandler.mAnnotHandler = mAnnotHandler;
        mAnnotHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerToolHandler(mToolHandler);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigureChangeListener);
            uiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
            uiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
            uiExtensionsManager.getDocumentManager().registerUndoEventListener(this.undoEventListener);

            boolean isLoadPencil = uiExtensionsManager.getConfig().modules.annotations.isLoadPencil;
            if (isLoadPencil) {
                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_HOME, ToolConstants.Pencil, mToolHandler.getToolSupply());
                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_DRAW, ToolConstants.Pencil, mToolHandler.getToolSupply());
                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_DRAW, ToolConstants.Highlighter, mToolHandler.getToolSupply());
            }
        }

        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerScaleGestureEventListener(mScaleGestureEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterScaleGestureEventListener(mScaleGestureEventListener);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterToolHandler(mToolHandler);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigureChangeListener);
            uiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
            uiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
            uiExtensionsManager.getDocumentManager().unregisterUndoEventListener(this.undoEventListener);
        }
        return true;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }

            if (mAnnotHandler.getPropertyBar() != null && mAnnotHandler.getPropertyBar().isShowing()) {
                mAnnotHandler.getPropertyBar().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {


        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigureChangeListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            mToolHandler.onConfigurationChanged(newConfig);
        }
    };

    private final IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {

        @Override
        public void onAnnotsWillFlatten(PDFDoc doc) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.reset();
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mAnnotHandler.updateTheme();
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.updateTheme();
            }
        }
    };

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.reset();
            }
        }

        @Override
        public void onPanelClosed() {
        }
    };

    private final PDFViewCtrl.IScaleGestureEventListener mScaleGestureEventListener = new PDFViewCtrl.IScaleGestureEventListener() {
        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.reset();
            }
            return false;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
        }
    };

    private AbstractUndo.IUndoEventListener undoEventListener = new AbstractUndo.IUndoEventListener() {
        @Override
        public void itemWillAdd(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemAdded(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemWillRemoved(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemRemoved(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willUndo(DocumentManager dm, IUndoItem item) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.reset();
            }
        }

        @Override
        public void undoFinished(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willRedo(DocumentManager dm, IUndoItem item) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() == mToolHandler) {
                mToolHandler.reset();
            }
        }

        @Override
        public void redoFinished(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willClearUndo(DocumentManager dm) {
        }

        @Override
        public void clearUndoFinished(DocumentManager dm) {
        }
    };

}
