/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.graphics.PointF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Path;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;

public abstract class InkUndoItem extends AnnotUndoItem {
    ArrayList<String> mGroupNMList = new ArrayList<>();

    ArrayList<Float> mPSIPressures;
    ArrayList<Float> mOldPSIPressures;

    Path mPath;
    Path mOldPath;
    InkAnnotHandler mAnnotHandler;
    boolean isFromEraser;
    int mPencilType;

    public InkUndoItem(InkAnnotHandler annotHandler, PDFViewCtrl pdfViewCtrl) {
        mAnnotHandler = annotHandler;
        mPdfViewCtrl = pdfViewCtrl;
    }
}

class InkAddUndoItem extends InkUndoItem {

    public InkAddUndoItem(InkAnnotHandler annotHandler, PDFViewCtrl pdfViewCtrl) {
        super(annotHandler, pdfViewCtrl);
    }

    @Override
    public boolean undo() {
        InkDeleteUndoItem undoItem = new InkDeleteUndoItem(mAnnotHandler, mPdfViewCtrl);
        undoItem.mNM = mNM;
        undoItem.mPageIndex = mPageIndex;
        undoItem.mPencilType = mPencilType;
        undoItem.mPath = mPath;
        if (mPSIPressures != null)
            undoItem.mPSIPressures = new ArrayList<>(mPSIPressures);
        try {
            undoItem.mPath = mPath;
            PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Ink)) {
                return false;
            }
            ArrayList<String> groupNMList = GroupManager.getInstance().getGroupUniqueIDs(mPdfViewCtrl, annot);
            undoItem.mGroupNMList = groupNMList;
            mGroupNMList = groupNMList;

            mAnnotHandler.removeAnnot(annot, undoItem, false, null);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean redo() {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Ink, AppUtil.toFxRectF(mBBox)), Annot.e_Ink);

            mAnnotHandler.addAnnot(mPageIndex, (Ink) annot, this, false, true, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        if (mGroupNMList.size() > 0) {
                            GroupManager.getInstance().setAnnotGroup(mPdfViewCtrl, page, mGroupNMList);
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotGrouped(page, AppAnnotUtil.getAnnotsByNMs(page, mGroupNMList));
                        }
                    }
                }
            });
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }
}

class InkModifyUndoItem extends InkUndoItem {
    public InkModifyUndoItem(InkAnnotHandler annotHandler, PDFViewCtrl pdfViewCtrl) {
        super(annotHandler, pdfViewCtrl);
    }

    @Override
    public boolean undo(Event.Callback callback) {
        return modifyAnnot(true, callback);
    }

    @Override
    public boolean redo(Event.Callback callback) {
        return modifyAnnot(false, callback);
    }

    private boolean modifyAnnot(boolean userOldValue, Event.Callback callback) {
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Ink)) {
                return false;
            }

            mAnnotHandler.modifyAnnot((Ink) annot, this, userOldValue, false, true, callback);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean undo() {
        return undo(null);
    }

    @Override
    public boolean redo() {
        return redo(null);
    }
}

class InkDeleteUndoItem extends InkUndoItem {

    public InkDeleteUndoItem(InkAnnotHandler annotHandler, PDFViewCtrl pdfViewCtrl) {
        super(annotHandler, pdfViewCtrl);
    }

    @Override
    public boolean undo(final Event.Callback callback) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Ink annot = (Ink) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Ink, AppUtil.toFxRectF(mBBox)), Annot.e_Ink);
            InkAddUndoItem undoItem = new InkAddUndoItem(mAnnotHandler, mPdfViewCtrl);
            undoItem.mNM = mNM;
            undoItem.mPageIndex = mPageIndex;
            undoItem.mAuthor = mAuthor;
            undoItem.mFlags = mFlags;
            undoItem.mSubject = mSubject;
            undoItem.mCreationDate = mCreationDate;
            undoItem.mModifiedDate = mModifiedDate;
            undoItem.mColor = mColor;
            undoItem.mOpacity = mOpacity;
            undoItem.mLineWidth = mLineWidth;
            undoItem.mIntent = mIntent;
            undoItem.mBBox = mBBox;
            undoItem.mContents = mContents;
            undoItem.isFromEraser = isFromEraser;
            undoItem.mGroupNMList = mGroupNMList;
            undoItem.mReplys = mReplys;
            undoItem.mPencilType = mPencilType;
            undoItem.mPath = mPath;
            undoItem.mPSIPressures = mPSIPressures;

            mAnnotHandler.addAnnot(mPageIndex, annot, undoItem, false, true, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (mGroupNMList.size() > 0) {
                        GroupManager.getInstance().setAnnotGroup(mPdfViewCtrl, page, mGroupNMList);
                        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                        documentManager.onAnnotGrouped(page, AppAnnotUtil.getAnnotsByNMs(page, mGroupNMList));
                    }

                    if (callback != null)
                        callback.result(event, success);
                }
            });
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean redo(Event.Callback callback) {
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Ink)) {
                return false;
            }

            mAnnotHandler.removeAnnot(annot, this, false, callback);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean undo() {
        return undo(null);
    }

    @Override
    public boolean redo() {
        return redo(null);
    }
}