/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink.ocr;

import android.graphics.PointF;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.SuccessContinuation;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.mlkit.vision.digitalink.recognition.Ink;
import com.google.mlkit.vision.digitalink.recognition.Ink.Point;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

/**
 * Manages the recognition logic and the content that has been added to the current page.
 */
public class StrokeManager {

    /**
     * Interface to register to be notified of changes in the recognized content.
     */
    public interface ContentChangedListener {

        /**
         * This method is called when the recognized content changes.
         */
        void onInkContentChanged(boolean isConvert);
    }

    /**
     * Interface to register to be notified of changes in the status.
     */
    public interface StatusChangedListener {

        /**
         * This method is called when the recognized content changes.
         */

        void onStatusChanged();
    }

    /**
     * Interface to register to be notified of changes in the downloaded model state.
     */
    public interface DownloadedModelsChangedListener {

        /**
         * This method is called when the downloaded models changes.
         */
        void onDownloadedModelsChanged(Set<String> downloadedLanguageTags);


    }

    @VisibleForTesting
    static final long CONVERSION_TIMEOUT_MS = 1000;
    private static final String TAG = "MLKD.StrokeManager";
    // This is a constant that is used as a message identifier to trigger the timeout.
    private static final int TIMEOUT_TRIGGER = 1;
    // For handling recognition and model downloading.
    private RecognitionTask recognitionTask = null;
    @VisibleForTesting
    ModelManager modelManager = new ModelManager();
    // Managing the recognition queue.
    private final List<RecognitionTask.RecognizedInk> content = new ArrayList<>();
    // Managing ink currently drawn.
    private Ink.Stroke.Builder strokeBuilder = Ink.Stroke.builder();
    private Ink.Builder inkBuilder = Ink.builder();
    private boolean stateChangedSinceLastRequest = false;
    @Nullable
    private ContentChangedListener contentChangedListener = null;
    @Nullable
    private StatusChangedListener statusChangedListener = null;
    @Nullable
    private DownloadedModelsChangedListener downloadedModelsChangedListener = null;

    private boolean triggerRecognitionAfterInput = true;
    private boolean clearCurrentInkAfterRecognition = true;
    private String status = "";

    public void setTriggerRecognitionAfterInput(boolean shouldTrigger) {
        triggerRecognitionAfterInput = shouldTrigger;
    }

    public void setClearCurrentInkAfterRecognition(boolean shouldClear) {
        clearCurrentInkAfterRecognition = shouldClear;
    }

    // Handler to handle the UI Timeout.
    // This handler is only used to trigger the UI timeout. Each time a UI interaction happens,
    // the timer is reset by clearing the queue on this handler and sending a new delayed message (in
    // addNewTouchEvent).
    private Handler uiHandler = new Handler(new Handler.Callback() {
        @Override
        public boolean handleMessage(Message msg) {
            if (msg.what == TIMEOUT_TRIGGER) {
                Log.i(TAG, "Handling timeout trigger.");
                commitResult();
                return true;
            }
            // In the current use this statement is never reached because we only ever send
            // TIMEOUT_TRIGGER messages to this handler.
            // This line is necessary because otherwise Java's static analysis doesn't allow for
            // compiling. Returning false indicates that a message wasn't handled.
            return false;
        }
    });

    private void setStatus(String newStatus) {
        status = newStatus;
        if (statusChangedListener != null) {
            statusChangedListener.onStatusChanged();
        }
    }

    private void commitResult() {
        if (recognitionTask.done() && recognitionTask.result() != null) {
            content.add(recognitionTask.result());
            setStatus("Successful recognition: " + recognitionTask.result().text);
            if (clearCurrentInkAfterRecognition) {
                resetCurrentInk();
            }
        }
        if (contentChangedListener != null) {
            contentChangedListener.onInkContentChanged(recognitionTask.done() && recognitionTask.result() != null);
        }
    }

    public void reset() {
        Log.i(TAG, "reset");
        resetCurrentInk();
        content.clear();
        if (recognitionTask != null && !recognitionTask.done()) {
            recognitionTask.cancel();
        }
        setStatus("");
    }

    private void resetCurrentInk() {
        inkBuilder = Ink.builder();
        strokeBuilder = Ink.Stroke.builder();
        stateChangedSinceLastRequest = false;
    }

    public Ink getCurrentInk() {
        return inkBuilder.build();
    }

    /**
     * This method is called when a new touch event happens on the drawing client and notifies the
     * StrokeManager of new content being added.
     *
     * <p>This method takes care of triggering the UI timeout and scheduling recognitions on the
     * background thread.
     *
     * @return whether the touch event was handled.
     */
    public boolean addNewTouchEvent(MotionEvent event) {
        int action = event.getActionMasked();
        float x = event.getX();
        float y = event.getY();
        long t = System.currentTimeMillis();

        // A new event happened -> clear all pending timeout messages.
        uiHandler.removeMessages(TIMEOUT_TRIGGER);

        switch (action) {
            case MotionEvent.ACTION_DOWN:
            case MotionEvent.ACTION_MOVE:
                strokeBuilder.addPoint(Point.create(x, y, t));
                break;
            case MotionEvent.ACTION_UP:
                strokeBuilder.addPoint(Point.create(x, y, t));
                inkBuilder.addStroke(strokeBuilder.build());
                strokeBuilder = Ink.Stroke.builder();
                stateChangedSinceLastRequest = true;
                if (triggerRecognitionAfterInput) {
                    recognize();
                }
                break;
            default:
                // Indicate touch event wasn't handled.
                return false;
        }

        return true;
    }

    public boolean addPoint(PDFViewCtrl pdfViewCtrl, ArrayList<ArrayList<ArrayList<PointF>>> inkListss, int index) {
        // A new event happened -> clear all pending timeout messages.
        long t = System.currentTimeMillis();
        uiHandler.removeMessages(TIMEOUT_TRIGGER);

        int rotation = 0;
        float height;
        float width;
        try {
            PDFPage page = pdfViewCtrl.getDoc().getPage(index);
            rotation = (page.getRotation() + pdfViewCtrl.getViewRotation()) % 4;

            height = page.getHeight();
            width = page.getWidth();

            if (rotation != 0) {
                PointF pageCenter = new PointF(page.getWidth() / 2f, page.getHeight() / 2f);
                for (int i = 0; i < inkListss.size(); i++) {
                    ArrayList<ArrayList<PointF>> inkList  = inkListss.get(i);
                    for (int j = 0; j < inkList.size(); j++) {
                        ArrayList<PointF> points  = inkList.get(j);
                        rotatePoints(points, pageCenter, rotation);
                    }
                }
            }
        } catch (PDFException ignored){
            height = pdfViewCtrl.getDisplayViewHeight();
            width = pdfViewCtrl.getDisplayViewWidth();
        }

        for (ArrayList<ArrayList<PointF>> lists : inkListss) {
            for (ArrayList<PointF> inkList : lists) {
                for (PointF pointF : inkList) {
                    if (t == 0)
                        t = System.currentTimeMillis();
                    else
                        t = t + 200;
                    float x = pointF.x;
                    float y = pointF.y;

                    float x_new = x;
                    float y_new = height - y;
//                    pdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, index);
                    Point point;
//                    if(x == 0) {
                        point = Point.create(x_new, y_new,t);
//                    }else{
//                        float dx = pointF.x - x;
//                        float dy = pointF.y - y;
//                        double sq = Math.sqrt(dx * dx + dy * dy);
//                        point =Ink.Point.create(pointF.x, pointF.y, t);
//                    }
                    strokeBuilder.addPoint(point);
//                    x = pointF.x;
//                    y = pointF.y;
                }
                t = t + 600;
                inkBuilder.addStroke(strokeBuilder.build());
                strokeBuilder = Ink.Stroke.builder();
                stateChangedSinceLastRequest = true;
            }
        }
        return true;
    }

    public static void rotatePoints(ArrayList<PointF> points,PointF center,int rotation) {
        float angle = -90f * rotation;
        double angleRad = Math.toRadians(angle);

//        PointF center = getCenterPoint(points);
        for (PointF p : points) {
            float translatedX = p.x - center.x;
            float translatedY = p.y - center.y;

            float rotatedX = (float) (translatedX * Math.cos(angleRad) - translatedY * Math.sin(angleRad));
            float rotatedY = (float) (translatedX * Math.sin(angleRad) + translatedY * Math.cos(angleRad));

            p.x = rotatedX + center.x;
            p.y = rotatedY + center.y;
        }
    }

    // Listeners to update the drawing and status.
    public void setContentChangedListener(ContentChangedListener contentChangedListener) {
        this.contentChangedListener = contentChangedListener;
    }

    public void setStatusChangedListener(StatusChangedListener statusChangedListener) {
        this.statusChangedListener = statusChangedListener;
    }

    public void setDownloadedModelsChangedListener(
            DownloadedModelsChangedListener downloadedModelsChangedListener) {
        this.downloadedModelsChangedListener = downloadedModelsChangedListener;
    }

    public List<RecognitionTask.RecognizedInk> getContent() {
        return content;
    }

    public String getStatus() {
        return status;
    }

    // Model downloading / deleting / setting.

    public void setActiveModel(String languageTag) {
        setStatus(modelManager.setModel(languageTag));
    }

    public Task<Void> deleteActiveModel() {
        return modelManager
                .deleteActiveModel()
                .addOnSuccessListener(new OnSuccessListener<String>() {
                    @Override
                    public void onSuccess(String s) {
                        refreshDownloadedModelsStatus();
                    }
                })
                .onSuccessTask(new SuccessContinuation<String, Void>() {
                    @NonNull
                    @Override
                    public Task<Void> then(String status) throws Exception {
                        setStatus(status);
                        return Tasks.forResult(null);
                    }
                });
    }
    public Task<Void> deleteActiveModel(String languageTag) {
        return modelManager
                .deleteActiveModel(languageTag)
                .addOnSuccessListener(new OnSuccessListener<String>() {
                    @Override
                    public void onSuccess(String s) {
                        refreshDownloadedModelsStatus();
                    }
                })
                .onSuccessTask(new SuccessContinuation<String, Void>() {
                    @NonNull
                    @Override
                    public Task<Void> then(String status) throws Exception {
                        setStatus(status);
                        return Tasks.forResult(null);
                    }
                });
    }


    public Task<Void> download() {
        setStatus("Download started.");
        return modelManager
                .download()
                .addOnSuccessListener(new OnSuccessListener<String>() {
                    @Override
                    public void onSuccess(String s) {
                        refreshDownloadedModelsStatus();
                    }
                })
                .onSuccessTask(new SuccessContinuation<String, Void>() {
                    @NonNull
                    @Override
                    public Task<Void> then(String status) throws Exception {
                        setStatus(status);
                        return Tasks.forResult(null);
                    }
                });
    }

    public Task<Void> download(String languageTag) {
        setStatus("Download started.");
        return modelManager
                .download(languageTag)
                .addOnSuccessListener(new OnSuccessListener<String>() {
                    @Override
                    public void onSuccess(String s) {
                        refreshDownloadedModelsStatus();
                    }
                })
                .onSuccessTask(new SuccessContinuation<String, Void>() {
                    @NonNull
                    @Override
                    public Task<Void> then(String status) throws Exception {
                        setStatus(status);
                        return Tasks.forResult(null);
                    }
                });
    }

    // Recognition-related.

    public Task<String> recognize() {

        if (!stateChangedSinceLastRequest || inkBuilder.isEmpty()) {
            setStatus("No recognition, ink unchanged or empty");
            if (contentChangedListener != null) {
                contentChangedListener.onInkContentChanged(false);
            }
            return Tasks.forResult(null);
        }
        if (modelManager.getRecognizer() == null) {
            setStatus("Recognizer not set");
            if (contentChangedListener != null) {
                contentChangedListener.onInkContentChanged(false);
            }
            return Tasks.forResult(null);
        }

        return modelManager
                .checkIsModelDownloaded()
                .onSuccessTask(new SuccessContinuation<Boolean, String>() {
                    @NonNull
                    @Override
                    public Task<String> then(Boolean result) throws Exception {
                        if (!result) {
                            setStatus("Model not downloaded yet");
                            if (contentChangedListener != null) {
                                contentChangedListener.onInkContentChanged(false);
                            }
                            return Tasks.forResult(null);
                        }

                        stateChangedSinceLastRequest = false;
                        recognitionTask =
                                new RecognitionTask(modelManager.getRecognizer(), inkBuilder.build());
                        uiHandler.sendMessageDelayed(
                                uiHandler.obtainMessage(TIMEOUT_TRIGGER), CONVERSION_TIMEOUT_MS);
                        return recognitionTask.run();
                    }
                });
    }

    public void refreshDownloadedModelsStatus() {
        modelManager
                .getDownloadedModelLanguages()
                .addOnSuccessListener(new OnSuccessListener<Set<String>>() {
                    @Override
                    public void onSuccess(Set<String> strings) {
                        if (downloadedModelsChangedListener != null) {
                            downloadedModelsChangedListener.onDownloadedModelsChanged(strings);
                        }
                    }
                });
    }
}
