/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.link;

import android.app.Activity;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.DrawFilter;
import android.graphics.Paint;
import android.graphics.PaintFlagsDrawFilter;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.Layout;
import android.text.StaticLayout;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.SparseArray;
import android.view.MotionEvent;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Library;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.FileSpec;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.sdk.pdf.actions.GotoAction;
import com.foxit.sdk.pdf.actions.JavaScriptAction;
import com.foxit.sdk.pdf.actions.LaunchAction;
import com.foxit.sdk.pdf.actions.RemoteGotoAction;
import com.foxit.sdk.pdf.actions.URIAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Link;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.concurrent.Callable;

import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_B;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_L;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_LB;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_LT;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_R;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_RB;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_RT;
import static com.foxit.uiextensions.annots.redaction.RedactAnnotHandler.CTR_T;

public class LinkAnnotHandler implements AnnotHandler {
    protected Context mContext;
    protected boolean isDocClosed = false;
    private final Paint mPaint;
    private final int mType;
    private final PDFViewCtrl mPdfViewCtrl;
    private Destination mDestination;
    private final UIExtensionsManager mExtensionManager;
    private boolean mIsLinkEditMode;
    private final TextPaint mTextPaint;
    private final int mMinDrawTextSize;
    private final int mMaxDrawTextSize;
    private final RectF mInvalidateDocRectF = new RectF();
    private SparseArray<LinkInfo> mLinkInfoArray;
    private Disposable mDisposable;
    private boolean mIsScalingPage = false;

    static class LinkInfo {
        int pageIndex;
        ArrayList<LinkItem> links;

        public LinkInfo(int pageIndex) {
            this.pageIndex = pageIndex;
            links = new ArrayList<>();
        }

        static class LinkItem {
            int objNum = -1;
            float borderWidth = -1;
            RectF rectF;
            LinkAction linkAction;

            static class LinkAction {
                int actionType;
                int pageNumber;
                String uri;
            }
        }
    }

    LinkAnnotHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mPaint = new Paint();
        mType = Annot.e_Link;
        mLinkInfoArray = new SparseArray<>();
        mExtensionManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mTextPaint = new TextPaint();
        mMinDrawTextSize = (int) AppResource.getDimension(mContext, R.dimen.ux_text_size_6sp);
        mMaxDrawTextSize = (int) AppResource.getDimension(mContext, R.dimen.ux_text_size_14sp);

        mPaintBbox = new Paint();
        mPaintBbox.setAntiAlias(true);
        mPaintBbox.setColor(DEFAULT_BORDER_COLOR);
        mPaintBbox.setStyle(Paint.Style.STROKE);
        mPaintBbox.setStrokeWidth(AppAnnotUtil.getInstance(context).getAnnotBBoxStrokeWidth());
        mPaintBbox.setPathEffect(AppAnnotUtil.getAnnotBBoxPathEffect());

        mCtlPtPaint = new Paint();
    }

    protected PDFViewCtrl.IPageEventListener getPageEventListener() {
        return mPageEventListener;
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            if (!success || index == dstIndex)
                return;

            if (mLinkInfoArray.size() == 0)
                return;

            SparseArray<LinkInfo> newArrs = new SparseArray<>();
            int size = mLinkInfoArray.size();
            for (int i = 0; i < size; i++) {
                int pageIndex = mLinkInfoArray.keyAt(i);
                LinkInfo linkInfo = mLinkInfoArray.valueAt(i);

                if (index < dstIndex) {
                    if (pageIndex <= dstIndex && pageIndex > index) {
                        pageIndex -= 1;
                    } else if (pageIndex == index) {
                        pageIndex = dstIndex;
                    }
                } else {
                    if (pageIndex >= dstIndex && pageIndex < index) {
                        pageIndex += 1;
                    } else if (pageIndex == index) {
                        pageIndex = dstIndex;
                    }
                }
                newArrs.put(pageIndex, linkInfo);
            }
            mLinkInfoArray = newArrs;
        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            if (!success || mLinkInfoArray.size() == 0)
                return;

            for (int i = 0; i < pageIndexes.length; i++) {
                int removeIndex = pageIndexes[i] - i;

                for (int j = mLinkInfoArray.size() - 1; j >= 0; j--) {
                    int pageIndex = mLinkInfoArray.keyAt(j);
                    LinkInfo linkInfo = mLinkInfoArray.valueAt(j);

                    if (pageIndex == removeIndex) {
                        linkInfo.links.clear();
                        mLinkInfoArray.remove(pageIndex);
                    } else if (pageIndex > removeIndex) {
                        mLinkInfoArray.remove(pageIndex);
                        mLinkInfoArray.put(pageIndex - 1, linkInfo);
                    }
                }
            }
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            if (!success || mLinkInfoArray.size() == 0)
                return;

            int offsetIndex = 0;
            for (int i = 0; i < range.length / 2; i++) {
                offsetIndex += range[2 * i + 1];
            }

            SparseArray<LinkInfo> newArrs = new SparseArray<>();
            int size = mLinkInfoArray.size();
            for (int i = 0; i < size; i++) {
                int pageIndex = mLinkInfoArray.keyAt(i);
                LinkInfo linkInfo = mLinkInfoArray.valueAt(i);
                if (pageIndex >= dstIndex) {
                    pageIndex += offsetIndex;
                }
                newArrs.put(pageIndex, linkInfo);
            }
            mLinkInfoArray = newArrs;
        }
    };

    protected Destination getDestination() {
        return mDestination;
    }

    protected void setDestination(Destination mDestination) {
        this.mDestination = mDestination;
    }

    protected void searchPageLinks(int pageIndex) {
        if (mPdfViewCtrl.getDoc() == null) return;

        if (mDisposable != null && !mDisposable.isDisposed()) {
            mDisposable.dispose();
        }

        LinkInfo linkInfo = mLinkInfoArray.get(pageIndex);
        if (linkInfo == null) {
            linkInfo = new LinkInfo(pageIndex);
            mLinkInfoArray.put(pageIndex, linkInfo);
        } else {
            linkInfo.pageIndex = -1;
            linkInfo.links.clear();
        }

        mDisposable = searchLinkAnnot(pageIndex)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<LinkInfo>() {
                    @Override
                    public void accept(LinkInfo linkInfo) throws Exception {
                        if (linkInfo != null) {
                            mLinkInfoArray.put(linkInfo.pageIndex, linkInfo);

                            if (mPdfViewCtrl.isPageVisible(linkInfo.pageIndex)) {
                                mPdfViewCtrl.invalidate(new Rect(0, 0, 10, 10));
                            }
                        }
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) throws Exception {
                    }
                });
    }

    private Single<LinkInfo> searchLinkAnnot(final int pageIndex) {
        return Single.fromCallable(new Callable<LinkInfo>() {
            @Override
            public LinkInfo call() throws Exception {
                try {
                    if (mPdfViewCtrl.getDoc() == null) {
                        return null;
                    }
                    LinkInfo linkInfo = new LinkInfo(pageIndex);
                    PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                    int count = page.getAnnotCount();
                    Annot annot;
                    for (int i = 0; i < count; i++) {
                        if (mPdfViewCtrl.getDoc() == null) {
                            break;
                        }

                        if (mDisposable.isDisposed()) {
                            break;
                        }

                        annot = page.getAnnot(i);
                        if (annot == null || annot.isEmpty()) continue;
                        if (annot.getType() == Annot.e_Link) {
                            LinkInfo.LinkItem linkItem = getLinkInfo(annot);
                            linkInfo.links.add(linkItem);
                        }
                        annot.delete();
                    }
                    page.delete();
                    return linkInfo;
                } catch (PDFException e) {
                    e.printStackTrace();
                }
                return null;
            }
        });
    }

    @Override
    public int getType() {
        return mType;
    }

    @Override
    public boolean annotCanAnswer(Annot annot) {
        try {
            if (annot.getType() == mType) {
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public RectF getAnnotBBox(Annot annot) {
        try {
            return AppUtil.toRectF(annot.getRect());
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return null;
    }

    @Override
    public boolean isHitAnnot(Annot annot, PointF point) {
        try {
            int pageIndex = annot.getPage().getIndex();
            RectF rectF = getAnnotBBox(annot);
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
            return rectF.contains(point.x, point.y);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public void onAnnotSelected(Annot annot, boolean reRender) {
        try {
            Link link = (Link) annot;
            if (link == null || link.isEmpty())
                return;
            setLinkEditable(AnnotPermissionUtil.canModifyAnnot(mExtensionManager.getDocumentManager(), link));
            prepareAnnotMenu(annot);
            int _pageIndex = annot.getPage().getIndex();
            RectF annotRectF = AppUtil.toRectF(annot.getRect());
            mInvalidateDocRectF.set(annotRectF);

            if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                RectF invalidateRect = new RectF(annotRectF);
                mPdfViewCtrl.convertPdfRectToPageViewRect(invalidateRect, invalidateRect, _pageIndex);
                RectF menuRect = new RectF(invalidateRect);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(menuRect, menuRect, _pageIndex);
                mPdfViewCtrl.refresh(_pageIndex, AppDmUtil.rectFToRect(invalidateRect));
                mAnnotMenu.show(menuRect);
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private ArrayList<Integer> mMenuItems;
    private AnnotMenu mAnnotMenu;

    private void prepareAnnotMenu(final Annot annot) {
        if (mAnnotMenu == null) {
            mAnnotMenu = new AnnotMenuImpl(mContext, mPdfViewCtrl);
        }
        resetAnnotationMenuResource(annot);
        mAnnotMenu.setMenuItems(mMenuItems);
        final Link link = (Link) annot;
        mAnnotMenu.setListener(new AnnotMenu.ClickListener() {
            @Override
            public void onAMClick(int btType) {
                if (btType == AnnotMenu.AM_BT_DELETE) {
                    if (link == mExtensionManager.getDocumentManager().getCurrentAnnot()) {
                        removeAnnot(link, true, null);
                    }
                } else if (btType == AnnotMenu.AM_BT_EDIT) {
                    LinkToolHandler handler = getLinkToolHandler();
                    if (handler != null) {
                        handler.showSettingDialog(new MatchDialog.DismissListener() {
                            @Override
                            public void onDismiss() {
                                if (mExtensionManager.getDocumentManager().getCurrentAnnot() == link) {
                                    try {
                                        int pageindex = link.getPage().getIndex();
                                        RectF annotRectF = AppUtil.toRectF(link.getRect());
                                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageindex);
                                        RectF menuRect = new RectF();
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, menuRect, pageindex);
                                        mAnnotMenu.show(menuRect);
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                        });
                    }
                    if (mAnnotMenu != null && mAnnotMenu.isShowing()) {
                        mAnnotMenu.dismiss();
                    }
                }
            }
        });
    }

    private LinkToolHandler getLinkToolHandler() {
        ToolHandler handler = mExtensionManager.getToolHandlerByType(ToolHandler.TH_TYPE_LINK);
        if (handler instanceof LinkToolHandler) {
            return (LinkToolHandler) handler;
        }
        return null;
    }

    private void resetAnnotationMenuResource(Annot annot) {
        if (mMenuItems == null) {
            mMenuItems = new ArrayList<>();
        } else {
            mMenuItems.clear();
        }
        if (mExtensionManager.getDocumentManager().canAddAnnot() && mExtensionManager.isEnableModification()) {
            if (AnnotPermissionUtil.canModifyAnnot(mExtensionManager.getDocumentManager(), annot))
                mMenuItems.add(AnnotMenu.AM_BT_EDIT);
            if (!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot))
                    && AnnotPermissionUtil.canDeleteAnnot(mExtensionManager.getDocumentManager(), annot)) {
                mMenuItems.add(AnnotMenu.AM_BT_DELETE);
            }
        }
    }

    @Override
    public void removeAnnot(Annot annot, boolean addUndo, Event.Callback result) {
        removeLink(annot, addUndo, result);
    }

    private void removeLink(final Annot annot, final boolean addUndo, final Event.Callback result) {
        final DocumentManager documentManager = mExtensionManager.getDocumentManager();
        if (documentManager.getCurrentAnnot() != null && AppAnnotUtil.isSameAnnot(annot, documentManager.getCurrentAnnot())) {
            documentManager.setCurrentAnnot(null, false);
        }

        try {
            final PDFPage page = annot.getPage();
            final int pageIndex = page.getIndex();

            final LinkDeleteUndoItem undoItem = new LinkDeleteUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mBBox = AppUtil.toRectF(annot.getRect());
            undoItem.mBorderInfo = annot.getBorderInfo();
            undoItem.mPageIndex = pageIndex;
            RectFArray rectFArray = new RectFArray();
            rectFArray.add(annot.getRect());
            undoItem.mRectFArray = rectFArray;
            undoItem.mAction = ((Link) annot).getAction();

            undoItem.mPDFDict = AppAnnotUtil.clonePDFDict(annot.getDict());

            documentManager.onAnnotWillDelete(page, annot);
            LinkEvent event = new LinkEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Link) annot, mPdfViewCtrl);
            if (documentManager.isMultipleSelectAnnots()) {
                if (result != null) {
                    result.result(event, true);
                }
                return;
            }

            final int objNum = annot.getDict().getObjNum();
            final RectF annotRect = AppUtil.toRectF(annot.getRect());
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        documentManager.onAnnotDeleted(page, annot);
                        if (addUndo) {
                            documentManager.addUndoItem(undoItem);
                        }
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            if (documentManager.getCurrentAnnot() != null) {
                                documentManager.setCurrentAnnot(null, false);
                            }
                            removeLinkInfo(objNum, pageIndex);
                            documentManager.setDocModified(true);

                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, pageIndex);
                            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(annotRect));
                        }
                    }
                    if (result != null) {
                        result.result(null, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onAnnotDeselected(Annot annot, boolean reRender) {
        mAnnotMenu.dismiss();
        mInvalidateDocRectF.setEmpty();
        mMenuItems.clear();
        try {
            int pageIndex = annot.getPage().getIndex();
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                RectF annotRectF = AppUtil.toRectF(annot.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(annotRectF));
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void addAnnot(int pageIndex, AnnotContent content, boolean addUndo, Event.Callback result) {
    }

    @Override
    public void modifyAnnot(Annot annot, AnnotContent content, boolean addUndo, Event.Callback result) {
    }

    public boolean onTouchEvent(int pageIndex, MotionEvent e, Annot annot) {
        return false;
    }

    protected void onDrawForControls(Canvas canvas) {
        Annot annot = mExtensionManager.getDocumentManager().getCurrentAnnot();
        if (!(annot instanceof Link)) return;
        if (mExtensionManager.getCurrentAnnotHandler() != LinkAnnotHandler.this) return;

        try {
            int annotPageIndex = annot.getPage().getIndex();
            if (mPdfViewCtrl.isPageVisible(annotPageIndex)) {
                RectF deviceRt = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, deviceRt, annotPageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, deviceRt, annotPageIndex);
                mAnnotMenu.update(deviceRt);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (isDocClosed || mPdfViewCtrl.isDynamicXFA()) return;
        if (!mExtensionManager.isLinkHighlightEnabled())
            return;

        LinkInfo linkInfo = mLinkInfoArray.get(pageIndex);
        if (linkInfo == null) {
            searchPageLinks(pageIndex);
            return;
        }

        if (linkInfo.links.isEmpty()) return;
        canvas.save();
        canvas.setDrawFilter(mDrawFilter);
        Rect clipRect = canvas.getClipBounds();
        try {
            int linkCount = linkInfo.links.size();
            if (linkCount > 100 && mIsScalingPage) {
                canvas.restore();
                return;
            }

            mPaint.setColor((int) mExtensionManager.getLinkHighlightColor());
            ArrayList<LinkInfo.LinkItem> invalidLinks = new ArrayList<>();
            for (int i = 0; i < linkCount; i++) {
                LinkInfo.LinkItem item = linkInfo.links.get(i);
                int objNum = item.objNum;
                if (objNum == -1) {
                    invalidLinks.add(item);
                    continue;
                }

                RectF rectF = new RectF(item.rectF);
                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                if (rectF.intersect(new RectF(clipRect))) {
                    Annot curAnnot = mExtensionManager.getDocumentManager().getCurrentAnnot();
                    int currAnnotObjNum = curAnnot == null ? 0 : curAnnot.getDict().getObjNum();
                    if (mLinkEditable && AppAnnotUtil.isSameAnnot(objNum, currAnnotObjNum)) {
                        drawControlViewForLink(pageIndex, canvas, item.borderWidth, rectF);
                    } else {
                        canvas.drawRect(rectF, mPaint);
                    }
                    if (mIsLinkEditMode) {
                        LinkInfo.LinkItem.LinkAction linkAction = item.linkAction;
                        if (linkAction == null) {
                            continue;
                        }
                        String content = "";
                        if (linkAction.actionType == Action.e_TypeGoto) {
                            content = String.format(AppResource.getString(mContext, R.string.to_page_format_text), linkAction.pageNumber);
                        } else if (linkAction.actionType == Action.e_TypeURI) {
                            if (!TextUtils.isEmpty(linkAction.uri)) {
                                content = String.format(AppResource.getString(mContext, R.string.link_to_format_text), linkAction.uri);
                            }
                        }

                        RectF invalidRectF = new RectF();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, invalidRectF, pageIndex);
                        if (!invalidRectF.isEmpty() && AppAnnotUtil.isSameAnnot(item.objNum, currAnnotObjNum)) {
                            rectF.set(invalidRectF);
                        }
                        drawTextForLink(canvas, rectF, content);
                    }
                }
            }

            if (!invalidLinks.isEmpty())
                linkInfo.links.removeAll(invalidLinks);
            canvas.restore();
            canvas.setDrawFilter(null);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void drawTextForLink(Canvas canvas, RectF rectF, String content) {
        if (TextUtils.isEmpty(content)) return;
        mTextPaint.setColor(AppResource.getColor(mContext, R.color.t2));
        mTextPaint.setAntiAlias(true);
        mTextPaint.setStyle(Paint.Style.FILL);
        StaticLayout layout = null;
        for (int i = mMaxDrawTextSize; i >= mMinDrawTextSize; i--) {
            mTextPaint.setTextSize(i);
            layout = new StaticLayout(content, mTextPaint,
                    (int) (rectF.width()), Layout.Alignment.ALIGN_CENTER,
                    1, 0, false);
            if (layout.getWidth() <= rectF.width() && layout.getHeight() <= rectF.height()) {
                break;
            } else {
                layout = null;
            }
        }
        if (layout == null) {
            return;
        }
        canvas.save();
        canvas.translate(rectF.left, rectF.centerY() - layout.getHeight() / 2f);
        layout.draw(canvas);
        canvas.restore();
    }

    private int mDeltaXY;

    private void drawControlViewForLink(int pageIndex, Canvas canvas, float borderWidth, RectF rectF) {
        if (borderWidth == -1) return;
        mDeltaXY = (int) (mCtlPtLineWidth + mCtlPtRadius * 2 + 2);// Judging border value

        canvas.save();
        float thickness = thicknessOnPageView(pageIndex, borderWidth);

        RectF drawRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, drawRectF, pageIndex);
        if (drawRectF.isEmpty()) {
            mPdfViewCtrl.convertPageViewRectToPdfRect(rectF, mInvalidateDocRectF, pageIndex);
            drawRectF.set(rectF);
        }
        drawControlPoints(canvas, drawRectF);
        drawControlImaginary(canvas, drawRectF);
        drawRectF.inset(thickness, thickness);
        canvas.drawRect(drawRectF, mPaint);
        canvas.restore();
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        return false;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, Annot annot) {
        if (!mPdfViewCtrl.isPageVisible(pageIndex) || mPdfViewCtrl.getUIExtensionsManager() == null || !(annot instanceof Link))
            return false;

        if (!mExtensionManager.isLinksEnabled()) return false;

        if (mIsLinkEditMode && mLinkEditable) {
            showEditableLink(pageIndex, annot, motionEvent);
            return true;
        }

        UIExtensionsManager.ILinkEventListener ILinkEventListener = mExtensionManager.getLinkEventListener();
        if (ILinkEventListener != null) {
            UIExtensionsManager.LinkInfo emLinkInfo = new UIExtensionsManager.LinkInfo();
            emLinkInfo.link = annot;
            emLinkInfo.linkType = UIExtensionsManager.LINKTYPE_ANNOT;
            if (ILinkEventListener.onLinkTapped(emLinkInfo)) return true;
        }

        try {
            Action annotAction = ((Link) annot).getAction();
            return executeAction(((Link) annot), annotAction);
        } catch (PDFException e1) {
            if (e1.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
            e1.printStackTrace();
        }
        return true;
    }

    protected boolean executeAction(Link link, Action action) {
        try {
            if (action == null || action.isEmpty()) {
                return false;
            }

            boolean ret = doAction(link, action);
            for (int i = 0; i < action.getSubActionCount(); i++) {
                Action subAction = action.getSubAction(i);
                ret = executeAction(link, subAction);
            }
            return ret;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean doAction(Link link, Action action) {
        try {
            int type = action.getType();
            switch (type) {
                case Action.e_TypeGoto: {
                    GotoAction gotoAction = new GotoAction(action);
                    final Destination destination = gotoAction.getDestination();
                    if (destination == null || destination.isEmpty()) {
                        mPdfViewCtrl.gotoPage(0, 0, 0);
                    } else {
                        final int pageIndex = destination.getPageIndex(mPdfViewCtrl.getDoc());
                        if (pageIndex < 0 || pageIndex > mPdfViewCtrl.getPageCount() - 1)
                            return false;

                        final PointF destPt = LinkUtil.getDestinationPoint(mPdfViewCtrl.getDoc(), destination);
                        mPdfViewCtrl.gotoPage(pageIndex);
                        if (destPt != null) {
                            mPdfViewCtrl.post(new Runnable() {
                                @Override
                                public void run() {
                                    try {
                                        int zoomMode = destination.getZoomMode();
                                        float viewWidth = mPdfViewCtrl.getWidth();
                                        float viewHeight = mPdfViewCtrl.getHeight();
                                        if (zoomMode == Destination.e_ZoomFitRect) {
                                            RectF rectF = new RectF(destination.getLeft(), destination.getTop(), destination.getRight(), destination.getBottom());
                                            if (Math.abs(rectF.width()) != 0 && Math.abs(rectF.height()) != 0){
                                                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);

                                                float scaleX = viewWidth / Math.abs(rectF.width());
                                                float scaleY = viewHeight / Math.abs(rectF.height());
                                                float scale = Math.min(scaleX, scaleY);
                                                if (Math.abs(scale - 1.0f) > 0.1f) {
                                                    float zoomScale = Math.min(
                                                            mPdfViewCtrl.getMaxZoomLimit(),
                                                            Math.max(mPdfViewCtrl.getMinZoomLimit(), scale * mPdfViewCtrl.getZoom()
                                                            ));
                                                    mPdfViewCtrl.setZoom(zoomScale);
                                                }
                                            }
                                        }

                                        PointF devicePt = new PointF();
                                        mPdfViewCtrl.convertPdfPtToPageViewPt(destPt, devicePt, pageIndex);
//
                                        PointF centerPoint = new PointF(viewWidth / 2.0f, viewHeight / 2.0f);
                                        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(centerPoint, centerPoint, pageIndex);

                                        float deltaX = centerPoint.x - devicePt.x;
                                        float deltaY = centerPoint.y - devicePt.y;
                                        mPdfViewCtrl.scrollView(-deltaX, -deltaY);
                                    } catch (PDFException e) {
                                        System.out.println(e.getMessage());
                                    }
                                }
                            });
                        }
                    }
                    break;
                }
                case Action.e_TypeURI: {
                    Activity context = mExtensionManager.getAttachedActivity();
                    if (context == null) return false;
                    URIAction uriAction = new URIAction(action);
                    String uri = uriAction.getURI();
                    if (uri.toLowerCase().startsWith("mailto:")) {
                        AppUtil.mailTo(context, uri);
                    } else {
                        AppUtil.openUrl(context, uri);
                    }
                    break;
                }
                case Action.e_TypeLaunch:
                case Action.e_TypeGoToR: {
                    FileSpec fileSpec;
                    RemoteGotoAction remoteGotoAction = null;
                    if (type == Action.e_TypeGoToR) {
                        remoteGotoAction = new RemoteGotoAction(action);
                        fileSpec = remoteGotoAction.getFileSpec();
                    } else {
                        LaunchAction launchAction = new LaunchAction(action);
                        fileSpec = launchAction.getFileSpec();
                    }

                    if (fileSpec == null || fileSpec.isEmpty()) return false;
                    String fileName = fileSpec.getFileName();
                    String path = mPdfViewCtrl.getFilePath();
                    if (path != null && path.lastIndexOf("/") > 0) {
                        path = path.substring(0, path.lastIndexOf("/") + 1);
                        path += fileName;
                        File file = new File(path);
                        if (!file.exists()) {
                            InputStream in = null;
                            OutputStream out = null;
                            String externalPath = AppStorageManager.getInstance(mContext).toExternalPathFromScopedCache(path);
                            try {
                                in = mContext.getContentResolver().openInputStream(AppFileUtil.toDocumentUriFromPath(externalPath));
                                out = new FileOutputStream(file);
                                AppFileUtil.copy(in, out);
                            } catch (Exception e) {
                                e.printStackTrace();
                            } finally {
                                AppFileUtil.closeQuietly(in);
                                AppFileUtil.closeQuietly(out);
                            }
                        }
                        if (file.exists() && file.isFile()) {
                            mExtensionManager.exitPanZoomMode();
                            if (path.length() > 4) {
                                String subfix = path.substring(path.length() - 4);
                                if (subfix.equalsIgnoreCase(".pdf")) {
                                    mPdfViewCtrl.cancelAllTask();
                                    clear();

                                    mExtensionManager.changeState(ReadStateConfig.STATE_NORMAL);
                                    mExtensionManager.getDocumentManager().setCurrentAnnot(null);
                                    mExtensionManager.getDocumentManager().clearUndoRedo();
                                    mExtensionManager.getDocumentManager().setDocModified(false);

                                    boolean isNullDestination = false;
                                    if (type == Action.e_TypeGoToR) {
                                        Destination destination = remoteGotoAction.getDestination();
                                        if (destination == null || destination.isEmpty()) {
                                            isNullDestination = true;
                                        }
                                        setDestination(destination);
                                    }
                                    String openPath = AppFileUtil.getAdaptedFilePath(mContext, path);
                                    mPdfViewCtrl.openDoc(openPath, null);
                                    return !isNullDestination;
                                }
                            }
                            Activity context = mExtensionManager.getAttachedActivity();
                            if (context == null) return false;
                            AppIntentUtil.openFile(context, path);
                            return true;
                        }
                        Activity context = mExtensionManager.getAttachedActivity();
                        if (context == null) return false;
                        AppUtil.alert(context,
                                AppResource.getString(mContext.getApplicationContext(), R.string.annot_link_alert_title),
                                AppResource.getString(mContext.getApplicationContext(), R.string.annot_link_alert_prompt),
                                AppUtil.ALERT_OK);
                    }
                    return true;
                }
                case Action.e_TypeJavaScript:
                    Library.killJSTimer();
                    if (link != null && !link.isEmpty()) {
                        link.executeJavaScriptAction(new JavaScriptAction(action));
                    }
                    return true;
                case Action.e_TypeUnknown:
                    return false;
                default:
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        return true;
    }

    public void clear() {
        mIsScalingPage = false;
        mLinkInfoArray.clear();
        if (mDisposable != null && !mDisposable.isDisposed()) {
            mDisposable.dispose();
        }
    }

    private LinkInfo.LinkItem getLinkInfo(Annot annot) {
        try {
            Link link = new Link(annot);
            LinkInfo.LinkItem item = new LinkInfo.LinkItem();
            item.objNum = link.getDict().getObjNum();
            item.borderWidth = link.getBorderInfo().getWidth();
            item.rectF = AppUtil.toRectF(link.getRect());

            Action action = link.getAction();
            if (action != null && !action.isEmpty()) {
                LinkInfo.LinkItem.LinkAction linkAction = new LinkInfo.LinkItem.LinkAction();
                linkAction.actionType = action.getType();
                if (linkAction.actionType == Action.e_TypeGoto) {
                    GotoAction gotoAction = new GotoAction(action);
                    Destination destination = gotoAction.getDestination();
                    int pageNumber = 1;
                    if (destination != null && !destination.isEmpty()) {
                        pageNumber = destination.getPageIndex(mPdfViewCtrl.getDoc()) + 1;
                        destination.delete();
                    }
                    gotoAction.delete();

                    linkAction.pageNumber = pageNumber;
                } else if (linkAction.actionType == Action.e_TypeURI) {
                    URIAction uriAction = new URIAction(action);
                    String uri = uriAction.getURI();
                    if (!TextUtils.isEmpty(uri)) {
                        linkAction.uri = uri;
                    }
                    uriAction.delete();
                }
                action.delete();

                item.linkAction = linkAction;
            }
            link.delete();
            return item;
        } catch (PDFException ignored) {
        }
        return null;
    }

    public void setLinkEditMode(boolean isEditMode, int pageIndex) {
        mIsLinkEditMode = isEditMode;
        if (mAnnotMenu != null && mAnnotMenu.isShowing()) {
            mLinkEditable = false;
            mExtensionManager.getDocumentManager().setCurrentAnnot(null);
        }
        if (mLinkInfoArray.get(pageIndex) != null) {
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                mPdfViewCtrl.refresh(pageIndex, new Rect());
            }
        }
    }

    private boolean mLinkEditable;

    public void setLinkEditable(boolean editable) {
        this.mLinkEditable = editable;
    }

    private final RectF mEditablePageViewRect = new RectF(0, 0, 0, 0);
    private final Paint mPaintBbox;
    private final Paint mCtlPtPaint;
    private final RectF mMapBounds = new RectF();
    private final float mCtlPtLineWidth = 2;
    private final float mCtlPtRadius = 5;
    private final RectF mThicknessRectF = new RectF();
    private final Path mImaginaryPath = new Path();
    private final DrawFilter mDrawFilter = new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);
    private static final int DEFAULT_BORDER_COLOR = PropertyBar.PB_COLORS_TOOL_DEFAULT[0] | 0xFF000000;

    private void showEditableLink(int pageIndex, Annot link, MotionEvent motionEvent) {
        PointF pageViewPoint = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
        if (link == mExtensionManager.getDocumentManager().getCurrentAnnot()) {
            try {
                if (pageIndex != link.getPage().getIndex() || !isHitAnnot(link, pageViewPoint)) {
                    mExtensionManager.getDocumentManager().setCurrentAnnot(null);
                    setLinkEditable(false);
                } else {
                    onAnnotSelected(link, false);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        } else {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(link);
        }
    }

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mThicknessRectF.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mThicknessRectF, mThicknessRectF, pageIndex);
        return Math.abs(mThicknessRectF.width());
    }

    private int isTouchControlPoint(RectF rect, float x, float y) {
        PointF[] ctlPts = calculateControlPoints(rect);
        RectF area = new RectF();
        int ret = -1;
        for (int i = 0; i < ctlPts.length; i++) {
            area.set(ctlPts[i].x, ctlPts[i].y, ctlPts[i].x, ctlPts[i].y);
            float mCtlPtTouchExt = 20;
            area.inset(-mCtlPtTouchExt, -mCtlPtTouchExt);
            if (area.contains(x, y)) {
                ret = i + 1;
            }
        }
        return ret;
    }

    private PointF[] calculateControlPoints(RectF rect) {
        rect.sort();
        mMapBounds.set(rect);
        mMapBounds.inset(-mCtlPtRadius - mCtlPtLineWidth / 2f, -mCtlPtRadius - mCtlPtLineWidth / 2f);// control rect
        PointF p1 = new PointF(mMapBounds.left, mMapBounds.top);
        PointF p2 = new PointF((mMapBounds.right + mMapBounds.left) / 2, mMapBounds.top);
        PointF p3 = new PointF(mMapBounds.right, mMapBounds.top);
        PointF p4 = new PointF(mMapBounds.right, (mMapBounds.bottom + mMapBounds.top) / 2);
        PointF p5 = new PointF(mMapBounds.right, mMapBounds.bottom);
        PointF p6 = new PointF((mMapBounds.right + mMapBounds.left) / 2, mMapBounds.bottom);
        PointF p7 = new PointF(mMapBounds.left, mMapBounds.bottom);
        PointF p8 = new PointF(mMapBounds.left, (mMapBounds.bottom + mMapBounds.top) / 2);
        return new PointF[]{p1, p2, p3, p4, p5, p6, p7, p8};
    }

    private void drawControlPoints(Canvas canvas, RectF rectBBox) {
        PointF[] ctlPts = calculateControlPoints(rectBBox);
        mCtlPtPaint.setStrokeWidth(mCtlPtLineWidth);

        for (PointF ctlPt : ctlPts) {
            mCtlPtPaint.setColor(Color.WHITE);
            mCtlPtPaint.setStyle(Paint.Style.FILL);
            canvas.drawCircle(ctlPt.x, ctlPt.y, mCtlPtRadius, mCtlPtPaint);
            mCtlPtPaint.setColor(DEFAULT_BORDER_COLOR);
            mCtlPtPaint.setStyle(Paint.Style.STROKE);
            canvas.drawCircle(ctlPt.x, ctlPt.y, mCtlPtRadius, mCtlPtPaint);
        }
    }

    private void drawControlImaginary(Canvas canvas, RectF rectBBox) {
        PointF[] ctlPts = calculateControlPoints(rectBBox);
        mPaintBbox.setStrokeWidth(mCtlPtLineWidth);
        mPaintBbox.setColor(DEFAULT_BORDER_COLOR);
        mImaginaryPath.reset();
        pathAddLine(mImaginaryPath, ctlPts[0].x + mCtlPtRadius, ctlPts[0].y, ctlPts[1].x - mCtlPtRadius, ctlPts[1].y);
        pathAddLine(mImaginaryPath, ctlPts[1].x + mCtlPtRadius, ctlPts[1].y, ctlPts[2].x - mCtlPtRadius, ctlPts[2].y);
        pathAddLine(mImaginaryPath, ctlPts[2].x, ctlPts[2].y + mCtlPtRadius, ctlPts[3].x, ctlPts[3].y - mCtlPtRadius);
        pathAddLine(mImaginaryPath, ctlPts[3].x, ctlPts[3].y + mCtlPtRadius, ctlPts[4].x, ctlPts[4].y - mCtlPtRadius);
        pathAddLine(mImaginaryPath, ctlPts[4].x - mCtlPtRadius, ctlPts[4].y, ctlPts[5].x + mCtlPtRadius, ctlPts[5].y);
        pathAddLine(mImaginaryPath, ctlPts[5].x - mCtlPtRadius, ctlPts[5].y, ctlPts[6].x + mCtlPtRadius, ctlPts[6].y);
        pathAddLine(mImaginaryPath, ctlPts[6].x, ctlPts[6].y - mCtlPtRadius, ctlPts[7].x, ctlPts[7].y + mCtlPtRadius);
        pathAddLine(mImaginaryPath, ctlPts[7].x, ctlPts[7].y - mCtlPtRadius, ctlPts[0].x, ctlPts[0].y + mCtlPtRadius);
        canvas.drawPath(mImaginaryPath, mPaintBbox);
    }

    private void pathAddLine(Path path, float start_x, float start_y, float end_x, float end_y) {
        path.moveTo(start_x, start_y);
        path.lineTo(end_x, end_y);
    }

    public void removeLinkInfo(int objNum, int pageIndex) {
        LinkInfo linkInfo = mLinkInfoArray.get(pageIndex);
        if (linkInfo != null) {
            for (int i = 0; i < linkInfo.links.size(); i++) {
                if (linkInfo.links.get(i).objNum == objNum) {
                    linkInfo.links.remove(i);
                    break;
                }
            }
        }
    }

    public void reLoadLinkInfo(Annot annot, int pageIndex) {
        try {
            LinkInfo linkInfo = mLinkInfoArray.get(pageIndex);
            if (linkInfo == null) {
                linkInfo = new LinkInfo(pageIndex);
                mLinkInfoArray.put(pageIndex, linkInfo);
            } else {
                this.removeLinkInfo(annot.getDict().getObjNum(), pageIndex);
            }

            LinkInfo.LinkItem linkItem = getLinkInfo(annot);
            linkInfo.links.add(linkItem);
        } catch (PDFException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean mIsTouchingView;
    private int mTouchPointPosition;
    private int mPageIndex;

    public boolean onDown(int pageIndex, MotionEvent e) {
        mPageIndex = pageIndex;
        PointF pageViewPoint = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, e);
        RectF viewRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, viewRectF, pageIndex);

        mIsTouchingView = viewRectF.contains(pageViewPoint.x, pageViewPoint.y);
        mTouchPointPosition = isTouchControlPoint(viewRectF, pageViewPoint.x, pageViewPoint.y);
        mEditablePageViewRect.set(viewRectF);
        boolean handled = false;
        if (mIsTouchingView) {
            handled = true;
        } else if (mTouchPointPosition != -1) {
            handled = true;
        }
        return handled;
    }

    public boolean onScroll(MotionEvent e2, float distanceX, float distanceY) {
        boolean handled = false;
        if (mIsTouchingView) {
            translateEditableViewRect(-distanceX, -distanceY);
            handled = true;
        } else if (mTouchPointPosition != -1) {
            scaleViewRect(mTouchPointPosition, e2);
            handled = true;
        }
        if (handled && mAnnotMenu.isShowing()) {
            mAnnotMenu.dismiss();
        }
        return handled;
    }

    private void scaleViewRect(int position, MotionEvent e2) {
        Rect pageDisplayRect = mPdfViewCtrl.getPageViewRect(mPageIndex);
        RectF pageViewRectF = new RectF();
        mPdfViewCtrl.convertDisplayViewRectToPageViewRect(new RectF(pageDisplayRect), pageViewRectF, mPageIndex);
        pageViewRectF.inset(mDeltaXY, mDeltaXY);
        PointF viewPoint = new PointF();

        RectF invalidRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, invalidRectF, mPageIndex);
        PointF leftTopPoint = new PointF(invalidRectF.left, invalidRectF.top);
        PointF rightBottomPoint = new PointF(invalidRectF.right, invalidRectF.bottom);
        RectF viewDisplayRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mEditablePageViewRect, viewDisplayRect, mPageIndex);
        if (position == CTR_L) {
            onControlLeft(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_T) {
            onControlTop(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_B) {
            onControlBottom(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_R) {
            onControlRight(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_LT) {
            onControlLeft(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
            onControlTop(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_LB) {
            onControlLeft(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
            onControlBottom(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_RT) {
            onControlRight(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
            onControlTop(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (position == CTR_RB) {
            onControlRight(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
            onControlBottom(e2, viewPoint, leftTopPoint, rightBottomPoint, viewDisplayRect);
        }
        if (leftTopPoint.x < pageViewRectF.left) {
            leftTopPoint.x = pageViewRectF.left;
        }
        if (rightBottomPoint.x > pageViewRectF.right) {
            rightBottomPoint.x = pageViewRectF.right;
        }
        if (leftTopPoint.y < pageViewRectF.top) {
            leftTopPoint.y = pageViewRectF.top;
        }
        if (rightBottomPoint.y > pageViewRectF.bottom) {
            rightBottomPoint.y = pageViewRectF.bottom;
        }
        invalidRectF.set(leftTopPoint.x, leftTopPoint.y, rightBottomPoint.x, rightBottomPoint.y);
        invalidRectF.sort();
        mPdfViewCtrl.convertPageViewRectToPdfRect(invalidRectF, mInvalidateDocRectF, mPageIndex);

        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalidRectF, invalidRectF, mPageIndex);
        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalidRectF));
    }

    private void onControlBottom(MotionEvent e2, PointF viewPoint, PointF leftTopPoint, PointF rightBottomPoint, RectF viewDisplayRect) {
        float changeY;
        changeY = convertPageViewPoint(0, e2.getY(), viewPoint).y;
        if (e2.getY() < viewDisplayRect.top) {
            leftTopPoint.y = changeY;
        } else {
            rightBottomPoint.y = changeY;
        }
    }

    private void onControlTop(MotionEvent e2, PointF viewPoint, PointF leftTopPoint, PointF rightBottomPoint, RectF viewDisplayRect) {
        float changeY;
        changeY = convertPageViewPoint(0, e2.getY(), viewPoint).y;
        if (e2.getY() > viewDisplayRect.bottom) {
            rightBottomPoint.y = changeY;
        } else {
            leftTopPoint.y = changeY;
        }
    }

    private void onControlRight(MotionEvent e2, PointF viewPoint, PointF leftTopPoint, PointF rightBottomPoint, RectF viewDisplayRect) {
        float changeX;
        changeX = convertPageViewPoint(e2.getX(), 0, viewPoint).x;
        if (e2.getX() < viewDisplayRect.left) {
            leftTopPoint.x = changeX;
        } else {
            rightBottomPoint.x = changeX;
        }
    }

    private void onControlLeft(MotionEvent e2, PointF viewPoint, PointF leftTopPoint, PointF rightBottomPoint, RectF viewDisplayRect) {
        float changeX;
        changeX = convertPageViewPoint(e2.getX(), 0, viewPoint).x;
        if (e2.getX() > viewDisplayRect.right) {
            rightBottomPoint.x = changeX;
        } else {
            leftTopPoint.x = changeX;
        }
    }

    private PointF convertPageViewPoint(float x, float y, PointF viewPoint) {
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(new PointF(x, y), viewPoint, mPageIndex);
        return viewPoint;
    }


    private void translateEditableViewRect(float distanceX, float distanceY) {
        RectF currentPageRectF = new RectF(mPdfViewCtrl.getPageViewRect(mPageIndex));
        RectF invalidRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, invalidRectF, mPageIndex);

        RectF displayRectF = new RectF();
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalidRectF, displayRectF, mPageIndex);
        currentPageRectF.inset(mDeltaXY, mDeltaXY);
        displayRectF.offset(distanceX, distanceY);
        float newLeft;
        float newTop;
        if (displayRectF.left >= currentPageRectF.left && displayRectF.left <= currentPageRectF.right - displayRectF.width()) {
            invalidRectF.offset(distanceX, 0);
        } else {
            if (distanceX > 0) {
                newLeft = currentPageRectF.right - displayRectF.width();
            } else {
                newLeft = currentPageRectF.left;
            }
            PointF devicePoint = new PointF(newLeft, 0);
            PointF viewPoint = new PointF();
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devicePoint, viewPoint, mPageIndex);
            invalidRectF.offsetTo(viewPoint.x, invalidRectF.top);
        }
        if (displayRectF.top >= currentPageRectF.top && displayRectF.top <= currentPageRectF.bottom - displayRectF.height()) {
            invalidRectF.offset(0, distanceY);
        } else {
            if (distanceY > 0) {
                newTop = currentPageRectF.bottom - displayRectF.height();
            } else {
                newTop = currentPageRectF.top;
            }
            PointF devicePoint = new PointF(0, newTop);
            PointF viewPoint = new PointF();
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devicePoint, viewPoint, mPageIndex);
            invalidRectF.offsetTo(invalidRectF.left, viewPoint.y);
        }
        mPdfViewCtrl.convertPageViewRectToPdfRect(invalidRectF, mInvalidateDocRectF, mPageIndex);

        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalidRectF, invalidRectF, mPageIndex);
        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalidRectF));
    }

    public boolean onTouchUpOrCancel(int pageIndex) {
        boolean handled = false;
        if (!(mExtensionManager.getDocumentManager().getCurrentAnnot() instanceof Link))
            return false;
        Link link = (Link) mExtensionManager.getDocumentManager().getCurrentAnnot();
        if (mIsTouchingView) {
            modifyLink(pageIndex, link);
            handled = true;
        } else if (mTouchPointPosition != -1) {
            modifyLink(pageIndex, link);
            handled = true;
        }
        if (handled && !mAnnotMenu.isShowing()) {
            RectF menuRect = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mInvalidateDocRectF, menuRect, pageIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(menuRect, menuRect, pageIndex);
            mAnnotMenu.show(menuRect);
        }
        return handled;
    }

    public void modifyLink(int pageIndex, Link link) {
        RectF moveRectF = new RectF(mInvalidateDocRectF);
        modifyLink(link, pageIndex + 1, null, moveRectF, null);
    }

    public void modifyLink(final Link link, final int pageNumber, String uri, RectF rectF, final Event.Callback callback) {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc == null) {
            return;
        }
        try {
            final LinkModifyUndoItem undoItem = new LinkModifyUndoItem(mPdfViewCtrl);
            if (TextUtils.isEmpty(link.getUniqueID())) {
                link.setUniqueID(AppDmUtil.randomUUID(null));
            }
            final int pageIndex = pageNumber - 1;
            if (rectF == null) {
                undoItem.mUndoAction = link.getAction();
                if (uri == null) {
                    GotoAction action = new GotoAction(Action.create(doc, Action.e_TypeGoto));
                    action.setDestination(Destination.createFitPage(doc, pageIndex));
                    link.setAction(action);
                } else {
                    URIAction action = new URIAction(Action.create(doc, Action.e_TypeURI));
                    action.setURI(uri);
                    link.setAction(action);
                }
                undoItem.mRedoAction = link.getAction();
                undoItem.mAction = link.getAction();
            } else {
                undoItem.mUndoBbox = AppUtil.toRectF(link.getRect());
                undoItem.mRedoBbox = rectF;
                undoItem.mBBox = rectF;
            }
            undoItem.mNM = link.getUniqueID();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();

            final RectF oldRectF = AppUtil.toRectF(link.getRect());
            LinkEvent event = new LinkEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, link, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        DocumentManager documentManager = mExtensionManager.getDocumentManager();
                        try {
                            documentManager.onAnnotModified(mPdfViewCtrl.getDoc().getPage(mPageIndex), link);
                            documentManager.addUndoItem(undoItem);
                            reLoadLinkInfo(link, mPageIndex);

                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                RectF newRect = AppUtil.toRectF(link.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(newRect, newRect, pageIndex);
                                mPdfViewCtrl.convertPdfRectToPageViewRect(oldRectF, oldRectF, pageIndex);
                                newRect.union(oldRectF);
                                mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(newRect));
                            }
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                    if (callback != null) {
                        callback.result(event, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
            UIToast.getInstance(mContext).show(R.string.rv_panel_annot_failed, Toast.LENGTH_SHORT);
        }
    }


    public void onScaleBegin() {
        mIsScalingPage = true;
    }


    public void onScaleEnd() {
        mIsScalingPage = false;
    }

}
