/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.link;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.PointF;
import android.view.ScaleGestureDetector;
import android.view.View;

import androidx.fragment.app.FragmentActivity;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.BaseDialogFragment;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.thumbnail.SelectPageFragment;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;

public class LinkModule implements Module {
    private LinkAnnotHandler mAnnotHandler;

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private LinkToolHandler mLinkToolHandler;
    private SelectPageFragment mSelectPageFragment;

    public LinkModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_LINK;
    }

    public LinkAnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    public void clear() {
        mAnnotHandler.isDocClosed = true;
        mAnnotHandler.clear();
    }

    @Override
    public boolean loadModule() {
        mAnnotHandler = new LinkAnnotHandler(mContext, mPdfViewCtrl);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mAnnotHandler.getPageEventListener());
        mPdfViewCtrl.registerRecoveryEventListener(mRecoveryListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerScaleGestureEventListener(mSaleGestureEventListener);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager extensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
            extensionsManager.registerAnnotHandler(mAnnotHandler);
            mLinkToolHandler = new LinkToolHandler(mContext, mPdfViewCtrl);
            extensionsManager.registerToolHandler(mLinkToolHandler);
            if (!extensionsManager.getConfig().permissions.disableLink)
                extensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_EDIT, ToolConstants.Link, mLinkToolHandler.getToolSupply());
            extensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
            extensionsManager.registerToolHandlerChangedListener(mHandlerChangedListener);
            extensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            extensionsManager.registerThemeEventListener(mThemeEventListener);
            extensionsManager.registerModule(this);
        }
        return true;
    }

    private void initSelectPageFragment() {
        if (mSelectPageFragment != null) return;
        mSelectPageFragment = new SelectPageFragment();
        mSelectPageFragment.init(mPdfViewCtrl, AppResource.getString(mContext, R.string.rv_page_present_thumbnail), true);
        mSelectPageFragment.setDismissListener(new BaseDialogFragment.DismissListener() {
            @Override
            public void onDismiss() {
                mSelectPageFragment = null;
            }
        });
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mAnnotHandler.getPageEventListener());
        mPdfViewCtrl.unregisterRecoveryEventListener(mRecoveryListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterScaleGestureEventListener(mSaleGestureEventListener);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterToolHandler(mLinkToolHandler);
            uiExtensionsManager.unregisterToolHandlerChangedListener(mHandlerChangedListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
        }
        return true;
    }


    PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocOpened(PDFDoc pdfDoc, int i) {
            if (i != Constants.e_ErrSuccess) return;
            mAnnotHandler.isDocClosed = false;

            Destination destination = mAnnotHandler.getDestination();
            try {
                if (destination != null) {
                    if (!destination.isEmpty()) {
                        PointF destPt = AppUtil.getDestinationPoint(mPdfViewCtrl.getDoc(), destination);
                        PointF devicePt = new PointF();
                        int _pageIndex = destination.getPageIndex(mPdfViewCtrl.getDoc());
                        if (_pageIndex > mPdfViewCtrl.getPageCount() -1) return;

                        if (!mPdfViewCtrl.convertPdfPtToPageViewPt(destPt, devicePt, _pageIndex)) {
                            devicePt.set(0, 0);
                        }
                        mPdfViewCtrl.gotoPage(_pageIndex, devicePt.x, devicePt.y);
                    } else {
                        mPdfViewCtrl.gotoPage(0, 0, 0);
                    }
                    mAnnotHandler.setDestination(null);
                }
            } catch (PDFException e) {
                if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                    mPdfViewCtrl.recoverForOOM();
                }
                e.printStackTrace();
            }
        }

        @Override
        public void onDocWillClose(PDFDoc pdfDoc) {
            mAnnotHandler.isDocClosed = true;
            mAnnotHandler.clear();
        }

    };

    PDFViewCtrl.IRecoveryEventListener mRecoveryListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            mAnnotHandler.isDocClosed = true;
            mAnnotHandler.clear();
        }

        @Override
        public void onRecovered() {
        }
    };

    private final IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotWillApply(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            try {
                mAnnotHandler.searchPageLinks(page.getIndex());
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    public void showSelectPages(int pageNumber, View.OnClickListener rightClickListener) {
        if (mUiExtensionsManager == null) {
            return;
        }
        Activity activity = ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
        if (activity == null) {
            return;
        }

        if (!(activity instanceof FragmentActivity)) {
            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }

        FragmentActivity act = (FragmentActivity) activity;
        SelectPageFragment contentFragment = (SelectPageFragment) act.getSupportFragmentManager().findFragmentByTag(SelectPageFragment.class.getSimpleName());
        if (contentFragment == null) {
            initSelectPageFragment();
            contentFragment = mSelectPageFragment;
        }
        mSelectPageFragment.setSelectedPagePosition(pageNumber - 1);
        mSelectPageFragment.setOnTitleRightClickListener(rightClickListener);
        AppDialogManager.getInstance().showAllowManager(contentFragment, act.getSupportFragmentManager(), SelectPageFragment.class.getSimpleName(), null);
    }

    public void dismissSelectPageFragment() {
        if (mSelectPageFragment != null && !mSelectPageFragment.isHidden()) {
            mSelectPageFragment.dismiss();
        }
    }

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mLinkToolHandler != null) {
                mLinkToolHandler.dismissDialog();
            }
            if (mSelectPageFragment != null) {
                mSelectPageFragment.dismiss();
                mSelectPageFragment = null;
            }
        }
    };
    private final UIExtensionsManager.ToolHandlerChangedListener mHandlerChangedListener = new UIExtensionsManager.ToolHandlerChangedListener() {
        @Override
        public void onToolHandlerChanged(ToolHandler oldToolHandler, ToolHandler newToolHandler) {
            if (newToolHandler == mLinkToolHandler && oldToolHandler != mLinkToolHandler) {
                mAnnotHandler.setLinkEditMode(true, mPdfViewCtrl.getCurrentPage());
            } else if (newToolHandler != mLinkToolHandler && oldToolHandler == mLinkToolHandler) {
                mAnnotHandler.setLinkEditMode(false, mPdfViewCtrl.getCurrentPage());
            }
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mLinkToolHandler != null) {
                mLinkToolHandler.onConfigurationChanged(newConfig);
            }
        }
    };

    private final PDFViewCtrl.IScaleGestureEventListener mSaleGestureEventListener = new PDFViewCtrl.IScaleGestureEventListener() {
        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            if (mAnnotHandler != null) {
                mAnnotHandler.onScaleBegin();
            }
            return false;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
            if (mAnnotHandler != null) {
                mAnnotHandler.onScaleEnd();
            }
        }
    };

    public LinkToolHandler getToolHandler(){
        return mLinkToolHandler;
    }

    public boolean jumpToActionDest(Action action) {
        return mAnnotHandler.executeAction(null, action);
    }

}
