/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.link;


import static com.foxit.uiextensions.utils.AppAnnotUtil.ANNOT_SELECT_TOLERANCE;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.Patterns;
import android.view.GestureDetector;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.sdk.pdf.actions.GotoAction;
import com.foxit.sdk.pdf.actions.URIAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Link;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import androidx.appcompat.view.menu.MenuView;

public class LinkToolHandler implements ToolHandler {

    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUIExtensionsManager;
    private final GestureDetector mGestureDetector;

    private boolean mIsContinue;
    private int mPageIndex = 0;
    private final Paint mPaint;
    private final RectF mSelectRectF = new RectF();
    private final Rect mInvalidateRect = new Rect();
    private int mPageView_W;
    private int mPageView_H;
    private boolean mIsSelectingArea;
    private UIMatchDialog mDialog;
    private LinkAnnotHandler mLinkHandler;

    public LinkToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mGestureDetector = new GestureDetector(context, new GestureDetector.SimpleOnGestureListener() {
            @Override
            public void onLongPress(MotionEvent e) {
            }

            @Override
            public boolean onSingleTapConfirmed(MotionEvent e) {
                LinkAnnotHandler linkAnnotHandler = getLinkHandler();
                if (linkAnnotHandler == null) {
                    return false;
                }
                linkAnnotHandler.setLinkEditable(true);
                PointF pdfPoint = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, mPageIndex, e);
                Annot annot = mUIExtensionsManager.getDocumentManager().getCurrentAnnot();
                if (annot == null || annot.isEmpty()) {
                    PDFPage page;
                    try {
                        page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
                        if (page != null && !page.isEmpty()) {
                            Annot annotation = getLink(page, pdfPoint);
                            if (annotation == null)
                                annotation = page.getAnnotAtDevicePoint(AppUtil.toFxPointF(pdfPoint), ANNOT_SELECT_TOLERANCE,
                                        AppUtil.toMatrix2D(mPdfViewCtrl.getDisplayMatrix(mPageIndex)));
                            annot = AppAnnotUtil.createAnnot(annotation);
                        }
                    } catch (PDFException pdfException) {
                        pdfException.printStackTrace();
                        return false;
                    }
                }
                return linkAnnotHandler.onSingleTapConfirmed(mPageIndex, e, annot);
            }

            @Override
            public boolean onDown(MotionEvent e) {
                mPageView_W = mPdfViewCtrl.getPageViewWidth(mPageIndex);
                mPageView_H = mPdfViewCtrl.getPageViewHeight(mPageIndex);
                mLinkHandler = getLinkHandler();
                boolean handled = false;
                if (mLinkHandler != null) {
                    handled = mLinkHandler.onDown(mPageIndex, e);
                }
                mDownCaptured = true;
                if (handled) return true;
                return !mUIExtensionsManager.defaultTouchEvent(mPageIndex, e);
            }

            @Override
            public boolean onScroll(MotionEvent e1, MotionEvent e2, float distanceX, float distanceY) {
                boolean handled = false;
                if (mLinkHandler != null) {
                    handled = mLinkHandler.onScroll(e2, distanceX, distanceY);
                }
                if (handled) return true;
                mSelectRectF.left = Math.min(e1.getX(), e2.getX());
                mSelectRectF.top = Math.min(e1.getY(), e2.getY());
                mSelectRectF.right = mSelectRectF.left + Math.abs(e1.getX() - e2.getX());
                mSelectRectF.bottom = mSelectRectF.top + Math.abs(e1.getY() - e2.getY());
                mPdfViewCtrl.convertDisplayViewRectToPageViewRect(mSelectRectF, mSelectRectF, mPageIndex);
                if (mSelectRectF.left < 0 )
                    mSelectRectF.left = 0;
                if (mSelectRectF.top < 0 )
                    mSelectRectF.top = 0;
                if (mSelectRectF.right > mPageView_W )
                    mSelectRectF.right = mPageView_W;
                if (mSelectRectF.bottom > mPageView_H )
                    mSelectRectF.bottom = mPageView_H;

                mSelectRectF.roundOut(mInvalidateRect);
                mPdfViewCtrl.invalidate(mInvalidateRect);
                mIsSelectingArea = true;
                if (mUIExtensionsManager.getDocumentManager().getCurrentAnnot() instanceof Link) {
                    mUIExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
                return true;
            }
        });
        mPaint = new Paint();
        mPaint.setStyle(Paint.Style.FILL);
    }

    private Annot getLink(PDFPage page, PointF pageViewPoint) {
        Annot annot = null;
        PointF pdfPoint = new PointF();
        if (page != null && !page.isEmpty()) {
            try {
                int count = page.getAnnotCount();
                for (int i = 0; i < count; i++) {
                    annot = page.getAnnot(i);
                    if (annot.getType() != Annot.e_Link) {
                        annot = null;
                        continue;
                    }
                    mPdfViewCtrl.convertPageViewPtToPdfPt(pageViewPoint, pdfPoint, page.getIndex());
                    if (!annot.getRect().contains(pdfPoint.x, pdfPoint.y)) {
                        annot = null;
                    } else {
                        break;
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return annot;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_LINK;
    }

    @Override
    public void onActivate() {
        mUIExtensionsManager.registerThemeEventListener(mThemeEventListener);
    }

    @Override
    public void onDeactivate() {
        mUIExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        if (mDialog != null){
            mDialog.release();
        }
        dismissDialog();
        mPageNumberEditText = null;
        mWebLinkEditText = null;
        mSelectPageView = null;
        mSelectWebView = null;
        mSelectThumbnail = null;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        return onLinkTouchEvent(pageIndex, motionEvent);
    }

    boolean mDownCaptured = false;

    private boolean onLinkTouchEvent(int pageIndex, MotionEvent ev) {
        mPageIndex = pageIndex;
        if(ev.getActionMasked() == MotionEvent.ACTION_POINTER_DOWN){
            MotionEvent downEvent = MotionEvent.obtain(ev);
            downEvent.setAction(MotionEvent.ACTION_DOWN);
            boolean handled = mPdfViewCtrl.defaultTouchEvent(downEvent);
            downEvent.recycle();
            mDownCaptured = false;
            return handled;
        }
        if (ev.getPointerCount() > 1) {
            return mPdfViewCtrl.defaultTouchEvent(ev);
        }
        boolean handled;
        if (!mDownCaptured && (ev.getActionMasked() ==  MotionEvent.ACTION_CANCEL
                || ev.getActionMasked() ==  MotionEvent.ACTION_UP )){
            handled = false;
        }else {
            handled = mGestureDetector.onTouchEvent(ev);
        }
        if (!handled) {
            switch (ev.getActionMasked()) {
                case MotionEvent.ACTION_CANCEL:
                case MotionEvent.ACTION_UP:
                    if (!mDownCaptured) return mPdfViewCtrl.defaultTouchEvent(ev);
                    if (mLinkHandler != null) {
                        handled = mLinkHandler.onTouchUpOrCancel(pageIndex);
                    }
                    if (handled) return true;
                    if (mIsSelectingArea) {
                        showSettingDialog(null);
                        mIsSelectingArea = false;
                        mPdfViewCtrl.invalidate(mInvalidateRect);
                    }
                    return true;
            }
        }
        return handled;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        return true;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinue;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinue = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mIsSelectingArea && mPageIndex == pageIndex) {
            canvas.save();
            mPaint.setColor((int) mUIExtensionsManager.getLinkHighlightColor());
            canvas.drawRect(mSelectRectF, mPaint);
            canvas.restore();
        }
    }

    private IToolSupply mToolSupply;

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new LinkToolSupply(mContext);
        return mToolSupply;
    }

    private class LinkToolSupply extends ToolSupplyImpl {

        public LinkToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.edit_tool_link;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            return null;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_LINK;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            if (itemBean.toolItem.isSelected()) {
                setContinueAddAnnot(false);
                UIToast.getInstance(mContext).showCustomToast(AppResource.getString(mContext, R.string.edit_add_link_toast_content));
                mUIExtensionsManager.setCurrentToolHandler(LinkToolHandler.this);
            } else {
                if (mUIExtensionsManager.getCurrentToolHandler() == LinkToolHandler.this) {
                    mUIExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
        }

        @Override
        public PropertyBar getPropertyBar() {
            return null;
        }

    }

    private int mLinkPageNumber = -1;
    private String mLinkUri = null;

    public void dismissDialog() {
        if (mDialog != null) {
            mDialog.dismiss();
            mDialog = null;
        }
    }

    public void showSettingDialog(final MatchDialog.DismissListener dismissListener) {
        Activity activity = mPdfViewCtrl.getAttachedActivity();
        if (activity == null) return;
        if (mDialog == null) {
            UIMatchDialog dialog = new UIMatchDialog(activity);
            dialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
            dialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
            dialog.setTitle(AppResource.getString(mContext, R.string.settings));
            dialog.getTitleBar().setMiddleButtonCenter(true);
            dialog.setRightButtonVisible(View.VISIBLE);
            View mSettingView = createSettingView();
            dialog.setContentView(mSettingView);
            dialog.setBackgroundColor(ThemeConfig.getInstance(mContext).getB1());
            dialog.setListener(new MatchDialog.DialogListener() {
                @Override
                public void onResult(long btType) {
                }

                @Override
                public void onBackClick() {
                    if (mDialog != null)
                        mDialog.dismiss();
                }

                @Override
                public void onTitleRightButtonClick() {
                    Annot link = mUIExtensionsManager.getDocumentManager().getCurrentAnnot();
                    if (mSelectPageView.getVisibility() == View.VISIBLE) {
                        if (getPageNumber() > mPdfViewCtrl.getPageCount() || getPageNumber() < 1) {
                            UIToast.getInstance(mContext).show(R.string.invalid_page_count, Toast.LENGTH_SHORT);
                            updatePageNumber(mPageIndex + 1);
                            return;
                        }
                        if (link == null) {
                            addLink(getPageNumber(), null);
                        } else if (link instanceof Link) {
                            modifyLink((Link) link, getPageNumber(), null);
                        }
                    } else {
                        if (!Patterns.WEB_URL.matcher(getUri()).matches()) {
                            UIToast.getInstance(mContext).show(R.string.invalid_link_format, Toast.LENGTH_SHORT);
                            return;
                        }
                        if (link == null) {
                            addLink(getPageNumber(), getUri());
                        } else if (link instanceof Link) {
                            modifyLink((Link) link, getPageNumber(), getUri());
                        }
                    }
                }
            });
            mDialog = dialog;
        }

        mDialog.setOnDLDismissListener(new MatchDialog.DismissListener() {
            @Override
            public void onDismiss() {
                if (dismissListener != null)
                    dismissListener.onDismiss();
            }
        });

        int pageNumber = mPageIndex + 1;
        mLinkPageNumber = -1;
        mLinkUri = null;
        String uri = "";
        Annot link = mUIExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (link instanceof Link) {
            try {
                Action action = ((Link) link).getAction();
                if (action.getType() == Action.e_TypeGoto) {
                    Destination destination = new GotoAction(action).getDestination();
                    pageNumber = destination.getPageIndex(mPdfViewCtrl.getDoc()) + 1;
                    mLinkPageNumber = pageNumber;
                } else if (action.getType() == Action.e_TypeURI) {
                    uri = new URIAction(action).getURI();
                    mLinkUri = uri;
                }

            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        updatePageNumber(pageNumber);
        mWebLinkEditText.setText(uri);
        mWebLinkEditText.setSelection(uri.length());
        if (!TextUtils.isEmpty(uri)) {
            switchToPageMode(false);
        }
        mDialog.showDialog();
    }

    private void modifyLink(final Link link, int pageNumber, String uri) {
        if (getLinkHandler() != null) {
            getLinkHandler().modifyLink(link, pageNumber, uri, null, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        if (mDialog != null)
                            mDialog.dismiss();
                    }
                }
            });
        }
    }

    private void addLink(int pageNumber, String uri) {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc == null) {
            return;
        }
        try {
            final LinkAddUndoItem undoItem = new LinkAddUndoItem(mPdfViewCtrl);
            PDFPage page = doc.getPage(mPageIndex);
            mPdfViewCtrl.convertPageViewRectToPdfRect(mSelectRectF, mSelectRectF, mPageIndex);
            final Link link = (Link) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Link, AppUtil.toFxRectF(mSelectRectF)));
            BorderInfo borderInfo = new BorderInfo();
            borderInfo.setWidth(0);
            link.setBorderInfo(borderInfo);
            if (TextUtils.isEmpty(link.getUniqueID()))
                link.setUniqueID(AppDmUtil.randomUUID(null));
            RectFArray rect_array = new RectFArray();
            rect_array.add(link.getRect());
            if (uri == null) {
                GotoAction action = new GotoAction(Action.create(doc, Action.e_TypeGoto));
                action.setDestination(Destination.createFitPage(doc, pageNumber - 1));
                link.setAction(action);
            } else {
                URIAction action = new URIAction(Action.create(doc, Action.e_TypeURI));
                action.setURI(uri);
                link.setAction(action);
            }
            undoItem.mPageIndex = mPageIndex;
            undoItem.mRectFArray = rect_array;
            undoItem.mNM = link.getUniqueID();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mAction = link.getAction();
            undoItem.mBorderInfo = link.getBorderInfo();
            LinkEvent event = new LinkEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, link, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
                        try {
                            uiExtensionsManager.getDocumentManager().onAnnotAdded(mPdfViewCtrl.getDoc().getPage(mPageIndex), link);
                            uiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            LinkAnnotHandler handler = getLinkHandler();
                            if (handler != null) {
                                handler.reLoadLinkInfo(link, mPageIndex);
                            }
                            mPdfViewCtrl.invalidate();
                            if (!isContinueAddAnnot()) {
                                mUIExtensionsManager.setCurrentToolHandler(null);
                            }
                            if (mDialog != null)
                                mDialog.dismiss();
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
            UIToast.getInstance(mContext).show(R.string.rv_panel_annot_failed, Toast.LENGTH_SHORT);
        }
    }

    public void AI_addLink(int pageIndex, int pageNumber, String uri, RectF rect) {
        mSelectRectF.set(rect);
        mPageIndex = pageIndex;
        addLink(pageNumber, uri);
    }
    private View mSelectPageView;
    private View mSelectWebView;
    private View mSelectThumbnail;
    private EditText mPageNumberEditText;
    private EditText mWebLinkEditText;
    private Drawable mClearDrawable;

    private View createSettingView() {
        View view = View.inflate(mContext, R.layout.edit_link_settings_layout, null);
        mSelectPageView = view.findViewById(R.id.select_page);
        mSelectWebView = view.findViewById(R.id.select_web);
        mSelectThumbnail = view.findViewById(R.id.select_from_thumbnail);
        mSelectThumbnail.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                final LinkModule module = (LinkModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_LINK);
                if (module == null) return;
                int pageNumber = getPageNumber();
                module.showSelectPages(pageNumber, new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (v.getTag() == null) return;
                        int[] selectIndexes = (int[]) v.getTag();
                        int pageNumber = selectIndexes[0] + 1;
                        updatePageNumber(pageNumber);
                        module.dismissSelectPageFragment();
                    }
                });
            }
        });
        mPageNumberEditText = view.findViewById(R.id.page_number_edit_text);
        mWebLinkEditText = view.findViewById(R.id.web_link_edit_text);
        mWebLinkEditText.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mWebLinkEditText.setHintTextColor(AppResource.getColor(mContext, R.color.t2));
        mPageNumberEditText.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mPageNumberEditText.setHintTextColor(AppResource.getColor(mContext, R.color.t2));
        if (AppDisplay.isPad()) { // SDKRD-9313
            mPageNumberEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            mWebLinkEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        initEdiTextView(mPageNumberEditText);
        initEdiTextView(mWebLinkEditText);
        resetSettingViewTint();
        return view;
    }

    public int getPageNumber() {
        int pageNumber = mPageIndex;
        try {
            pageNumber = Integer.parseInt(mPageNumberEditText.getText().toString());
        } catch (Exception ignored) {
        }
        return pageNumber;
    }

    private String getUri() {
        return mWebLinkEditText.getText().toString();
    }

    private void updatePageNumber(int pageNumber) {
        switchToPageMode(true);
        mPageNumberEditText.setText(String.format(AppResource.getString(mContext, R.string.number_format_text), pageNumber));
        mPageNumberEditText.setSelection(mPageNumberEditText.getText().length());
    }

    private void updateLinkURI(String uri) {
        switchToPageMode(false);
        mWebLinkEditText.setText(String.format(AppResource.getString(mContext, R.string.link_to_format_text), uri));
        mWebLinkEditText.setSelection(mPageNumberEditText.getText().length());
    }


    @SuppressLint("ClickableViewAccessibility")
    private void initEdiTextView(final EditText editText) {
        mClearDrawable = editText.getCompoundDrawables()[2];
        if (mClearDrawable == null) {
            mClearDrawable = AppResource.getDrawable(mContext, R.drawable.rd_clear_search);
        }
        mClearDrawable.setBounds(0, 0, mClearDrawable.getIntrinsicWidth(), mClearDrawable.getIntrinsicHeight());
        editText.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getAction() == MotionEvent.ACTION_UP) {
                    switchToPageMode(mPageNumberEditText == v);
                    if (editText.getCompoundDrawables()[2] != null) {
                        boolean isTouchRight = event.getX() > (editText.getWidth() - editText.getTotalPaddingRight()) && (event.getX() < ((editText.getWidth() - editText.getPaddingRight())));
                        if (isTouchRight) {
                            editText.setText("");
                        }
                    }
                }
                return false;
            }
        });

        editText.addTextChangedListener(new TextWatcher() {

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                validateInput(editText);
            }

            @Override
            public void afterTextChanged(Editable s) {
            }
        });
    }

    public void validateInput(EditText editText) {
        boolean isLinkToPage = mSelectPageView.getVisibility() == View.VISIBLE;
        boolean isPageNumberEditText = mPageNumberEditText == editText;
        if (isLinkToPage && !isPageNumberEditText || !isLinkToPage && isPageNumberEditText) return;
        String text = editText.getText().toString();
        if (!TextUtils.isEmpty(text)) {
            mDialog.setRightButtonEnable(true);
            editText.setCompoundDrawables(null, null, mClearDrawable, null);
        } else {
            mDialog.setRightButtonEnable(false);
            editText.setCompoundDrawables(null, null, null, null);
        }
        if (isLinkToPage && mLinkPageNumber != -1 && mLinkPageNumber == getPageNumber()) {
            mDialog.setRightButtonEnable(false);
        } else if (!isLinkToPage && !TextUtils.isEmpty(mLinkUri) && mLinkUri.contentEquals(getUri())) {
            mDialog.setRightButtonEnable(false);
        }
    }

    private void resetSettingViewTint() {
        ThemeUtil.setBackgroundTintList(mSelectPageView, ThemeUtil.getPrimaryTextColor(mContext));
        ThemeUtil.setBackgroundTintList(mSelectWebView, ThemeUtil.getPrimaryTextColor(mContext));
        ThemeUtil.setBackgroundTintList(mSelectThumbnail, ThemeUtil.getPrimaryTextColor(mContext));
    }

    private void switchToPageMode(boolean isPage) {
        mSelectPageView.setVisibility(isPage ? View.VISIBLE : View.INVISIBLE);
        mSelectWebView.setVisibility(!isPage ? View.VISIBLE : View.INVISIBLE);
        validateInput(isPage ? mPageNumberEditText : mWebLinkEditText);
        if (isPage) {
            mPageNumberEditText.requestFocus();
            mWebLinkEditText.setCompoundDrawables(null, null, null, null);
        } else {
            mWebLinkEditText.requestFocus();
            mPageNumberEditText.setCompoundDrawables(null, null, null, null);
        }
    }

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mDialog != null) {
                mDialog.themeColorChanged();
                resetSettingViewTint();
            }
        }
    };

    private LinkAnnotHandler getLinkHandler() {
        if (mLinkHandler == null && mUIExtensionsManager.getAnnotHandlerByType(Annot.e_Link) != null) {
            mLinkHandler = (LinkAnnotHandler) mUIExtensionsManager.getAnnotHandlerByType(Annot.e_Link);
        }
        return mLinkHandler;
    }

    public void onConfigurationChanged(Configuration newConfig) {
        if (mDialog != null && mDialog.isShowing()) {
            mDialog.resetWH();
            mDialog.showDialog();
        }
    }

    public void deleteCurLink() {
        Annot curAnnot = mUIExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (curAnnot == null) return;
        mLinkHandler.removeAnnot(curAnnot, true, null);
    }

    public void editCurLink() {
        Annot curAnnot = mUIExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (curAnnot == null) return;
        showSettingDialog(null);
    }
}
