/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.link;


import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Link;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

public abstract class LinkUndoItem extends AnnotUndoItem {
    RectFArray mRectFArray;
    int mBorderColor;
    PDFDictionary mPDFDict;
    Action mAction;
    BorderInfo mBorderInfo;
}

class LinkAddUndoItem extends LinkUndoItem {

    public LinkAddUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo() {
        LinkDeleteUndoItem undoItem = new LinkDeleteUndoItem(mPdfViewCtrl);
        undoItem.mNM = mNM;
        undoItem.mPageIndex = mPageIndex;

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Link)) {
                return false;
            }

            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null, false);
            }

            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotWillDelete(page, annot);

            final int objNum = annot.getDict().getObjNum();
            LinkEvent deleteEvent = new LinkEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Link) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(deleteEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                        uiExtensionsManager.getDocumentManager().onAnnotDeleted(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            RectF deviceRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, mPageIndex);

                            AnnotHandler annotHandler = uiExtensionsManager.getAnnotHandlerByType(Annot.e_Link);
                            if (annotHandler instanceof LinkAnnotHandler) {
                                LinkAnnotHandler handler = (LinkAnnotHandler) annotHandler;
                                handler.removeLinkInfo(objNum, mPageIndex);
                            }
                            mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(deviceRectF));
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }

    @Override
    public boolean redo() {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Link link = (Link) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Link, mRectFArray.getAt(0)));
            link.setAction(mAction);
            link.setBorderInfo(mBorderInfo);
            link.setUniqueID(mNM);
            LinkEvent addEvent = new LinkEvent(EditAnnotEvent.EVENTTYPE_ADD, this, link, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, link);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link) instanceof LinkAnnotHandler) {
                                LinkAnnotHandler handler = (LinkAnnotHandler) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link);
                                handler.reLoadLinkInfo(link, mPageIndex);
                            }
                            mPdfViewCtrl.invalidate();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }
}

class LinkModifyUndoItem extends LinkUndoItem {
    public RectF mUndoBbox;
    public int mUndoBorderColor;
    public RectF mRedoBbox;
    public int mRedoBorderColor;
    public Action mUndoAction;
    public Action mRedoAction;

    public LinkModifyUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo() {
        return modifyLink(mUndoBorderColor, mUndoBbox, mUndoAction);
    }

    @Override
    public boolean redo() {
        return modifyLink(mRedoBorderColor, mRedoBbox, mRedoAction);
    }

    private boolean modifyLink(int borderColor, RectF bbox, Action action) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Link)) {
                return false;
            }

//            final RectF oldBbox = AppUtil.toRectF(annot.getRect());
            mBorderColor = borderColor;
            if (bbox != null)
                mBBox = new RectF(bbox);
            mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            if (action != null)
                mAction = action;
            LinkEvent modifyEvent = new LinkEvent(EditAnnotEvent.EVENTTYPE_MODIFY, this, (Link) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(modifyEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                        }
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotModified(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link) instanceof LinkAnnotHandler) {
                                LinkAnnotHandler handler = (LinkAnnotHandler) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link);
                                handler.reLoadLinkInfo(annot, mPageIndex);
                            }
                            mPdfViewCtrl.invalidate();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

}

class LinkDeleteUndoItem extends LinkUndoItem {

    public LinkDeleteUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo(Event.Callback callback) {
        LinkAddUndoItem undoItem = new LinkAddUndoItem(mPdfViewCtrl);
        undoItem.mPageIndex = mPageIndex;
        undoItem.mNM = mNM;
        undoItem.mModifiedDate = mModifiedDate;
        undoItem.mBBox = new RectF(mBBox);
        undoItem.mRectFArray = mRectFArray;
        undoItem.mBorderColor = mBorderColor;
        undoItem.mPDFDict = mPDFDict;

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Link link = (Link) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Link, mRectFArray.getAt(0)));
            link.setUniqueID(mNM);
            LinkEvent addEvent = new LinkEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, link, mPdfViewCtrl);
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (uiExtensionsManager.getDocumentManager().isMultipleSelectAnnots()) {
                if (callback != null) {
                    callback.result(addEvent, true);
                }
                return true;
            }
            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, link);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            try {
                                RectF annotRectF = AppUtil.toRectF(link.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, mPageIndex);
                                if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link) instanceof LinkAnnotHandler) {
                                    LinkAnnotHandler handler = (LinkAnnotHandler) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link);
                                    handler.reLoadLinkInfo(link, mPageIndex);
                                }
                                mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(annotRectF));
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean redo(Event.Callback callback) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
            final Annot annot = documentManager.getAnnot(page, mNM);
            if (!(annot instanceof Link)) {
                return false;
            }

            if (annot == documentManager.getCurrentAnnot()) {
                documentManager.setCurrentAnnot(null, false);
            }

            documentManager.onAnnotWillDelete(page, annot);
            LinkEvent deleteEvent = new LinkEvent(EditAnnotEvent.EVENTTYPE_DELETE, this, (Link) annot, mPdfViewCtrl);
            if (documentManager.isMultipleSelectAnnots()) {
                if (callback != null) {
                    callback.result(deleteEvent, true);
                }
                return true;
            }

            final int objNum = annot.getDict().getObjNum();
            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            EditAnnotTask task = new EditAnnotTask(deleteEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotDeleted(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            RectF deviceRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, mPageIndex);
                            if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link) instanceof LinkAnnotHandler) {
                                LinkAnnotHandler handler = (LinkAnnotHandler) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_Link);
                                handler.removeLinkInfo(objNum, mPageIndex);
                            }
                            mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(deviceRectF));
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean undo() {
        return undo(null);
    }

    @Override
    public boolean redo() {
        return redo(null);
    }
}