/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia;


import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.res.Configuration;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.media.MediaPlayer;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.SeekBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.graphics.drawable.DrawableCompat;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

public class AudioPlayView extends LinearLayout implements IAudioPlayView {

    private SeekBar mediaPlayer_seekbar;
    private ImageView mediaPlayer_dragView;
    private TextView mediaPlayer_pastTime;
    private TextView mediaPlayer_totalTime;
    private ImageView mediaPlayer_playbtn;
    private ImageView mediaPlayer_slowbtn;
    private ImageView mediaPlayer_speedbtn;
    private ImageView mediaPlayer_stopbtn;
    private View mediaPlayer_view;

    private AudioPlayService mAudioPlayer;
    private boolean isBindService = false;

    private float mMinPlayerPositionY = -1;
    private float mMaxPlayerPositionY = -1;
    private final Context mContext;
    private PDFViewCtrl mViewCtrl;
    private int mRepeatCount = 1;

    public AudioPlayView(@NonNull Context context) {
        super(context);
        mContext = context;
        init();
    }

    public AudioPlayView(@NonNull Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mContext = context;
        mViewCtrl = pdfViewCtrl;
        init();
    }

    private void init() {
        mediaPlayer_view = View.inflate(mContext, R.layout.audio_play_layout, this);
        mediaPlayer_view.setVisibility(View.INVISIBLE);
        setMoveBound();

        mediaPlayer_view.setOnTouchListener(new OnTouchListener() {
            private float dX;
            private float dY;
            private float playerMaxPositionX;
            private float playerMaxPositionY;
            private float x;
            private float y;

            @Override
            public boolean onTouch(View view, MotionEvent event) {
                switch (event.getActionMasked()) {
                    case MotionEvent.ACTION_DOWN:
                        dX = view.getX() - event.getRawX();
                        dY = view.getY() - event.getRawY();
                        playerMaxPositionX = ((View) getParent()).getWidth() - view.getWidth();
                        playerMaxPositionY = mMaxPlayerPositionY - view.getHeight();
                        return true;
                    case MotionEvent.ACTION_MOVE:
                        x = Math.max(Math.min(event.getRawX() + dX, playerMaxPositionX), 0);
                        if (x != view.getX())
                            view.setX(x);
                        y = Math.max(Math.min(event.getRawY() + dY, playerMaxPositionY), mMinPlayerPositionY);
                        if (y != view.getY())
                            view.setY(y);
                        return true;
                    default:
                        break;
                }
                return false;
            }
        });

        mediaPlayer_dragView = mediaPlayer_view.findViewById(R.id.audio_drag_view);
        mediaPlayer_pastTime = mediaPlayer_view.findViewById(R.id.audio_play_pasttime);
        mediaPlayer_totalTime = mediaPlayer_view.findViewById(R.id.audio_play_totaltime);
        mediaPlayer_playbtn = mediaPlayer_view.findViewById(R.id.audio_play_pause);
        mediaPlayer_slowbtn = mediaPlayer_view.findViewById(R.id.audio_play_slow);
        mediaPlayer_speedbtn = mediaPlayer_view.findViewById(R.id.audio_play_speed);
        mediaPlayer_stopbtn = mediaPlayer_view.findViewById(R.id.audio_play_stop);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mediaPlayer_dragView.setForceDarkAllowed(false);
            mediaPlayer_slowbtn.setForceDarkAllowed(false);
            mediaPlayer_playbtn.setForceDarkAllowed(false);
            mediaPlayer_stopbtn.setForceDarkAllowed(false);
            mediaPlayer_speedbtn.setForceDarkAllowed(false);
        }
        //init play status
        mediaPlayer_seekbar = mediaPlayer_view.findViewById(R.id.audio_play_seekbar);
        mediaPlayer_seekbar.setOnSeekBarChangeListener(mSeekBarChangedListener);
        updateSeekBarProgressDrawable();

        mediaPlayer_playbtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                if (mAudioPlayer.isPlaying()) {
                    mediaPlayer_playbtn.setImageResource(R.drawable.ic_audio_player_play);
                    mAudioPlayer.pause();
                } else {
                    mediaPlayer_playbtn.setImageResource(R.drawable.ic_audio_player_pause);
                    mAudioPlayer.seekTo(mAudioPlayer.getCurrentPosition());
                    try {
                        mAudioPlayer.start();
                        mHandler.sendEmptyMessage(CHANGE_UI_STATE);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        });

        mediaPlayer_stopbtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                release();
            }
        });

        mediaPlayer_slowbtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                int pos = mAudioPlayer.getCurrentPosition();
                pos -= 5000;

                if (pos < 0) {
                    pos = 0;
                }
                mAudioPlayer.seekTo(pos);
                mediaPlayer_seekbar.setProgress(pos);
                mediaPlayer_pastTime.setText(timeParse(pos));
            }
        });

        mediaPlayer_speedbtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {

                int pos = mAudioPlayer.getCurrentPosition();
                pos += 5000;

                if (pos > mAudioPlayer.getDuration()) {
                    pos = mAudioPlayer.getDuration();
                }
                mAudioPlayer.seekTo(pos);
                mediaPlayer_seekbar.setProgress(pos);
                mediaPlayer_pastTime.setText(timeParse(pos));
            }
        });
    }

    private void setMoveBound() {
        if (mViewCtrl == null) return;
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mViewCtrl.getUIExtensionsManager();
        mMinPlayerPositionY = ((MainFrame) uiExtensionsManager.getMainFrame()).getTopBarHeight();
        if (AppDisplay.isPad()) {
            mMaxPlayerPositionY = AppDisplay.getRawScreenHeight() - AppDisplay.getRealNavBarHeight();
        } else {
            mMaxPlayerPositionY = AppDisplay.getRawScreenHeight() - AppDisplay.getNavBarHeight()
                    - ((MainFrame) uiExtensionsManager.getMainFrame()).getBottomBarHeight();
        }
    }

    private void updateSeekBarProgressDrawable() {
        Drawable drawable = mediaPlayer_seekbar.getProgressDrawable();
        if (drawable != null) {
            drawable = ((LayerDrawable) drawable).getDrawable(1);
            DrawableCompat.setTintMode(drawable, PorterDuff.Mode.SRC_IN);
            DrawableCompat.setTint(drawable, ThemeConfig.getInstance(mContext).getPrimaryColor());
        }
    }

    private boolean fromUser;
    private final SeekBar.OnSeekBarChangeListener mSeekBarChangedListener = new SeekBar.OnSeekBarChangeListener() {
        @Override
        public void onProgressChanged(SeekBar seekBar, int i, boolean b) {
            if (fromUser) {
                mAudioPlayer.seekTo(mediaPlayer_seekbar.getProgress());
                mediaPlayer_pastTime.setText(timeParse(mAudioPlayer.getCurrentPosition()));
            }
        }

        @Override
        public void onStartTrackingTouch(SeekBar seekBar) {
            fromUser = true;
        }

        @Override
        public void onStopTrackingTouch(SeekBar seekBar) {
            fromUser = false;
        }
    };

    private String timeParse(long duration) {
        String time = "";
        long minute = duration / 60000;
        long seconds = duration % 60000;
        long second = Math.round((float) seconds / 1000);
        if (minute < 10) {
            time += "0";
        }
        time += minute + ":";
        if (second < 10) {
            time += "0";
        }
        time += second;
        return time;
    }

    private void startAudioService() {
        if (isBindService) {
            stopAudioService();
        }
        Intent intent = new Intent(mContext, AudioPlayService.class);
        mContext.startService(intent);
        isBindService = true;
        mContext.bindService(intent, mServiceConnection, Context.BIND_AUTO_CREATE);
    }

    private void stopAudioService() {
        Intent intent = new Intent(mContext, AudioPlayService.class);
        if (isBindService) {
            mContext.unbindService(mServiceConnection);
            isBindService = false;
        }
        mContext.stopService(intent);
        ((UIExtensionsManager)mViewCtrl.getUIExtensionsManager()).getPhoneStateBroadCaseReceiver().unregisterListener(mPhoneStateListener);
        mAudioPlayer = null;
    }

    private PhoneStateBroadCastReceiver.IPhoneStateListener mPhoneStateListener = new PhoneStateBroadCastReceiver.IPhoneStateListener() {
        @Override
        public void onOutGoingCall() {
            if (mAudioPlayer != null) {
                mAudioPlayer.mute(true);
            }
        }

        @Override
        public void onIncomingCall() {
            if (mAudioPlayer != null) {
                mAudioPlayer.mute(true);
            }
        }

        @Override
        public void onCallIdle() {
            if (mAudioPlayer != null) {
                mAudioPlayer.mute(false);
            }
        }
    };

    private final ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            mAudioPlayer = ((AudioPlayService.AudioPlayBinder) service).getService();
            ((UIExtensionsManager)mViewCtrl.getUIExtensionsManager()).getPhoneStateBroadCaseReceiver().registerListener(mPhoneStateListener);
            mHandler.sendEmptyMessage(SERVICE_CONNECTED);
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            mHandler.sendEmptyMessage(SERVICE_DISCONNECTED);
            ((UIExtensionsManager)mViewCtrl.getUIExtensionsManager()).getPhoneStateBroadCaseReceiver().unregisterListener(mPhoneStateListener);
        }
    };

    private static final int START_SERVICE = 111;
    private static final int STOP_SERVICE = 222;
    private static final int SERVICE_CONNECTED = 333;
    private static final int SERVICE_DISCONNECTED = 444;
    private static final int CHANGE_UI_STATE = 555;
    private String mPlayFilePath;

    private final Handler mHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            switch (msg.what) {
                case START_SERVICE:
                    startAudioService();
                    break;
                case STOP_SERVICE:
                    stopAudioService();
                    break;
                case SERVICE_CONNECTED:
                    preparePlayer();
                    break;
                case SERVICE_DISCONNECTED:
                    break;
                case CHANGE_UI_STATE:
                    changeUIState();
                    break;
                default:
                    break;
            }
        }
    };

    private OnPreparedListener mPreparedListener;

    public void startPlayAudio(String path, OnPreparedListener preparedListener) {
        mPreparedListener = preparedListener;
        mPlayFilePath = path;
        if (isBindService) {
            mAudioPlayer.stop();
            mediaPlayer_view.setVisibility(View.INVISIBLE);
            preparePlayer();
        } else {
            mHandler.sendEmptyMessage(START_SERVICE);
        }
    }

    @Override
    public void release() {
        if (mediaPlayer_view == null) return;
        mediaPlayer_view.setVisibility(View.INVISIBLE);
        if (mAudioPlayer != null) {
            mAudioPlayer.stop();
            mHandler.sendEmptyMessage(STOP_SERVICE);
        }
    }

    @Override
    public void startPlay(final String path) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mViewCtrl.getUIExtensionsManager();
        uiExtensionsManager.requestPhoneStatePermission(AppResource.getString(mContext, R.string.audio_fun_name), new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                showProgressDlg();
                startPlayAudio(path, new AudioPlayView.OnPreparedListener() {
                    @Override
                    public void onPrepared(boolean success, MediaPlayer mp) {
                        dismissProgressDlg();
                        if (success) {
                            if (mViewCtrl == null) return;

                            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mViewCtrl.getUIExtensionsManager();
                            if (uiExtensionsManager.getState() == ReadStateConfig.STATE_TTS) {
                                TTSModule ttsModule = (TTSModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS);
                                if (ttsModule != null)
                                    ttsModule.onKeyBack();
                            }
                        } else {
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_document_open_failed));
                        }
                    }
                });
            }
        });
    }

    @Override
    public boolean isPlayIng() {
        if (mAudioPlayer != null) {
            return mAudioPlayer.isPlaying();
        }
        return false;
    }

    @Override
    public void updateLayout() {
        if (View.VISIBLE == mediaPlayer_view.getVisibility()) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    changeUIState();
                }
            }, 100);
        }
    }

    @Override
    public void onThemeColorChanged() {
        updateSeekBarProgressDrawable();
    }

    @Override
    public View getPlayView() {
        return this;
    }

    private void preparePlayer() {
        if (mediaPlayer_view == null) return;
        try {
            mAudioPlayer.prepare(mPlayFilePath, new AudioPlayView.OnPreparedListener() {
                @Override
                public void onPrepared(boolean success, MediaPlayer mp) {
                    if (mPreparedListener != null) {
                        mPreparedListener.onPrepared(success, mp);
                    }
                    if (success) {
                        mediaPlayer_view.setVisibility(View.VISIBLE);
                        mediaPlayer_seekbar.setMax(mAudioPlayer.getDuration());
                        mediaPlayer_seekbar.setProgress(0);
                        mediaPlayer_pastTime.setText(timeParse(0));

                        //set totalTime
                        mediaPlayer_totalTime.setText(timeParse(mAudioPlayer.getDuration()));
                        //play file
                        mAudioPlayer.start();
                        mHandler.sendEmptyMessage(CHANGE_UI_STATE);
                    } else {
                        mediaPlayer_view.setVisibility(View.INVISIBLE);
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
            mediaPlayer_view.setVisibility(View.INVISIBLE);
        }
    }

    public void changeUIState() {
        if (mAudioPlayer == null || (mediaPlayer_view != null && !mediaPlayer_view.isShown()))
            return;

        if (!mAudioPlayer.isPlaying()) {
            mediaPlayer_playbtn.setImageResource(R.drawable.ic_audio_player_play);
            if (mAudioPlayer.getCurrentPosition() + 1000 > mAudioPlayer.getDuration()) {
                //end
                mediaPlayer_pastTime.setText(timeParse(mAudioPlayer.getDuration()));
                try {
                    mRepeatCount--;

                    mAudioPlayer.seekTo(0);
                    mediaPlayer_pastTime.setText(timeParse(0));
                    mediaPlayer_seekbar.setProgress(0);
                    if (mRepeatCount <= 0) {
                        mAudioPlayer.pause();
                    } else {
                        mAudioPlayer.start();
                        mHandler.sendEmptyMessage(CHANGE_UI_STATE);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } else {
            mediaPlayer_playbtn.setImageResource(R.drawable.ic_audio_player_pause);
            mediaPlayer_pastTime.setText(timeParse(mAudioPlayer.getCurrentPosition()));
            mediaPlayer_seekbar.setProgress(mAudioPlayer.getCurrentPosition());

            mHandler.sendEmptyMessageDelayed(CHANGE_UI_STATE, 100);
        }
    }

    public View getContentView() {
        return mediaPlayer_view;
    }

    public interface OnPreparedListener {
        void onPrepared(boolean success, MediaPlayer mp);
    }

    private ViewGroup mParenView;

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        if (mParenView == null) {
            mParenView = (ViewGroup) getParent();
        }
        final int rootWidth = mParenView.getWidth();
        final int rootHeight = mParenView.getHeight();

        final Rect rect = new Rect();
        getGlobalVisibleRect(rect);

        final float scaleT = (float) rect.top / rootHeight;
        final float scaleB = (float) (rootHeight - rect.bottom) / rootHeight;

        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
            @Override
            public void run() {
                setMoveBound();

                int newRootWidth = mParenView.getWidth();
                int newRootHeight = mParenView.getHeight();
                float scaleL = (float) rect.left / (rootWidth - rect.width());
                int newLeft = (int) (scaleL * (newRootWidth - rect.width()));

                float y;
                if (scaleT < scaleB) {
                    y = newRootHeight * scaleT;
                } else {
                    y = (newRootHeight - newRootHeight * scaleB) - getHeight();
                }
                mediaPlayer_view.setY(Math.min(mMaxPlayerPositionY - getHeight(), y));
                mediaPlayer_view.setX(newLeft);
            }
        }, 300);
    }

    private FxProgressDialog mProgressDlg;

    private void showProgressDlg() {
        if (mViewCtrl == null) return;

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mViewCtrl.getUIExtensionsManager();
        if (mProgressDlg == null && uiExtensionsManager.getAttachedActivity() != null) {
            mProgressDlg = new FxProgressDialog(uiExtensionsManager.getAttachedActivity(), AppResource.getString(mContext, R.string.fx_string_opening));
        }

        if (mProgressDlg != null && !mProgressDlg.isShowing()) {
            mProgressDlg.show();
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            mProgressDlg.dismiss();
            mProgressDlg = null;
        }
    }

    @Override
    public void setRepeatCount(int count) { //The repeat count to be set. 0 means repeat forever.
        // This value should not be negative value.
        if (count == 0) {
            mRepeatCount = Integer.MAX_VALUE;
        } else {
            mRepeatCount = count;
        }
    }

}
