/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.screen.image;


import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Canvas;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IPolicyEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.multimedia.screen.multimedia.MultimediaManager;
import com.foxit.uiextensions.config.uisettings.annotations.annots.ImageConfig;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;

import java.util.List;

public class PDFImageModule implements Module, PropertyBar.PropertyChangeListener {

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private PDFImageAnnotHandler mAnnotHandler;
    private PDFImageToolHandler mToolHandler;

    public PDFImageModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this.mContext = context;
        this.mPdfViewCtrl = pdfViewCtrl;
        this.mUiExtensionsManager = uiExtensionsManager;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    @Override
    public String getName() {
        return MODULE_NAME_IMAGE;
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new PDFImageToolHandler(mContext, mPdfViewCtrl);
        mToolHandler.setPropertyChangeListener(this);
        mToolHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));

        mAnnotHandler = new PDFImageAnnotHandler(mContext, mPdfViewCtrl);
        mAnnotHandler.setPropertyChangeListener(this);
        mAnnotHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));

        int opacity = 100;
        int rotation = Constants.e_Rotation0;
        int[] rotations = new int[]{0, 90, 180, 270};
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerToolHandler(mToolHandler);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.registerPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);

            ImageConfig config = ((UIExtensionsManager) mUiExtensionsManager).getConfig()
                    .uiSettings.annotations.image;
            opacity = (int) (config.opacity * 100);
            int temp = config.rotation;
            for (int i = 0; i < rotations.length; i++) {
                if (rotations[i] == temp) {
                    rotation = i;
                    break;
                }
            }

            uiExtensionsManager.getToolsManager()
                    .addToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Image_Annotation, mToolHandler.getToolSupply());
            ((MainFrame) uiExtensionsManager.getMainFrame()).resetToolbar(ToolbarItemConfig.ITEM_COMMENT_TAB);
        }

        mToolHandler.setRotation(rotation);
        mToolHandler.setOpacity(opacity);
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        List<Integer> toolItems = ((UIExtensionsManager) mUiExtensionsManager).getToolsManager().getToolItems(UIToolBaseBar.USAGE_TYPE_COMMENT);
        if (toolItems.contains(ToolConstants.Image_Annotation)) {
            ((UIExtensionsManager) mUiExtensionsManager).getToolsManager().removeToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Image_Annotation, mToolHandler.getToolSupply());

            ((MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame()).resetToolbar(ToolbarItemConfig.ITEM_COMMENT_TAB);
        }

        mAnnotHandler.removePropertyBarListener();
        mToolHandler.removePropertyBarListener();

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterToolHandler(mToolHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterAnnotHandler(mAnnotHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterPolicyEventListener(mPolicyEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLifecycleListener(mLifecycleEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterConfigurationChangedListener(mConfigurationChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }
        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }

            if (mAnnotHandler.getPropertyBar() != null && mAnnotHandler.getPropertyBar().isShowing()) {
                mAnnotHandler.getPropertyBar().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private boolean needHandleByTool(UIExtensionsManager uiExtensionsManager, AnnotHandler annotHandler) {
        return uiExtensionsManager.getCurrentToolHandler() == mToolHandler
                && (mAnnotHandler != annotHandler || mAnnotHandler.getAnnotMenu().isShowing());
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_OPACITY) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.setOpacity(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onOpacityValueChanged(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.setOpacity(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_ROTATION) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.setRotation(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onRotationValueChanged(value == 0 ? value : 4 - value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.setRotation(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
    }

    @Override
    public void onValueChanged(long property, String value) {
    }

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() instanceof PDFImageToolHandler) {
                mToolHandler.onDrawForControls(pageIndex, canvas);
            }
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            MultimediaManager.getInstance().clear();
        }

    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mAnnotHandler.updateTheme();
            mToolHandler.updateTheme();
        }
    };

    private final IPolicyEventListener mPolicyEventListener = new IPolicyEventListener() {
        @Override
        public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() instanceof PDFImageToolHandler) {
                mToolHandler.onRequestPermissionsResult(requestCode, permissions, grantResults);
            }
        }
    };

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() instanceof PDFImageToolHandler) {
                mToolHandler.onActivityResult(act, requestCode, resultCode, data);
            }
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler() instanceof PDFImageToolHandler) {
                mToolHandler.onConfigurationChanged(newConfig);
            }
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            if (property == PropertyBar.PROPERTY_OPACITY) {
                mToolHandler.getImageInfo().setOpacity((int) value);
            } else if (property == PropertyBar.PROPERTY_ROTATION) {
                mToolHandler.getImageInfo().setRotation((int) value);
            }
        }

        @Override
        public int getToolType() {
            return ToolConstants.Image_Annotation;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_COMMENT_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
            mToolHandler.resetPropertyBar();
        }
    };

}
