/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.sound;


import android.app.Activity;
import android.content.Context;
import android.graphics.Canvas;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.multimedia.AudioPlayView;
import com.foxit.uiextensions.annots.multimedia.IAudioPlayView;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

public class SoundModule implements Module {
    private final Context mContext;
    private final PDFViewCtrl mPDFViewCtrl;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private SoundAnnotHandler mAnnotHandler;
    private IAudioPlayView mAudioPlayView;

    public SoundModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this.mContext = context;
        mPDFViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return MODULE_NAME_SOUND;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            mAnnotHandler = new SoundAnnotHandler(this, mContext, mPDFViewCtrl);

            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        }
        mPDFViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPDFViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        }
        mPDFViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPDFViewCtrl.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            if (mAudioPlayView != null)
                mAudioPlayView.release();
            mAnnotHandler.release();
        }

    };

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onResume(Activity act) {
            if (mAudioPlayView != null)
                mAudioPlayView.updateLayout();
        }

        @Override
        public void onHiddenChanged(boolean hidden) {
            if (!hidden) {
                if (mAudioPlayView != null)
                    mAudioPlayView.updateLayout();
            }
        }

        @Override
        public void onDestroy(Activity act) {
            if (mAudioPlayView != null)
                mAudioPlayView.release();
            mAnnotHandler.release();
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mAudioPlayView != null) {
                mAudioPlayView.onThemeColorChanged();
            }
        }
    };

    public IAudioPlayView createPlayView() {
        if (mAudioPlayView == null) {
            mAudioPlayView = new AudioPlayView(mContext, mPDFViewCtrl);

            UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            MainFrame mainFrame = (MainFrame) extensionsManager.getMainFrame();
            if (mainFrame != null) {
                RelativeLayout parent = (RelativeLayout) extensionsManager.getRootView().getParent();
                RelativeLayout.LayoutParams lp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
                lp.addRule(RelativeLayout.CENTER_IN_PARENT);
                lp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                if (AppDisplay.isPad()) {
                    lp.setMargins(0, 0, 0, (int) AppResource.getDimension(mContext, R.dimen.ux_toolbar_height_pad) + AppDisplay.dp2px(16));
                } else {
                    lp.setMargins(0, 0, 0, AppDisplay.dp2px(110));
                }
                parent.addView(mAudioPlayView.getPlayView(), lp);
            }
        }
        return mAudioPlayView;
    }

    public IAudioPlayView getAudioPlayView() {
        return mAudioPlayView;
    }

}
