/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.redaction;


import android.text.TextUtils;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.Popup;
import com.foxit.sdk.pdf.annots.Redact;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.List;

public class RedactEvent extends EditAnnotEvent {

    public RedactEvent(int eventType, RedactUndoItem undoItem, Redact redact, PDFViewCtrl pdfViewCtrl) {
        mType = eventType;
        mUndoItem = undoItem;
        mAnnot = redact;
        mPdfViewCtrl = pdfViewCtrl;
    }

    public RedactEvent(int eventType, RedactUndoItem undoItem, List<Annot> redactList, PDFViewCtrl pdfViewCtrl) {
        mType = eventType;
        mUndoItem = undoItem;
        mAnnotList = redactList;
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean add() {
        boolean result = false;
        if (notEmptyList()) {
            return batchAdd();
        }
        try {
            result = addAnnot(mAnnot, mUndoItem);
            if (result)
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return result;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return result;
    }

    @Override
    public boolean modify() {
        if (mAnnot == null || !(mAnnot instanceof Redact)) {
            return false;
        }

        try {
            Redact annot = (Redact) mAnnot;
            RedactModifyUndoItem undoItem = (RedactModifyUndoItem) mUndoItem;

            // overlaytext
            DefaultAppearance da = annot.getDefaultAppearance();
            Font font = da.getFont();
            String overlayText = annot.getOverlayText();
            if (!TextUtils.isEmpty(overlayText) || font != null && !font.isEmpty()) {
                da.setFlags(undoItem.mDaFlags);
                da.setText_color(undoItem.mTextColor);
                da.setFont(undoItem.mFont);
                da.setText_size(undoItem.mFontSize);
                annot.setDefaultAppearance(da);
                annot.setOverlayText(undoItem.mOverlayText);
            }
            annot.move(AppUtil.toFxRectF(mUndoItem.mBBox));
            annot.setModifiedDateTime(AppDmUtil.currentDateToDocumentDate());
            if (undoItem.mContents != null) {
                annot.setContent(undoItem.mContents);
            }
            annot.setBorderColor(undoItem.mBorderColor);
            annot.setApplyFillColor(undoItem.mApplyFillColor);
            annot.resetAppearanceStream();
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean delete() {
        boolean result = false;
        if (notEmptyList()) {
            return batchDelete();
        }
        try {
            result = removeAnnot(mAnnot,false);
            if (result)
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return result;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return result;
    }

    private boolean removeAnnot(Annot annot,boolean cleanPopup) throws PDFException {
        if (!(annot instanceof Redact)) {
            return false;
        }
        PDFPage page = annot.getPage();
        if (cleanPopup){
            Popup popup = ((Markup)annot).getPopup();
            if (!popup.isEmpty()){
                annot.getPage().removeAnnot(popup);
            }
        }
        ((Markup) annot).removeAllReplies();
        return page.removeAnnot(annot);
    }

    private boolean addAnnot(Annot annot, AnnotUndoItem undo) throws PDFException {
        if (!(annot instanceof Redact)) {
            return false;
        }
        RedactUndoItem undoItem = (RedactUndoItem) undo;
        Redact redact = (Redact) annot;
        if (undoItem.mPDFDict != null) {
            boolean reset = AppAnnotUtil.resetPDFDict(redact, undoItem.mPDFDict);
            if (reset) {
                if (undoItem.mReplys != null)
                    undoItem.mReplys.addReply(redact, undoItem.mReplys);
                return true;
            }
        }

        if (undoItem.mQuadPointsArray != null && undoItem.mQuadPointsArray.getSize() > 0) {
            redact.setQuadPoints(undoItem.mQuadPointsArray);
        }
        if (undoItem.mContents != null) {
            redact.setContent(undoItem.mContents);
        }
        if (undoItem.mCreationDate != null && AppDmUtil.isValidDateTime(undoItem.mCreationDate)) {
            redact.setCreationDateTime(undoItem.mCreationDate);
        }
        if (undoItem.mModifiedDate != null && AppDmUtil.isValidDateTime(undoItem.mModifiedDate)) {
            redact.setModifiedDateTime(undoItem.mModifiedDate);
        }
        if (undoItem.mAuthor != null) {
            redact.setTitle(undoItem.mAuthor);
        }
        if (undoItem.mSubject != null) {
            redact.setSubject(undoItem.mSubject);
        }
        if (undoItem.mOverlayText != null) {
            DefaultAppearance da = new DefaultAppearance();
            da.set(undoItem.mDaFlags, undoItem.mFont, undoItem.mFontSize, undoItem.mTextColor);
            redact.setDefaultAppearance(da);
            redact.setOverlayText(undoItem.mOverlayText);
        }
        redact.setFlags(undoItem.mFlags);
        redact.setBorderColor(undoItem.mBorderColor);
        if (undoItem.mFillColor != 0)
            redact.setFillColor(undoItem.mFillColor);
        redact.setApplyFillColor(undoItem.mApplyFillColor);
        redact.setUniqueID(undoItem.mNM);
        if (undoItem.mReplys != null)
            undoItem.mReplys.addReply(redact, undoItem.mReplys);
        redact.resetAppearanceStream();
        return true;
    }

    private boolean notEmptyList() {
        return mAnnotList != null && !mAnnotList.isEmpty();
    }

    private boolean batchAdd() {
        boolean result;
        boolean needSetModified = false;
        boolean succeeded;
        try {
            int size = mAnnotList.size();
            result = true;
            for (int i = 0; i < size; i++) {
                succeeded = addAnnot(mAnnotList.get(i), mUndoItem.mUndoItemList.get(i));
                if (!needSetModified) {
                    needSetModified = succeeded;
                }
                result = result && succeeded;

            }
            if (needSetModified)
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(needSetModified);
        } catch (PDFException e) {
            result = false;
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return result;
    }

    private boolean batchDelete() {
        boolean result;
        boolean needSetModified = false;
        boolean succeeded;
        try {
            result = true;
            for (Annot annot : mAnnotList) {
                succeeded = removeAnnot(annot,true);
                if (!needSetModified) {
                    needSetModified = succeeded;
                }
                result = result && succeeded;
            }
            if (needSetModified)
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
        } catch (PDFException e) {
            e.printStackTrace();
            result = false;
        }
        return result;
    }
}
