/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.redaction;


import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.addon.Redaction;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.sdk.pdf.annots.Redact;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.List;

public abstract class RedactUndoItem extends AnnotUndoItem {
    QuadPointsArray mQuadPointsArray;

    RectFArray mRectFArray;
    Font mFont;
    float mFontSize;
    int mTextColor;
    int mBorderColor;
    int mFillColor;
    int mApplyFillColor;
    int mDaFlags;
    String mOverlayText;
    PDFDictionary mPDFDict;
}

class RedactAddUndoItem extends RedactUndoItem {

    public RedactAddUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo() {
        if (notEmptyUndoList()) {
            return batchUndo();
        }
        RedactDeleteUndoItem undoItem = new RedactDeleteUndoItem(mPdfViewCtrl);
        undoItem.mNM = mNM;
        undoItem.mPageIndex = mPageIndex;

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Redact)) {
                return false;
            }

            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null, false);
            }

            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotWillDelete(page, annot);

            RedactEvent deleteEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Redact) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(deleteEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotDeleted(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            RectF deviceRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, mPageIndex);
                            mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(deviceRectF));
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }

    @Override
    public boolean redo() {
        try {
            if (notEmptyUndoList()) {
                return batchRedo();
            }
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Redaction redaction = new Redaction(mPdfViewCtrl.getDoc());
            final Redact annot = redaction.markRedactAnnot(page, mRectFArray);

            RedactEvent addEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_ADD, this, annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            try {
                                RectF annotRect = AppUtil.toRectF(annot.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, mPageIndex);
                                mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(annotRect));
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean batchUndo() {
        try {
            RedactDeleteUndoItem undoItem = new RedactDeleteUndoItem(mPdfViewCtrl);
            RedactDeleteUndoItem deleteUndoItem;
            RedactAddUndoItem addUndoItem;
            undoItem.mUndoItemList = new ArrayList<>();
            List<Annot> annotList = new ArrayList<>();
            final List<RectF> rectFList = new ArrayList<>();
            int pageIndex;
            String mNM;
            Annot annot;
            for (int i = 0; i < mUndoItemList.size(); i++) {
                addUndoItem = (RedactAddUndoItem) mUndoItemList.get(i);
                deleteUndoItem = new RedactDeleteUndoItem(mPdfViewCtrl);
                deleteUndoItem.mNM = addUndoItem.mNM;
                deleteUndoItem.mPageIndex = addUndoItem.mPageIndex;
                undoItem.mUndoItemList.add(deleteUndoItem);
                pageIndex = addUndoItem.mPageIndex;
                mNM = addUndoItem.mNM;
                annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(pageIndex, mNM);
                annotList.add(annot);
                rectFList.add(AppUtil.toRectF(annot.getRect()));
            }
            RedactEvent deleteEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, annotList, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(deleteEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        int size = rectFList.size();
                        int pageIndex;
                        for (int i = 0; i < size; i++) {
                            pageIndex = mUndoItemList.get(i).mPageIndex;
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotsDeleted(mPdfViewCtrl.getDoc());
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                RectF deviceRectF = new RectF();
                                mPdfViewCtrl.convertPdfRectToPageViewRect(rectFList.get(i), deviceRectF, pageIndex);
                                mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(deviceRectF));
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    private boolean batchRedo() {
        try {
            Redaction redaction = new Redaction(mPdfViewCtrl.getDoc());
            PDFPage page;
            RedactAddUndoItem undoItem;
            List<Annot> redactList = new ArrayList<>();
            final List<RectF> rectFList = new ArrayList<>();
            Redact redact;
            for (AnnotUndoItem item : mUndoItemList) {
                undoItem = (RedactAddUndoItem) item;
                page = mPdfViewCtrl.getDoc().getPage(item.mPageIndex);
                redact = redaction.markRedactAnnot(page, undoItem.mRectFArray);
                redactList.add(redact);
                rectFList.add(AppUtil.toRectF(redact.getRect()));
            }
            RedactEvent addEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_ADD, this, redactList, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        int size = rectFList.size();
                        int pageIndex;
                        for (int i = 0; i < size; i++) {
                            pageIndex = mUndoItemList.get(i).mPageIndex;
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotsAdded(mPdfViewCtrl.getDoc());
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                RectF deviceRectF = new RectF();
                                mPdfViewCtrl.convertPdfRectToPageViewRect(rectFList.get(i), deviceRectF, pageIndex);
                                mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(deviceRectF));
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }
}

class RedactModifyUndoItem extends RedactUndoItem {
    public int mUndoApplyFillColor;
    public String mUndoContents;
    public String mUndoOverlayText;
    public Font mUndoFont;
    public float mUndoFontSize;
    public int mUndoTextColor;
    public int mUndoDaFlags;
    public RectF mUndoBbox;
    public int mUndoBorderColor;

    public int mRedoApplyFillColor;
    public String mRedoContents;
    public String mRedoOverlayText;
    public Font mRedoFont;
    public float mRedoFontSize;
    public int mRedoTextColor;
    public int mRedoDaFlags;
    public RectF mRedoBbox;
    public int mRedoBorderColor;

    public RedactModifyUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo() {
        return modifyAnnot(mUndoApplyFillColor, mUndoBorderColor, mUndoDaFlags,
                mUndoTextColor, mUndoFontSize, mUndoFont, mUndoContents, mUndoOverlayText, mUndoBbox);
    }

    @Override
    public boolean redo() {
        return modifyAnnot(mRedoApplyFillColor, mRedoBorderColor, mRedoDaFlags,
                mRedoTextColor, mRedoFontSize, mRedoFont, mRedoContents, mRedoOverlayText, mRedoBbox);
    }

    private boolean modifyAnnot(int applyFillColor, int borderColor, int flags, int textColor, float fontSize,
                                Font font, String content, String overlayText, RectF bbox) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            final Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getAnnot(page, mNM);
            if (!(annot instanceof Redact)) {
                return false;
            }

            final RectF oldBbox = AppUtil.toRectF(annot.getRect());
            mApplyFillColor = applyFillColor;
            mBorderColor = borderColor;
            mTextColor = textColor;
            mFontSize = fontSize;
            mFont = font;
            mOverlayText = overlayText;
            mBBox = new RectF(bbox);
            mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            mContents = content;
            mDaFlags = flags;

            RedactEvent modifyEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_MODIFY, this, (Redact) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(modifyEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                        }

                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotModified(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            try {
                                RectF annotRect = AppUtil.toRectF(annot.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, mPageIndex);
                                mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(annotRect));

                                mPdfViewCtrl.convertPdfRectToPageViewRect(oldBbox, oldBbox, mPageIndex);
                                mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(oldBbox));
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

}

class RedactDeleteUndoItem extends RedactUndoItem {

    public RedactDeleteUndoItem(PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean undo(Event.Callback callback) {
        RedactAddUndoItem undoItem = new RedactAddUndoItem(mPdfViewCtrl);
        undoItem.mPageIndex = mPageIndex;
        undoItem.mNM = mNM;
        undoItem.mAuthor = mAuthor;
        undoItem.mFlags = mFlags;
        undoItem.mSubject = mSubject;
        undoItem.mCreationDate = mCreationDate;
        undoItem.mModifiedDate = mModifiedDate;
        undoItem.mBBox = new RectF(mBBox);
        undoItem.mColor = mColor;
        undoItem.mContents = mContents;
        undoItem.mReplys = mReplys;

        undoItem.mRectFArray = mRectFArray;
        undoItem.mQuadPointsArray = new QuadPointsArray(mQuadPointsArray);
        undoItem.mApplyFillColor = mApplyFillColor;
        undoItem.mFillColor = mFillColor;
        undoItem.mBorderColor = mBorderColor;
        undoItem.mTextColor = mTextColor;
        undoItem.mFontSize = mFontSize;
        undoItem.mFont = mFont;
        undoItem.mDaFlags = mDaFlags;
        undoItem.mOverlayText = mOverlayText;

        undoItem.mPDFDict = mPDFDict;

        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            Redaction redaction = new Redaction(mPdfViewCtrl.getDoc());
            final Redact annot = redaction.markRedactAnnot(page, mRectFArray);
            RedactEvent addEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (uiExtensionsManager.getDocumentManager().isMultipleSelectAnnots()) {
                if (callback != null) {
                    callback.result(addEvent, true);
                }
                return true;
            }
            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            try {
                                RectF annotRectF = AppUtil.toRectF(annot.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, mPageIndex);
                                mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(annotRectF));
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean redo(Event.Callback callback) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(mPageIndex);
            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
            final Annot annot = documentManager.getAnnot(page, mNM);
            if (!(annot instanceof Redact)) {
                return false;
            }

            if (annot == documentManager.getCurrentAnnot()) {
                documentManager.setCurrentAnnot(null, false);
            }

            documentManager.onAnnotWillDelete(page, annot);
            RedactEvent deleteEvent = new RedactEvent(EditAnnotEvent.EVENTTYPE_DELETE, this, (Redact) annot, mPdfViewCtrl);
            if (documentManager.isMultipleSelectAnnots()) {
                if (callback != null) {
                    callback.result(deleteEvent, true);
                }
                return true;
            }

            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            EditAnnotTask task = new EditAnnotTask(deleteEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotDeleted(page, annot);
                        if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                            RectF deviceRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, mPageIndex);
                            mPdfViewCtrl.refresh(mPageIndex, AppDmUtil.rectFToRect(deviceRectF));
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean undo() {
        return undo(null);
    }

    @Override
    public boolean redo() {
        return redo(null);
    }
}



