/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.square;

import android.content.Context;
import android.graphics.Canvas;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;

public class SquareModule implements Module, PropertyBar.PropertyChangeListener {

    private SquareAnnotHandler mAnnotHandler;
    private SquareToolHandler mToolHandler;

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public SquareModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    private boolean needHandleByTool(UIExtensionsManager uiExtensionsManager, AnnotHandler annotHandler) {
        return uiExtensionsManager.getCurrentToolHandler() == mToolHandler
                && (mAnnotHandler != annotHandler || mAnnotHandler.getAnnotMenu().isShowing());
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.changeCurrentColor(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onColorValueChanged(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentColor(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_OPACITY) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.changeCurrentOpacity(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onOpacityValueChanged(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentOpacity(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_FILL_COLOR) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.changeCurrentFillColor(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onFillColorValueChanged(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentFillColor(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_LINEWIDTH) {
            if (needHandleByTool(uiExtensionsManager, currentAnnotHandler)) {
                mToolHandler.changeCurrentThickness(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onLineWidthValueChanged(value);
            } else if (mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentThickness(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, String value) {
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_SQUARE;
    }

    @Override
    public boolean loadModule() {
        mAnnotHandler = new SquareAnnotHandler(mContext, mPdfViewCtrl);
        mToolHandler = new SquareToolHandler(mContext, mPdfViewCtrl);

        mAnnotHandler.setPropertyChangeListener(this);
        mToolHandler.setPropertyChangeListener(this);
        mAnnotHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerToolHandler(mToolHandler);
            ((UIExtensionsManager) mUiExtensionsManager).registerAnnotHandler(mAnnotHandler);
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);

            Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
            mToolHandler.changeCurrentColor(config.uiSettings.annotations.rectangle.color);
            mToolHandler.changeCurrentFillColor(config.uiSettings.annotations.rectangle.fillColor);
            mToolHandler.changeCurrentOpacity((int) (config.uiSettings.annotations.rectangle.opacity * 100));
            mToolHandler.changeCurrentThickness(config.uiSettings.annotations.rectangle.thickness);
            ((UIExtensionsManager) mUiExtensionsManager).getToolsManager()
                    .addToolItem(UIToolBaseBar.USAGE_TYPE_DRAW, ToolConstants.Rectangle, mToolHandler.getToolSupply());
        }
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mToolHandler.removePropertyListener();
        mAnnotHandler.removePropertyListener();

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterToolHandler(mToolHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterAnnotHandler(mAnnotHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }

        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        return true;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }

            if (mAnnotHandler.getPropertyBar() != null && mAnnotHandler.getPropertyBar().isShowing()) {
                mAnnotHandler.getPropertyBar().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mAnnotHandler.updateTheme();
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mToolHandler.mColor = (int) value;
            } else if (property == PropertyBar.PROPERTY_OPACITY) {
                mToolHandler.mOpacity = (int) value;
            } else if (property == PropertyBar.PROPERTY_LINEWIDTH) {
                mToolHandler.mThickness = (float) value;
            } else if (property == PropertyBar.PROPERTY_FILL_COLOR) {
                mToolHandler.mFillColor = (int) value;
            }
        }

        @Override
        public int getToolType() {
            return ToolConstants.Rectangle;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_DRAWING_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
            mToolHandler.resetPropertyBar();
        }
    };

}
