/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.RectF;
import android.os.Environment;
import android.text.TextUtils;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Image;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.AdditionalAction;
import com.foxit.sdk.pdf.actions.JavaScriptAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.IconProviderCallback;
import com.foxit.sdk.pdf.annots.Screen;
import com.foxit.sdk.pdf.annots.ShadingColor;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.annots.stamp.customstamp.BaseStampBean;
import com.foxit.uiextensions.annots.stamp.customstamp.ImageStampBean;
import com.foxit.uiextensions.annots.stamp.customstamp.TextStampBean;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.UUID;

public class DynamicStampIconProvider extends IconProviderCallback {

    private Context mContext;
    private HashMap<String, PDFDoc> mDocMap;
    private HashMap<PDFDoc, PDFPage> mDocPagePair;
    private HashMap<String, BaseStampBean> mCustomStampBean;
    private String id = UUID.randomUUID().toString();
    private String version;//"Version 8.1";
    private int pageIndex = 0;

    private static DynamicStampIconProvider instance;

    public static DynamicStampIconProvider getInstance(Context context) {
        if (instance == null) {
            instance = new DynamicStampIconProvider(context);
        }
        return instance;
    }

    private DynamicStampIconProvider(Context context) {
        mContext = context;
        mDocMap = new HashMap<String, PDFDoc>();
        mDocPagePair = new HashMap<PDFDoc, PDFPage>();
        mCustomStampBean = new HashMap<>();
        try {
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            version = pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
    }

    public void addDocMap(String key, PDFDoc pdfDoc) {
        if (key == null || key.trim().length() < 1) {
            return;
        }

        if (mDocMap.get(key) == null) {
            mDocMap.put(key, pdfDoc);
        }
    }

    public PDFDoc getDoc(String key) {
        if (key == null || key.trim().length() < 1) {
            return null;
        }
        return mDocMap.get(key);
    }

    @Override
    public void release() {
        for (PDFDoc pdfDoc : mDocMap.values()) {
            pdfDoc.delete();
        }

        mCustomStampBean.clear();
        mDocPagePair.clear();
        mDocMap.clear();
    }

    @Override
    public String getProviderID() {
        return id;
    }

    @Override
    public String getProviderVersion() {
        return version;
    }

    @Override
    public boolean hasIcon(int annotType, String iconName) {
        return Annot.e_Stamp == annotType;
    }

    @Override
    public boolean canChangeColor(int annotType, String iconName) {
        return true;
    }

    @Override
    public PDFPage getIcon(int annotType, String iconName, int color) {
        if (mDocMap == null)
            return null;

        if (mDocMap.get(iconName + annotType) == null && !addIcon(iconName)) {
            return null;
        }

        if (annotType == Annot.e_Stamp) {
            try {
                PDFDoc pdfDoc = mDocMap.get(iconName + annotType);
                if (pdfDoc == null) return null;
                if (mDocPagePair.get(pdfDoc) != null) {
                    return mDocPagePair.get(pdfDoc);
                }
                PDFPage page = pdfDoc.getPage(pageIndex);
                mDocPagePair.put(pdfDoc, page);
                return page;
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    public boolean addIcon(RectF bbox, String iconName) {
        BaseStampBean customStamp = mCustomStampBean.get(iconName);
        if (customStamp != null) {
            if (customStamp instanceof TextStampBean)
                return addTextIcon((TextStampBean) customStamp);
            else if (customStamp instanceof ImageStampBean)
                return addImageIcon(Math.abs(bbox.width()), Math.abs(bbox.height()), (ImageStampBean) customStamp);
        }
        return addIcon(iconName);
    }

    private boolean addIcon(String iconName) {
        BaseStampBean customStamp = mCustomStampBean.get(iconName);
        if (customStamp != null) {
            if (customStamp instanceof TextStampBean) {
                return addTextIcon((TextStampBean) customStamp);
            } else if (customStamp instanceof ImageStampBean) {
                RectF rectF = ((ImageStampBean) customStamp).mTempAnnotRectF;
                float width = 50;
                float height = 100;
                if (rectF != null && AppUtil.isEmptyPDFRect(rectF)) {
                    width = Math.abs(rectF.width());
                    height = Math.abs(rectF.height());
                }
                return addImageIcon(width, height, (ImageStampBean) customStamp);
            }
        }

        if (mDocMap.get(iconName + Annot.e_Stamp) == null) {
            FileOutputStream fos = null;
            try {
                int stampType = StampUtil.getStampTypeByName(iconName);

                String stampFileName = null;
                String assetsPath = null;
                if (stampType >= 0 && stampType <= 11) {
                    assetsPath = "StandardStamps/";
                    stampFileName = assetsPath + iconName + ".pdf";

                } else if (stampType >= 12 && stampType <= 16) {
                    assetsPath = "SignHere/";
                    stampFileName = assetsPath + iconName + ".pdf";
                } else if (stampType >= 17 && stampType <= 21) {
                    assetsPath = "DynamicStamps/";
                    stampFileName = assetsPath + iconName.substring(4) + ".pdf";
                }

                if (stampFileName == null) {
                    assetsPath = "StandardStamps/";
                    stampFileName = "StandardStamps/Approved.pdf";
                }

                InputStream is = mContext.getAssets().open(stampFileName);
                byte[] buffer = new byte[1 << 13];
                String path = AppFileUtil.getDiskCachePath(mContext);
                if (TextUtils.isEmpty(path)) {
                    path = Environment.getExternalStorageDirectory().getPath() + "/FoxitSDK/";
                } else if (!path.endsWith("/")) {
                    path += "/";
                }
                String dirPath = path + assetsPath;
                File dir = new File(dirPath);
                if (!dir.exists()) {
                    dir.mkdirs();
                }
                path += stampFileName;
                File file = new File(path);
                fos = new FileOutputStream(file);
                int n = 0;
                while (-1 != (n = is.read(buffer))) {
                    fos.write(buffer, 0, n);
                }

                PDFDoc pdfDoc = new PDFDoc(path);
                pdfDoc.load(null);
                mDocMap.put(iconName + Annot.e_Stamp, pdfDoc);
                is.close();
                return true;
            } catch (PDFException | IOException e) {
                e.printStackTrace();
            } finally {
                if (fos != null) {
                    try {
                        fos.flush();
                        fos.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        } else {
            return true;
        }
        return false;
    }

    private boolean addTextIcon(TextStampBean textStampBean) {
        if (textStampBean == null) return false;

        String key = String.valueOf(textStampBean.mModifiedTime);
        PDFDoc doc = mDocMap.get(key + Annot.e_Stamp);
        if (doc == null) {
            FileOutputStream fos = null;
            InputStream is = null;
            try {
                String assetsPath = "CustomStamps/";
                String stampFileName;
                if (textStampBean.mColor == StampConstants.Custom_Colors[0]) {
                    stampFileName = assetsPath + "custom_red.pdf";
                } else if (textStampBean.mColor == StampConstants.Custom_Colors[1]) {
                    stampFileName = assetsPath + "custom_green.pdf";
                } else if (textStampBean.mColor == StampConstants.Custom_Colors[2]) {
                    stampFileName = assetsPath + "custom_blue.pdf";
                } else if (textStampBean.mColor == StampConstants.Custom_Colors[3]) {
                    stampFileName = assetsPath + "custom_purple.pdf";
                } else {
                    stampFileName = assetsPath + "custom_red.pdf";
                }

                String cachePath = AppFileUtil.getDiskCachePath(mContext);
                if (TextUtils.isEmpty(cachePath)) {
                    cachePath = Environment.getExternalStorageDirectory().getPath() + "/FoxitSDK/";
                } else if (!cachePath.endsWith("/")) {
                    cachePath += "/";
                }

                String cacheFile = cachePath + stampFileName;
                File file = new File(cacheFile);
                if (!file.exists()) {
                    File dir = new File(cachePath + assetsPath);
                    if (!dir.exists()) {
                        dir.mkdirs();
                    }

                    is = mContext.getAssets().open(stampFileName);
                    byte[] buffer = new byte[1 << 13];
                    fos = new FileOutputStream(file);
                    int n;
                    while (-1 != (n = is.read(buffer))) {
                        fos.write(buffer, 0, n);
                    }
                }

                doc = new PDFDoc(cacheFile);
                int errorCode = doc.load(null);
                if (errorCode != Constants.e_ErrSuccess) {
                    return false;
                }
                mDocMap.put(String.valueOf(textStampBean.mModifiedTime) + Annot.e_Stamp, doc);
            } catch (PDFException | IOException e) {
                return false;
            } finally {
                try {
                    if (is != null) {
                        is.close();
                    }
                    if (fos != null) {
                        fos.flush();
                        fos.close();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }

        try {
            PDFPage page = doc.getPage(0);
            if (textStampBean.mShowAuthor || textStampBean.mShowDate || textStampBean.mShowTime) {
                // remove old title field -- on the center
                Widget centerWidget = new Widget(page.getAnnot(2));
                centerWidget.getField().reset();

                //title field -- on the top
                setValue(page.getAnnot(1), textStampBean.mText);

                //date field -- on the bottom
                Annot annot = page.getAnnot(0);
                Widget widget = new Widget(annot);
                AdditionalAction additionalAction = new AdditionalAction(widget.getField());
                Action action = Action.create(doc, Action.e_TypeJavaScript);
                JavaScriptAction javaScriptAction = new JavaScriptAction(action);
                javaScriptAction.setScript(StampUtil.getCustomDateScript(textStampBean));
                additionalAction.setAction(AdditionalAction.e_TriggerFieldRecalculateValue, javaScriptAction);
            } else {
                // remove date field
                Annot annot = page.getAnnot(0);
                Widget widget = new Widget(annot);
                AdditionalAction additionalAction = new AdditionalAction(widget.getField());
                if (!additionalAction.isEmpty()) {
                    additionalAction.removeAction(AdditionalAction.e_TriggerFieldRecalculateValue);
                }
                widget.getField().reset();

                // remove old title field -- on the top
                Widget topWidget = new Widget(page.getAnnot(1));
                topWidget.getField().reset();

                //title field -- on the center
                setValue(page.getAnnot(2), textStampBean.mText);
            }
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public boolean addImageIcon(float width, float height, String iconName) {
        BaseStampBean customStamp = mCustomStampBean.get(iconName);
        if (customStamp instanceof ImageStampBean) {
            return addImageIcon(width, height, (ImageStampBean) customStamp);
        }
        return false;
    }

    private boolean addImageIcon(float width, float height, ImageStampBean stampBean) {
        if (width == 0 || height == 0) return false;
        if (stampBean == null) return false;

        stampBean.mTempAnnotRectF = new RectF(0, 0, width, height);
        String key = String.valueOf(stampBean.mModifiedTime);
        PDFDoc doc = mDocMap.get(key + Annot.e_Stamp);
        if (doc == null) {
            try {
//                doc = new PDFDoc(stampBean.mCachePDFFile); //todo
//                int errorCode = doc.load(null);
//                if (errorCode != Constants.e_ErrSuccess) {
                    doc = new PDFDoc();
                    PDFPage page = doc.insertPage(0, width, height);
                    Annot annot = page.addAnnot(Annot.e_Screen, new com.foxit.sdk.common.fxcrt.RectF(0, 0, width, height));
                    Screen screen = new Screen(annot);
                    Image image;
                    File file = new File(stampBean.mCacheFile);
                    if (file.exists()) {
                        image = new Image(stampBean.mCacheFile);
                        int rotation = AppFileUtil.readPictureRotation(stampBean.mCacheFile);
                        screen.setRotation(rotation);
                    } else {
                        Bitmap bitmap = BitmapFactory.decodeResource(AppUtil.getApplicationContext().getResources(),
                                R.drawable.custom_stamp_image_empty_icon);
                        image = new Image();
                        image.addFrame(bitmap);
                    }
                    screen.setImage(image, 0, 0);
                    screen.setOpacity(stampBean.mOpacity / 100f);
                    screen.resetAppearanceStream();
//                }

                mDocMap.put(String.valueOf(stampBean.mModifiedTime) + Annot.e_Stamp, doc);
                return true;
            } catch (PDFException e) {
                return false;
            }
        } else {
            try {
                PDFPage page = mDocPagePair.get(doc);
                if (page == null) {
                    page = doc.getPage(0);
                }
                if (page.getAnnotCount() > 0) {
                    Annot annot = page.getAnnot(0);
                    Screen screen = new Screen(annot);
                    screen.setOpacity(stampBean.mOpacity / 100f);
                    screen.resetAppearanceStream();
                }
                return true;
            } catch (PDFException e) {
                return false;
            }
        }
    }

    @Override
    public boolean getShadingColor(int annotType, String iconName, int refColor,
                                   int shadingIndex, ShadingColor shadingColor) {
        return false;
    }

    @Override
    public float getDisplayWidth(int annotType, String iconName) {
        return 0;
    }

    @Override
    public float getDisplayHeight(int annotType, String iconName) {
        return 0;
    }

    private void setValue(Annot annot, String value) throws PDFException {
        Widget widget = new Widget(annot);
        widget.getField().setValue(value);
        widget.resetAppearanceStream();
    }

    public void addCustomStamp(String key, BaseStampBean stampBean) {
        mCustomStampBean.put(key, stampBean);
    }

}
