/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp;

import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.Stamp;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;

public class StampEvent extends EditAnnotEvent {

    public StampEvent(int eventType, StampUndoItem undoItem, Stamp stamp, PDFViewCtrl pdfViewCtrl) {
        mType = eventType;
        mUndoItem = undoItem;
        mAnnot = stamp;
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean add() {
        if (mAnnot == null || !(mAnnot instanceof Stamp)) {
            return false;
        }
        Stamp annot = (Stamp) mAnnot;
        StampAddUndoItem undoItem = (StampAddUndoItem) mUndoItem;
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        try {
            if (undoItem.mPDFDict != null) {
                boolean reset = AppAnnotUtil.resetPDFDict(annot, undoItem.mPDFDict);
                if (reset) {
                    if (undoItem.mReplys != null)
                        undoItem.mReplys.addReply(annot, undoItem.mReplys);
                    uiExtensionsManager.getDocumentManager().setDocModified(true);
                    return true;
                }
            }

            annot.setUniqueID(undoItem.mNM);
            if (undoItem.mReplys != null)
                undoItem.mReplys.addReply(annot, undoItem.mReplys);
            annot.setFlags(undoItem.mFlags);
            if (undoItem.mCreationDate != null && AppDmUtil.isValidDateTime(undoItem.mCreationDate)) {
                annot.setCreationDateTime(undoItem.mCreationDate);
            }

            if (undoItem.mModifiedDate != null && AppDmUtil.isValidDateTime(undoItem.mModifiedDate)) {
                annot.setModifiedDateTime(undoItem.mModifiedDate);
            }

            if (undoItem.mAuthor != null) {
                annot.setTitle(undoItem.mAuthor);
            }

            if (undoItem.mSubject != null) {
                annot.setSubject(undoItem.mSubject);
            }

            if (undoItem.mIconName != null) {
                annot.setIconName(undoItem.mIconName);
            }
            if (undoItem.mContents == null) {
                undoItem.mContents = "";
            }
            annot.setContent(undoItem.mContents);
            annot.setRotation(undoItem.mRotation);

            DynamicStampIconProvider iconProvider = DynamicStampIconProvider.getInstance(mPdfViewCtrl.getContext());
            if (undoItem.mStampType == StampConstants.CUSTOM_IMAGE_STAMP) {
                float width;
                float height;
                if (undoItem.mRotation == 90 || undoItem.mRotation == 270) {
                    width = Math.abs(undoItem.mBBox.height());
                    height = Math.abs(undoItem.mBBox.width());
                } else {
                    width = Math.abs(undoItem.mBBox.width());
                    height = Math.abs(undoItem.mBBox.height());
                }

                if (!iconProvider.addImageIcon(width, height, undoItem.mIconName)) {
                    return false;
                }
            } else {
                if (!iconProvider.addIcon(undoItem.mBBox, undoItem.mIconName)) {
                    return false;
                }
            }

            RectF rectF = undoItem.mBBox;
            RectF annotRectF = AppUtil.toRectF(annot.getRect());
            if (!annotRectF.equals(rectF))
                annot.move(AppUtil.toFxRectF(rectF));
            annot.resetAppearanceStream();
            uiExtensionsManager.getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return false;
    }

    @Override
    public boolean modify() {
        if (mAnnot == null || !(mAnnot instanceof Stamp)) {
            return false;
        }

        StampModifyUndoItem undoItem = (StampModifyUndoItem) mUndoItem;
        Stamp annot = (Stamp) mAnnot;
        try {
            if (undoItem.mModifiedDate != null) {
                annot.setModifiedDateTime(undoItem.mModifiedDate);
            }
            if (undoItem.mContents == null) {
                undoItem.mContents = "";
            }

            if (undoItem.mOperType == StampUndoItem.OPER_ROTATE) {
                if (undoItem.mRotation != annot.getRotation())
                    annot.setRotation(undoItem.mRotation);
            } else if (undoItem.mOperType == StampUndoItem.OPER_FRAME) {
                annot.move(AppUtil.toFxRectF(undoItem.mBBox));
            } else {
                if (undoItem.mRotation != annot.getRotation())
                    annot.setRotation(undoItem.mRotation);
                annot.move(AppUtil.toFxRectF(undoItem.mBBox));
            }
            annot.setContent(undoItem.mContents);
            annot.resetAppearanceStream();
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return false;
    }

    @Override
    public boolean delete() {
        if (mAnnot == null || !(mAnnot instanceof Stamp)) {
            return false;
        }

        try {
            ((Markup) mAnnot).removeAllReplies();
            PDFPage page = mAnnot.getPage();
            page.removeAnnot(mAnnot);
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }
}
