/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.view.ViewGroup;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IPolicyEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;

public class StampModule implements Module {

    private StampToolHandler mToolHandlerSTP;
    private StampAnnotHandler mAnnotHandlerSTP;

    private final Context mContext;
    private ViewGroup mParent;
    private final PDFViewCtrl mPdfViewCtrl;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public StampModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_STAMP;
    }

    @Override
    public boolean loadModule() {
        mToolHandlerSTP = new StampToolHandler(mContext, mPdfViewCtrl);
        mAnnotHandlerSTP = new StampAnnotHandler(mContext, mParent, mPdfViewCtrl);
        mAnnotHandlerSTP.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandlerSTP.setToolHandler(mToolHandlerSTP);

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerToolHandler(mToolHandlerSTP);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandlerSTP);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.registerPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);

            uiExtensionsManager.getToolsManager()
                    .addToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Stamp, mToolHandlerSTP.getToolSupply());
        }
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;

            uiExtensionsManager.unregisterToolHandler(mToolHandlerSTP);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandlerSTP);
            uiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.unregisterPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);

        mParent = null;
        return false;
    }

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandlerSTP.onDrawForControls(canvas);
        }
    };

    public ToolHandler getToolHandler() {
        return mToolHandlerSTP;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandlerSTP;
    }

    PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandlerSTP.getAnnotMenu() != null && mAnnotHandlerSTP.getAnnotMenu().isShowing()) {
                mAnnotHandlerSTP.getAnnotMenu().dismiss();
            }

            if (mToolHandlerSTP.getPropertyBar() != null && mToolHandlerSTP.getPropertyBar().isShowing()) {
                mToolHandlerSTP.getPropertyBar().dismiss();
            }

            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().reInit();
        }

        @Override
        public void onRecovered() {
            mToolHandlerSTP.initAnnotIconProvider();
        }
    };

    private final IPolicyEventListener mPolicyEventListener = new IPolicyEventListener() {
        @Override
        public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            ToolHandler currentToolHandler = uiExtensionsManager.getCurrentToolHandler();
            if (currentToolHandler == mToolHandlerSTP || mToolHandlerSTP.getCustomPropertyListener() != null) {
                mToolHandlerSTP.onRequestPermissionsResult(requestCode, permissions, grantResults);
            }
        }
    };

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            ToolHandler currentToolHandler = uiExtensionsManager.getCurrentToolHandler();
            if (currentToolHandler == mToolHandlerSTP || mToolHandlerSTP.getCustomPropertyListener() != null) {
                mToolHandlerSTP.onActivityResult(act, requestCode, resultCode, data);
            }
        }

        @Override
        public void onStop(Activity act) {
            if (mToolHandlerSTP != null) {
                mToolHandlerSTP.saveCustomStamps();
            }
        }

    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            ToolHandler currentToolHandler = uiExtensionsManager.getCurrentToolHandler();
            if (currentToolHandler == mToolHandlerSTP || mToolHandlerSTP.getCustomPropertyListener() != null) {
                mToolHandlerSTP.onConfigurationChanged(newConfig);
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            ToolHandler currentToolHandler = uiExtensionsManager.getCurrentToolHandler();
            if (currentToolHandler == mToolHandlerSTP || mToolHandlerSTP.getCustomPropertyListener() != null) {
                mToolHandlerSTP.updateTheme();
            }
        }
    };

}
