/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp.adapter;


import android.content.Context;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.GradientDrawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.stamp.IStampItemClickListener;
import com.foxit.uiextensions.annots.stamp.StampUtil;
import com.foxit.uiextensions.annots.stamp.customstamp.TextStampBean;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppResource;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;

public class TextStampAdapter extends SuperAdapter<TextStampBean> {

    private IStampItemClickListener<TextStampBean> mItemClickListener;

    private final List<TextStampBean> mItems;
    private final List<TextStampBean> mSelectedItems;
    private boolean mIsEditState = false;
    private TextStampBean mLastSelectedItem = null;
    private final UIExtensionsManager mUiExtensionsManager;

    public TextStampAdapter(Context context, PDFViewCtrl viewCtrl, List<TextStampBean> items) {
        super(context);
        mItems = items;
        mUiExtensionsManager = (UIExtensionsManager) viewCtrl.getUIExtensionsManager();
        mSelectedItems = new ArrayList<>();
    }

    public void addTextItem(TextStampBean stampBean) {
        int index = mItems.indexOf(stampBean);
        if (index == -1) {
            mItems.add(stampBean);
        }
        notifyUpdateData();
    }

    public List<TextStampBean> getItems() {
        return mItems;
    }

    public void removeSelectedItems() {
        if (mSelectedItems.contains(mLastSelectedItem)) {
            mLastSelectedItem = null;
        }
        mItems.removeAll(mSelectedItems);
        mSelectedItems.clear();
        notifyUpdateData();
    }

    public void selectAll(boolean selectAll) {
        mSelectedItems.clear();
        for (TextStampBean textStampBean : mItems) {
            textStampBean.mSelected = selectAll;
            if (selectAll)
                mSelectedItems.add(textStampBean);
        }
        notifyUpdateData();
    }

    public boolean isSelectAll() {
        return mSelectedItems.size() == mItems.size();
    }

    public List<TextStampBean> getSelectedItems() {
        return mSelectedItems;
    }

    public TextStampBean getLastSelectedItem() {
        return mLastSelectedItem;
    }

    public void setSelectedItem(TextStampBean stampBean) {
        if (stampBean == mLastSelectedItem) return;

        if (mLastSelectedItem != null) {
            mLastSelectedItem.mSelected = false;
            int index = mItems.indexOf(mLastSelectedItem);
            notifyItemChanged(index);
            mLastSelectedItem = null;
        }

        if (stampBean != null) {
            stampBean.mSelected = true;
            int index = mItems.indexOf(stampBean);
            notifyItemChanged(index);
            mLastSelectedItem = stampBean;
        }
    }

    public void setSelectedItem(long createTime) {
        if (mLastSelectedItem != null) {
            if (mLastSelectedItem.mCreateTime == createTime) return;

            mLastSelectedItem.mSelected = false;
            int index = mItems.indexOf(mLastSelectedItem);
            notifyItemChanged(index);
            mLastSelectedItem = null;
        }

        if (createTime > 0) {
            for (int i = 0; i < mItems.size(); i++) {
                TextStampBean stampBean = mItems.get(i);
                if (stampBean.mCreateTime == createTime) {
                    stampBean.mSelected = true;
                    notifyItemChanged(i);
                    mLastSelectedItem = stampBean;
                    break;
                }
            }
        }
    }

    public void setSelectedItem(int position) {
        if (mLastSelectedItem != null) {
            int index = mItems.indexOf(mLastSelectedItem);
            if (position == index) return;

            mLastSelectedItem.mSelected = false;
            notifyItemChanged(index);
            mLastSelectedItem = null;
        }

        if (position >= 0 && position < mItems.size()) {
            TextStampBean itemBean = mItems.get(position);
            if (!itemBean.mSelected) {
                itemBean.mSelected = true;
                notifyItemChanged(position);
            }
            mLastSelectedItem = itemBean;
        }
    }

    public void setEditState(boolean isEditState) {
        mIsEditState = isEditState;
        if (isEditState) {
            if (mLastSelectedItem != null)
                mLastSelectedItem.mSelected = false;
        } else {
            if (mLastSelectedItem != null)
                mLastSelectedItem.mSelected = true;
            if (mSelectedItems.size() > 0) {
                for (TextStampBean textStampBean : mSelectedItems) {
                    if (textStampBean != mLastSelectedItem) {
                        textStampBean.mSelected = false;
                    }
                }
                mSelectedItems.clear();
            }
        }
        notifyUpdateData();
    }

    public boolean isEditSate(){
        return mIsEditState;
    }

    public boolean isEmpty() {
        return mItems.size() == 0;
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public TextStampBean getDataItem(int position) {
        return mItems.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.fx_stamp_text_item_layout, parent, false);
        return new TextStampViewHolder(itemView);
    }

    @Override
    public int getItemCount() {
        return mItems.size();
    }

    class TextStampViewHolder extends SuperViewHolder {
        private final View mItemView;
        private final TextView mCustomTextTv;
        private final TextView mCustomTextCenterTv;
        private final TextView mDescTv;
        private final RelativeLayout mColorView;
        private final CheckBox mSelectedView;

        public TextStampViewHolder(View itemView) {
            super(itemView);
            mItemView = itemView.findViewById(R.id.custom_text_item_item_view);
            mColorView = itemView.findViewById(R.id.custom_text_item_rl);
            mCustomTextTv = itemView.findViewById(R.id.custom_text_item_title);
            mCustomTextCenterTv = itemView.findViewById(R.id.custom_text_item_center_title);
            mDescTv = itemView.findViewById(R.id.custom_text_item_desc);
            mSelectedView = itemView.findViewById(R.id.custom_text_item_checkview);

            mItemView.setOnClickListener(this);
            mSelectedView.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            TextStampBean stampBean = (TextStampBean) data;
            if (isShowDescView(stampBean)){
                mCustomTextTv.setVisibility(View.VISIBLE);
                mDescTv.setVisibility(View.VISIBLE);
                mCustomTextCenterTv.setVisibility(View.GONE);
                mCustomTextTv.setText(stampBean.mText);
                mDescTv.setText(StampUtil.getCustomDateString(mUiExtensionsManager.getAnnotAuthor(), stampBean));
            }else {
                mCustomTextTv.setVisibility(View.GONE);
                mDescTv.setVisibility(View.GONE);
                mCustomTextCenterTv.setVisibility(View.VISIBLE);
                mCustomTextCenterTv.setText(stampBean.mText);
            }

            if (stampBean.mSelected) {
                GradientDrawable drawable = new GradientDrawable();
                drawable.setCornerRadius(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_4dp));
                drawable.setColor(ThemeConfig.getInstance(getContext()).getB2());
                drawable.setStroke(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_list_divider_height),
                        ThemeConfig.getInstance(getContext()).getPrimaryColor());
                mItemView.setBackground(drawable);
            } else {
                if (mIsEditState) {
                    GradientDrawable drawable = new GradientDrawable();
                    drawable.setCornerRadius(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_4dp));
                    drawable.setColor(ThemeConfig.getInstance(getContext()).getB2());
                    mItemView.setBackground(drawable);
                } else {
                    mItemView.setBackground(null);
                }
            }
            mColorView.getBackground().setColorFilter(
                    new PorterDuffColorFilter(stampBean.mColor, PorterDuff.Mode.SRC_IN));
            mSelectedView.setChecked(stampBean.mSelected);
            mSelectedView.setVisibility(mIsEditState ? View.VISIBLE : View.GONE);
        }

        @Override
        public void onClick(View v) {
            int position = getAdapterPosition();
            if (position == -1) return;

            int id = v.getId();
            if (id == R.id.custom_text_item_item_view) {
                if (mIsEditState) {
                    TextStampBean selectedBean = mItems.get(position);
                    if (mItemClickListener != null)
                        mItemClickListener.onItemClick(true, position, selectedBean);
                } else {
                    TextStampBean selectedBean = mItems.get(position);
                    if (mLastSelectedItem == selectedBean) {
                        if (mItemClickListener != null)
                            mItemClickListener.onItemClick(false, position, selectedBean);
                        return;
                    }

                    if (mLastSelectedItem != null) {
                        mLastSelectedItem.mSelected = false;
                        int index = mItems.indexOf(mLastSelectedItem);
                        notifyItemChanged(index);
                    }
                    selectedBean.mSelected = true;
                    notifyItemChanged(position);
                    mLastSelectedItem = selectedBean;
                    if (mItemClickListener != null)
                        mItemClickListener.onItemClick(false, position, selectedBean);
                }
            } else if (id == R.id.custom_text_item_checkview) {
                TextStampBean itemBean = mItems.get(position);
                if (mSelectedItems.contains(itemBean)) {
                    itemBean.mSelected = false;
                    mSelectedItems.remove(itemBean);
                } else {
                    itemBean.mSelected = true;
                    mSelectedItems.add(itemBean);
                }
                notifyItemChanged(position);
                if (mItemClickListener != null)
                    mItemClickListener.onCheckedChanged(position, mSelectedItems);
            }
        }

        private boolean isShowDescView(TextStampBean stampBean) {
            return stampBean.mShowAuthor || stampBean.mShowDate || stampBean.mShowTime;
        }

    }

    public void setItemClickListener(IStampItemClickListener<TextStampBean> itemClickListener) {
        mItemClickListener = itemClickListener;
    }

}
