/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.textmarkup.highlight;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.ClipboardManager;
import android.text.TextUtils;
import android.view.MotionEvent;
import android.view.View;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Highlight;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.sdk.pdf.annots.TextMarkup;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.common.UIAnnotFlatten;
import com.foxit.uiextensions.annots.common.UIAnnotFrame;
import com.foxit.uiextensions.annots.common.UIAnnotReply;
import com.foxit.uiextensions.annots.common.UIMagnifierView;
import com.foxit.uiextensions.annots.freetext.FtUtil;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContent;
import com.foxit.uiextensions.annots.textmarkup.TextSelector;
import com.foxit.uiextensions.annots.textmarkup.TextmarkupConstants;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.modules.tts.TTSInfo;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.modules.tts.TTSUtils;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;

import static com.foxit.uiextensions.annots.freetext.FtUtil.CTRLPTTOUCHEXT;


class HighlightAnnotHandler implements AnnotHandler {

    private Context mContext;
    private Paint mPaintBbox;
    private Paint mPaintAnnot;
    private Paint mPaintCtr;

    private AnnotMenu mAnnotMenu;
    private PropertyBar mAnnotPropertyBar;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private Annot mLastAnnot;
    private ArrayList<Integer> mMenuItems;
    private AppAnnotUtil mAppAnnotUtil;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private int mBBoxSpace;
    private int mPaintBoxOutset;
    private boolean mIsAnnotModified;
    private boolean mIsEditProperty;
    private boolean mbAreaHi;

    private RectF mModifyBbox = new RectF();
    private int mModifyColor;
    private int mModifyOpacity;
    private int mModifyAnnotColor;
    private QuadPointsArray mModifyQuadPointsArray;

    private TextSelector mTextSelector;
    private int ctrlPoint = 0;
    private int mCurrentIndex;
    public Bitmap mHandlerBitmap;
    private UIMagnifierView mMagnifierView;
    private ArrayList<RectF> mModifyRects = new ArrayList<>();

    public HighlightAnnotHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mAppAnnotUtil = AppAnnotUtil.getInstance(context);
        mBBoxSpace = AppAnnotUtil.getAnnotBBoxSpace();
        mPaintBbox = new Paint();
        mPaintBbox.setAntiAlias(true);
        mPaintBbox.setStyle(Paint.Style.STROKE);
        mPaintBbox.setStrokeWidth(mAppAnnotUtil.getAnnotBBoxStrokeWidth());
        mPaintBbox.setPathEffect(mAppAnnotUtil.getAnnotBBoxPathEffect());

        mPaintAnnot = new Paint();
        mPaintAnnot.setAntiAlias(true);
        mPaintAnnot.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mPaintCtr = new Paint();
        mPaintCtr.setAntiAlias(true);
        mPaintCtr.setStyle(Paint.Style.FILL_AND_STROKE);

        mMenuItems = new ArrayList<Integer>();

        mPaintBoxOutset = AppResource.getDimensionPixelSize(mContext, R.dimen.annot_highlight_paintbox_outset);

        mTextSelector = new TextSelector(mPdfViewCtrl);
        mHandlerBitmap = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.rv_textselect_handler);
    }

    public void setAnnotMenu(AnnotMenu annotMenu) {
        mAnnotMenu = annotMenu;
    }

    public AnnotMenu getAnnotMenu() {
        return mAnnotMenu;
    }

    void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    @Override
    public int getType() {
        return Annot.e_Highlight;
    }

    @Override
    public boolean annotCanAnswer(Annot annot) {
        return true;
    }

    @Override
    public RectF getAnnotBBox(Annot annot) {
        try {
            return AppUtil.toRectF(annot.getRect());
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean isHitAnnot(Annot annot, PointF point) {
        try {
            int pageIndex = annot.getPage().getIndex();
            if (mTextSelector != null && mTextSelector.getRectFList().size() > 0) {
                for (RectF rectF : mTextSelector.getRectFList()) {
                    RectF textRectF = new RectF();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, textRectF, pageIndex);
                    boolean ret = textRectF.contains(point.x, point.y);
                    if (!ret) continue;
                    return true;
                }
            } else {
                RectF rectF = getAnnotBBox(annot);
                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                return rectF.contains(point.x, point.y);
            }
        } catch (PDFException ignored) {
        }
        return false;
    }

    private static final int HANDLE_AREA = 10;

    private int isControlPoint(int pageIndex, PointF point) {
        if (mTextSelector != null && mTextSelector.getRectFList().size() > 0) {
            RectF start = new RectF();
            RectF end = new RectF();

            mPdfViewCtrl.convertPdfRectToPageViewRect(new RectF(mTextSelector.getRectFList().get(0)), start, pageIndex);
            mPdfViewCtrl.convertPdfRectToPageViewRect(new RectF(mTextSelector.getRectFList().get(mTextSelector.getRectFList().size() - 1)), end, pageIndex);
            start.set(start.left, start.top, start.left, start.bottom);
            end.set(end.right, end.top, end.right, end.bottom);
            start.inset(-HANDLE_AREA, -HANDLE_AREA);
            end.inset(-HANDLE_AREA, -HANDLE_AREA);

            Rect leftCtlRect = calcLeftControlPointRect(pageIndex);
            Rect rightCtlRect = calcRightControlPointRect(pageIndex);
            leftCtlRect.inset(-HANDLE_AREA, -HANDLE_AREA);
            rightCtlRect.inset(-HANDLE_AREA, -HANDLE_AREA);

            if (start.contains(point.x, point.y) || leftCtlRect.contains((int) point.x, (int) point.y))
                return 1;
            if (end.contains(point.x, point.y) || rightCtlRect.contains((int) point.x, (int) point.y))
                return 2;
        }
        return 0;

    }

    private Rect calcLeftControlPointRect(int pageIndex) {
        RectF start = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(new RectF(mTextSelector.getRectFList().get(0)), start, pageIndex);

        int startLeft = (int) start.left - mHandlerBitmap.getWidth();
        int startTop = (int) start.top - mHandlerBitmap.getHeight();

        startLeft = Math.max(0, startLeft);

        Rect rect = new Rect(startLeft, startTop,
                startLeft + mHandlerBitmap.getWidth(), startTop + mHandlerBitmap.getHeight());
        return rect;
    }

    Rect calcRightControlPointRect(int pageIndex) {
        RectF end = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(new RectF(mTextSelector.getRectFList().get(mTextSelector.getRectFList().size() - 1)), end, pageIndex);

        int endRight = (int) end.right + mHandlerBitmap.getWidth();
        int endBottom = (int) end.bottom + mHandlerBitmap.getHeight();

        RectF cropRect = mPdfViewCtrl.getCropRect(pageIndex);
        if (cropRect != null) {
            mPdfViewCtrl.convertPdfRectToPageViewRect(cropRect, cropRect, pageIndex);
            endRight = (int) Math.min(endRight, cropRect.right);
        }

        Rect rect = new Rect(endRight - mHandlerBitmap.getWidth(), endBottom - mHandlerBitmap.getHeight(),
                endRight, endBottom);
        return rect;
    }

    public RectF mTmpRect = new RectF();
    private RectF mTmpDesRect = new RectF();

    private void invalidateTouch(int pageIndex, TextSelector selectInfo) {
        if (selectInfo == null) return;
        RectF rectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(selectInfo.getBbox(), rectF, pageIndex);
        RectF rF = calculate(rectF, mTmpRect);
        Rect rect = new Rect();
        rF.roundOut(rect);
        extendInvalidateRect(rect);
        mPdfViewCtrl.invalidate(rect);
        mTmpRect.set(rectF);
    }

    public void extendInvalidateRect(Rect rect) {
        rect.top -= mHandlerBitmap.getHeight();
        rect.bottom += mHandlerBitmap.getHeight();
        rect.left -= mHandlerBitmap.getWidth() / 2;
        rect.right += mHandlerBitmap.getWidth() / 2;
        rect.inset(-20, -20);
    }

    public RectF calculate(RectF desRectF, RectF srcRectF) {
        if (srcRectF.isEmpty()) return desRectF;
        int count = 0;
        if (desRectF.left == srcRectF.left && desRectF.top == srcRectF.top) count++;
        if (desRectF.right == srcRectF.right && desRectF.top == srcRectF.top) count++;
        if (desRectF.left == srcRectF.left && desRectF.bottom == srcRectF.bottom) count++;
        if (desRectF.right == srcRectF.right && desRectF.bottom == srcRectF.bottom) count++;
        mTmpDesRect.set(desRectF);
        if (count == 2) {
            mTmpDesRect.union(srcRectF);
            RectF rectF = new RectF();
            rectF.set(mTmpDesRect);
            mTmpDesRect.intersect(srcRectF);
            rectF.intersect(mTmpDesRect);
            return rectF;
        } else if (count == 3 || count == 4) {
            return mTmpDesRect;
        } else {
            mTmpDesRect.union(srcRectF);
            return mTmpDesRect;
        }
    }

    private boolean OnSelectMove(int pageIndex, PointF point, TextSelector selectInfo) {
        if (selectInfo == null) return false;
        if (mCurrentIndex != pageIndex) return false;
        PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
        try {
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            int index = textPage.getIndexAtPos(point.x, point.y, 10);
            if (index < 0) return false;

            int start = mTextSelector.getStart();
            int end = mTextSelector.getEnd();
            if (ctrlPoint == 1) {
                if (index <= selectInfo.getEnd())
                    start = index;
            } else if (ctrlPoint == 2) {
                if (index >= selectInfo.getStart())
                    end = index;
            }
            mTextSelector.computeSelected(page, start, end);
        } catch (PDFException e) {
            return false;
        }
        return true;
    }

    private void initMagnifierView() {
        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
        if (mMagnifierView == null) {
            mMagnifierView = new UIMagnifierView(mContext.getApplicationContext());
        }
        mMagnifierView.setTargetView(mPdfViewCtrl);
        mMagnifierView.setVisibility(View.GONE);
        mainFrame.getContentView().addView(mMagnifierView);
    }

    private void removeMagnifierView() {
        if (mMagnifierView != null) {
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
            mainFrame.getContentView().removeView(mMagnifierView);
            mMagnifierView.setTargetView(null);
        }
    }

    private RectF mDownRect = new RectF();
    private PointF mDownPoint = new PointF();
    private PointF mLastPoint = new PointF();

    private int mPageView_W;
    private int mPageView_H;
    private int mCurrentCtr = FtUtil.CTR_NONE;
    private int mLastOper = FtUtil.OPER_DEFAULT;
    private float mOffset;
    private boolean mTouchCaptured = false;

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent, Annot annot) {
        try {
            PointF pagePt = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            float evX = pagePt.x;
            float evY = pagePt.y;
            int action = motionEvent.getAction();
            switch (action) {
                case MotionEvent.ACTION_DOWN: {
                    if (annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex()) {
                        if (mbAreaHi) {
                            mCurrentCtr = getTouchControlPoint(pageIndex, annot, evX, evY);
                            mDownRect = AppUtil.toRectF(annot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(mDownRect, mDownRect, pageIndex);

                            if (mCurrentCtr >= 0 && mCurrentCtr <= 7) {
                                mLastOper = UIAnnotFrame.OP_SCALE;
                            } else if (mDownRect.contains(evX, evY)) {
                                mCurrentCtr = FtUtil.CTR_TEXTBBOX;
                                mLastOper = UIAnnotFrame.OP_TRANSLATE;
                            } else {
                                return false;
                            }
                            mPageView_W = mPdfViewCtrl.getPageViewWidth(pageIndex);
                            mPageView_H = mPdfViewCtrl.getPageViewHeight(pageIndex);
                            mDownPoint.set(evX, evY);
                            mLastPoint.set(evX, evY);
                            mTouchCaptured = true;
                            return true;
                        } else {
                            mCurrentIndex = pageIndex;
                            ctrlPoint = isControlPoint(pageIndex, pagePt);
                            if (ctrlPoint != 0) {
                                mMagnifierView.setVisibility(View.VISIBLE);
                                mMagnifierView.onTouchEvent(motionEvent);
                                mAnnotMenu.dismiss();
                                mTouchCaptured = true;
                                return true;
                            }
                        }
                    }
                }
                return false;
                case MotionEvent.ACTION_MOVE: {
                    if (mTouchCaptured
                            && pageIndex == annot.getPage().getIndex()
                            && annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && mUiExtensionsManager.getDocumentManager().canAddAnnot()
                            && mUiExtensionsManager.isEnableModification()
                            && AnnotPermissionUtil.canModifyAnnot(mUiExtensionsManager.getDocumentManager(), annot)) {
                        if (mbAreaHi) {
                            if (evX < 0) evX = 0;
                            if (evY < 0) evY = 0;
                            if (evX > mPageView_W) evX = mPageView_W;
                            if (evY > mPageView_H) evY = mPageView_H;

                            if (evX != mLastPoint.x || evY != mLastPoint.y) {
                                RectF pageViewRect = AppUtil.toRectF(annot.getRect());
                                mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewRect, pageViewRect, pageIndex);

                                RectF rectInv = new RectF(pageViewRect);
                                RectF rectChanged = new RectF(pageViewRect);

                                switch (mLastOper) {
                                    case UIAnnotFrame.OP_SCALE: {
                                        Matrix matrix = FtUtil.calculateScaleMatrix(mCurrentCtr, pageViewRect,
                                                mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                                        Matrix matrix2 = FtUtil.calculateScaleMatrix(mCurrentCtr, pageViewRect, evX - mDownPoint.x, evY
                                                - mDownPoint.y);
                                        rectInv = new RectF(pageViewRect);
                                        RectF rect2 = new RectF(pageViewRect);
                                        matrix.mapRect(rectInv);
                                        matrix2.mapRect(rect2);

                                        rectInv.union(rectChanged);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectInv, rectInv, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectInv));

                                        float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, 8);
                                        PointF adjustXY = FtUtil.adjustScalePointF(mCurrentCtr, mPdfViewCtrl, pageIndex, rect2, deltaXY);

                                        RectF rectInViewerF = new RectF(rectChanged);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectInViewerF, rectInViewerF, pageIndex);
                                        if (mAnnotMenu.isShowing()) {
                                            mAnnotMenu.dismiss();
                                            mAnnotMenu.update(rectInViewerF);
                                        }

                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                        break;
                                    }
                                    case UIAnnotFrame.OP_TRANSLATE: {
                                        rectInv.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                                        rectChanged.offset(evX - mDownPoint.x, evY - mDownPoint.y);
                                        float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, 2);
                                        float adjustx = 0;
                                        float adjusty = 0;
                                        if (rectChanged.left < deltaXY) {
                                            adjustx = -rectChanged.left + deltaXY;
                                        }
                                        if (rectChanged.top < deltaXY) {
                                            adjusty = -rectChanged.top + deltaXY;
                                        }
                                        if (rectChanged.right > mPageView_W - deltaXY) {
                                            adjustx = mPageView_W - rectChanged.right - deltaXY;
                                        }
                                        if (rectChanged.bottom > mPageView_H - deltaXY) {
                                            adjusty = mPageView_H - rectChanged.bottom - deltaXY;
                                        }
                                        if (rectChanged.top < deltaXY && rectChanged.bottom > mPageView_H - deltaXY) {
                                            adjusty = -rectChanged.top + deltaXY;
                                        }
                                        rectChanged.offset(adjustx, adjusty);

                                        rectInv.union(rectChanged);
                                        rectInv.inset(-mBBoxSpace - mOffset, -mBBoxSpace - mOffset);

                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectInv, rectInv, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectInv));

                                        RectF rectInViewerF = new RectF(rectChanged);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectInViewerF, rectInViewerF, pageIndex);
                                        if (mAnnotMenu.isShowing()) {
                                            mAnnotMenu.dismiss();
                                            mAnnotMenu.update(rectInViewerF);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustx, adjusty);
                                        break;
                                    }
                                    default:
                                        break;
                                }

                            }
                        } else {
                            if (ctrlPoint != 0) {
                                mIsAnnotModified = true;
                                PointF docPt = new PointF();
                                mPdfViewCtrl.convertPageViewPtToPdfPt(pagePt, docPt, pageIndex);
                                OnSelectMove(pageIndex, docPt, mTextSelector);
                                invalidateTouch(pageIndex, mTextSelector);

                                mMagnifierView.onTouchEvent(motionEvent);
                            }
                        }
                        return true;
                    }
                    return false;
                }
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL: {
                    if (mTouchCaptured
                            && annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex()) {

                        if (mbAreaHi) {
                            RectF pageViewRect = AppUtil.toRectF(annot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewRect, pageViewRect, pageIndex);

                            switch (mLastOper) {
                                case UIAnnotFrame.OP_SCALE: {
                                    Matrix matrix = FtUtil.calculateScaleMatrix(mCurrentCtr, pageViewRect,
                                            mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                                    RectF rectBBox = new RectF(pageViewRect);
                                    matrix.mapRect(rectBBox);

                                    RectF rectViewer = new RectF(rectBBox);
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectViewer, rectViewer, pageIndex);

                                    mIsAnnotModified = true;
                                    mPdfViewCtrl.convertPageViewRectToPdfRect(rectBBox, rectBBox, pageIndex);
                                    mModifyBbox.set(rectBBox);

                                    QuadPoints quadPoint = new QuadPoints();
                                    quadPoint.setFirst(AppUtil.toFxPointF(mModifyBbox.left, mModifyBbox.top));
                                    quadPoint.setSecond(AppUtil.toFxPointF(mModifyBbox.right, mModifyBbox.top));
                                    quadPoint.setThird(AppUtil.toFxPointF(mModifyBbox.left, mModifyBbox.bottom));
                                    quadPoint.setFourth(AppUtil.toFxPointF(mModifyBbox.right, mModifyBbox.bottom));
                                    mModifyQuadPointsArray.removeAll();
                                    mModifyQuadPointsArray.add(quadPoint);
                                    ((Highlight) annot).setQuadPoints(mModifyQuadPointsArray);
                                    annot.resetAppearanceStream();
                                    resetMenuItems(annot, mUiExtensionsManager);
                                    mAnnotMenu.show(rectViewer);
                                    break;
                                }

                                case UIAnnotFrame.OP_TRANSLATE: {
                                    RectF rectChanged = new RectF(pageViewRect);
                                    rectChanged.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);

                                    RectF rectInViewerF = new RectF(rectChanged);
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectInViewerF, rectInViewerF, pageIndex);

                                    if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                        mIsAnnotModified = true;
                                        mPdfViewCtrl.convertPageViewRectToPdfRect(rectChanged, rectChanged, pageIndex);
                                        mModifyBbox.set(rectChanged);

                                        QuadPoints quadPoint = new QuadPoints();
                                        quadPoint.setFirst(AppUtil.toFxPointF(mModifyBbox.left, mModifyBbox.top));
                                        quadPoint.setSecond(AppUtil.toFxPointF(mModifyBbox.right, mModifyBbox.top));
                                        quadPoint.setThird(AppUtil.toFxPointF(mModifyBbox.left, mModifyBbox.bottom));
                                        quadPoint.setFourth(AppUtil.toFxPointF(mModifyBbox.right, mModifyBbox.bottom));
                                        mModifyQuadPointsArray.removeAll();
                                        mModifyQuadPointsArray.add(quadPoint);
                                        ((Highlight) annot).setQuadPoints(mModifyQuadPointsArray);
                                        annot.resetAppearanceStream();
                                        resetMenuItems(annot, mUiExtensionsManager);
                                    }

                                    if (mAnnotMenu.isShowing()) {
                                        mAnnotMenu.update(rectInViewerF);
                                    } else {
                                        mAnnotMenu.show(rectInViewerF);
                                    }
                                    break;
                                }
                            }
                        } else {
                            mMagnifierView.setVisibility(View.GONE);
                            RectF annotRectF = AppUtil.toRectF(annot.getRect());
                            RectF deviceRt = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRt, pageIndex);
                            Rect rect = rectRoundOut(deviceRt, 0);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, annotRectF, pageIndex);
                            mAnnotMenu.show(annotRectF);
                            mPdfViewCtrl.refresh(pageIndex, rect);

                            mModifyBbox.set(mTextSelector.getBbox());
                            mModifyRects.clear();
                            mModifyRects.addAll(mTextSelector.getRectFList());

                            mModifyQuadPointsArray.removeAll();
                            for (int i = 0; i < mModifyRects.size(); i++) {
                                RectF rF = mModifyRects.get(i);
//                                RectF rF = new RectF();
                                QuadPoints quadPoint = new QuadPoints();
                                quadPoint.setFirst(AppUtil.toFxPointF(rF.left, rF.top));
                                quadPoint.setSecond(AppUtil.toFxPointF(rF.right, rF.top));
                                quadPoint.setThird(AppUtil.toFxPointF(rF.left, rF.bottom));
                                quadPoint.setFourth(AppUtil.toFxPointF(rF.right, rF.bottom));
                                mModifyQuadPointsArray.add(quadPoint);
                            }
                            ((Highlight) annot).setQuadPoints(mModifyQuadPointsArray);
                            annot.resetAppearanceStream();
                        }
                        mTouchCaptured = false;
                        mDownPoint.set(0, 0);
                        mLastPoint.set(0, 0);
                        mLastOper = UIAnnotFrame.OP_DEFAULT;
                        mCurrentCtr = FtUtil.CTR_NONE;
                        return true;
                    }
                    mTouchCaptured = false;
                    mDownPoint.set(0, 0);
                    mLastPoint.set(0, 0);
                    mLastOper = UIAnnotFrame.OP_DEFAULT;
                    mCurrentCtr = FtUtil.CTR_NONE;
                    return false;
                }
                default:
                    break;
            }
        } catch (PDFException ignored) {
        }
        return false;
    }

    private int getTouchControlPoint(int pageIndex, Annot annot, float x, float y) {
        try {
            RectF touchRectF = AppUtil.toRectF(annot.getRect());
            mPdfViewCtrl.convertPdfRectToPageViewRect(touchRectF, touchRectF, pageIndex);
            RectF frmRect = new RectF(touchRectF);
            float[] ctlPts = FtUtil.calculateTextControlPoints(frmRect);
            RectF area = new RectF();
            for (int i = 0; i < ctlPts.length / 2; i++) {
                area.set(ctlPts[i * 2], ctlPts[i * 2 + 1], ctlPts[i * 2], ctlPts[i * 2 + 1]);
                area.inset(-widthOnPageView(mPdfViewCtrl, pageIndex, CTRLPTTOUCHEXT * 2),
                        -widthOnPageView(mPdfViewCtrl, pageIndex, CTRLPTTOUCHEXT * 2));
                if (area.contains(x, y)) {
                    return i;
                }
            }
        } catch (PDFException ignored) {
        }
        return FtUtil.CTR_NONE;
    }

    private float widthOnPageView(PDFViewCtrl pdfViewCtrl, int pageIndex, float width) {
        RectF rectF = new RectF(0, 0, width, width);
        pdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        return Math.abs(rectF.width());
    }

    private Rect mRect = new Rect();
    private RectF mRectF = new RectF();

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        Annot annot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (!(annot instanceof Highlight)) return;
        if (!mPdfViewCtrl.isPageVisible(pageIndex)) return;
        try {
            int annotPageIndex = annot.getPage().getIndex();
            //update page
            if (pageIndex != annotPageIndex) return;

            if (AppAnnotUtil.equals(mLastAnnot, annot)) {
                canvas.save();
                if (mbAreaHi) {
                    RectF frameRectF = new RectF();
                    RectF rect = AppUtil.toRectF(annot.getRect());
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, pageIndex);

                    Matrix matrix = new Matrix();
                    switch (mLastOper) {
                        case UIAnnotFrame.OP_TRANSLATE:
                            rect.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                            break;
                        case UIAnnotFrame.OP_SCALE:
                            matrix = FtUtil.calculateScaleMatrix(mCurrentCtr, rect, mLastPoint.x - mDownPoint.x, mLastPoint.y
                                    - mDownPoint.y);
                            matrix.mapRect(rect);
                            break;
                    }
                    frameRectF.set(rect);

                    mPaintBbox.setColor(Color.RED);
                    mPaintBbox.setStrokeWidth(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH));
                    canvas.drawRect(frameRectF, mPaintAnnot);
                    canvas.drawRect(frameRectF, mPaintBbox);

                    mPaintCtr.setStrokeWidth(FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, FtUtil.DEFAULT_BORDER_WIDTH));
                    mPaintCtr.setColor(Color.WHITE);
                    mPaintCtr.setStyle(Paint.Style.FILL);
                    float[] ctlPts = FtUtil.calculateTextControlPoints(frameRectF);
                    float radius = AppDisplay.dp2px(FtUtil.RADIUS);
                    for (int i = 0; i < ctlPts.length; i += 2) {
                        canvas.drawCircle(ctlPts[i], ctlPts[i + 1], radius, mPaintCtr);
                        canvas.drawCircle(ctlPts[i], ctlPts[i + 1], radius, mPaintCtr);
                    }

                    mPaintCtr.setColor(Color.RED);
                    mPaintCtr.setStyle(Paint.Style.STROKE);
                    for (int i = 0; i < ctlPts.length; i += 2) {
                        canvas.drawCircle(ctlPts[i], ctlPts[i + 1], radius, mPaintCtr);
                        canvas.drawCircle(ctlPts[i], ctlPts[i + 1], radius, mPaintCtr);
                    }
                } else {
                    if (mTextSelector.getRectFList().size() == 0) return;

                    Rect leftCtlRect = calcLeftControlPointRect(pageIndex);
                    Rect rightCtlRect = calcRightControlPointRect(pageIndex);
                    canvas.drawBitmap(mHandlerBitmap, leftCtlRect.left, leftCtlRect.top, null);
                    canvas.drawBitmap(mHandlerBitmap, rightCtlRect.left, rightCtlRect.top, null);

                    Rect clipRect = canvas.getClipBounds();
                    for (RectF rect : mTextSelector.getRectFList()) {
                        mRectF.setEmpty();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(rect, mRectF, pageIndex);
                        mRectF.round(mRect);
                        if (mRect.intersect(clipRect)) {
                            canvas.drawRect(mRect, mPaintAnnot);
                        }
                    }
                }
                canvas.restore();
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        return onSingleTapOrLongPress(pageIndex, motionEvent, annot);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, Annot annot) {
        return onSingleTapOrLongPress(pageIndex, motionEvent, annot);
    }

    private boolean onSingleTapOrLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        try {
            PointF pointF = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            if (annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
                if (pageIndex == annot.getPage().getIndex() && isHitAnnot(annot, pointF)) {
                    return true;
                } else {
                    mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
            } else {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(annot);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        Annot curAnnot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        return !AppAnnotUtil.isSameAnnot(curAnnot, annot);
    }


    private int mUndoColor;
    private int mUndoOpacity;
    private RectF mUndoRectF;
    private QuadPointsArray mUndoQuadPointsArray;
    private String mUndoContents;
    private int[] mPBColors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];

    private int getPBCustomColor() {
        int color = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        return color;
    }

    public void setPropertyBar(PropertyBar propertyBar) {
        mAnnotPropertyBar = propertyBar;
    }

    public PropertyBar getPropertyBar() {
        return mAnnotPropertyBar;
    }

    private String getTextFormTextMarkup(TextMarkup textMarkup) {
        if (textMarkup == null || textMarkup.isEmpty()) return null;
        try {
            PDFPage page = textMarkup.getPage();
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            return textPage.getTextUnderAnnot(textMarkup);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "";
    }

    @Override
    public void onAnnotSelected(final Annot annot, boolean needInvalid) {
        setUndoItemCallback(mUndoItemCallback);

        try {
            final Highlight highlight = (Highlight) annot;
            mUndoColor = annot.getBorderColor();
            mUndoOpacity = (int) (highlight.getOpacity() * 255f + 0.5f);
            mUndoRectF = AppUtil.toRectF(annot.getRect());
            mUndoContents = annot.getContent();
            mUndoQuadPointsArray = highlight.getQuadPoints();

            mAnnotPropertyBar.setArrowVisible(false);
            mPaintBbox.setColor(annot.getBorderColor() | 0xFF000000);
            mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(mUndoColor, mUndoOpacity));
            mModifyQuadPointsArray = new QuadPointsArray();
            mbAreaHi = annot.getDict().hasKey(TextmarkupConstants.AREA_HIGHLIGHT) || TextmarkupConstants.AREA_HIGHLIGHT.equals(highlight.getIntent());
            if (mbAreaHi) {
                mOffset = FtUtil.widthOnPageView(mPdfViewCtrl, annot.getPage().getIndex(), CTRLPTTOUCHEXT * 4);
            } else {
                initMagnifierView();
                QuadPoints startQP = highlight.getQuadPoints().getAt(0);
                QuadPoints endQP = startQP;
                if (highlight.getQuadPoints().getSize() > 1) {
                    int size = (int) highlight.getQuadPoints().getSize();
                    endQP = highlight.getQuadPoints().getAt(size - 1);
                }

                TextPage textPage = new TextPage(annot.getPage(), TextPage.e_ParseTextNormal);
                int pageRotation = annot.getPage().getRotation();
                PointF startPt;
                PointF endPt;
                if (pageRotation == Constants.e_Rotation90 || pageRotation == Constants.e_Rotation270) {
                    startPt = new PointF(startQP.getThird().getX() + (startQP.getFourth().getX() - startQP.getThird().getX()) / 2,
                            startQP.getThird().getY() + 2);
                    endPt = new PointF(endQP.getFirst().getX() + (endQP.getSecond().getX() - endQP.getFirst().getX()) / 2,
                            endQP.getFirst().getY() - 2);
                } else {
                    startPt = new PointF(startQP.getFirst().getX() + 1,
                            startQP.getFirst().getY() /*+ (startQP.getThird().getY() - startQP.getFirst().getY()) / 2 */ - 3);
                    endPt = new PointF(endQP.getFourth().getX() - 1,
                            endQP.getFourth().getY() /*- (endQP.getFourth().getY() - endQP.getSecond().getY()) / 2*/ + 3);
                }
                int start = textPage.getIndexAtPos(startPt.x, startPt.y, 10);
                int end = textPage.getIndexAtPos(endPt.x, endPt.y, 10);
                if (start == -1 || end == -1) {
                    mTextSelector.setBbox(new RectF(mUndoRectF));
                    mTextSelector.getRectFList().clear();
                    mTextSelector.getRectFList().add(mTextSelector.getBbox());
                } else {
                    mTextSelector.clear();
                    mTextSelector.computeSelected(annot.getPage(), start, end);
                }
            }

            final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            resetMenuItems(annot, uiExtensionsManager);
            mAnnotMenu.setListener(new AnnotMenu.ClickListener() {

                @Override
                public void onAMClick(int type) {
                    try {
                        if (AnnotMenu.AM_BT_COPY == type) {
                            @SuppressWarnings("deprecation")
                            ClipboardManager clipboard = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
                            clipboard.setText(getTextFormTextMarkup((TextMarkup) annot));
                            AppAnnotUtil.toastAnnotCopy(mContext);
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        } else if (AnnotMenu.AM_BT_DELETE == type) {
                            deleteAnnot(annot, true, null);
                        } else if (AnnotMenu.AM_BT_STYLE == type) {
                            mAnnotMenu.dismiss();
                            mIsEditProperty = true;
                            mAnnotPropertyBar.setEditable(AnnotPermissionUtil.canEditabled(mUiExtensionsManager.getDocumentManager(), annot));
                            System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, mPBColors, 0, mPBColors.length);
                            mPBColors[0] = getPBCustomColor();
                            mAnnotPropertyBar.setColors(mPBColors);
                            mAnnotPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, (int) annot.getBorderColor());
                            mAnnotPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, AppDmUtil.opacity255To100((int) (highlight.getOpacity() * 255f + 0.5f)));
                            mAnnotPropertyBar.reset(PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_OPACITY);
                            RectF annotRectF = new RectF(AppUtil.toRectF(annot.getRect()));
                            int _pageIndex = annot.getPage().getIndex();

                            RectF deviceRt = new RectF();
                            if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                                if (mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRt, _pageIndex)) {
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, annotRectF, _pageIndex);
                                }
                            }

                            RectF rectF = AppUtil.toGlobalVisibleRectF(uiExtensionsManager.getRootView(), annotRectF);
                            mAnnotPropertyBar.show(rectF, false);
                            mAnnotPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
                        } else if (AnnotMenu.AM_BT_COMMENT == type) {
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotReply.showComments(mPdfViewCtrl, uiExtensionsManager.getRootView(), annot);
                        } else if (AnnotMenu.AM_BT_REPLY == type) {
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotReply.replyToAnnot(mPdfViewCtrl, uiExtensionsManager.getRootView(), annot);
                        } else if (AnnotMenu.AM_BT_FLATTEN == type) {
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotFlatten.flattenAnnot(mPdfViewCtrl, annot);
                        } else if (AnnotMenu.AM_BT_TTS == type) {
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                            TTSModule ttsModule = (TTSModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS);
                            if (ttsModule != null) {
                                TTSInfo ttsInfo = TTSUtils.getTTSInfoFormTextMarkup((TextMarkup) annot, ttsModule);
                                if (ttsInfo == null || AppUtil.isEmpty(ttsInfo.mText)) {
                                    return;
                                }
                                ttsModule.speakFromTs(ttsInfo);
                            }
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            });

            int _pageIndex = annot.getPage().getIndex();
            RectF annotRectF = AppUtil.toRectF(annot.getRect());

            if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                RectF deviceRt = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRt, _pageIndex);
                Rect rect = rectRoundOut(deviceRt, 0);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, annotRectF, _pageIndex);
                mAnnotMenu.show(annotRectF);
                mPdfViewCtrl.refresh(_pageIndex, rect);
                if (annot == uiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
                    mLastAnnot = annot;
                }
            } else {
                mLastAnnot = annot;
            }

        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void resetMenuItems(Annot annot, UIExtensionsManager uiExtensionsManager) {
        mMenuItems.clear();
        boolean containsText = !TextUtils.isEmpty(getTextFormTextMarkup((TextMarkup) annot));
        if (uiExtensionsManager.getDocumentManager().canCopy() && containsText)
            mMenuItems.add(AnnotMenu.AM_BT_COPY);
        if (!uiExtensionsManager.getDocumentManager().canAddAnnot()
                || !uiExtensionsManager.isEnableModification()) {
            mMenuItems.add(AnnotMenu.AM_BT_COMMENT);
        } else {
            mMenuItems.add(AnnotMenu.AM_BT_STYLE);
            mMenuItems.add(AnnotMenu.AM_BT_COMMENT);
            if (AnnotPermissionUtil.canReplyAnnot(uiExtensionsManager.getDocumentManager(), annot))
                mMenuItems.add(AnnotMenu.AM_BT_REPLY);
            if (AnnotPermissionUtil.canFlattenAnnot(uiExtensionsManager.getDocumentManager(), annot)) {
                mMenuItems.add(AnnotMenu.AM_BT_FLATTEN);
            }
            if (!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot))) {
                if (AnnotPermissionUtil.canDeleteAnnot(uiExtensionsManager.getDocumentManager(), annot))
                    mMenuItems.add(AnnotMenu.AM_BT_DELETE);
            }
        }

        if (uiExtensionsManager.getDocumentManager().canCopy()
                && uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS) != null
                && ((TTSModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_TTS)).isSupperTts()
                && containsText) {
            mMenuItems.add(AnnotMenu.AM_BT_TTS);
        }

        mAnnotMenu.setMenuItems(mMenuItems);
    }

    @Override
    public void onAnnotDeselected(final Annot annot, boolean needInvalid) {
        setUndoItemCallback(null);

        removeMagnifierView();
        mAnnotMenu.dismiss();
        mMenuItems.clear();

        if (mIsEditProperty) {
            mIsEditProperty = false;
            mAnnotPropertyBar.dismiss();
        }

        try {
            if (mIsAnnotModified && needInvalid) {
                if (mUndoColor != mModifyAnnotColor
                        || mUndoOpacity != mModifyOpacity
                        || !mUndoRectF.equals(mModifyBbox)) {
                    Highlight highlight = (Highlight) annot;
                    modifyAnnot(annot, highlight.getBorderColor(), (int) (highlight.getOpacity() * 255f + 0.5f),
                            mTextSelector.getText(annot.getPage()), highlight.getQuadPoints(), null, true, null);
                }
            } else if (mIsAnnotModified) {
                try {
                    annot.setBorderColor(mUndoColor);
                    ((Highlight) annot).setQuadPoints(mUndoQuadPointsArray);
                    ((Highlight) annot).setOpacity(mUndoOpacity / 255f);
                    annot.resetAppearanceStream();
                } catch (PDFException e) {
                    if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                        mPdfViewCtrl.recoverForOOM();
                    }
                    return;
                }
            }
            mIsAnnotModified = false;
            if (needInvalid) {
                try {
                    int _pageIndex = annot.getPage().getIndex();
                    if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                        RectF rectF = new RectF(AppUtil.toRectF(annot.getRect()));
                        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, _pageIndex);
                        Rect rect = rectRoundOut(rectF, 2);
                        mPdfViewCtrl.refresh(_pageIndex, rect);
                        mLastAnnot = null;
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }

                return;
            }
            mLastAnnot = null;
        } catch (PDFException ignored) {
        }

        mModifyQuadPointsArray = null;
    }

    @Override
    public void removeAnnot(Annot annot, boolean addUndo, Event.Callback result) {
        deleteAnnot(annot, addUndo, result);
    }

    public void modifyAnnotColor(int color) {
        Annot annot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (annot == null) return;
        mModifyColor = color & 0xFFFFFF;
        try {
            mModifyOpacity = (int) (((Highlight) annot).getOpacity() * 255f + 0.5f);

            mModifyAnnotColor = mModifyColor;
            if (annot.getBorderColor() != mModifyAnnotColor) {
                mIsAnnotModified = true;
                annot.setBorderColor(mModifyAnnotColor);
                ((Highlight) annot).setOpacity(mModifyOpacity / 255f);
                PDFViewCtrl.lock();
                annot.resetAppearanceStream();
                PDFViewCtrl.unlock();
                mPaintBbox.setColor(mModifyAnnotColor | 0xFF000000);
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(color, mModifyOpacity));
                invalidateForToolModify(annot);
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    public void modifyAnnotOpacity(int opacity) {
        Annot annot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (annot == null) return;
        try {
            mModifyColor = (int) annot.getBorderColor() & 0xFFFFFF;
            mModifyOpacity = opacity;

            mModifyAnnotColor = mModifyColor;
            if ((int) (((Highlight) annot).getOpacity() * 255f + 0.5f) != mModifyOpacity) {
                mIsAnnotModified = true;
                annot.setBorderColor(mModifyAnnotColor);
                ((Highlight) annot).setOpacity(mModifyOpacity / 255f);
                PDFViewCtrl.lock();
                annot.resetAppearanceStream();
                PDFViewCtrl.unlock();
                mPaintBbox.setColor(mModifyAnnotColor | 0xFF000000);
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply((int) annot.getBorderColor(), mModifyOpacity));
                invalidateForToolModify(annot);
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void modifyAnnot(final Annot annot,
                             int color,
                             int opacity,
                             String contents,
                             QuadPointsArray quadPointsArray,
                             DateTime modifyDate,
                             final boolean addUndo,
                             final Event.Callback callback) {
        try {
            final PDFPage page = annot.getPage();
            if (page == null) {
                if (callback != null) {
                    callback.result(null, false);
                }
                return;
            }

            final HighlightModifyUndoItem undoItem = new HighlightModifyUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mPageIndex = page.getIndex();
            undoItem.mColor = color;
            undoItem.mOpacity = opacity / 255f;
            undoItem.mModifiedDate = (modifyDate == null) ? AppDmUtil.currentDateToDocumentDate() : modifyDate;
            undoItem.mContents = contents;

            undoItem.mRedoColor = color;
            undoItem.mRedoOpacity = opacity / 255f;
            undoItem.mRedoContents = contents;

            undoItem.mUndoColor = mUndoColor;
            undoItem.mUndoOpacity = mUndoOpacity / 255f;
            undoItem.mUndoContents = mUndoContents;

            undoItem.mRedoQuadPointsArray = quadPointsArray;
            undoItem.mUndoQuadPointsArray = mUndoQuadPointsArray;
            HighlightEvent event = new HighlightEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, (Highlight) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        mUiExtensionsManager.getDocumentManager().onAnnotModified(page, annot);
                        if (addUndo) {
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                        } else {
                            try {
                                if (mPdfViewCtrl.isPageVisible(page.getIndex())) {
                                    RectF annotRectF = new RectF(AppUtil.toRectF(annot.getRect()));
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, page.getIndex());
                                    mPdfViewCtrl.refresh(page.getIndex(), AppDmUtil.rectFToRect(annotRectF));
                                }
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }

                    }

                    if (callback != null) {
                        callback.result(null, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);

        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void deleteAnnot(final Annot annot, final boolean addUndo, final Event.Callback result) {
        final DocumentManager documentManager = mUiExtensionsManager.getDocumentManager();
        if (documentManager.getCurrentAnnot() != null && AppAnnotUtil.isSameAnnot(annot, documentManager.getCurrentAnnot())) {
            documentManager.setCurrentAnnot(null, false);
        }

        try {
            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            final PDFPage page = annot.getPage();
            if (page == null) {
                if (result != null) {
                    result.result(null, false);
                }
                return;
            }
            final int pageIndex = page.getIndex();
            final HighlightDeleteUndoItem undoItem = new HighlightDeleteUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mPageIndex = pageIndex;
            undoItem.mbAreaHi = mbAreaHi;
            undoItem.quadPointsArray = ((Highlight) annot).getQuadPoints();

            documentManager.onAnnotWillDelete(page, annot);
            HighlightEvent event = new HighlightEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Highlight) annot, mPdfViewCtrl);
            if (documentManager.isMultipleSelectAnnots()) {
                if (result != null) {
                    result.result(event, true);
                }
                return;
            }
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        documentManager.onAnnotDeleted(page, annot);
                        if (addUndo) {
                            documentManager.addUndoItem(undoItem);
                        }

                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF deviceRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, pageIndex);
                            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(deviceRectF));
                            if (annot == documentManager.getCurrentAnnot()) {
                                mLastAnnot = null;
                            }
                        } else {
                            if (annot == documentManager.getCurrentAnnot()) {
                                mLastAnnot = null;
                            }
                        }
                    }

                    if (result != null) {
                        result.result(null, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void invalidateForToolModify(Annot annot) {
        try {
            int pageIndex = annot.getPage().getIndex();
            if (!mPdfViewCtrl.isPageVisible(pageIndex)) return;
            RectF rectF = new RectF(AppUtil.toRectF(annot.getRect()));
            RectF pvRect = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, pvRect, pageIndex);
            Rect rect = rectRoundOut(pvRect, mBBoxSpace);
            rect.inset(-mPaintBoxOutset, -mPaintBoxOutset);
            mPdfViewCtrl.refresh(pageIndex, rect);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Rect rectRoundOut(RectF rectF, int roundSize) {
        Rect rect = new Rect();
        rectF.roundOut(rect);
        rect.inset(-roundSize, -roundSize);
        return rect;
    }

    @Override
    public void addAnnot(int pageIndex, AnnotContent contentSupplier, boolean addUndo, Event.Callback result) {
        if (mToolHandler != null) {
            if (!(contentSupplier instanceof TextMarkupContent)) {
                mToolHandler.setFromSelector(true);
            }
            mToolHandler.addAnnot(pageIndex, addUndo, contentSupplier, result);
        } else {
            if (result != null) {
                result.result(null, false);
            }
        }
    }

    private HighlightToolHandler mToolHandler;

    public void setToolHandler(HighlightToolHandler toolHandler) {
        mToolHandler = toolHandler;
    }

    @Override
    public void modifyAnnot(Annot annot, AnnotContent content, boolean addUndo, Event.Callback result) {
        if (content == null) {
            if (result != null) {
                result.result(null, false);
            }
            return;
        }
        try {
            mUndoColor = (int) annot.getBorderColor();
            mUndoOpacity = (int) (((Highlight) annot).getOpacity() * 255f + 0.5f);
            mUndoContents = annot.getContent();
            annot.setContent(content.getContents() != null ? content.getContents() : "");

            if (mLastAnnot == annot) {
                mPaintBbox.setColor(content.getColor() | 0xFF000000);
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(content.getColor(), content.getOpacity()));
            }
            modifyAnnot(annot, content.getColor(), content.getOpacity(), annot.getContent(),
                    ((Highlight) annot).getQuadPoints(), content.getModifiedDate(), addUndo, result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void onDrawForControls(Canvas canvas) {
        Annot annot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        if (!(annot instanceof Highlight)) return;
        if (mUiExtensionsManager.getCurrentAnnotHandler() != this)
            return;
        try {
            int annotPageIndex = annot.getPage().getIndex();

            if (mPdfViewCtrl.isPageVisible(annotPageIndex)) {
                mRectF.set(AppUtil.toRectF(annot.getRect()));
                RectF deviceRt = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(mRectF, deviceRt, annotPageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, mRectF, annotPageIndex);
                if (mIsEditProperty) {
                    RectF rectF = AppUtil.toGlobalVisibleRectF(mUiExtensionsManager.getRootView(), mRectF);
                    mAnnotPropertyBar.update(rectF);
                }
                mAnnotMenu.update(mRectF);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void removeProbarListener() {
        mPropertyChangeListener = null;
    }

    void updateTheme() {
        if (mAnnotPropertyBar != null)
            mAnnotPropertyBar.updateTheme();
    }

    private void setUndoItemCallback(UndoModule.IUndoItemCallback callback) {
        UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
        if (undoModule != null) {
            undoModule.setUndoItemCallback(callback);
        }
    }

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mIsAnnotModified) {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mIsAnnotModified) {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };


}
