/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.textmarkup.underline;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.ClipboardManager;
import android.view.MotionEvent;
import android.view.View;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.sdk.pdf.annots.TextMarkup;
import com.foxit.sdk.pdf.annots.Underline;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.common.UIAnnotFlatten;
import com.foxit.uiextensions.annots.common.UIAnnotReply;
import com.foxit.uiextensions.annots.common.UIMagnifierView;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContent;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContentAbs;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupUtil;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.modules.tts.TTSInfo;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.modules.tts.TTSUtils;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;


class UnderlineAnnotHandler implements AnnotHandler {

    private Context mContext;
    private Paint mPaintBbox;
    private AnnotMenu mAnnotMenu;
    private ArrayList<Integer> mMenuItems;

    private int mModifyColor;
    private int mModifyOpacity;
    private int mModifyAnnotColor;
    private boolean mIsAnnotModified;
    private Annot mLastAnnot;
    private int mBBoxSpace;
    private UnderlineToolHandler mUnderlineToolHandler;

    private boolean mIsEditProperty;
    private PropertyBar mAnnotPropertyBar;

    private PDFViewCtrl mPdfViewCtrl;
    private AppAnnotUtil mAppAnnotUtil;

    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private UIExtensionsManager mUiExtensionsManager;
    private Paint mPaintCtr;
    private Paint mPaintAnnot;

    private QuadPointsArray mModifyQuadPointsArray;
    private int mPaintBoxOutset;
    private int ctrlPoint = 0;
    private final RectF mModifyBbox = new RectF();
    private int mCurrentIndex;
    public Bitmap mHandlerBitmap;
    private UIMagnifierView mMagnifierView;
    public SelectInfo mSelectInfo;
    private boolean mbShouldViewCtrlDraw = false;

    void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    public UnderlineAnnotHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mAppAnnotUtil = AppAnnotUtil.getInstance(context);
        mBBoxSpace = AppAnnotUtil.getAnnotBBoxSpace();
        mPaintBbox = new Paint();
        mPaintBbox.setAntiAlias(true);
        mPaintBbox.setStyle(Paint.Style.STROKE);
        mPaintBbox.setStrokeWidth(mAppAnnotUtil.getAnnotBBoxStrokeWidth());
        mPaintBbox.setPathEffect(AppAnnotUtil.getAnnotBBoxPathEffect());

        mPaintAnnot = new Paint();
        mPaintAnnot.setAntiAlias(true);
        //mPaintAnnot.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mPaintCtr = new Paint();
        mPaintCtr.setAntiAlias(true);
        mPaintCtr.setStyle(Paint.Style.FILL_AND_STROKE);

        mPaintBoxOutset = AppResource.getDimensionPixelSize(mContext, R.dimen.annot_highlight_paintbox_outset);

        mHandlerBitmap = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.rv_textselect_handler);

        mMenuItems = new ArrayList<>();

        mSelectInfo = new SelectInfo();
    }

    public void setAnnotMenu(AnnotMenu annotMenu) {
        mAnnotMenu = annotMenu;
    }

    public AnnotMenu getAnnotMenu() {
        return mAnnotMenu;
    }

    public void setPropertyBar(PropertyBar propertyBar) {
        mAnnotPropertyBar = propertyBar;
    }

    public PropertyBar getPropertyBar() {
        return mAnnotPropertyBar;
    }

    public void setToolHandler(UnderlineToolHandler toolHandler) {
        mUnderlineToolHandler = toolHandler;
    }

    @Override
    public int getType() {
        return Annot.e_Underline;
    }

    @Override
    public boolean annotCanAnswer(Annot annot) {
        return true;
    }

    @Override
    public RectF getAnnotBBox(Annot annot) {
        try {
            return new RectF(AppUtil.toRectF(annot.getRect()));
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean isHitAnnot(Annot annot, PointF point) {
        try {
            RectF rectF = getAnnotBBox(annot);
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, annot.getPage().getIndex());
            return rectF.contains(point.x, point.y);
        } catch (PDFException ignored) {
        }
        return false;
    }

    private static final int HANDLE_AREA = 10;

    private int isControlPoint(PointF point, int pageIndex) {
        if (mSelectInfo != null && mSelectInfo.mRectArray.size() > 0) {
            RectF start = new RectF(mSelectInfo.mRectArray.get(0));
            mPdfViewCtrl.convertPdfRectToPageViewRect(start, start, pageIndex);
            start.right = start.left;
            
            RectF end = new RectF(mSelectInfo.mRectArray.get(mSelectInfo.mRectArray.size() - 1));
            mPdfViewCtrl.convertPdfRectToPageViewRect(end, end, pageIndex);
            end.left = end.right;
            
            start.inset(-HANDLE_AREA, -HANDLE_AREA);
            end.inset(-HANDLE_AREA, -HANDLE_AREA);

            Rect leftCtlRect = calcLeftControlPointRect(pageIndex);
            Rect rightCtlRect = calcRightControlPointRect(pageIndex);
            leftCtlRect.inset(-HANDLE_AREA, -HANDLE_AREA);
            rightCtlRect.inset(-HANDLE_AREA, -HANDLE_AREA);

            if (start.contains(point.x, point.y) || leftCtlRect.contains((int) point.x, (int) point.y))
                return 1;
            if (end.contains(point.x, point.y) || rightCtlRect.contains((int) point.x, (int) point.y))
                return 2;
        }
        return 0;

    }

    private Rect calcLeftControlPointRect(int pageIndex) {
        RectF start = new RectF(mSelectInfo.mRectArray.get(0));
        PointF startPoint = new PointF(start.left, start.top);
        mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, pageIndex);

        int startLeft = (int) startPoint.x - mHandlerBitmap.getWidth();
        int startTop = (int) startPoint.y - mHandlerBitmap.getHeight();
        startLeft = Math.max(0, startLeft);
        Rect rect = new Rect(startLeft, startTop,
                startLeft + mHandlerBitmap.getWidth(), startTop + mHandlerBitmap.getHeight());
        return rect;
    }

    private Rect calcRightControlPointRect(int pageIndex) {
        RectF end = new RectF(mSelectInfo.mRectArray.get(mSelectInfo.mRectArray.size() - 1));
        PointF endPoint = new PointF(end.right, end.bottom);
        mPdfViewCtrl.convertPdfPtToPageViewPt(endPoint, endPoint, pageIndex);

        int endRight = (int) endPoint.x + mHandlerBitmap.getWidth();
        int endBottom = (int) endPoint.y + mHandlerBitmap.getHeight();
        RectF cropRect = mPdfViewCtrl.getCropRect(mCurrentIndex);
        if (cropRect != null) {
            mPdfViewCtrl.convertPdfRectToPageViewRect(cropRect, cropRect, mCurrentIndex);
            endRight = (int) Math.min(endRight, cropRect.right);
        }

        Rect rect = new Rect(endRight - mHandlerBitmap.getWidth(), endBottom - mHandlerBitmap.getHeight(),
                endRight, endBottom);
        return rect;
    }

    public void extendInvalidateRect(Rect rect) {
        rect.top -= mHandlerBitmap.getHeight();
        rect.bottom += mHandlerBitmap.getHeight();
        rect.left -= mHandlerBitmap.getWidth() / 2;
        rect.right += mHandlerBitmap.getWidth() / 2;
        rect.inset(-20, -20);
    }

    public RectF mTmpRect = new RectF();
    private RectF mTmpDesRect = new RectF();

    private void invalidateTouch(int pageIndex, SelectInfo selectInfo) {
        if (selectInfo == null) return;
        RectF rectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(selectInfo.mBBox, rectF, pageIndex);
        RectF rF = calculate(rectF, mTmpRect);
        Rect rect = new Rect();
        rF.roundOut(rect);
        extendInvalidateRect(rect);
        mPdfViewCtrl.invalidate(rect);
        mTmpRect.set(rectF);
    }

    public RectF calculate(RectF desRectF, RectF srcRectF) {
        if (srcRectF.isEmpty()) return desRectF;
        int count = 0;
        if (desRectF.left == srcRectF.left && desRectF.top == srcRectF.top) count++;
        if (desRectF.right == srcRectF.right && desRectF.top == srcRectF.top) count++;
        if (desRectF.left == srcRectF.left && desRectF.bottom == srcRectF.bottom) count++;
        if (desRectF.right == srcRectF.right && desRectF.bottom == srcRectF.bottom) count++;
        mTmpDesRect.set(desRectF);
        if (count == 2) {
            mTmpDesRect.union(srcRectF);
            RectF rectF = new RectF();
            rectF.set(mTmpDesRect);
            mTmpDesRect.intersect(srcRectF);
            rectF.intersect(mTmpDesRect);
            return rectF;
        } else if (count == 3 || count == 4) {
            return mTmpDesRect;
        } else {
            mTmpDesRect.union(srcRectF);
            return mTmpDesRect;
        }
    }

    private boolean onSelectMove(int pageIndex, PointF point, SelectInfo selectInfo) {
        if (selectInfo == null) return false;
        if (mCurrentIndex != pageIndex) return false;
        try {
            PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            if (page == null) {
                return false;
            }
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);

            PointF pagePt = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(point, pagePt, pageIndex);
            int index = textPage.getIndexAtPos(pagePt.x, pagePt.y, 10);
            if (index < 0) return false;

            if (ctrlPoint == 1) {
                if (index <= selectInfo.mEndChar)
                    mSelectInfo.mStartChar = index;

            } else if (ctrlPoint == 2) {
                if (index >= selectInfo.mStartChar)
                    mSelectInfo.mEndChar = index;
            }
            mSelectInfo.mContent = textPage.getChars(mSelectInfo.mStartChar, mSelectInfo.mEndChar - mSelectInfo.mStartChar + 1);
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
            return false;
        }
        return true;
    }


    protected void selectCountRect(int pageIndex, SelectInfo selectInfo) {
        if (selectInfo == null) return;

        int start = selectInfo.mStartChar;
        int end = selectInfo.mEndChar;
        if (start == end && start == -1) return;
        if (end < start) {
            int tmp = end;
            end = start;
            start = tmp;
        }

        selectInfo.mRectArray.clear();
        selectInfo.mRectVert.clear();
        try {
            PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            if (page == null) {
                return;
            }

            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            int count = textPage.getTextRectCount(start, end - start + 1);
            for (int i = 0; i < count; i++) {
                RectF crect = AppUtil.toRectF(textPage.getTextRect(i));
                int rotate = textPage.getBaselineRotation(i);
                boolean vert = rotate == 1 || rotate == 3;
                mSelectInfo.mRectArray.add(crect);
                mSelectInfo.mRectVert.add(vert);
                mSelectInfo.mRotation.add(rotate);
                if (i == 0) {
                    selectInfo.mBBox = new RectF(crect);
                } else {
                    AppUtil.unionFxRectF(selectInfo.mBBox, crect);
                }
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }
    
    private void initMagnifierView() {
        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
        if (mMagnifierView == null) {
            mMagnifierView = new UIMagnifierView(mContext.getApplicationContext());
        }
        mMagnifierView.setTargetView(mPdfViewCtrl);
        mMagnifierView.setVisibility(View.GONE);
        mainFrame.getContentView().addView(mMagnifierView);
    }

    private void removeMagnifierView() {
        if (mMagnifierView != null) {
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
            mainFrame.getContentView().removeView(mMagnifierView);
            mMagnifierView.setTargetView(null);
        }
    }

    private final int[] mPBColors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];

    public int getPBCustomColor() {
        return PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
    }

    private QuadPointsArray mUndoQuadPointsArray;
    private RectF mUndoRectF;

    public class SelectInfo {
        public boolean mIsFromTS;
        public int mStartChar;
        public int mEndChar;
        public String mContent;
        public RectF mBBox;
        public ArrayList<RectF> mRectArray;
        public ArrayList<Boolean> mRectVert;
        public ArrayList<Integer> mRotation;

        public SelectInfo() {
            mBBox = new RectF();
            mRectArray = new ArrayList<>();
            mRectVert = new ArrayList<>();
            mRotation = new ArrayList<>();
        }

        public void clear() {
            mIsFromTS = false;
            mStartChar = mEndChar = -1;
            mBBox.setEmpty();
            mRectArray.clear();
            mContent = null;
        }
    }

    private void resetLineData() {
        mSelectInfo.mStartChar = mSelectInfo.mEndChar = -1;
        mSelectInfo.mRectArray.clear();
        mSelectInfo.mBBox.setEmpty();
        mSelectInfo.mRectVert.clear();
        mSelectInfo.mRotation.clear();
        mTmpRect.setEmpty();
    }

    @Override
    public void onAnnotSelected(final Annot annot, boolean needInvalid) {
        setUndoItemCallback(mUndoItemCallback);

        try {
            mbShouldViewCtrlDraw = false;
            mTmpUndoColor = (int) annot.getBorderColor();
            mTmpUndoOpacity = (int) (((Markup) annot).getOpacity() * 255f + 0.5f);
            mTmpUndoContents = annot.getContent();
            mPaintBbox.setColor(mTmpUndoColor | 0xFF000000);
            mPaintAnnot.setColor(mTmpUndoColor | 0xFF000000);

            mUndoRectF = AppUtil.toRectF(annot.getRect());
            mUndoQuadPointsArray = ((Underline) annot).getQuadPoints();

            resetLineData();

            PointF startPoint = new PointF();
            PointF endPoint = new PointF();

            PointF startPDFPoint = new PointF(
                    mUndoQuadPointsArray.getAt(0).getFirst().getX() + 2,
                    mUndoQuadPointsArray.getAt(0).getFirst().getY() + (mUndoQuadPointsArray.getAt(0).getThird().getY() - mUndoQuadPointsArray.getAt(0).getFirst().getY()) / 2 - 3);

            PointF endPDFPoint = new PointF(
                    mUndoQuadPointsArray.getAt(mUndoQuadPointsArray.getSize() - 1).getFourth().getX() - 2,
                    mUndoQuadPointsArray.getAt(mUndoQuadPointsArray.getSize() - 1).getFourth().getY() - (mUndoQuadPointsArray.getAt(mUndoQuadPointsArray.getSize() - 1).getFourth().getY() - mUndoQuadPointsArray.getAt(mUndoQuadPointsArray.getSize() - 1).getSecond().getY()) / 2 + 3);

            mPdfViewCtrl.convertPdfPtToPageViewPt(startPDFPoint,
                    startPoint
                    , annot.getPage().getIndex());
            mPdfViewCtrl.convertPdfPtToPageViewPt(endPDFPoint,
                    endPoint
                    , annot.getPage().getIndex());
            onSelectDown(annot.getPage().getIndex(), startPoint, endPoint, mSelectInfo);
            selectCountRect(annot.getPage().getIndex(), mSelectInfo);
            if (mSelectInfo.mRectArray.size() == 0) {
                mbShouldViewCtrlDraw = true;
                RectF bbox = new RectF(mUndoRectF);
                mSelectInfo.mBBox = new RectF(bbox);
                mSelectInfo.mRectArray.clear();
                mSelectInfo.mRectArray.add(bbox);
            }

            mAnnotPropertyBar.setArrowVisible(false);

            mModifyQuadPointsArray = new QuadPointsArray();
            mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(mTmpUndoColor, mTmpUndoOpacity));
            initMagnifierView();

            resetMenuItems(annot);
            mAnnotMenu.setMenuItems(mMenuItems);
            mAnnotMenu.setListener(new AnnotMenu.ClickListener() {
                @Override
                public void onAMClick(int btType) {
                    try {
                        UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                        if (btType == AnnotMenu.AM_BT_COPY) {
                            ClipboardManager clipboard = null;
                            clipboard = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
                            clipboard.setText(annot.getContent());
                            AppAnnotUtil.toastAnnotCopy(mContext);
                            extensionsManager.getDocumentManager().setCurrentAnnot(null);
                        } else if (btType == AnnotMenu.AM_BT_DELETE) {
                            deleteAnnot(annot, true, null);
                        } else if (btType == AnnotMenu.AM_BT_STYLE) {
                            mAnnotMenu.dismiss();
                            mIsEditProperty = true;
                            mAnnotPropertyBar.setEditable(AnnotPermissionUtil.canEditabled(mUiExtensionsManager.getDocumentManager(), annot));
                            System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, mPBColors, 0, mPBColors.length);
                            mPBColors[0] = getPBCustomColor();
                            mAnnotPropertyBar.setColors(mPBColors);
                            mAnnotPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, (int) annot.getBorderColor());
                            mAnnotPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, AppDmUtil.opacity255To100((int) (((Markup) annot).getOpacity() * 255f + 0.5f)));
                            mAnnotPropertyBar.reset(PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_OPACITY);
                            RectF annotRectF = new RectF();
                            int _pageIndex = annot.getPage().getIndex();

                            if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                                RectF tempRect = new RectF(AppUtil.toRectF(annot.getRect()));
                                mPdfViewCtrl.convertPdfRectToPageViewRect(tempRect, annotRectF, _pageIndex);
                                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, _pageIndex);
                            }
                            RectF rectF = AppUtil.toGlobalVisibleRectF(extensionsManager.getRootView(), annotRectF);
                            mAnnotPropertyBar.show(rectF, false);
                            mAnnotPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
                        } else if (btType == AnnotMenu.AM_BT_COMMENT) {
                            extensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotReply.showComments(mPdfViewCtrl, extensionsManager.getRootView(), annot);
                        } else if (btType == AnnotMenu.AM_BT_REPLY) {
                            extensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotReply.replyToAnnot(mPdfViewCtrl, extensionsManager.getRootView(), annot);
                        } else if (AnnotMenu.AM_BT_FLATTEN == btType) {
                            extensionsManager.getDocumentManager().setCurrentAnnot(null);
                            UIAnnotFlatten.flattenAnnot(mPdfViewCtrl, annot);
                        } else if (AnnotMenu.AM_BT_TTS == btType) {
                            extensionsManager.getDocumentManager().setCurrentAnnot(null);
                            TTSModule ttsModule = (TTSModule) extensionsManager.getModuleByName(Module.MODULE_NAME_TTS);
                            if (ttsModule != null) {
                                TTSInfo ttsInfo = TTSUtils.getTTSInfoFormTextMarkup((TextMarkup) annot, ttsModule);
                                if (ttsInfo == null || AppUtil.isEmpty(ttsInfo.mText)) {
                                    return;
                                }
                                ttsModule.speakFromTs(ttsInfo);
                            }
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            });
            RectF annotRectF = AppUtil.toRectF(annot.getRect());
            int _pageIndex = annot.getPage().getIndex();

            if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                RectF tempRect = new RectF(AppUtil.toRectF(annot.getRect()));
                mPdfViewCtrl.convertPdfRectToPageViewRect(tempRect, annotRectF, _pageIndex);
                Rect rect = TextMarkupUtil.rectRoundOut(annotRectF, 0);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, _pageIndex);
                mPdfViewCtrl.refresh(_pageIndex, rect);
                if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                    mLastAnnot = annot;
                }

            } else {
                mLastAnnot = annot;
            }
            mAnnotMenu.show(annotRectF);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onAnnotDeselected(Annot annot, boolean needInvalid) {
        setUndoItemCallback(null);

        try {
            String annotContent = mSelectInfo.mContent;

            mSelectInfo.clear();
            mTmpRect.setEmpty();
            mCurrentIndex = -1;
            removeMagnifierView();
            mAnnotMenu.dismiss();
            mMenuItems.clear();
            if (mIsEditProperty) {
                mIsEditProperty = false;
                mAnnotPropertyBar.dismiss();
            }

            if (mIsAnnotModified && needInvalid) {
                Underline underline = (Underline) annot;
                if (mTmpUndoColor != mModifyAnnotColor || mTmpUndoOpacity != mModifyOpacity || !mUndoRectF.equals(mModifyBbox)) {
                    modifyAnnot(annot, underline.getBorderColor(), (int) (underline.getOpacity() * 255f + 0.5f),annotContent,
                            underline.getQuadPoints(), null, true, null);
                }
            } else if (mIsAnnotModified) {
                annot.setBorderColor(mTmpUndoColor);
                ((Markup) annot).setOpacity(mTmpUndoOpacity);
                ((Underline) annot).setQuadPoints(mUndoQuadPointsArray);
                annot.resetAppearanceStream();
            }
            if (needInvalid) {
                mIsAnnotModified = false;
                int _pageIndex = annot.getPage().getIndex();

                if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                    RectF rectF = new RectF(AppUtil.toRectF(annot.getRect()));
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, _pageIndex);
                    Rect rect = TextMarkupUtil.rectRoundOut(rectF, 0);
                    mPdfViewCtrl.refresh(_pageIndex, rect);
                }
                return;
            }
            mLastAnnot = null;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        mModifyQuadPointsArray = null;
    }

    private void modifyAnnot(final Annot annot, int color, int opacity, String content,
                             QuadPointsArray quadPointsArray, DateTime modifyDate, final boolean addUndo, final Event.Callback callback) {
        try {
            final PDFPage page = annot.getPage();
            if (null == page) return;

            if (modifyDate == null) {
                modifyDate = AppDmUtil.currentDateToDocumentDate();
                annot.setBorderColor(mModifyAnnotColor);
            } else {
                annot.setBorderColor(color);
                ((Markup) annot).setOpacity(opacity / 255f);
            }

            final int _pageIndex = page.getIndex();
            final UnderlineModifyUndoItem undoItem = new UnderlineModifyUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mPageIndex = _pageIndex;
            undoItem.mColor = color;
            undoItem.mOpacity = opacity / 255f;
            undoItem.mModifiedDate = modifyDate;
            undoItem.mContents = content;

            undoItem.mRedoColor = color;
            undoItem.mRedoOpacity = opacity / 255f;
            undoItem.mRedoContent = content;

            undoItem.mUndoColor = mTmpUndoColor;
            undoItem.mUndoOpacity = mTmpUndoOpacity / 255f;
            undoItem.mUndoContent = mTmpUndoContents;

            undoItem.mRedoQuadPointsArray = quadPointsArray;
            undoItem.mUndoQuadPointsArray = mUndoQuadPointsArray;

            UnderlineEvent event = new UnderlineEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, (Underline) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotModified(page, annot);
                        if (addUndo) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        }
                        if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                            try {
                                RectF annotRectF = new RectF(AppUtil.toRectF(annot.getRect()));
                                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, _pageIndex);
                                mPdfViewCtrl.refresh(_pageIndex, AppDmUtil.rectFToRect(annotRectF));
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }

                        }
                    }

                    if (callback != null) {
                        callback.result(null, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void deleteAnnot(final Annot annot, final boolean addUndo, final Event.Callback result) {
        final DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (documentManager.getCurrentAnnot() != null && AppAnnotUtil.isSameAnnot(annot, documentManager.getCurrentAnnot())) {
            documentManager.setCurrentAnnot(null, false);
        }

        try {
            final RectF annotRectF = AppUtil.toRectF(annot.getRect());
            final PDFPage page = annot.getPage();
            final int _pageIndex = page.getIndex();
            final UnderlineDeleteUndoItem undoItem = new UnderlineDeleteUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mPageIndex = _pageIndex;
            undoItem.mQuadPoints = ((Underline) annot).getQuadPoints();

            documentManager.onAnnotWillDelete(page, annot);
            UnderlineEvent event = new UnderlineEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Underline) annot, mPdfViewCtrl);
            if (documentManager.isMultipleSelectAnnots()) {
                if (result != null) {
                    result.result(event, true);
                }
                return;
            }
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        documentManager.onAnnotDeleted(page, annot);
                        if (addUndo) {
                            documentManager.addUndoItem(undoItem);
                        }
                        RectF deviceRectF = new RectF();
                        if (mPdfViewCtrl.isPageVisible(_pageIndex)) {
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRectF, _pageIndex);
                            mPdfViewCtrl.refresh(_pageIndex, AppDmUtil.rectFToRect(deviceRectF));
                        }
                        if (annot == documentManager.getCurrentAnnot()) {
                            mLastAnnot = null;
                        }
                    }

                    if (result != null) {
                        result.result(event, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    @Override
    public void addAnnot(int pageIndex, AnnotContent contentSupplier, boolean addUndo, Event.Callback result) {
        if (mUnderlineToolHandler != null) {
            if (contentSupplier instanceof TextMarkupContent) {
                mUnderlineToolHandler.addAnnot(pageIndex, addUndo, contentSupplier, result);
            } else {
                TextMarkupContentAbs tmSelector = (TextMarkupContentAbs) contentSupplier;
                UnderlineToolHandler.SelectInfo info = mUnderlineToolHandler.mSelectInfo;
                info.clear();
                info.mIsFromTS = true;
                info.mStartChar = tmSelector.getTextSelector().getStart();
                info.mEndChar = tmSelector.getTextSelector().getEnd();
                mUnderlineToolHandler.setFromSelector(true);
                mUnderlineToolHandler.selectCountRect(contentSupplier.getPageIndex(), info);
                mUnderlineToolHandler.onSelectRelease(contentSupplier.getPageIndex(), info, result);
            }
        }
    }

    @Override
    public void removeAnnot(Annot annot, boolean addUndo, Event.Callback result) {
        deleteAnnot(annot, addUndo, result);
    }

    private boolean onSelectDown(int pageIndex, PointF startPoint, PointF endPoint, SelectInfo selectInfo) {
        if (selectInfo == null) return false;
        try {
            mCurrentIndex = pageIndex;
            selectInfo.mRectArray.clear();
            selectInfo.mStartChar = selectInfo.mEndChar = -1;
            PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
            if (page == null) {
                return false;
            }
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);

            PointF pagePt = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(startPoint, pagePt, pageIndex);
            int startIndex = textPage.getIndexAtPos(pagePt.x, pagePt.y, 10);
            if (startIndex >= 0) {
                selectInfo.mStartChar = selectInfo.mEndChar = startIndex;
            }
            mPdfViewCtrl.convertPageViewPtToPdfPt(endPoint, pagePt, pageIndex);
            int endIndex = textPage.getIndexAtPos(pagePt.x, pagePt.y, 10);
            if (endIndex >= 0) {
                selectInfo.mEndChar = endIndex;
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
            return false;
        }
        return true;
    }

    private boolean mTouchCaptured = false;

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent, Annot annot) {
        try {
            PointF pagePt = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            int action = motionEvent.getAction();
            switch (action) {
                case MotionEvent.ACTION_DOWN: {
                    if (annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex()) {
                        mCurrentIndex = pageIndex;
                        ctrlPoint = isControlPoint(pagePt, pageIndex);
                        if (ctrlPoint != 0) {
                            mMagnifierView.setVisibility(View.VISIBLE);
                            mMagnifierView.onTouchEvent(motionEvent);
                            mAnnotMenu.dismiss();
                            mTouchCaptured = true;
                            return true;
                        }
                    }
                }
                return false;
                case MotionEvent.ACTION_MOVE: {
                    if (mTouchCaptured
                            && pageIndex == annot.getPage().getIndex()
                            && annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && mUiExtensionsManager.getDocumentManager().canAddAnnot()
                            && mUiExtensionsManager.isEnableModification()
                            && AnnotPermissionUtil.canModifyAnnot(mUiExtensionsManager.getDocumentManager(), annot)) {
                        if (ctrlPoint != 0) {
                            mIsAnnotModified = true;
                            onSelectMove(pageIndex, pagePt, mSelectInfo);
                            selectCountRect(pageIndex, mSelectInfo);
                            invalidateTouch(pageIndex, mSelectInfo);
                            mMagnifierView.onTouchEvent(motionEvent);
                        }
                        return true;
                    }
                    return false;
                }
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL: {
                    if (mTouchCaptured
                            && annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex()) {
                        mMagnifierView.setVisibility(View.GONE);

                        RectF annotRectF = AppUtil.toRectF(annot.getRect());
                        RectF deviceRt = new RectF();
                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, deviceRt, pageIndex);
                        Rect rect = TextMarkupUtil.rectRoundOut(deviceRt, 0);
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, annotRectF, pageIndex);
                        mAnnotMenu.show(annotRectF);
                        mPdfViewCtrl.refresh(pageIndex, rect);

                        mModifyBbox.set(mSelectInfo.mBBox);
                        mModifyQuadPointsArray.removeAll();
                        for (int i = 0; i < mSelectInfo.mRectArray.size(); i++) {
                            RectF rF = mSelectInfo.mRectArray.get(i);
                            QuadPoints quadPoint = new QuadPoints();
                            if (mSelectInfo.mRectVert.get(i)) {
                                quadPoint.setFirst(AppUtil.toFxPointF(rF.left, rF.top));
                                quadPoint.setSecond(AppUtil.toFxPointF(rF.left, rF.bottom));
                                quadPoint.setThird(AppUtil.toFxPointF(rF.right, rF.top));
                                quadPoint.setFourth(AppUtil.toFxPointF(rF.right, rF.bottom));
                            } else {
                                quadPoint.setFirst(AppUtil.toFxPointF(rF.left, rF.top));
                                quadPoint.setSecond(AppUtil.toFxPointF(rF.right, rF.top));
                                quadPoint.setThird(AppUtil.toFxPointF(rF.left, rF.bottom));
                                quadPoint.setFourth(AppUtil.toFxPointF(rF.right, rF.bottom));
                            }
                            mModifyQuadPointsArray.add(quadPoint);
                        }
                        ((Underline) annot).setQuadPoints(mModifyQuadPointsArray);
                        annot.resetAppearanceStream();
                        mTouchCaptured = false;
                        return true;
                    }
                    mTouchCaptured = false;
                    return false;
                }
                default:
                    break;
            }
        } catch (PDFException ignored) {
        }
        return false;
    }


    private int mTmpUndoColor;
    private int mTmpUndoOpacity;
    private String mTmpUndoContents;
    private RectF mRectF = new RectF();
    private final Rect mRect = new Rect();

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (!(annot instanceof Markup)) return;
        if (!mPdfViewCtrl.isPageVisible(pageIndex)) return;

        try {
            if (annot.getPage().getIndex() != pageIndex) return;
            if (AppAnnotUtil.equals(mLastAnnot, annot)) {

                if (mSelectInfo.mRectArray.size() == 0) return;
                if (mbShouldViewCtrlDraw) {
                    RectF rectF = AppUtil.toRectF(annot.getRect());
                    RectF deviceRt = new RectF();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, deviceRt, pageIndex);
                    Rect rectBBox = TextMarkupUtil.rectRoundOut(deviceRt, mBBoxSpace);
                    canvas.save();
                    canvas.drawRect(rectBBox, mPaintBbox);
                    canvas.restore();
                } else {
                    canvas.save();
                    Rect leftCtlRect = calcLeftControlPointRect(pageIndex);
                    Rect rightCtlRect = calcRightControlPointRect(pageIndex);
                    canvas.drawBitmap(mHandlerBitmap, leftCtlRect.left, leftCtlRect.top, null);
                    canvas.drawBitmap(mHandlerBitmap, rightCtlRect.left, rightCtlRect.top, null);
                    canvas.restore();

                    Rect clipRect = canvas.getClipBounds();
                    int i = 0;
                    PointF startPointF = new PointF();
                    PointF endPointF = new PointF();
                    int pageRotation;
                    try {
                        pageRotation = mPdfViewCtrl.getDoc().getPage(pageIndex).getRotation();
                    } catch (PDFException e) {
                        pageRotation = 0;
                    }

                    for (RectF rect : mSelectInfo.mRectArray) {
                        mPdfViewCtrl.convertPdfRectToPageViewRect(rect, mRectF, pageIndex);
                        mRectF.round(mRect);

                        if (mRect.intersect(clipRect)) {
                            if (i < mSelectInfo.mRectVert.size()) {
                                int rotation = (mSelectInfo.mRotation.get(i) + pageRotation + mPdfViewCtrl.getViewRotation()) % 4;
                                boolean vert = rotation == 1 || rotation == 3;

                                //reset Paint width
                                if ((rect.top - rect.bottom) > (rect.right - rect.left)) {
                                    TextMarkupUtil.resetDrawLineWidth(mPdfViewCtrl, pageIndex, mPaintAnnot, rect.right, rect.left);
                                } else {
                                    TextMarkupUtil.resetDrawLineWidth(mPdfViewCtrl, pageIndex, mPaintAnnot, rect.top, rect.bottom);
                                }

                                if (vert) {
                                    if (rotation == 3) {
                                        startPointF.x = mRectF.right - (mRectF.right - mRectF.left) / 16f;
                                    } else {
                                        startPointF.x = mRectF.left + (mRectF.right - mRectF.left) / 16f;
                                    }

                                    startPointF.y = mRectF.top;
                                    endPointF.x = startPointF.x;
                                    endPointF.y = mRectF.bottom;
                                } else {
                                    if (rotation == 0) {
                                        startPointF.y = mRectF.bottom + (mRectF.top - mRectF.bottom) / 16f;
                                    } else {
                                        startPointF.y = mRectF.top - (mRectF.top - mRectF.bottom) / 16f;
                                    }
                                    startPointF.x = mRectF.left;
                                    endPointF.x = mRectF.right;
                                    endPointF.y = startPointF.y;
                                }

                                canvas.save();
                                canvas.drawLine(startPointF.x, startPointF.y, endPointF.x, endPointF.y, mPaintAnnot);
                                canvas.restore();
                            }
                        }
                        i++;
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        return onSingleTapOrLongPress(pageIndex, motionEvent, annot);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, Annot annot) {
        return onSingleTapOrLongPress(pageIndex, motionEvent, annot);
    }

    private boolean onSingleTapOrLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        try {
            PointF pointF = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            if (annot == mUiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
                if (pageIndex == annot.getPage().getIndex() && isHitAnnot(annot, pointF)) {
                    return true;
                } else {
                    mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
            } else {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(annot);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        if (mbShouldViewCtrlDraw) return true;
        Annot curAnnot = mUiExtensionsManager.getDocumentManager().getCurrentAnnot();
        return !AppAnnotUtil.isSameAnnot(curAnnot, annot);
    }

    public void onDrawForControls(Canvas canvas) {
        Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (!(curAnnot instanceof Markup)) return;
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentAnnotHandler() != this)
            return;

        try {
            int annotPageIndex = curAnnot.getPage().getIndex();
            if (mPdfViewCtrl.isPageVisible(annotPageIndex)) {
                mRectF = new RectF(AppUtil.toRectF(curAnnot.getRect()));
                RectF deviceRt = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(mRectF, deviceRt, annotPageIndex);

                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(deviceRt, mRectF, annotPageIndex);
                if (mIsEditProperty) {
                    RectF rect = AppUtil.toGlobalVisibleRectF(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getRootView(), mRectF);
                    mAnnotPropertyBar.update(rect);
                }
                mAnnotMenu.update(mRectF);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public void modifyAnnotColor(int color) {
        Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (annot == null) return;
        try {
            mModifyColor = color & 0xFFFFFF;
            mModifyOpacity = (int) (((Markup) annot).getOpacity() * 255f + 0.5f);
            mModifyAnnotColor = mModifyColor;

            if (annot.getBorderColor() != mModifyAnnotColor) {
                mIsAnnotModified = true;
                annot.setBorderColor(mModifyAnnotColor);
                ((Markup) annot).setOpacity(mModifyOpacity / 255f);
                PDFViewCtrl.lock();
                annot.resetAppearanceStream();
                PDFViewCtrl.unlock();
                mPaintBbox.setColor(mModifyAnnotColor | 0xFF000000);
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(color, mModifyOpacity));
                invalidateForToolModify(annot);
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void invalidateForToolModify(Annot annot) {
        try {
            int pageIndex = annot.getPage().getIndex();
            if (!mPdfViewCtrl.isPageVisible(pageIndex)) return;
            RectF rectF = new RectF(AppUtil.toRectF(annot.getRect()));
            RectF pvRect = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, pvRect, pageIndex);
            Rect rect = TextMarkupUtil.rectRoundOut(pvRect, mBBoxSpace);
            rect.inset(-mPaintBoxOutset, -mPaintBoxOutset);
            mPdfViewCtrl.refresh(pageIndex, rect);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void modifyAnnotOpacity(int opacity) {
        Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (annot == null) return;
        try {
            mModifyColor = (int) annot.getBorderColor() & 0xFFFFFF;
            mModifyOpacity = opacity;

            mModifyAnnotColor = mModifyColor;
            if ((int) (((Markup) annot).getOpacity() * 255f + 0.5f) != mModifyOpacity) {
                mIsAnnotModified = true;
                annot.setBorderColor(mModifyAnnotColor);
                ((Markup) annot).setOpacity(mModifyOpacity / 255f);
                PDFViewCtrl.lock();
                annot.resetAppearanceStream();
                PDFViewCtrl.unlock();
                mPaintBbox.setColor(mModifyAnnotColor | 0xFF000000);
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply((int) annot.getBorderColor(), mModifyOpacity));
                invalidateForToolModify(annot);
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    public void resetMenuItems(Annot annot) {
        mMenuItems.clear();
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (documentManager.canCopy()) {
            mMenuItems.add(AnnotMenu.AM_BT_COPY);
        }
        if (!documentManager.canAddAnnot() || !((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()) {
            mMenuItems.add(AnnotMenu.AM_BT_COMMENT);
        } else {
            mMenuItems.add(AnnotMenu.AM_BT_STYLE);
            mMenuItems.add(AnnotMenu.AM_BT_COMMENT);
            if (AnnotPermissionUtil.canReplyAnnot(documentManager, annot))
                mMenuItems.add(AnnotMenu.AM_BT_REPLY);
            if (AnnotPermissionUtil.canFlattenAnnot(documentManager, annot)) {
                mMenuItems.add(AnnotMenu.AM_BT_FLATTEN);
            }
            if (!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot))) {
                if (AnnotPermissionUtil.canDeleteAnnot(documentManager, annot))
                    mMenuItems.add(AnnotMenu.AM_BT_DELETE);
            }
        }
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canCopy()
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getModuleByName(Module.MODULE_NAME_TTS) != null
                && ((TTSModule) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getModuleByName(Module.MODULE_NAME_TTS)).isSupperTts()) {
            mMenuItems.add(AnnotMenu.AM_BT_TTS);
        }
    }

    @Override
    public void modifyAnnot(Annot annot, AnnotContent content, boolean addUndo, Event.Callback result) {
        if (content == null) return;
        try {
            mTmpUndoColor = (int) annot.getBorderColor();
            mTmpUndoOpacity = (int) (((Markup) annot).getOpacity() * 255f + 0.5f);
            mTmpUndoContents = annot.getContent();
            annot.setContent(content.getContents() != null ? content.getContents() : "");
            if (mLastAnnot == annot) {
                mPaintBbox.setColor(content.getColor());
                mPaintAnnot.setColor(AppDmUtil.calColorByMultiply(content.getColor(), content.getOpacity()));
            }
            modifyAnnot(annot, content.getColor(), content.getOpacity(), annot.getContent(),
                    ((Underline) annot).getQuadPoints(), content.getModifiedDate(), addUndo, result);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected void removeProbarListener() {
        mPropertyChangeListener = null;
    }

    void updateTheme() {
        if (mAnnotPropertyBar != null)
            mAnnotPropertyBar.updateTheme();
    }

    private void setUndoItemCallback(UndoModule.IUndoItemCallback callback) {
        UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
        if (undoModule != null) {
            undoModule.setUndoItemCallback(callback);
        }
    }

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mIsAnnotModified) {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mIsAnnotModified) {
                mUiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };

}