/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.browser.behavior;


import android.animation.ValueAnimator;
import android.content.Context;
import android.util.AttributeSet;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.core.view.ViewCompat;

public class NestedContentScrollBehavior extends CoordinatorLayout.Behavior<View> {

    private int mHeaderHeight;
    private View mContentView;
    private ValueAnimator mValueAnimator;

    public NestedContentScrollBehavior(Context context) {
        this(context, null);
    }

    public NestedContentScrollBehavior(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    @Override
    public boolean onLayoutChild(@NonNull CoordinatorLayout parent, @NonNull View child, int layoutDirection) {
        parent.onLayoutChild(child, layoutDirection);

        mContentView = child;
        View headerView = parent.getChildAt(0);
        mHeaderHeight = headerView.getMeasuredHeight();

        if (child.getTag() instanceof Integer) {
            int tag = (int) child.getTag();
            if (tag == NestedHeaderBehavior.FIXED) {
                CoordinatorLayout.LayoutParams lp = (CoordinatorLayout.LayoutParams) child.getLayoutParams();
                if (lp.topMargin != mHeaderHeight) {
                    lp.topMargin = mHeaderHeight;
                    child.setLayoutParams(lp);
                }
                return true;
            }
        }
        CoordinatorLayout.LayoutParams lp = (CoordinatorLayout.LayoutParams) child.getLayoutParams();
        if (lp.topMargin != 0) {
            lp.topMargin = 0;
            child.setLayoutParams(lp);
        }
        ViewCompat.offsetTopAndBottom(child, mHeaderHeight);
        return true;
    }

    @Override
    public boolean onStartNestedScroll(@NonNull CoordinatorLayout coordinatorLayout, @NonNull View child,
                                       @NonNull View directTargetChild, @NonNull View target, int axes, int type) {
        return (axes & ViewCompat.SCROLL_AXIS_VERTICAL) != 0;
    }

    @Override
    public void onNestedPreScroll(@NonNull CoordinatorLayout coordinatorLayout, @NonNull View child,
                                  @NonNull View target, int dx, int dy, @NonNull int[] consumed, int type) {
        super.onNestedPreScroll(coordinatorLayout, child, target, dx, dy, consumed, type);
        stopAutoRebound();

        if (child.getTag() instanceof Integer) {
            int tag = (int) child.getTag();
            if (tag == NestedHeaderBehavior.FIXED) {
                child.setTranslationY(0f);
                return;
            }
        }

        if (dy > 0) {
            float translationY = child.getTranslationY() - dy;
            if (translationY >= -mHeaderHeight) {
                consumed[1] = dy;
                child.setTranslationY(translationY);
            } else {
                consumed[1] = mHeaderHeight + (int) child.getTranslationY();
                child.setTranslationY(-mHeaderHeight);
            }
        }
    }

    @Override
    public void onNestedScroll(@NonNull CoordinatorLayout coordinatorLayout, @NonNull View child,
                               @NonNull View target, int dxConsumed, int dyConsumed, int dxUnconsumed,
                               int dyUnconsumed, int type, @NonNull int[] consumed) {
        super.onNestedScroll(coordinatorLayout, child, target, dxConsumed, dyConsumed, dxUnconsumed, dyUnconsumed, type, consumed);
        stopAutoRebound();

        if (child.getTag() instanceof Integer) {
            int tag = (int) child.getTag();
            if (tag == NestedHeaderBehavior.HIDE) {
                child.setTranslationY(-mHeaderHeight);
                return;
            }
        }

        if (child.getTag() instanceof Integer) {
            int tag = (int) child.getTag();
            if (tag == NestedHeaderBehavior.FIXED) {
                child.setTranslationY(0f);
                return;
            }
        }

        float translationY = child.getTranslationY();
        if (type == ViewCompat.TYPE_NON_TOUCH || translationY >= 0 || dyUnconsumed > 0) {
            return;
        }

        if (dyUnconsumed < 0) {
            float newTranslationY = child.getTranslationY() - dyUnconsumed;
            child.setTranslationY(newTranslationY <= 0 ? newTranslationY : 0f);
        }
    }


    @Override
    public void onStopNestedScroll(@NonNull CoordinatorLayout coordinatorLayout, @NonNull View child,
                                   @NonNull View target, int type) {
        super.onStopNestedScroll(coordinatorLayout, child, target, type);
        if (child.getTranslationY() >= 0 || child.getTranslationY() <= -mHeaderHeight) {
            return;
        }

        if (child.getTranslationY() <= -mHeaderHeight * 0.5f) {
            stopAutoRebound();
            startAutoRebound((int) child.getTranslationY(), -mHeaderHeight);
        } else {
            stopAutoRebound();
            startAutoRebound(child.getTranslationY(), 0);
        }
    }

    private void startAutoRebound(float current, float target) {
        mValueAnimator = ValueAnimator.ofFloat(current, target);
        mValueAnimator.setDuration(400).start();
        mValueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator animation) {
                if (mContentView != null)
                    mContentView.setTranslationY((float) animation.getAnimatedValue());
            }
        });
    }

    private void stopAutoRebound() {
        if (mValueAnimator != null) {
            mValueAnimator.cancel();
            mValueAnimator.removeAllUpdateListeners();
            mValueAnimator = null;
        }
    }

}
