/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;


import android.content.Context;
import android.content.DialogInterface;
import android.os.Bundle;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

public abstract class BaseDialogFragment extends DialogFragment {

    protected Context mContext;
    protected boolean mbFullScreen = true;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mContext = getActivity().getApplicationContext();
        int theme;
        if (SystemUiHelper.getInstance().isStatusBarShown(getActivity())) {
            SystemUiHelper.getInstance().setAllowedHideSystemUI(getActivity(), false);
//            SystemUiHelper.getInstance().showStatusBar(getActivity());
            theme = R.style.BaseDialogTheme;
            mbFullScreen = false;
        } else {
            if (SystemUiHelper.getInstance().isFullScreenMode(getActivity())) {
                SystemUiHelper.getInstance().setAllowedHideSystemUI(getActivity(), false);
            } else {
                SystemUiHelper.getInstance().setAllowedHideSystemUI(getActivity(), true);
            }
            theme = R.style.DialogFullScreen;
            mbFullScreen = true;
        }
        setStyle(STYLE_NO_TITLE, theme);
        if (getPDFViewCtrl() != null && getPDFViewCtrl().getUIExtensionsManager() != null)
            ((UIExtensionsManager) getPDFViewCtrl().getUIExtensionsManager()).registerLayoutChangeListener(mLayoutChangeListener);
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View contentView = onCreateView(inflater, container);
        if (contentView != null) {
            return contentView;
        } else {
            return super.onCreateView(inflater, container, savedInstanceState);
        }
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
            @Override
            public void run() {
                if (SystemUiHelper.getInstance().isFullScreen() && getDialog() != null)
                    SystemUiHelper.getInstance().hideSystemUI(getDialog().getWindow());
            }
        }, 100);
    }

    protected abstract View onCreateView(LayoutInflater inflater, ViewGroup container);

    protected abstract PDFViewCtrl getPDFViewCtrl();

    protected abstract void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight);

    @Override
    public void onStart() {
        super.onStart();
        refreshWindowLayout();
    }

    @Override
    public void onDetach() {
        super.onDetach();
        if (getPDFViewCtrl() == null) return;

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) getPDFViewCtrl().getUIExtensionsManager();
        if (uiExtensionsManager != null) {
            SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), true);
            uiExtensionsManager.unregisterLayoutChangeListener(mLayoutChangeListener);
        }
    }

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (oldWidth != newWidth || oldHeight != newHeight) {
                refreshWindowLayout();
                BaseDialogFragment.this.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
            }
        }
    };

    private void refreshWindowLayout() {
        if (getPDFViewCtrl() == null) return;
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) getPDFViewCtrl().getUIExtensionsManager();
        if (uiExtensionsManager == null) return;

        View rootView = uiExtensionsManager.getRootView();
        int[] location = new int[2];
        rootView.getLocationOnScreen(location);
        int x = location[0];
        int y = location[1];
        int width = rootView.getWidth();
        int height = rootView.getHeight();

        Window window = getDialog().getWindow();
        WindowManager.LayoutParams windowParams = window.getAttributes();
        windowParams.gravity = Gravity.LEFT | Gravity.TOP;
        windowParams.dimAmount = 0.0f;
        windowParams.x = x;
        windowParams.y = y;
        windowParams.width = width;
        windowParams.height = height;
        window.setAttributes(windowParams);
        getDialog().setCanceledOnTouchOutside(true);
    }

    @Override
    public void dismiss() {
        super.dismiss();
    }

    private DismissListener mDismissListener;

    public void setDismissListener(DismissListener dismissListener) {
        mDismissListener = dismissListener;
    }

    public interface DismissListener {
        void onDismiss();
    }

    public boolean isFullScreen() {
        return mbFullScreen;
    }

    @Override
    public void onDismiss(@NonNull DialogInterface dialog) {
        super.onDismiss(dialog);
        if (mDismissListener != null)
            mDismissListener.onDismiss();
        AppDialogManager.getInstance().remove(this);
    }
}
