/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.os.Build;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.View;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

public class CircleProgressBar extends View {

    private int mProgress = 10;
    private int mMax = 100;
    private float mAngle;
    private int mPrimaryColor;
    private int mSecondaryColor;
    private int mCircleWidth;
    private Paint mCirclePaint;
    private Paint mTextPaint;
    private int mTextSize = 18;
    private int mTextColor = Color.BLACK;
    private long mUiThreadId;
    private RefreshProgressRunnable mRefreshProgressRunnable;

    public CircleProgressBar(Context context) {
        this(context, null);
    }

    public CircleProgressBar(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public CircleProgressBar(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initialize(context, attrs, defStyleAttr);
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public CircleProgressBar(Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        initialize(context, attrs, defStyleAttr);
    }

    private void initialize(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        mUiThreadId = Thread.currentThread().getId();
        TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.CircleProgressBar,
                defStyleAttr, 0);

        for (int i = 0; i < typedArray.getIndexCount(); i++) {
            int attr = typedArray.getIndex(i);
            if (attr == R.styleable.CircleProgressBar_primaryColor) {
                mPrimaryColor = typedArray.getColor(attr, AppResource.getColor(context, R.color.b2));
            } else if (attr == R.styleable.CircleProgressBar_secondaryColor) {
                mSecondaryColor = typedArray.getColor(attr, AppResource.getColor(context, R.color.main_color));
            } else if (attr == R.styleable.CircleProgressBar_circleWidth) {
                mCircleWidth = typedArray.getDimensionPixelSize(attr, (int) TypedValue.applyDimension(
                        TypedValue.COMPLEX_UNIT_DIP, AppResource.getDimension(context, R.dimen.ux_toolbar_button_interval_small_pad), getResources().getDisplayMetrics()));
            } else if (attr == R.styleable.CircleProgressBar_textSize) {
                mTextSize = typedArray.getDimensionPixelSize(attr, (int) TypedValue.applyDimension(
                        TypedValue.COMPLEX_UNIT_DIP, AppResource.getDimension(context, R.dimen.ux_text_circle_progress_bar), getResources().getDisplayMetrics()));
            } else if (attr == R.styleable.CircleProgressBar_textColor) {
                mTextColor = typedArray.getColor(attr, Color.BLACK);
            }
        }
        typedArray.recycle();

        mCirclePaint = new Paint();
        mCirclePaint.setAntiAlias(true);
        mCirclePaint.setDither(true);
        mCirclePaint.setStrokeWidth(mCircleWidth);

        mTextPaint = new Paint();
        mTextPaint.setAntiAlias(true);
        mTextPaint.setDither(true);
        mTextPaint.setTextAlign(Paint.Align.CENTER);
        mTextPaint.setColor(mTextColor);
        mTextPaint.setTextSize(mTextSize);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = MeasureSpec.getSize(widthMeasureSpec);
        int height = MeasureSpec.getSize(heightMeasureSpec);
        setMeasuredDimension(Math.min(width, height), Math.min(width, height));
    }

    private RectF mOval = new RectF();

    @Override
    protected void onDraw(Canvas canvas) {
        int center = getWidth() / 2;
        int radius = center - mCircleWidth / 2;

        //draw circle
        mCirclePaint.setColor(mPrimaryColor);
        mCirclePaint.setStyle(Paint.Style.STROKE);
        canvas.drawCircle(center, center, radius, mCirclePaint);

        // draw arc
        mCirclePaint.setColor(mSecondaryColor);
        mCirclePaint.setStrokeCap(Paint.Cap.ROUND);
        mOval.set(center - radius, center - radius, center + radius, center + radius);
        mAngle = mProgress * 360.0f / mMax * 1.0f;
        canvas.drawArc(mOval, -90, mAngle, false, mCirclePaint);

        //draw text
        int nPercent = mProgress * 100 / mMax;
        String percent = String.format("%d", nPercent) + "%";
        mTextPaint.setStrokeWidth(0);
        Paint.FontMetricsInt fontMetrics = mTextPaint.getFontMetricsInt();

        onMeasureText(percent);

        int baseline = center + (fontMetrics.bottom - fontMetrics.top) / 2 - fontMetrics.bottom;
        canvas.drawText(percent, center, baseline, mTextPaint);


    }

    public void onMeasureText(String percent) {
        if (mTextPaint.measureText(percent) + mCircleWidth*2 > mOval.width()) {
            mTextPaint.setTextSize(mTextPaint.getTextSize() - 1);
            onMeasureText(percent);
        }
    }


    public void setProgress(int progress) {
        if (progress > mMax || mProgress == progress) return;
        mProgress = progress;
        refreshProgress();
    }

    public int getProgress() {
        return mProgress;
    }

    public void setMax(int max) {
        mMax = max;
    }

    public int getMax() {
        return mMax;
    }

    public void setPrimaryColor(int primaryColor) {
        mPrimaryColor = primaryColor;
        refreshProgress();
    }

    public void setSecondaryColor(int secondaryColor) {
        mSecondaryColor = secondaryColor;
        refreshProgress();
    }

    public void setCircleWidth(int circleWidth) {
        mCircleWidth = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, circleWidth, getResources().getDisplayMetrics());
        mCirclePaint.setStrokeWidth(circleWidth);

        refreshProgress();
    }

    public void setTextColor(int textColor) {
        mTextColor = textColor;
        refreshProgress();
    }

    public void setTextSize(int textSize) {
        mTextSize = textSize;
        refreshProgress();
    }

    private void refreshProgress() {
        if (mUiThreadId == Thread.currentThread().getId()) {
            invalidate();
        } else {
            if (mRefreshProgressRunnable == null) {
                mRefreshProgressRunnable = new RefreshProgressRunnable();
            }
            AppThreadManager.getInstance().runOnUiThread(mRefreshProgressRunnable);
        }
    }

    private class RefreshProgressRunnable implements Runnable {
        public void run() {
            synchronized (CircleProgressBar.this) {
                invalidate();
            }
        }
    }
}
