/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.graphics.Typeface;
import android.os.Build;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.SystemUiHelper;

public class UIMatchDialog extends Dialog implements MatchDialog {
    protected Context mContext;
    private View mView;
    private RelativeLayout mRootView;
    protected LinearLayout mTitleView;
    private View mTitleShadowLine;

    private BaseBar mTitleBar;
    private IBaseItem mBackImageItem;
    private IBaseItem mBackTextItem;
    protected IBaseItem mRightTextItem;
    private IBaseItem mTitleItem;
    private boolean mShowSeparateLine = true;
    private boolean mFullscreen = false;
    private int mTitleStyle = DLG_TITLE_STYLE_BG_DEFAULT;

    private LinearLayout mContentViewRoot;
    private LinearLayout mContentView;
    private LinearLayout mButtonsViewRoot;
    private LinearLayout mButtonsView;

    private Button mDlg_bt_cancel;
    private Button mDlg_bt_skip;
    private Button mDlg_bt_replace;
    private Button mDlg_bt_copy;
    private Button mDlg_bt_move;
    private Button mDlg_bt_ok;
    private Button mDlg_bt_open_only;
    private Button mDlg_bt_upload;

    private DialogListener mDialogListener;
    protected DismissListener mDismissListener;
    protected OnCancelListener mOnCancelListener;

    private int mHeight = -100;
    private int mButtonHeight;
    private boolean mShowMask = false;
    private int mBackStyle = IMAGE_BACK;
    protected Activity mActivity;
    private boolean mAutoRestSystemUi = true;
    private boolean mIsStatusBarShown = false;
    protected boolean mIsBackDismiss = false;

    public UIMatchDialog(Context context) {
        // If theme = R.style.rd_dialog_fullscreen_style, new a fullscreen dialog.
        // If theme = 0, it will be not a fullscreen dialog.
        this(context, AppDisplay.isPad() ? 0 :
                (SystemUiHelper.getInstance().isStatusBarShown((Activity) context)
                        ? R.style.rd_dialog_no_title_style : R.style.rd_dialog_fullscreen_style), true);
    }

    public UIMatchDialog(Context context, boolean showSeparateLine) {
        this(context, AppDisplay.isPad() ? 0 :
                (SystemUiHelper.getInstance().isStatusBarShown((Activity) context)
                        ? R.style.rd_dialog_no_title_style : R.style.rd_dialog_fullscreen_style), showSeparateLine);
    }

    public UIMatchDialog(Context context, int theme) {
        // set theme = 0, it will be not a fullscreen dialog.
        // set theme = R.style.rd_dialog_fullscreen_style, it will be a fullscreen dialog.
        this(context, theme, true);
    }

    public UIMatchDialog(Context context, int theme, boolean showSeparateLine) {
        // set theme = 0, it will be not a fullscreen dialog.
        // set theme = R.style.rd_dialog_fullscreen_style, it will be a fullscreen dialog.
        super(context, theme == 0 ? AppTheme.getDialogTheme() : theme);
        this.mContext = context.getApplicationContext();
        if (context instanceof Activity) {
            this.mActivity = (Activity) context;
        }
        this.mShowSeparateLine = showSeparateLine;
        if (theme == R.style.rd_dialog_fullscreen_style || theme == R.style.rd_dialog_no_title_style)
            mFullscreen = true;
        else
            mFullscreen = false;
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (mActivity.getWindow() != null && mActivity.getWindow().getStatusBarColor() != getWindow().getStatusBarColor()){
                SystemUiHelper.getInstance().setStatusBarColor(getWindow(), mActivity.getWindow().getStatusBarColor());
            }
        }
        initView();
    }

    private void initView() {
        mButtonHeight = (int) (mContext.getResources().getDimension(R.dimen.ux_dialog_button_height));

        mView = LayoutInflater.from(mContext).inflate(R.layout.dlg_root, null, false);
        mRootView = (RelativeLayout) mView.findViewById(R.id.dlg_root);
        mTitleView = (LinearLayout) mRootView.findViewById(R.id.dlg_top_title);
        mTitleView.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mContentViewRoot = (LinearLayout) mRootView.findViewById(R.id.dlg_contentview_root);
        mContentView = (LinearLayout) mRootView.findViewById(R.id.dlg_contentview);
        mButtonsViewRoot = (LinearLayout) mRootView.findViewById(R.id.dlg_buttonview);
        mTitleShadowLine = mRootView.findViewById(R.id.dlg_top_title_shadow_line);
        mTitleShadowLine.setVisibility(View.GONE);
        if (mFullscreen) {
            mTitleStyle = DLG_TITLE_STYLE_BG_DEFAULT;
            if (mShowSeparateLine) {
                mTitleBar = new TopBarImpl(mContext);
            } else {
                mTitleBar = new BaseBarImpl(mContext);
            }
        } else {
            mTitleStyle = DLG_TITLE_STYLE_BG_WHITE;
            mTitleBar = new BaseBarImpl(mContext);
            if (mShowSeparateLine) {
                mTitleShadowLine.setVisibility(View.VISIBLE);
            } else {
                mTitleShadowLine.setVisibility(View.GONE);
            }
            mTitleBar.setHeight(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_topbar_height));
        }
        mTitleBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));
        mTitleBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mBackImageItem = new BaseItemImpl(mContext);
        mBackImageItem.setDisplayStyle(IBaseItem.ItemType.Item_Image);
        mBackImageItem.setImageResource(R.drawable.rd_back_vector);
        mBackImageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        mBackImageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mDialogListener != null) {
                    mDialogListener.onBackClick();
                }
                if (!mIsBackDismiss) {
                    UIMatchDialog.this.dismiss();
                }
            }
        });

        mBackTextItem = new BaseItemImpl(mContext);
        mBackTextItem.setText(AppResource.getString(mContext, R.string.fx_string_cancel));
        mBackTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mBackTextItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mBackTextItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mDialogListener != null) {
                    mDialogListener.onBackClick();
                }
                if (!mIsBackDismiss) {
                    UIMatchDialog.this.dismiss();
                }
            }
        });
        mBackTextItem.getContentView().setVisibility(View.GONE);

        mRightTextItem = new BaseItemImpl(mContext);
        mRightTextItem.setText(AppResource.getString(mContext, R.string.fx_string_done));
        mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mRightTextItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mRightTextItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mDialogListener != null) {
                    mDialogListener.onTitleRightButtonClick();
                }
            }
        });
        mRightTextItem.getContentView().setVisibility(View.GONE);

        mTitleItem = new BaseItemImpl(mContext);
        mTitleItem.setDisplayStyle(IBaseItem.ItemType.Item_Text);
        mTitleItem.setText("");
        mTitleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mTitleItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
        mTitleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));

        mTitleBar.addView(mBackTextItem, BaseBar.TB_Position.Position_LT);
        mTitleBar.addView(mBackImageItem, BaseBar.TB_Position.Position_LT);
        mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
        mTitleBar.addView(mRightTextItem, BaseBar.TB_Position.Position_RB);
        mTitleView.addView(mTitleBar.getContentView());
        if (!SystemUiHelper.getInstance().isStatusBarShown(mActivity)) {
            if (getWindow() != null)
                getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
        }
        if (mTitleStyle == DLG_TITLE_STYLE_BG_DEFAULT) {
            setTitleDefaultStyle();
        } else if (mTitleStyle == DLG_TITLE_STYLE_BG_WHITE) {
            setTitleStyleWhite();
        } else {
            setTitleDefaultStyle();
        }
        super.setContentView(mView);

        if (!mFullscreen) {
            WindowManager.LayoutParams params = getWindow().getAttributes();
            params.width = AppDisplay.getDialogWidth();
            params.height = getDialogHeight();
            getWindow().setAttributes(params);
        }

        setCanceledOnTouchOutside(true);
        setOnDismissListener(new OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                mIsStatusBarShown = false;
                if (mAutoRestSystemUi
                        && SystemUiHelper.getInstance().isFullScreen()
                        && SystemUiHelper.getInstance().isAllowedHideSystemUI()) {
                    if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(mActivity);
                    else
                        SystemUiHelper.getInstance().hideStatusBar(mActivity);
                }
                if (mShowMask)
                    mShowMask = false;
                if (mDismissListener != null)
                    mDismissListener.onDismiss();
                AppDialogManager.getInstance().dismiss(UIMatchDialog.this);
            }
        });

        setOnCancelListener(new OnCancelListener() {
            @Override
            public void onCancel(DialogInterface dialog) {
                if (mOnCancelListener != null)
                    mOnCancelListener.onCancel(dialog);
            }
        });
    }

    @Override
    public void setFullScreenWithStatusBar() {
        mIsStatusBarShown = true;
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN);
        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
    }

    @Override
    public void setButton(long buttons) {
        if (buttons == DIALOG_NO_BUTTON) {
            mButtonsViewRoot.removeAllViews();
            mButtonsViewRoot.setPadding(0, 0, 0, AppDisplay.dp2px(5.0f));
        } else {
            mButtonsViewRoot.removeAllViews();
            ImageView separator = new ImageView(mContext);
            separator.setLayoutParams(new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT, 1));
            separator.setImageResource(R.color.p1);
            mButtonsViewRoot.addView(separator);

            mButtonsView = new LinearLayout(mContext);
            mButtonsView.setLayoutParams(new LinearLayout.LayoutParams(LinearLayout.LayoutParams.MATCH_PARENT, LinearLayout.LayoutParams.WRAP_CONTENT));
            mButtonsView.setOrientation(LinearLayout.HORIZONTAL);
            mButtonsView.setGravity(Gravity.CENTER_VERTICAL);
            mButtonsViewRoot.addView(mButtonsView);

            if ((buttons & MatchDialog.DIALOG_CANCEL) == MatchDialog.DIALOG_CANCEL) {
                mDlg_bt_cancel = new Button(mContext);
                addButton(mDlg_bt_cancel, mContext.getString(R.string.fx_string_cancel), MatchDialog.DIALOG_CANCEL);
            }
            if ((buttons & MatchDialog.DIALOG_SKIP) == MatchDialog.DIALOG_SKIP) {
                mDlg_bt_skip = new Button(mContext);
                addButton(mDlg_bt_skip, mContext.getString(R.string.fm_paste_skip), MatchDialog.DIALOG_SKIP);
            }
            if ((buttons & MatchDialog.DIALOG_REPLACE) == MatchDialog.DIALOG_REPLACE) {
                mDlg_bt_replace = new Button(mContext);
                addButton(mDlg_bt_replace, mContext.getString(R.string.fm_paste_replace), MatchDialog.DIALOG_REPLACE);
            }
            if ((buttons & MatchDialog.DIALOG_COPY) == MatchDialog.DIALOG_COPY) {
                mDlg_bt_copy = new Button(mContext);
                addButton(mDlg_bt_copy, mContext.getString(R.string.fx_string_copy), MatchDialog.DIALOG_COPY);
            }
            if ((buttons & MatchDialog.DIALOG_MOVE) == MatchDialog.DIALOG_MOVE) {
                mDlg_bt_move = new Button(mContext);
                addButton(mDlg_bt_move, mContext.getString(R.string.fm_move), MatchDialog.DIALOG_MOVE);
            }
            if ((buttons & MatchDialog.DIALOG_OK) == MatchDialog.DIALOG_OK) {
                mDlg_bt_ok = new Button(mContext);
                addButton(mDlg_bt_ok, mContext.getString(R.string.fx_string_ok), MatchDialog.DIALOG_OK);
            }
            if ((buttons & MatchDialog.DIALOG_OPEN_ONLY) == MatchDialog.DIALOG_OPEN_ONLY) {
                mDlg_bt_open_only = new Button(mContext);
                addButton(mDlg_bt_open_only, mContext.getString(R.string.rv_emailreview_mergedlg_openbutton), MatchDialog.DIALOG_OPEN_ONLY);
            }
            if ((buttons & MatchDialog.DIALOG_UPLOAD) == MatchDialog.DIALOG_UPLOAD) {
                mDlg_bt_upload = new Button(mContext);
                addButton(mDlg_bt_upload, mContext.getString(R.string.cloud_toolbar_more_upload), MatchDialog.DIALOG_UPLOAD);
            }

            if (mButtonsView.getChildCount() > 0) {
                mButtonsViewRoot.setPadding(0, 0, 0, 0);
                mButtonsView.getChildAt(0).setVisibility(View.GONE);
                int buttonNum = mButtonsView.getChildCount() / 2;
                if (buttonNum == 1) {
                    mButtonsView.getChildAt(buttonNum * 2 - 1).setBackgroundResource(R.drawable.dialog_button_background_selector);
                } else if (buttonNum == 2) {
                    mButtonsView.getChildAt(1).setBackgroundResource(R.drawable.dialog_left_button_background_selector);
                    mButtonsView.getChildAt(3).setBackgroundResource(R.drawable.dialog_right_button_background_selector);
                } else if (buttonNum > 2) {
                    mButtonsView.getChildAt(1).setBackgroundResource(R.drawable.dialog_left_button_background_selector);
                    mButtonsView.getChildAt(buttonNum * 2 - 1).setBackgroundResource(R.drawable.dialog_right_button_background_selector);

                    for (int i = 2; i < buttonNum; i++) {
                        mButtonsView.getChildAt(i * 2 - 1).setBackgroundResource(R.drawable.dlg_bt_bg_selector);
                    }
                }
            } else {
                mButtonsViewRoot.setPadding(0, 0, 0, AppDisplay.dp2px(5.0f));
                separator.setVisibility(View.GONE);
            }

            if (mButtonsView.getChildCount() == 6 && ((Long) mButtonsView.getChildAt(5).getTag() == MatchDialog.DIALOG_OPEN_ONLY)) {
                LinearLayout.LayoutParams cancelLayoutParams = (LinearLayout.LayoutParams) mButtonsView.getChildAt(1).getLayoutParams();
                cancelLayoutParams.width = LinearLayout.LayoutParams.WRAP_CONTENT;
                mButtonsView.getChildAt(1).setLayoutParams(cancelLayoutParams);

                LinearLayout.LayoutParams okLayoutParams = (LinearLayout.LayoutParams) mButtonsView.getChildAt(3).getLayoutParams();
                okLayoutParams.width = LinearLayout.LayoutParams.WRAP_CONTENT;
                mButtonsView.getChildAt(3).setLayoutParams(okLayoutParams);

                LinearLayout.LayoutParams openOnlyLayoutParams = (LinearLayout.LayoutParams) mButtonsView.getChildAt(5).getLayoutParams();
                openOnlyLayoutParams.width = LinearLayout.LayoutParams.WRAP_CONTENT;
                mButtonsView.getChildAt(5).setLayoutParams(openOnlyLayoutParams);
            } else {
                if (mDlg_bt_cancel != null) {
                    LinearLayout.LayoutParams cancelLayoutParams = (LinearLayout.LayoutParams) mDlg_bt_cancel.getLayoutParams();
                    cancelLayoutParams.width = 0;
                    mDlg_bt_cancel.setLayoutParams(cancelLayoutParams);
                }
                if (mDlg_bt_ok != null) {
                    LinearLayout.LayoutParams cancelLayoutParams = (LinearLayout.LayoutParams) mDlg_bt_ok.getLayoutParams();
                    cancelLayoutParams.width = 0;
                    mDlg_bt_ok.setLayoutParams(cancelLayoutParams);
                }
            }
        }
    }

//    public void setDarkModeColor() {
//        mContentViewRoot.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//        mBackTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
//        mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
//        mBackImageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
//        mTitleItem.setTextColorResource(R.color.t4);
//        mTitleBar.setBackgroundColor(mContext.getResources().getColor(R.color.b2));
//        if (mFullscreen) {
//            mTitleStyle = DLG_TITLE_STYLE_BG_DEFAULT;
//            if (mShowSeparateLine) {
//                ((TopBarImpl) mTitleBar).setShowSolidLineColor(mContext.getResources().getColor(R.color.p1));
//            } else {
//                mTitleShadowLine.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
//            }
//        } else {
//            mTitleStyle = DLG_TITLE_STYLE_BG_WHITE;
//            mTitleBar = new BaseBarImpl(mContext);
//            if (!mShowSeparateLine) {
//                mTitleShadowLine.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
//            }
//        }
//    }

    private void addButton(Button button, String title, long buttonType) {
        ImageView separator_vertical = new ImageView(mContext);
        separator_vertical.setLayoutParams(new LinearLayout.LayoutParams(1, LinearLayout.LayoutParams.MATCH_PARENT));
        separator_vertical.setImageResource(R.color.p2);
        mButtonsView.addView(separator_vertical);

        button.setLayoutParams(new LinearLayout.LayoutParams(0, mButtonHeight, 1));
        button.setBackgroundResource(R.drawable.dlg_bt_bg_selector);
        button.setGravity(Gravity.CENTER);
        button.setText(title);
        button.setAllCaps(false);
        button.setTextSize(TypedValue.COMPLEX_UNIT_PX, mContext.getResources().getDimension(R.dimen.ux_text_size_14sp));
        button.setTextColor(mContext.getResources().getColor(R.color.dlg_bt_text_selector));
        button.setTag(buttonType);
        button.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mDialogListener != null) {
                    long btType = (Long) v.getTag();
                    mDialogListener.onResult(btType);
                }
            }
        });
        mButtonsView.addView(button);
    }

    @Override
    public boolean isShowing() {
        return super.isShowing();
    }

    private void prepareShow() {
        resetForHeight();
    }

    @Override
    public void showDialog() {
        prepareShow();
        resetSystemBars();
        AppDialogManager.getInstance().showAllowManager(this, null);
    }

    @Override
    public void showDialogNoManage() {
        prepareShow();
        resetSystemBars();
    }

    @Override
    public void showDialog(boolean showMask) {
        prepareShow();
        resetSystemBars();
        if (!isShowing()) {
            AppDialogManager.getInstance().showAllowManager(this, null);
            mShowMask = showMask;
        }
    }

    @Override
    public void dismiss() {
        if (isShowing()) {
            super.dismiss();
        }
    }

    @Override
    public ViewGroup getRootView() {
        return mRootView;
    }

    @Override
    public void setTitle(String title) {
        mTitleItem.setText(title);
    }

    @Override
    public void setContentView(View view) {
        if (view != null) {
            ViewGroup parent = (ViewGroup) view.getParent();
            if (parent != null) {
                parent.removeView(view);
            }
            mContentView.removeAllViews();
            mContentView.addView(view);
        }
    }

    @Override
    public void setStyle(int style) {
        mTitleStyle = style;
        if (style == DLG_TITLE_STYLE_BG_DEFAULT) {
            setTitleDefaultStyle();
        } else if (style == DLG_TITLE_STYLE_BG_WHITE) {
            setTitleStyleWhite();
        } else {
            mTitleStyle = DLG_TITLE_STYLE_BG_DEFAULT;
            setTitleDefaultStyle();
        }
    }

    private void setTitleDefaultStyle() {
        if (mFullscreen) {
            getWindow().setBackgroundDrawableResource(R.color.b1);
            mTitleBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        } else {
//            getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
//            mRootView.setPadding(0, 0, 0, AppDisplay.getInstance(mContext).dp2px(5.0f));
//            mRootView.setBackgroundResource(R.drawable.dlg_title_bg_4circle_corner_white);
            mTitleBar.setBackgroundResource(R.drawable.dlg_title_bg_circle_corner_blue);
        }
    }

    private void setTitleStyleWhite() {
        if (mFullscreen) {
            getWindow().setBackgroundDrawableResource(R.color.b1);
            mTitleBar.setBackgroundResource(R.color.b1);
        } else {
//            getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
//            mRootView.setBackgroundResource(R.drawable.dlg_title_bg_4circle_corner_white);
//            mRootView.setPadding(0, 0, 0, AppDisplay.getInstance(mContext).dp2px(5.0f));
            mTitleBar.setBackgroundResource(R.drawable.dlg_title_bg_circle_corner_white);
        }
    }

    @Override
    public void setBackButtonVisible(int visibility) {
        if (!AppDisplay.isPad() && mFullscreen) {
            if (visibility == View.VISIBLE) {
                mTitleBar.removeItem(mTitleItem);
                mTitleBar.addView(mTitleItem, mTitlePosition);
            } else {
                mTitleBar.removeItem(mTitleItem);
                mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
            }
        }

        if (mBackStyle == TEXT_BACK) {
            mBackTextItem.getContentView().setVisibility(visibility);
            mBackImageItem.getContentView().setVisibility(View.GONE);
        } else {
            mBackTextItem.getContentView().setVisibility(View.GONE);
            mBackImageItem.getContentView().setVisibility(visibility);
        }
    }

    @Override
    public void setBackButtonTintList(ColorStateList colorStateList) {
        if (mBackStyle == IMAGE_BACK) {
            mBackImageItem.setImageTintList(colorStateList);
        } else {
            mBackTextItem.setTextColor(colorStateList);
        }
    }

    @Override
    public void setRightButtonVisible(int visibility) {
        if (mRightTextItem != null)
            mRightTextItem.getContentView().setVisibility(visibility);
    }

    @Override
    public void setRightButtonText(String text) {
        if (mRightTextItem != null)
            mRightTextItem.setText(text);
    }

    @Override
    public void setRightButtonEnable(boolean enable) {
        if (mRightTextItem != null)
            mRightTextItem.setEnable(enable);
    }

    private void resetSystemBars() {
        if (mIsStatusBarShown) return;

        if (!AppDisplay.isPad() && getWindow() != null) {
            if (SystemUiHelper.getInstance().isFullScreen()) {
                if (!SystemUiHelper.getInstance().isFullScreenMode(getWindow())) {
                    SystemUiHelper.getInstance().hideStatusBar(getWindow());
                    getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
                }
            } else {
                if (!SystemUiHelper.getInstance().isFullScreenMode(mActivity)) {
                    SystemUiHelper.getInstance().showStatusBar(getWindow());
                    if (SystemUiHelper.getInstance().isFullScreenMode(getWindow()))
                        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
                }
            }
        }
    }

    @Override
    public void setBackButtonStyle(@BackButtonStyle int style) {
        mBackStyle = style;
        if (style == TEXT_BACK) {
            mTitleBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
            mBackTextItem.getContentView().setVisibility(View.VISIBLE);
            mBackImageItem.getContentView().setVisibility(View.GONE);
        } else {
            mTitleBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));
            mBackTextItem.getContentView().setVisibility(View.GONE);
            mBackImageItem.getContentView().setVisibility(View.VISIBLE);
        }
    }

    @Override
    public void setBackButtonText(CharSequence text) {
        if (mBackTextItem != null)
            mBackTextItem.setText(text);
    }

    @Override
    public void setButtonEnable(boolean enable, long buttons) {
        if ((buttons & MatchDialog.DIALOG_CANCEL) == MatchDialog.DIALOG_CANCEL) {
            setEnable(mDlg_bt_cancel, enable);
        }
        if ((buttons & MatchDialog.DIALOG_SKIP) == MatchDialog.DIALOG_SKIP) {
            setEnable(mDlg_bt_skip, enable);
        }
        if ((buttons & MatchDialog.DIALOG_REPLACE) == MatchDialog.DIALOG_REPLACE) {
            setEnable(mDlg_bt_replace, enable);
        }
        if ((buttons & MatchDialog.DIALOG_COPY) == MatchDialog.DIALOG_COPY) {
            setEnable(mDlg_bt_copy, enable);
        }
        if ((buttons & MatchDialog.DIALOG_MOVE) == MatchDialog.DIALOG_MOVE) {
            setEnable(mDlg_bt_move, enable);
        }
        if ((buttons & MatchDialog.DIALOG_OK) == MatchDialog.DIALOG_OK) {
            setEnable(mDlg_bt_ok, enable);
        }
        if ((buttons & MatchDialog.DIALOG_OPEN_ONLY) == MatchDialog.DIALOG_OPEN_ONLY) {
            setEnable(mDlg_bt_open_only, enable);
        }
        if ((buttons & MatchDialog.DIALOG_UPLOAD) == MatchDialog.DIALOG_UPLOAD) {
            setEnable(mDlg_bt_upload, enable);
        }
    }

    private void setEnable(Button button, boolean enable) {
        if (button != null) {
            if (enable) {
                button.setTextColor(mContext.getResources().getColor(R.color.dlg_bt_text_selector));
            } else {
                button.setTextColor(mContext.getResources().getColor(R.color.ux_bg_color_dialog_button_disabled));
            }
            button.setEnabled(enable);
        }
    }

    @Override
    public void setTitleShadowVisible(boolean visible) {
        if (visible) {
            mTitleShadowLine.setVisibility(View.VISIBLE);
        } else {
            mTitleShadowLine.setVisibility(View.GONE);
        }
    }

    @Override
    public void setTitleTextSize(int unit, float size) {
        mTitleItem.setTextSize(unit, size);
    }

    /* width: WindowManager.LayoutParams.WRAP_CONTENT, WindowManager.LayoutParams.MATCH_PARENT
     */
    @Override
    public void setWidth(int width) {
        if (!mFullscreen) {
            WindowManager.LayoutParams params = getWindow().getAttributes();
            params.width = width;
            getWindow().setAttributes(params);
        }
    }

    /* height: WindowManager.LayoutParams.WRAP_CONTENT, WindowManager.LayoutParams.MATCH_PARENT, int value
     */
    @Override
    public void setHeight(int height) {
        if (!mFullscreen) {
            mHeight = height;

            WindowManager.LayoutParams params = getWindow().getAttributes();
            if (mHeight >= getDialogHeight() || (mHeight <= 0 && mHeight != WindowManager.LayoutParams.WRAP_CONTENT && mHeight != WindowManager.LayoutParams.MATCH_PARENT)
            ) {
                params.height = getDialogHeight();
            } else {
                params.height = height;
            }
            getWindow().setAttributes(params);
        }
    }

    private void resetForHeight() {
        if (!mFullscreen && mHeight != -100) {
            mRootView.measure(0, 0);
            if (mHeight == WindowManager.LayoutParams.WRAP_CONTENT) {
                int maxTempHeight = getDialogHeight() - mTitleView.getMeasuredHeight();
                if (mButtonsView != null) {
                    if (mButtonsView.getChildCount() > 0) {
                        maxTempHeight = maxTempHeight - mButtonHeight - 1;
                    } else {
                        maxTempHeight = maxTempHeight - AppDisplay.dp2px(5.0f);
                    }
                }
                if (mTitleShadowLine.getVisibility() != View.GONE) {
                    maxTempHeight = maxTempHeight - AppDisplay.dp2px(1.5f);
                }

                if (mContentView.getMeasuredHeight() > maxTempHeight) {
                    WindowManager.LayoutParams params = getWindow().getAttributes();
                    params.height = getDialogHeight();
                    getWindow().setAttributes(params);
                } else {
                    RelativeLayout.LayoutParams contentViewRootParams = (RelativeLayout.LayoutParams) mContentViewRoot.getLayoutParams();
                    contentViewRootParams.addRule(RelativeLayout.ABOVE, 0);// remove ABOVE
                    mContentViewRoot.setLayoutParams(contentViewRootParams);

                    RelativeLayout.LayoutParams buttonViewParams = (RelativeLayout.LayoutParams) mButtonsViewRoot.getLayoutParams();
                    buttonViewParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM, 0);// remove ALIGN_PARENT_BOTTOM
                    buttonViewParams.addRule(RelativeLayout.BELOW, R.id.dlg_contentview_root);// add BELOW
                    mButtonsViewRoot.setLayoutParams(buttonViewParams);
                }
            }

        }
    }

    private BaseBar.TB_Position mTitlePosition = BaseBar.TB_Position.Position_LT;

    @Override
    public void setTitlePosition(BaseBar.TB_Position position) {
        if (position == BaseBar.TB_Position.Position_LT) {
            mTitlePosition = BaseBar.TB_Position.Position_LT;
            mTitleBar.removeItem(mTitleItem);
            mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_LT);
        } else if (position == BaseBar.TB_Position.Position_CENTER) {
            mTitlePosition = BaseBar.TB_Position.Position_CENTER;
            mTitleBar.removeItem(mTitleItem);
            mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
        } else if (position == BaseBar.TB_Position.Position_RB) {
            mTitlePosition = BaseBar.TB_Position.Position_RB;
            mTitleBar.removeItem(mTitleItem);
            mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_RB);
        } else {
            mTitlePosition = BaseBar.TB_Position.Position_CENTER;
            mTitleBar.removeItem(mTitleItem);
            mTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
        }
    }

    @Override
    public void setTitleBarVisiable(boolean visible) {
        mTitleBar.getContentView().setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public void setListener(DialogListener dialogListener) {
        this.mDialogListener = dialogListener;
    }

    @Override
    public void setOnDLDismissListener(DismissListener dismissListener) {
        mDismissListener = dismissListener;
    }

    @Override
    public void setOnDLCancelListener(OnCancelListener listener) {
        mOnCancelListener = listener;
    }

    @Override
    public void setBackgroundColor(int color) {
        mContentViewRoot.setBackgroundColor(color);
    }

    @Override
    public BaseBar getTitleBar() {
        return mTitleBar;
    }

    @Override
    public DialogListener getDialogListerner() {
        return mDialogListener;
    }

    @Override
    public void themeColorChanged() {
        if (mBackTextItem != null)
            mBackTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (mRightTextItem != null)
            mRightTextItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (mTitleItem != null)
            mTitleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        if (mTitleBar != null) {
            mTitleBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            if (mFullscreen) {
                if (mShowSeparateLine) {
                    ((TopBarImpl) mTitleBar).setShowSolidLineColor(mContext.getResources().getColor(R.color.p1));
                } else {
                    mTitleShadowLine.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
                }
            } else {
                if (!mShowSeparateLine) {
                    mTitleShadowLine.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
                }
            }
        }
        if (mTitleShadowLine != null)
            mTitleShadowLine.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
    }

    @Override
    public void resetWH() {
        if (!mFullscreen) {
            if (AppDevice.isChromeOs(mActivity))
                setWidth(AppDisplay.getDialogWidth());
            setHeight(getDialogHeight());
        }
    }

    public void titleTextAdaptive(){
        if(!AppDisplay.isPad()) {
            int leftWidth = Math.max(mBackTextItem.getContentView().getWidth(), mBackImageItem.getContentView().getWidth());
            int rightWidth = mRightTextItem.getContentView().getWidth();
            mTitleItem.setMaxWidth(AppDisplay.getScreenWidth() - Math.max(leftWidth, rightWidth) * 2 - AppDisplay.dp2px(32));
            mTitleItem.setEllipsize(TextUtils.TruncateAt.MIDDLE);
        }
    }

    public int getDialogHeight() {
        return AppDisplay.getDialogHeight();
    }

    public void setAutoResetSystemUiOnDismiss(boolean auto) {
        this.mAutoRestSystemUi = auto;
    }

    public void release(){
        mActivity = null;
    }
}
