/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.os.Build;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.SystemUiHelper;

import androidx.appcompat.widget.AppCompatImageView;
import androidx.fragment.app.FragmentActivity;

public class UIPopoverFrag extends UIPopupFragment {
    public static final int mBlackPopoverHeightDp = 36;


    public static UIPopoverFrag create(FragmentActivity act, View contentView, boolean blackStyle, boolean showDivider) {
        UIPopoverFrag popupDlg = new UIPopoverFrag();
        popupDlg.setArrowStyle(act.getApplicationContext(), blackStyle);
        popupDlg.setShowDivider(showDivider);
        if (blackStyle)
            popupDlg.init(act, contentView, "UI_BLACK_POPOVER_FRAGMENT", false, false);
        else
            popupDlg.init(act, contentView, "UI_WHITE_POPOVER_FRAGMENT", false, true);
        return popupDlg;
    }

    public static UIPopoverFrag create(FragmentActivity act, View contentView, String fragName, boolean blackStyle, boolean showDivider) {
        UIPopoverFrag popupDlg = new UIPopoverFrag();
        popupDlg.setArrowStyle(act.getApplicationContext(), blackStyle);
        popupDlg.setShowDivider(showDivider);
        if (blackStyle)
            popupDlg.init(act, contentView, fragName, false, !blackStyle);
        else
            popupDlg.init(act, contentView, fragName, false, !blackStyle);
        return popupDlg;
    }

    public static final int ARROW_AUTO = 0;
    public static final int ARROW_LEFT = 1;
    public static final int ARROW_TOP = 2;
    public static final int ARROW_RIGHT = 3;
    public static final int ARROW_BOTTOM = 4;
    public static final int ARROW_CENTER = 5;

    boolean mIsBlackStyle;
    boolean mIsShowDivider;
    int mArrowDistance = 12;
    int mArrowWidthDp = 8;
    int mArrowHeightDp = 20;
    int mArrowColor = Color.WHITE;
    int mArrowPosition = ARROW_LEFT;
    boolean mShowArrow = true;
    ArrowImageView mArrowView;
    int mCornerRadius;
    private Context mContext;

    void setArrowStyle(Context context, boolean blackStyle) {
        mIsBlackStyle = blackStyle;
        if (blackStyle) {
            mArrowDistance = 12;
            mArrowWidthDp = 8;
            mArrowHeightDp = 20;
            mArrowColor = AppResource.getColor(context, R.color.ux_color_black_popover_bg);
            mCornerRadius = AppDisplay.dp2px(8);
        } else {
            mArrowDistance = 12;
            mArrowWidthDp = 16;
            mArrowHeightDp = 30;
            mArrowColor = ThemeConfig.getInstance(mContext).getB1();
            mCornerRadius = AppDisplay.dp2px(3);
        }
    }

    void setShowDivider(boolean showDivider) {
        mIsShowDivider = showDivider;
    }

    @Override
    void init(FragmentActivity act, View contentView, String fragName, boolean withMask, boolean withShadow) {
        super.init(act, contentView, fragName, withMask, withShadow);

        mArrowView = new ArrowImageView(act.getApplicationContext());
        mContext = act.getApplicationContext();
        mArrowView.setMinimumWidth(AppDisplay.dp2px( mArrowHeightDp));
        mArrowView.setMaxWidth(AppDisplay.dp2px( mArrowHeightDp));
        mArrowView.setMinimumHeight(AppDisplay.dp2px( mArrowHeightDp));
        mArrowView.setMaxHeight(AppDisplay.dp2px( mArrowHeightDp));

        RelativeLayout.LayoutParams angleLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        angleLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        angleLp.addRule(RelativeLayout.CENTER_VERTICAL);
        mArrowView.setLayoutParams(angleLp);

        mRootView0.addView(mArrowView, 1);

        if (mIsBlackStyle) {
            mRootViewWithShadow.setBackgroundResource(R.drawable.black_popover_bg);
            ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                    AppDisplay.dp2px( mArrowWidthDp), 0, 0, 0);

            if (mIsShowDivider) {
                RelativeLayout dividerView = new RelativeLayout(act.getApplicationContext());
                RelativeLayout.LayoutParams dividerLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                dividerLp.setMargins(AppDisplay.dp2px( 8), 0, AppDisplay.dp2px( 8), 0);
                dividerView.setLayoutParams(dividerLp);
                dividerView.setBackgroundResource(R.color.b1);
                mRootViewWithShadow.addView(dividerView, 0);
            }
        } else {
            angleLp.setMargins(mShadowWidth - AppDisplay.dp2px( mArrowWidthDp), 0, 0, 0);
        }
    }

    public void setBackgroundResource(int resId) {
        mRootViewWithShadow.setBackgroundResource(resId);
    }

    public void setArrowColor(int color) {
        mArrowColor = color;
    }

    public void setShowArrow(boolean isShow) {
        mShowArrow = isShow;
    }

    public void setArrowForceDarkAllowed(boolean allowed) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            if (mArrowView != null) {
                mArrowView.setForceDarkAllowed(allowed);
            }
        }
    }

    public void showAtLocation(ViewGroup parent, Rect rect, int contentWidth, int contentHeight, int arrowPosition, int arrowDist) {
        int rootWidth = parent.getWidth();
        int rootHeight = parent.getHeight();
        int[] location = new int[2];
//        parent.getLocationOnScreen(location); // shouldn't consult the location on chromebook
//        parent.getLocationInWindow(location);
        int x = location[0];
        int y = location[1];
        if (SystemUiHelper.getInstance().isStatusBarShown(mAct)) {
            rect.offset(0, -SystemUiHelper.getInstance().getStatusBarHeight(mAct));
        }

        int width = contentWidth + getShadowLength() * 2;
        int height = contentHeight + getShadowLength() * 2;

        mArrowPosition = arrowPosition;
        mArrowDistance = arrowDist;

        if (mArrowPosition == ARROW_AUTO) {
            if (rect.top - height > 0) {
                mArrowPosition = ARROW_BOTTOM;
            } else if (rect.right + width < rootWidth + x) {
                mArrowPosition = ARROW_LEFT;
            } else if (rect.left - width > 0) {
                mArrowPosition = ARROW_RIGHT;
            } else if (rect.bottom + height < rootHeight + y) {
                mArrowPosition = ARROW_TOP;
            } else {
                mArrowPosition = ARROW_CENTER;
            }
        }

        int leftMargin = 0;
        int rightMargin = 0;
        int topMargin = 0;
        int bottomMargin = 0;

        if (mShowArrow) {
            RelativeLayout.LayoutParams angleLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            mArrowView.setLayoutParams(angleLp);
            mArrowView.setVisibility(View.VISIBLE);

            int screenInterval = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_screen_margin_text);

            if (mArrowPosition == ARROW_LEFT) {
                angleLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
                angleLp.addRule(RelativeLayout.CENTER_VERTICAL);

                if (rect.centerY() - height / 2 - y < screenInterval) {
                    topMargin = height / 2 + y - rect.centerY() + screenInterval;
                    angleLp.topMargin = height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - topMargin;
                    if (angleLp.topMargin < mCornerRadius + mShadowWidth) {
                        angleLp.topMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                } else if (rect.centerY() + height / 2 - y > rootHeight - screenInterval) {
                    bottomMargin = rect.centerY() + height / 2 - y - rootHeight + screenInterval;
                    angleLp.bottomMargin = height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - bottomMargin;
                    if (angleLp.bottomMargin < mCornerRadius + mShadowWidth) {
                        angleLp.bottomMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                } else {
                    angleLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                }

                if (mIsBlackStyle) {
                    width += AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                            AppDisplay.dp2px( mArrowWidthDp), 0, 0, 0);
                } else {
                    angleLp.leftMargin = mShadowWidth - AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                            0, 0, 0, 0);
                }
            } else if (mArrowPosition == ARROW_RIGHT) {
                angleLp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
                angleLp.addRule(RelativeLayout.CENTER_VERTICAL);

                if (rect.centerY() - height / 2 - y < screenInterval) {
                    topMargin = height / 2 + y - rect.centerY() + screenInterval;
                    angleLp.topMargin = height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - topMargin;
                    if (angleLp.topMargin < mCornerRadius + mShadowWidth) {
                        angleLp.topMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                } else if (rect.centerY() + height / 2 - y > rootHeight - screenInterval) {
                    bottomMargin = rect.centerY() + height / 2 - y - rootHeight + screenInterval;
                    angleLp.bottomMargin = height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - bottomMargin;
                    if (angleLp.bottomMargin < mCornerRadius + mShadowWidth) {
                        angleLp.bottomMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                } else {
                    angleLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                }

                if (mIsBlackStyle) {
                    width += AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                            0, 0, AppDisplay.dp2px( mArrowWidthDp), 0);
                } else {
                    angleLp.rightMargin = mShadowWidth - AppDisplay.dp2px(mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(0, 0, 0, 0); }
            } else if (mArrowPosition == ARROW_TOP) {
                angleLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                int arrowWidth = AppDisplay.dp2px( mArrowWidthDp);
                if (rect.centerX() - width / 2 - x < screenInterval) {
                    angleLp.leftMargin = rect.centerX() - AppDisplay.dp2px( mArrowHeightDp) / 2;
                    if (angleLp.leftMargin < mCornerRadius + mShadowWidth) {
                        angleLp.leftMargin = mIsBlackStyle || !mShowArrow ? (mCornerRadius + mShadowWidth) : (rect.left + mShadowWidth + arrowWidth / 2) - rect.centerX();
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
                } else if (rect.centerX() + width / 2 - x > rootWidth - screenInterval) {
                    int offset = rect.centerX() + width / 2 - x - rootWidth;
                    if (offset < 0) offset = 0;
                    angleLp.rightMargin = width / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - offset;
                    if (angleLp.rightMargin < mCornerRadius + mShadowWidth) {
                        int margin = rect.centerX() - (rect.right - mShadowWidth - arrowWidth / 2);
                        angleLp.rightMargin = mIsBlackStyle || !mShowArrow ? (mCornerRadius + mShadowWidth) : (margin > mShadowWidth ? margin : angleLp.rightMargin);
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
                } else {
                    angleLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                }

                if (mIsBlackStyle) {
                    height += AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                            0, AppDisplay.dp2px( mArrowWidthDp), 0, 0);
                } else {
                    angleLp.topMargin = mShadowWidth - AppDisplay.dp2px(mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(0, 0, 0, 0);
                }
            } else if (mArrowPosition == ARROW_BOTTOM) {
                angleLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);

                if (rect.centerX() - width / 2 - x < screenInterval) {
                    leftMargin = width / 2 + x + screenInterval - rect.centerX();
                    angleLp.leftMargin = width / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - leftMargin;
                    if (angleLp.leftMargin < mCornerRadius + mShadowWidth) {
                        angleLp.leftMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
                } else if (rect.centerX() + width / 2 - x > rootWidth - screenInterval) {
                    rightMargin = rect.centerX() + width / 2 - x - rootWidth + screenInterval;
                    angleLp.rightMargin = width / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2 - rightMargin;
                    if (angleLp.rightMargin < mCornerRadius + mShadowWidth) {
                        angleLp.rightMargin = mCornerRadius + mShadowWidth;
                    }
                    angleLp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
                } else {
                    angleLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                }

                if (mIsBlackStyle) {
                    height += AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(
                            0, 0, 0, AppDisplay.dp2px( mArrowWidthDp));
                } else {
                    angleLp.bottomMargin = mShadowWidth - AppDisplay.dp2px( mArrowWidthDp);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(0, 0, 0, 0);
                }
            } else if (mArrowPosition == ARROW_CENTER) {
                mArrowView.setVisibility(View.INVISIBLE);
            }
        } else {
            mArrowView.setVisibility(View.INVISIBLE);
        }

        setWidth(width);
        setHeight((int) Math.min(height, rootHeight * 0.85));

        if (!mShowArrow) {
            mArrowPosition = ARROW_AUTO;
        }

        if (mArrowPosition == ARROW_LEFT) {
            showAtLocation(null, Gravity.START | Gravity.TOP,
                    rect.right + mArrowDistance,
                    rect.centerY() - mHeight / 2 + topMargin - bottomMargin);
        } else if (mArrowPosition == ARROW_RIGHT) {
            showAtLocation(null, Gravity.START | Gravity.TOP,
                    rect.left - mArrowDistance - width,
                    rect.centerY() - mHeight / 2 + topMargin - bottomMargin);
        } else if (mArrowPosition == ARROW_TOP) {
            int screenInterval = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_screen_margin_text);
            int arrowWidth = AppDisplay.dp2px( mArrowHeightDp);
            int positionX;
            if (!mIsBlackStyle && mShowArrow) {
                if (rect.centerX() - width / 2 - x < screenInterval) {
                    positionX = Math.max(0, rect.centerX() - mWidth / 2);
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(rect.centerX() - (rect.left + mShadowWidth + arrowWidth / 2), 0, 0, 0);
                } else if (rect.centerX() + width / 2 - x > rootWidth) {
                    positionX = rect.centerX() - mWidth / 2;
                    ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams()).setMargins(0, 0, (rect.right - mShadowWidth - arrowWidth / 2) - rect.centerX(), 0);
                } else {
                    positionX = rect.centerX() - mWidth / 2;
                }
            } else {
                positionX = Math.max(0, rect.centerX() - mWidth / 2);
            }
            showAtLocation(null, Gravity.START | Gravity.TOP,
                    positionX,
                    rect.bottom + mArrowDistance);
        } else if (mArrowPosition == ARROW_BOTTOM) {
            showAtLocation(null, Gravity.START | Gravity.TOP,
                    rect.centerX() - mWidth / 2 + leftMargin - rightMargin,
                    rect.top - mArrowDistance - height);
        } else {
            x = Math.max(rect.centerX() - mWidth / 2, 0);
            y = rect.centerY() - mHeight / 2;
            RelativeLayout.LayoutParams params = ((RelativeLayout.LayoutParams) mRootViewWithShadow.getLayoutParams());
            int offset = mShadowWidth;
            params.leftMargin = 0;
            params.rightMargin = 0;
            if (x == 0) {
                params.leftMargin = -offset;
            } else if (x > rootWidth - mWidth) {
                params.rightMargin = -offset;
            }
            if (mAnchorMinPositionY != -1 && mAnchorMaxPositionY != -1) {
                if (y < mAnchorMinPositionY) {
                    y = mAnchorMinPositionY - mShadowWidth;
                } else if (y > mAnchorMaxPositionY - mHeight) {
                    y = mAnchorMaxPositionY - mHeight + mShadowWidth;
                }
            }
            params.setMargins(params.leftMargin, params.topMargin, params.rightMargin, params.bottomMargin);
            showAtLocation(null, Gravity.START | Gravity.TOP, x, y);
        }
    }

    public void update(ViewGroup parent, Rect rect, int contentWidth, int contentHeight) {
        showAtLocation(parent, rect, contentWidth, contentHeight, mArrowPosition, mArrowDistance);
    }

    public void update(ViewGroup parent, Rect rect, int contentWidth, int contentHeight, int arrowPosition) {
        showAtLocation(parent, rect, contentWidth, contentHeight, arrowPosition, mArrowDistance);
    }

    private int mAnchorMinPositionY = -1;
    private int mAnchorMaxPositionY = -1;

    public void setAnchorPositionYBoundary(int min, int max) {
        mAnchorMinPositionY = min;
        mAnchorMaxPositionY = max;
    }

    class ArrowImageView extends AppCompatImageView {
        Paint mPaint = new Paint();
        Path mPath = new Path();

        public ArrowImageView(Context context) {
            super(context);
        }

        public ArrowImageView(Context context, AttributeSet attrs) {
            super(context, attrs);
        }

        public ArrowImageView(Context context, AttributeSet attrs, int defStyleAttr) {
            super(context, attrs, defStyleAttr);
        }

        @Override
        public void onDraw(Canvas canvas) {
            super.onDraw(canvas);

            mPaint.setDither(true);
            mPaint.setAntiAlias(true);
            mPaint.setColor(mArrowColor);

            int width = getWidth();
            int height = getHeight();

            mPath.reset();
            switch (mArrowPosition) {
                case ARROW_LEFT:
                    mPath.moveTo(0, height / 2);
                    mPath.lineTo(AppDisplay.dp2px( mArrowWidthDp), height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2);
                    mPath.lineTo(AppDisplay.dp2px( mArrowWidthDp), height / 2 + AppDisplay.dp2px( mArrowHeightDp) / 2);
                    mPath.close();
                    break;
                case ARROW_RIGHT:
                    mPath.moveTo(width, height / 2);
                    mPath.lineTo(width - AppDisplay.dp2px( mArrowWidthDp), height / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2);
                    mPath.lineTo(width - AppDisplay.dp2px( mArrowWidthDp), height / 2 + AppDisplay.dp2px( mArrowHeightDp) / 2);
                    mPath.close();
                    break;
                case ARROW_TOP:
                    mPath.moveTo(width / 2, 0);
                    mPath.lineTo(width / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2, AppDisplay.dp2px( mArrowWidthDp));
                    mPath.lineTo(width / 2 + AppDisplay.dp2px( mArrowHeightDp) / 2, AppDisplay.dp2px( mArrowWidthDp));
                    mPath.close();
                    break;
                case ARROW_BOTTOM:
                    mPath.moveTo(width / 2, height);
                    mPath.lineTo(width / 2 - AppDisplay.dp2px( mArrowHeightDp) / 2, height - AppDisplay.dp2px( mArrowWidthDp));
                    mPath.lineTo(width / 2 + AppDisplay.dp2px( mArrowHeightDp) / 2, height - AppDisplay.dp2px( mArrowWidthDp));
                    mPath.close();
                    break;
                default:
            }

            canvas.drawPath(mPath, mPaint);
        }
    }
}
